/* -*- C++ -*-
 *
 * ---------------------------------------------------------------------
 * $Id: expr_iter.h,v 1.1.1.1.4.2 2004/05/05 18:01:40 drory Exp $
 * ---------------------------------------------------------------------
 *
 * Copyright (C) 2000-2002 Niv Drory <drory@usm.uni-muenchen.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA 
 *
 * ---------------------------------------------------------------------
 *
 */

/*! \class ltl::ExprIter
  This is a special version of an iterator useful for iterating over
  a ltl::TExpr<A,N> object, which is an iterator by itself, but which would
  be very inefficient to use directly, since it delegates ALL calls to
  ALL subnodes in an expression parse-tree. Here we define an ExprIter
  object which uses the 'remote-control' mechanism of the expr to iterate
  efficiently. See also comments in marray_iter.h and eval.h.
*/

#ifndef __LTL_MARRAY__
#error "<ltl/marray/expr_iter.h> must be included via <ltl/marray.h>, never alone!"
#endif



#ifndef __LTL_EXPR_ITER__
#define __LTL_EXPR_ITER__

#include <ltl/config.h>

LTL_BEGIN_NAMESPACE

template<class Expr, int N>
class TExpr;


template<class A, int N>
class ExprIter
{
   public:
      typedef std::forward_iterator_tag       iterator_category;
      typedef typename TExpr<A,N>::value_type value_type;
      typedef          int                    difference_type;
      typedef const value_type&               const_reference;
      typedef const value_type*               const_pointer;
      typedef       value_type&               reference;
      typedef       value_type*               pointer;

      enum { isVectorizable = TExpr<A,N>::isVectorizable };


      ExprIter( TExpr<A,N>& e )
            : e_(e), i_(*e.shape())
      { }

      ExprIter( TExpr<A,N>& e, bool end )
            : e_(e), i_()
      { }

      ExprIter( const ExprIter<A,N>& other )
            : e_(other.e_), i_(other.i_)
      { }

      inline ExprIter<A,N>& operator++()
      {
         i_.advance();
         e_.advance();
         if( i_.needAdvanceDim() )
         {
            i_.advanceDim();
            e_.advanceDim();
         }
         return *this;
      }

      void operator++( int )
      {
         ++(*this);
      }

      inline bool done() const
      {
         return i_.done();
      }

      inline value_type operator*() const
      {
         return *e_;
      }

#ifdef LTL_USE_SIMD
      inline typename VEC_TYPE(value_type) readVec( const int i ) const
      {
         return e_.readVec(i);
      }

      inline bool sameAlignmentAs( void* p ) const
      {
         return e_.sameAlignmentAs(p);
      }
#endif

      // FUDGE! but works for comparison with end iterator in STL algorithms
      inline bool operator==( const ExprIter<A,N>& other ) const
      {
         return done();
      }

   protected:
      TExpr<A,N>& e_;
      ShapeIter<N> i_;
};

LTL_END_NAMESPACE

#endif

