/* -*- C++ -*-
 *
 * ---------------------------------------------------------------------
 * $Id: marray_iter.h,v 1.2.2.6 2004/08/10 12:38:58 cag Exp $
 * ---------------------------------------------------------------------
 *
 * Copyright (C) 2000-2002 Niv Drory <drory@usm.uni-muenchen.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA 
 *
 * ---------------------------------------------------------------------
 *
 */

#ifndef __LTL_IN_FILE_MARRAY__
#error "<ltl/marray/marray_iter.h> must be included via <ltl/marray.h>, never alone!"
#endif


#ifndef __LTL_ITER__
#define __LTL_ITER__

#include <ltl/config.h>

LTL_BEGIN_NAMESPACE

template<class T, int N>
class MArray;

template<int N>
class Shape;

/*! \file marray_iter.h
  General iterators: a const and a non-const version.
  These iterators can deal with any array geometry/topology.
*/


struct _iter_end_tag
{ };


// -------------------------------------------------------------------
//! Const version.
// -------------------------------------------------------------------
template<class T, int N>
class MArrayIterConst
{

   public:
      typedef std::forward_iterator_tag             iterator_category;

      typedef int                                   difference_type;
      typedef typename MArray<T,N>::value_type      value_type;
      typedef typename MArray<T,N>::const_reference const_reference;
      typedef typename MArray<T,N>::const_pointer   const_pointer;
      typedef typename MArray<T,N>::reference       reference;
      typedef typename MArray<T,N>::pointer         pointer;

      enum { dims=N };
      enum { numIndexIter = 0 };
      enum { isVectorizable = 1 };

      MArrayIterConst( const MArray<T,N>& array );

      MArrayIterConst( const MArray<T,N>& array,
                       const _iter_end_tag& E )
	: stride_(array.stride(1)), shape_(array.shape())
      { // end iterator
         if( N==1 )
            data_ = last_[0] =
	      array.data() + array.nelements()*stride_;
         else
            data_ = 0;
      }

      MArrayIterConst( const MArrayIterConst<T,N>& other )
            : data_(other.data_), first_(other.first_), stride_(other.stride_),
            shape_(other.shape_)
      {
         //cout << "copy() \n";
         for( int i=0; i<N; ++i )
         {
            length_[i]  = other.length_[i];
            strides_[i] = other.strides_[i];
            stack_[i]   = other.stack_[i];
            last_[i]    = other.last_[i];
         }
      }

      // reset to first element
      void reset();

      // dereference with operator *
      inline value_type  operator*() const
      {
         LTL_ASSERT( !done(), "Dereferencing end iterator" );
         return *data_;
      }

      // increment the iterator
      // this method is almost never used, see comment below...
      inline MArrayIterConst<T,N>& operator++()
      {
         LTL_ASSERT( !done(), "Incrementing past end iterator" );
         advance(); // increment innermost loop

         if( N == 1 )  // will be const-folded away, dont worry ...
            return *this;

         if( data_ != last_[0] )
         {
            // We hit this case almost all the time.
            return *this;
         }

         advanceDim(); // hit end of row/columns/whatever, need to reset loops
         return *this;
      }

      void operator++( int )
      {
         ++(*this);
      } // breaks STL, but fast.
      // Otherwise one must copy iterator to save state ...

      // provide separate versions of advance() for incrementing the innermost
      // loop and needAdvanceDim()/advanceDim() for incrementing an outer loop
      // since when evaluating an expression involving more than one iterator
      // all terms MUST have the same geometry, so the loop structure is
      // identical and it's sufficent to check the end of loop condition on
      // one of the iterators ... all others are then 'remote controlled'
      // via the following methods.
      inline void advance()
      {
         data_ += stride_;
      }

      inline void advanceN( int n )
      {
         data_ += n * stride_;
      }

      inline void advanceWithStride1()
      {
         ++data_;
      }

      inline bool needAdvanceDim() const
      {
         if( N == 1 )
            return false;
         else
            return data_ == last_[0];
      }

      void advanceDim();

      // these are for 'loop unrolling' within evaluation of expression
      // templates.
      inline value_type readWithoutStride( const int i ) const
      {
         // we unroll the loop 4 times in eval.h, so we prefetch 4 elements
         // ahead ...
         LTL_PREFETCH_R(data_+4);
         return data_[i];
      }

      inline value_type readWithStride( const int i ) const
      {
         return data_[i*stride_];
      }

#ifdef LTL_USE_SIMD
      inline typename VEC_TYPE(value_type) readVec( const int i ) const
      {
         // we unroll the loop 4 times in eval.h, so we prefetch 4 vectors
         // ahead ...
         LTL_PREFETCH_R( & (((typename VEC_TYPE(value_type) *) data_)[i+4]) );
         return ((typename VEC_TYPE(value_type) *) data_)[i];
      }

      inline bool sameAlignmentAs( void* p ) const
      {
         return ( ((long)p & 0x0FL) == ((long)data_ & 0x0FL) );  
      }
#endif

      bool operator==( const MArrayIterConst<T,N>& other ) const
      {
         return data_ == other.data_;
      }

      bool operator!=( const MArrayIterConst<T,N>& other ) const
      {
         return data_ != other.data_;
      }

      // for 1-d arrays, we can avoid needAdvanceDim() since it's
      // equivalent to the done condition
      inline bool done() const
      {
         if( N == 1 )  // if() will be const-folded away ...
            return data_ == last_[0];
         else
            return !data_;
      }

      bool isConformable( const Shape<N>& other ) const
      {
         return shape_->isConformable( other );
      }

      void printRanges() const;

      const Shape<N>* shape() const
      {
         return shape_;
      }

      bool isStorageContiguous() const
      {
         return shape_->isStorageContiguous();
      }

      bool isStride1() const
      {
         return stride_==1;
      }

      value_type* restrict_ data() const
      {
         return data_;
      }

   protected:
#ifndef __xlC__
      value_type* restrict_ data_;
      value_type* restrict_ first_;
#else
      value_type* data_;
      value_type* first_;
#endif
      value_type* stack_[N], *last_[N];
      int strides_[N], length_[N];

      const int stride_;
      const Shape<N> *shape_;
};


//! The constructor implementation.
//
template<class T, int N>
MArrayIterConst<T,N>::MArrayIterConst( const MArray<T,N>& array )
      : data_(array.data()), first_(array.data()),
      stride_(array.stride(1)), shape_(array.shape())
{
   // copy information to avoid dereferencing array all the time
   for( int i=0; i<N; i++ )
   {
      strides_[i] = array.stride(i+1);
      length_[i] = array.length(i+1);
      stack_[i] = data_;
      last_[i] = data_ + length_[i] * strides_[i];
   }
}


//! Reset the iterator (back to first element).
//
template<class T, int N>
void MArrayIterConst<T,N>::reset()
{
   // reset data
   data_ = first_;

   // init stack and upper limits
   for( int i=0; i<N; i++ )
   {
      stack_[i] = data_;
      last_[i] = data_ + length_[i] * strides_[i];
   }
}


//! Reset stacks after we've hit the end of a dimension.
//
template<class T, int N>
void MArrayIterConst<T,N>::advanceDim()
{
   // We've hit the end of a row/column/whatever.  Need to
   // increment one of the loops over another dimension.
   int j=1;
   for( ; j<N; ++j )
   {
      data_ = stack_[j];
      data_ += strides_[j];
      if( data_ != last_[j] )
         break;
   }

   // are we finished?
   if ( j == N )
   {
      // Setting data_ to 0 indicates the end of the array has
      // been reached, and will match the end iterator.
      data_ = 0;
      return;
   }

   stack_[j] = data_;

   // Now reset all the last pointers
   for (--j; j >= 0; --j)
   {
      stack_[j] = data_;
      last_[j] = data_ + length_[j] * strides_[j];
   }
}

template<class T, int N>
void MArrayIterConst<T,N>::printRanges() const
{
   cerr << "Ranges: ";
   for(int i=0; i<N; i++)
      cerr << "(" << length_[i] << ")  ";
}



//! Non-const version simply by inheritance.
//
template<class T, int N>
class MArrayIter : public MArrayIterConst<T,N>
{
   public:

      MArrayIter( MArray<T,N>& array )
            : MArrayIterConst<T,N>( array )
      { }

      MArrayIter( MArray<T,N>& array, const _iter_end_tag& E )
            : MArrayIterConst<T,N>( array, E )
      { }

      MArrayIter( const MArrayIter<T,N>& other )
            : MArrayIterConst<T,N>( other )
      { }

      MArrayIter<T,N>& operator++()
      {
         MArrayIterConst<T,N>::operator++();
         return *this;
      }

      // dereference with operator *
      T& restrict_ operator*()
      {
         LTL_ASSERT( !(this->done()), "Dereferencing end iterator" );
         return *(this->data_);
      }
};

LTL_END_NAMESPACE

#endif // __LTL_ITER__
