/* -*- C++ -*-
 *
 * ---------------------------------------------------------------------
 * $Id: merge.h,v 1.1.1.1.4.5 2004/05/06 23:15:13 drory Exp $
 * ---------------------------------------------------------------------
 *
 * Copyright (C) 2000-2002 Niv Drory <drory@usm.uni-muenchen.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA 
 *
 * ---------------------------------------------------------------------
 *
 */

#ifndef __LTL_IN_FILE_MARRAY__
#error "<ltl/marray/merge.h> must be included via <ltl/marray.h>, never alone!"
#endif


#ifndef __LTL_MERGE__
#define __LTL_MERGE__

#include <ltl/config.h>

LTL_BEGIN_NAMESPACE

/*! \defgroup merge Merge Expressions

\ingroup marray_class

For array expressions A, T, and F: \n
\code
ltl::merge( A, T, F ); // elementwise gives T if A != 0 else F.
\endcode
E.g. merge( A, 1/A, 0 ) to avoid division by 0.
*/
template<class T1, class T2, class T3, int N>
class MergeExpr : public _et_parse_base
{
   public:
      typedef typename T2::value_type value_type;
      enum { dims=N };
      enum { numIndexIter = T1::numIndexIter + T2::numIndexIter +
                            T3::numIndexIter };
      enum { isVectorizable = 0 };

      MergeExpr( const T1& a, const T2& e1, const T3& e2 )
            : i1_(a), i2_(e1), i3_(e2)
      { }

      MergeExpr( const MergeExpr<T1,T2,T3,N>& other )
            :  i1_(other.i1_), i2_(other.i2_), i3_(other.i3_)
      { }

      value_type operator*() const
      {
         return *i1_ ? *i2_ : *i3_;
      }

      void operator++()
      {
         ++i1_;
         ++i2_;
         ++i3_;
      }

      value_type readWithoutStride( const int i ) const
      {
         return i1_.readWithoutStride(i) ?
                i2_.readWithoutStride(i) : i3_.readWithoutStride(i);
      }

      value_type readWithStride( const int i ) const
      {
         return i1_.readWithStride(i) ?
                i2_.readWithStride(i) : i3_.readWithStride(i);
      }

      void advance()
      {
         i1_.advance();
         i2_.advance();
         i3_.advance();
      }

      void advanceN( const int i )
      {
         i1_.advanceN(i);
         i2_.advanceN(i);
         i3_.advanceN(i);
      }

      void advanceWithStride1()
      {
         i1_.advanceWithStride1();
         i2_.advanceWithStride1();
         i3_.advanceWithStride1();
      }

      void advanceDim()
      {
         i1_.advanceDim();
         i2_.advanceDim();
         i3_.advanceDim();
      }

      bool isStorageContiguous() const
      {
         return i1_.isStorageContiguous() && i2_.isStorageContiguous() &&
                i3_.isStorageContiguous();
      }

      bool isStride1() const
      {
         return i1_.isStride1() && i2_.isStride1() && i3_.isStride1();
      }

      bool isConformable( const Shape<N>& other ) const
      {
         return i1_.isConformable( other ) &&
                i2_.isConformable( other ) &&
                i3_.isConformable( other );
      }

 #ifdef LTL_USE_SIMD
      inline T_VEC_TYPE(value_type) readVec( const int i ) const
      {
         return (T_VEC_TYPE(value_type))(0);
      }

      inline bool sameAlignmentAs( void* p ) const
      {
         return false;
      }
#endif

     void reset()
      {
         i1_.reset();
         i2_.reset();
         i3_.reset();
      }

      const Shape<N> *shape() const
      {
         return i1_.shape();
      }

   protected:
      T1 i1_;
      T2 i2_;
      T3 i3_;
};


// version for Expression, whatever, whatever
//
template<class T1, int N, class T2, class T3>
inline TExpr< MergeExpr<TExpr<T1,N>,
typename asExpr<T2>::value_type,
typename asExpr<T3>::value_type, N >, N >
merge( const TExpr<T1,N>& a, const T2& e1, const T3& e2 )
{
   typedef MergeExpr< TExpr<T1,N>,
   typename asExpr<T2>::value_type,
   typename asExpr<T3>::value_type, N > ExprT;
   return TExpr<ExprT,N>( ExprT(a,e1,e2) );
}


// version for MArray<T1,N>, whatever, whatever
//
template<class T1, int N, class T2, class T3>
inline TExpr< MergeExpr<typename MArray<T1,N>::ConstIterator,
typename asExpr<T2>::value_type,
typename asExpr<T3>::value_type, N >, N >
merge( const MArray<T1,N>& a, const T2& e1, const T3& e2 )
{
   typedef MergeExpr< typename MArray<T1,N>::ConstIterator,
   typename asExpr<T2>::value_type,
   typename asExpr<T3>::value_type, N > ExprT;
   return TExpr<ExprT,N>( ExprT(a.begin(),e1,e2) );
}

LTL_END_NAMESPACE

#endif // __LTL_MERGE__
