/* -*- C++ -*-
 *
 * ---------------------------------------------------------------------
 * $Id: where.h,v 1.3.2.1 2003/11/02 23:23:43 cag Exp $
 * ---------------------------------------------------------------------
 *
 * Copyright (C) 2000-2002 Niv Drory <drory@usm.uni-muenchen.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA 
 *
 * ---------------------------------------------------------------------
 *
 */

#ifndef __LTL_IN_FILE_MARRAY__
#error "<ltl/marray/where.h> must be included via <ltl/marray.h>, never alone!"
#endif


#ifndef __LTL_WHERE__
#define __LTL_WHERE__

#include <ltl/config.h>

LTL_BEGIN_NAMESPACE

template<class T,int N>
class MArray;

/* \class IndexList
A list of arbitrary ltl::MArray indices.
Can be used as an index to ltl::MArray yielding an ltl::MArray<T,1> object
containing the values.
*/

//! Used to implement the function ltl::where() which returns an ltl::IndexList.
template<int N>
class IndexList
{
   public:
      typedef FixedVector<int,N> Index;
      typedef typename list<Index>::const_iterator const_iterator;

      IndexList()
            : size_(0)
      { }

      IndexList( const IndexList<N>& other )
            : indexList_(other.indexList_), size_(other.size_)
      { }

      void add( const Index& v )
      {
         indexList_.push_back( v );
         ++size_;
      }

      const_iterator begin() const
      {
         return indexList_.begin();
      }

      const_iterator end() const
      {
         return indexList_.end();
      }

      size_t size() const
      {
         return size_;
      }

   protected:
      list<Index> indexList_;
      int size_;
};


template<class T1, int N>
IndexList<N> where( TExpr<T1,N> a )
{
   CHECK_CONFORM_MSG( a, a, "where()" );
   IndexList<N> l;

   IndexIter<typename TExpr<T1,N>::value_type, N> i( a.shape() );
   while( !i.done() )
   {
      if( *a )
         l.add( i.index() );
      a.advance();
      i.advance();
      if( i.needAdvanceDim() )
      {
         a.advanceDim();
         i.advanceDim();
      }
   }
   return l;
}

template<class T1, int N>
IndexList<N> where( const MArray<T1,N>& a )
{
   IndexList<N> l;

   MArrayIterConst<T1,N> iter = a.begin();
   IndexIter<T1, N> i( a );

   while( !i.done() )
   {
      if( *iter )
         l.add( i.index() );
      iter.advance();
      i.advance();
      if( iter.needAdvanceDim() )
      {
         iter.advanceDim();
         i.advanceDim();
      }
   }
   return l;
}

// dummy class for statements of the form
// A( IndexList ) = expr
// Need something to handle the assignment operator ...
template<class T, int N>
class IndexRef
{
   public:
      IndexRef( const IndexList<N>& il, MArray<T,N>& A )
            : il_(il), A_(A)
      { }

      template<class Expr>
      void operator=( TExpr<Expr,1> expr )
      {
         LTL_ASSERT( il_.size() == (expr.shape())->nelements(),
                     "Assignment to IndexList from MArray of different length" );

         typename IndexList<N>::const_iterator i=il_.begin(), e=il_.end();
         while( i != e )
         {
            A_( *i ) = static_cast<T>(*expr);
            expr.advance();
            ++i;
         }
      }

      template<class T2>
      void operator=( const MArray<T2,1>& expr )
      {
         LTL_ASSERT( il_.size() == expr.nelements(),
                     "Assignment to IndexList from MArray of different length" );

         typename IndexList<N>::const_iterator i=il_.begin();
         typename MArray<T2,1>::ConstIterator ie = expr.begin();
         while( !ie.done() )
         {
            A_( *i ) = static_cast<T>(*ie);
            ++ie;
            ++i;
         }
      }

      void operator=( const T expr )
      {
         typename IndexList<N>::const_iterator i=il_.begin(), e=il_.end();
         while( i != e )
         {
            A_( *i ) = expr;
            ++i;
         }
      }


   protected:
      const IndexList<N>& il_;
      MArray<T,N>& A_;
};

LTL_END_NAMESPACE

#endif // __LTL_WHERE__
