/* -*- C++ -*-
 *
 * ---------------------------------------------------------------------
 * $Id: marray_io.h,v 1.1.1.1.4.4 2004/04/23 19:21:28 drory Exp $
 * ---------------------------------------------------------------------
 *
 * Copyright (C) 2000-2002 Niv Drory <drory@usm.uni-muenchen.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA 
 *
 * ---------------------------------------------------------------------
 *
 */


#ifndef __LTL_MARRAYIO__
#define __LTL_MARRAYIO__

#include <ltl/config.h>

#include <string>
#include <iostream>

#include <ltl/marray.h>
#include <ltl/misc/exceptions.h>

using std::istream;
using std::ostream;
using std::string;
using std::endl;

LTL_BEGIN_NAMESPACE

template<class T, int N>
class MArray;
template<class T, int N>
class MArrayIter;
template<int N>
class Shape;
template<int N>
istream& operator>>( istream& is, Shape<N>& s );

/*! \addtogroup ma_stream_io
*/
//@{

/*! \relates ltl::MArray
  Write a 1-dim \c ltl::MArray to a stream.
*/
template<class T>
ostream& operator<<(ostream& os, const MArray<T,1>& A)
{
   os << "MArray<T,1> "<<*(A.shape()) << endl;
   os << "[ ";
   for (int i=A.minIndex(1); i <= A.maxIndex(1); ++i)
   {
      os << A(i) << " ";
   }
   os << "]";
   return os;
}

/*! \relates ltl::MArray
  Write a 2-dim \c ltl::MArray to a stream.
*/
template<class T>
ostream& operator<<(ostream& os, const MArray<T,2>& A)
{
   // rows x columns
   os << "MArray<T,2> "<<*(A.shape()) << endl;
   os << "[";
   for (int j=A.minIndex(2); j <= A.maxIndex(2); ++j)
   {
      os << "[ ";
      for (int i=A.minIndex(1); i <= A.maxIndex(1); ++i)
      {
         os << A(i,j) << " ";
      }
      os << "]";
      if( j<A.maxIndex(2) )
         os << endl << " ";
   }

   os << "]";

   return os;
}

/*! \relates ltl::MArray
  Write a 3-dim \c ltl::MArray to a stream.
*/
template<class T>
ostream& operator<<(ostream& os, const MArray<T,3>& A)
{
   os << "MArray<T,3> "<<*(A.shape()) << endl;
   os << "[";
   for( int k=A.minIndex(3); k <= A.maxIndex(3); ++k)
   {
      os << "[";
      for (int j=A.minIndex(2); j <= A.maxIndex(2); ++j)
      {
         os << "[ ";
         for (int i=A.minIndex(1); i <= A.maxIndex(1); ++i)
         {
            os << A(i,j,k) << " ";
         }
         os << "]";
         if( j<A.maxIndex(2) )
            os << endl << "  ";
      }
      os << "]";
      if( k<A.maxIndex(3) )
         os << endl << " ";
   }

   os << "]";

   return os;
}

/*! \relates ltl::MArray
  Write a 4-dim \c ltl::MArray to a stream.
*/
template<class T>
ostream& operator<<(ostream& os, const MArray<T,4>& A)
{
   os << "MArray<T,4> "<<*(A.shape()) << endl;
   os << "[";
   for( int l=A.minIndex(4); l <= A.maxIndex(4); ++l)
   {
      os << "[";
      for( int k=A.minIndex(3); k <= A.maxIndex(3); ++k)
      {
         os << "[";
         for (int j=A.minIndex(2); j <= A.maxIndex(2); ++j)
         {
            os << "[ ";
            for (int i=A.minIndex(1); i <= A.maxIndex(1); ++i)
            {
               os << A(i,j,k,l) << " ";
            }
            os << "]";
            if( j<A.maxIndex(2) )
               os << endl << "   ";
         }
         os << "]";
         if( k<A.maxIndex(3) )
            os << endl << "  ";
      }
      os << "]";
      if( l<A.maxIndex(4) )
         os << endl << " ";
   }

   os << "]";

   return os;
}

/*! \relates ltl::MArray
  Write a 5-dim \c ltl::MArray to a stream.
*/
template<class T>
ostream& operator<<(ostream& os, const MArray<T,5>& A)
{
   os << "MArray<T,5> "<<*(A.shape()) << endl;
   os << "[";
   for( int m=A.minIndex(5); m <= A.maxIndex(5); ++m)
   {
      os << "[";
      for( int l=A.minIndex(4); l <= A.maxIndex(4); ++l)
      {
         os << "[";
         for( int k=A.minIndex(3); k <= A.maxIndex(3); ++k)
         {
            os << "[";
            for (int j=A.minIndex(2); j <= A.maxIndex(2); ++j)
            {
               os << "[ ";
               for (int i=A.minIndex(1); i <= A.maxIndex(1); ++i)
               {
                  os << A(i,j,k,l,m) << " ";
               }
               os << "]";
               if( j<A.maxIndex(2) )
                  os << endl << "    ";
            }
            os << "]";
            if( k<A.maxIndex(3) )
               os << endl << "   ";
         }
         os << "]";
         if( l<A.maxIndex(4) )
            os << endl << "  ";
      }
      os << "]";
      if( m<A.maxIndex(5) )
         os << endl << " ";
   }

   os << "]";

   return os;
}

/*! \relates ltl::MArray
  Write a 6-dim \c ltl::MArray to a stream.
*/
template<class T>
ostream& operator<<(ostream& os, const MArray<T,6>& A)
{
   os << "MArray<T,6> "<<*(A.shape()) << endl;
   os << "[";
   for( int n=A.minIndex(6); n <= A.maxIndex(6); ++n)
   {
      os << "[";
      for( int m=A.minIndex(5); m <= A.maxIndex(5); ++m)
      {
         os << "[";
         for( int l=A.minIndex(4); l <= A.maxIndex(4); ++l)
         {
            os << "[";
            for( int k=A.minIndex(3); k <= A.maxIndex(3); ++k)
            {
               os << "[";
               for (int j=A.minIndex(2); j <= A.maxIndex(2); ++j)
               {
                  os << "[ ";
                  for (int i=A.minIndex(1); i <= A.maxIndex(1); ++i)
                  {
                     os << A(i,j,k,l,m,n) << " ";
                  }
                  os << "]";
                  if( j<A.maxIndex(2) )
                     os << endl << "    ";
               }
               os << "]";
               if( k<A.maxIndex(3) )
                  os << endl << "   ";
            }
            os << "]";
            if( l<A.maxIndex(4) )
               os << endl << "  ";
         }
         os << "]";
         if( m<A.maxIndex(5) )
            os << endl << " ";
      }
      os << "]";
      if( n<A.maxIndex(6) )
         os << endl << " ";
   }

   os << "]";

   return os;
}

/*! \relates ltl::MArray
  Write a 7-dim \c ltl::MArray to a stream.
*/
template<class T>
ostream& operator<<(ostream& os, const MArray<T,7>& A)
{
   os << "MArray<T,7> "<<*(A.shape()) << endl;
   os << "[";
   for( int p=A.minIndex(7); p <= A.maxIndex(7); ++p)
   {
      os << "[";
      for( int n=A.minIndex(6); n <= A.maxIndex(6); ++n)
      {
         os << "[";
         for( int m=A.minIndex(5); m <= A.maxIndex(5); ++m)
         {
            os << "[";
            for( int l=A.minIndex(4); l <= A.maxIndex(4); ++l)
            {
               os << "[";
               for( int k=A.minIndex(3); k <= A.maxIndex(3); ++k)
               {
                  os << "[";
                  for (int j=A.minIndex(2); j <= A.maxIndex(2); ++j)
                  {
                     os << "[ ";
                     for (int i=A.minIndex(1); i <= A.maxIndex(1); ++i)
                     {
                        os << A(i,j,k,l,m,n,p) << " ";
                     }
                     os << "]";
                     if( j<A.maxIndex(2) )
                        os << endl << "    ";
                  }
                  os << "]";
                  if( k<A.maxIndex(3) )
                     os << endl << "   ";
               }
               os << "]";
               if( l<A.maxIndex(4) )
                  os << endl << "  ";
            }
            os << "]";
            if( m<A.maxIndex(5) )
               os << endl << " ";
         }
         os << "]";
         if( n<A.maxIndex(6) )
            os << endl << " ";
      }
      os << "]";
      if( p<A.maxIndex(7) )
         os << endl << " ";
   }

   os << "]";

   return os;
}


/*! \relates ltl::MArray
  Read an \c ltl::MArray from a stream.
*/
template<class T, int N>
istream& operator>>( istream& is, MArray<T,N>& A )
{
   // read shape
   Shape<N> s;
   is >> s;
   A.realloc( s );

   T t;
   string tmp;

   typename MArray<T,N>::iterator i=A.begin();
   while( !i.done() )
   {
      is >> tmp;

      if( tmp[tmp.length()-1] != '[' )
         throw( IOException( "Format error while reading : '[' expected!" ) );
      for( int n=A.minIndex(1); n<=A.maxIndex(1); ++n, ++i )
      {
         if( i.done() )
            throw( IOException( "File too long while reading MArray!" ) );
         is >> t;

         *i = t;
         if( is.bad() )
            throw( IOException( "Format error while reading MArray!" ) );
      }
      is >> tmp;

      if( tmp[tmp.length()-1] != ']' )
         throw( IOException( "Format error while reading : ']' expected!" ) );
   }
   return is;
}

//@}

LTL_END_NAMESPACE

#endif // __LTL_MARRAYIO__

