/* -*- C++ -*-
*
* ---------------------------------------------------------------------
* $Id: applicops.h,v 1.4.2.5 2004/05/25 22:03:39 drory Exp $
* ---------------------------------------------------------------------
*
* Copyright (C) 2000-2002 Niv Drory <drory@usm.uni-muenchen.de>
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2, or (at your option)
* any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
*
* ---------------------------------------------------------------------
*
*/


#if !defined(__LTL_IN_FILE_MARRAY__) && !defined(__LTL_IN_FILE_FVECTOR__) && !defined(__LTL_IN_FILE_FMATRIX__)
#error "<ltl/misc/applicops.h> must be included via ltl headers, never alone!"
#endif


#ifndef __LTL_APPLICOPS_H__
#define __LTL_APPLICOPS_H__

#include <ltl/misc/type_promote.h>

LTL_BEGIN_NAMESPACE

/*! \file applicops.h

  Applicative template classes for use in expression templates.

  These represent operations in the expression parse tree. They hold
  the operation that is actually performed on the elements of the 
  container, and therefore take these as arguments.
  Each applicative template has a corresponding operator
  or function (defined in expr.h for MArrays) which takes the container
  itself and/or whole expressions as arguments and are used to build 
  up the parse tree.

  These objects defined here are used by all expression template engines 
  in LTL (\c Array, \c FVector, and \c FMatrix)

*/


//@{

//! This is just to keep everything together in class browsers
struct _et_applic_base
{ 
   enum { isVectorizable = 0 };
};


/*!
 *  applicative templates for binary operators returning the
 *  c-style promoted type of their inputs
 */
#define MAKE_BINAP(classname,op)                                        \
template<class T1, class T2>                                            \
struct classname : public _et_applic_base                               \
{                                                                       \
   typedef typename promotion_trait<T1,T2>::PType value_type;           \
   static inline value_type eval( const T1& a, const T2& b )            \
     { return a op b; }                                                 \
}


/*!
 *  applicative templates for binary operators returning a fixed unpromoted
 *  type (eg. bool, logical operations)
 */
#define MAKE_BINAP_RET(classname,ret_type,op)                   \
template<class T1, class T2>                                    \
struct classname : public _et_applic_base                       \
{                                                               \
   typedef ret_type value_type;                                 \
   static inline value_type eval( const T1& a, const T2& b )    \
     { return a op b; }                                         \
}


/*!
 *  applicative templates for binary functions returning a fixed unpromoted 
 *  type (e.g. stdlib math functions)
 */
#define MAKE_BINAP_FUNC(classname,ret_type,func)                \
template<class T1, class T2>                                    \
struct classname : public _et_applic_base                       \
{                                                               \
   typedef ret_type value_type;                                 \
   static inline ret_type eval( const T1& a, const T2& b )      \
     { return func( a, b ); }                                   \
}


/*!
 *  applicative templates for unary operators returning the same type
 *  as their input
 */
#define MAKE_UNAP(classname,op)                                         \
template<class T>                                                       \
struct classname : public _et_applic_base                               \
{                                                                       \
   typedef T value_type;                                                \
   static inline T eval( const T&  a )                                  \
     { return op(a); }                                                  \
}

/*!
 *   applicative templates for unary functions
 */
#define MAKE_UNAP_FUNC(classname,ret_type,op)                           \
template<class T>                                                       \
struct classname : public _et_applic_base                               \
{                                                                       \
   typedef ret_type value_type;                                         \
   static inline ret_type eval( const T& a )                            \
     { return op(a); }                                                  \
}


/*!
 *  standard operators
 *  these are common to MArray, FVector, and FMatrix
 */
MAKE_BINAP(__ltl_TAdd, +);
MAKE_BINAP(__ltl_TSub, -);
MAKE_BINAP(__ltl_TMul, *);
MAKE_BINAP(__ltl_TDiv, /);

MAKE_BINAP(__ltl_TBitAnd, &);
MAKE_BINAP(__ltl_TBitOr , |);
MAKE_BINAP(__ltl_TBitXor, ^);
MAKE_BINAP(__ltl_TMod   , %);

MAKE_BINAP_RET(__ltl_TAnd, bool, && );
MAKE_BINAP_RET(__ltl_TOr , bool, || );
MAKE_BINAP_RET(__ltl_TGT , bool, >  );
MAKE_BINAP_RET(__ltl_TLT , bool, <  );
MAKE_BINAP_RET(__ltl_TGE , bool, >= );
MAKE_BINAP_RET(__ltl_TLE , bool, <= );
MAKE_BINAP_RET(__ltl_TNE , bool, != );
MAKE_BINAP_RET(__ltl_TEQ , bool, == );

MAKE_UNAP( __ltl_TPlus,  + );
MAKE_UNAP( __ltl_TMinus, - );
MAKE_UNAP( __ltl_TNot,   ! );
MAKE_UNAP( __ltl_TNeg,   ~ );


/*!
 *
 * All standard library math functions
 *
 * acos        inverse trigonometric function
 * acosh       inverse hyperbolic function   
 * asin        inverse trigonometric function
 * asinh       inverse hyperbolic function   
 * atan        inverse trigonometric function
 * atanh       inverse hyperbolic function   
 * atan2       inverse trigonometric function
 * cabs        complex absolute value        
 * cbrt        cube root                     
 * ceil        integer no less than          
 * copysign    copy sign bit                 
 * cos         trigonometric function        
 * cosh        hyperbolic function           
 * erf         error function                
 * erfc        complementary error function  
 * exp         exponential                   
 * expm1       exp(x)-1                      
 * fabs        absolute value                
 * floor       integer no greater than       
 * hypot       Euclidean distance            
 * ilogb       exponent extraction           
 * infnan      signals exceptions
 * j0          bessel function               
 * j1          bessel function               
 * jn          bessel function               
 * lgamma      log gamma function; (formerly gamma.3m)
 * log         natural logarithm       
 * log10       logarithm to base 10    
 * log1p       log(1+x)                
 * pow         exponential x**y
 * remainder   remainder               
 * rint        round to nearest integer
 * scalbn      exponent adjustment     
 * sin         trigonometric function  
 * sinh        hyperbolic function     
 * sqrt        square root             
 * tan         trigonometric function  
 * tanh        hyperbolic function     
 * y0          bessel function         
 * y1          bessel function
 * yn          bessel function
 * 
 */
#ifdef LTL_USING_NAMESPACE
MAKE_UNAP_FUNC( __ltl_sin,   double, std::sin   );
MAKE_UNAP_FUNC( __ltl_cos,   double, std::cos   );
MAKE_UNAP_FUNC( __ltl_tan,   double, std::tan   );
MAKE_UNAP_FUNC( __ltl_asin,  double, std::asin  );
MAKE_UNAP_FUNC( __ltl_acos,  double, std::acos  );
MAKE_UNAP_FUNC( __ltl_atan,  double, std::atan  );
MAKE_UNAP_FUNC( __ltl_sinh,  double, std::sinh  );
MAKE_UNAP_FUNC( __ltl_cosh,  double, std::cosh  );
MAKE_UNAP_FUNC( __ltl_tanh,  double, std::tanh  );
MAKE_UNAP_FUNC( __ltl_exp,   double, std::exp   );
MAKE_UNAP_FUNC( __ltl_log,   double, std::log   );
MAKE_UNAP_FUNC( __ltl_log10, double, std::log10 );
MAKE_UNAP_FUNC( __ltl_sqrt,  double, std::sqrt  );
MAKE_UNAP_FUNC( __ltl_fabs,  double, std::fabs  );
MAKE_UNAP_FUNC( __ltl_floor, double, std::floor );
MAKE_UNAP_FUNC( __ltl_ceil,  double, std::ceil  );
MAKE_BINAP_FUNC( __ltl_pow,  double, std::pow   );
MAKE_BINAP_FUNC( __ltl_fmod, double, std::fmod  );
MAKE_BINAP_FUNC( __ltl_atan2,double, std::atan2 );
#else
MAKE_UNAP_FUNC( __ltl_sin,   double, ::sin   );
MAKE_UNAP_FUNC( __ltl_cos,   double, ::cos   );
MAKE_UNAP_FUNC( __ltl_tan,   double, ::tan   );
MAKE_UNAP_FUNC( __ltl_asin,  double, ::asin  );
MAKE_UNAP_FUNC( __ltl_acos,  double, ::acos  );
MAKE_UNAP_FUNC( __ltl_atan,  double, ::atan  );
MAKE_UNAP_FUNC( __ltl_sinh,  double, ::sinh  );
MAKE_UNAP_FUNC( __ltl_cosh,  double, ::cosh  );
MAKE_UNAP_FUNC( __ltl_tanh,  double, ::tanh  );
MAKE_UNAP_FUNC( __ltl_exp,   double, ::exp   );
MAKE_UNAP_FUNC( __ltl_log,   double, ::log   );
MAKE_UNAP_FUNC( __ltl_log10, double, ::log10 );
MAKE_UNAP_FUNC( __ltl_sqrt,  double, ::sqrt  );
MAKE_UNAP_FUNC( __ltl_fabs,  double, ::fabs  );
MAKE_UNAP_FUNC( __ltl_floor, double, ::floor );
MAKE_UNAP_FUNC( __ltl_ceil,  double, ::ceil  );
MAKE_BINAP_FUNC( __ltl_pow,  double, ::pow   );
MAKE_BINAP_FUNC( __ltl_fmod, double, ::fmod  );
MAKE_BINAP_FUNC( __ltl_atan2,double, ::atan2 );
#endif

/*! 
   IEEE math functions
   from C99 standard
*/
#ifdef HAVE_IEEE_MATH

MAKE_UNAP_FUNC( __ltl_asinh,  double, ::asinh  );
MAKE_UNAP_FUNC( __ltl_acosh,  double, ::acosh  );
MAKE_UNAP_FUNC( __ltl_atanh,  double, ::atanh  );
MAKE_UNAP_FUNC( __ltl_cbrt,   double, ::cbrt   );
MAKE_UNAP_FUNC( __ltl_expm1,  double, ::expm1  );
MAKE_UNAP_FUNC( __ltl_log1p,  double, ::log1p  );
MAKE_UNAP_FUNC( __ltl_erf,    double, ::erf    );
MAKE_UNAP_FUNC( __ltl_erfc,   double, ::erfc   );
MAKE_UNAP_FUNC( __ltl_j0,     double, ::j0     );
MAKE_UNAP_FUNC( __ltl_j1,     double, ::j1     );
MAKE_UNAP_FUNC( __ltl_y0,     double, ::y0     );
MAKE_UNAP_FUNC( __ltl_y1,     double, ::y1     );
MAKE_UNAP_FUNC( __ltl_lgamma, double, ::lgamma );
MAKE_UNAP_FUNC( __ltl_rint,   double, ::rint   );

MAKE_BINAP_FUNC( __ltl_hypot, double, ::hypot  );
#endif


/*!
   Optimal implementation of pow with integer exponent <= 8
*/
inline double pow2( const double x )
{
   return x*x;
}

inline double pow3( const double x )
{
   return x*x*x;
}

inline double pow4( const double x )
{
   const double tmp = x*x;
   return tmp*tmp;
}

inline double pow5( const double x )
{
   const double tmp = x*x;
   return tmp*tmp*x;
}

inline double pow6( const double x )
{
   const double tmp = x*x*x;
   return tmp*tmp;
}

inline double pow7( const double x )
{
   const double tmp = x*x;
   return tmp*tmp*tmp*x;
}

inline double pow8( const double x )
{
   const double tmp1 = x*x;
   const double tmp2 = tmp1*tmp1;
   return tmp2*tmp2;
}

MAKE_UNAP_FUNC( __ltl_pow2, double, pow2 );
MAKE_UNAP_FUNC( __ltl_pow3, double, pow3 );
MAKE_UNAP_FUNC( __ltl_pow4, double, pow4 );
MAKE_UNAP_FUNC( __ltl_pow5, double, pow5 );
MAKE_UNAP_FUNC( __ltl_pow6, double, pow6 );
MAKE_UNAP_FUNC( __ltl_pow7, double, pow7 );
MAKE_UNAP_FUNC( __ltl_pow8, double, pow8 );

//@}

LTL_END_NAMESPACE

#endif // __LTL_APPLOCOPS_H__
