/* -*- C++ -*-
 *
 * ---------------------------------------------------------------------
 * $Id: memblock.h,v 1.2.4.5 2004/07/10 23:18:56 drory Exp $
 * ---------------------------------------------------------------------
 *
 * Copyright (C) 2000-2002 Niv Drory <drory@usm.uni-muenchen.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA 
 *
 * ---------------------------------------------------------------------
 *
 */


#ifndef __LTL_MEMBLOCK__
#define __LTL_MEMBLOCK__

#include <ltl/config.h>
#include <cstdio>
#include <cstdlib>
#include <cstring>

// mmap file
#include <fcntl.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>

LTL_BEGIN_NAMESPACE

/*!
  Provides a reference-counted block of memory. This block
  may be referred to by multiple array objects. The memory
  is automatically deallocated when the last referring object is destructed.
*/
template<class T>
class MemoryBlock
{
   public:
      typedef T value_type;

      MemoryBlock()
         : data_(0), references_(0)
      { }

      //! Create new MemoryBlock from scratch.
      MemoryBlock( const int items )
         : data_(new value_type[items]), references_(0)
      { 
#ifdef LTL_DEBUG_MEMORY_BLOCK
         cerr << "MemoryBlock " << this
              << "   Allocating : " << data_ << endl;
#endif
      }
      
      //! Create MemoryBlock from pre-allocated memory.
      MemoryBlock( value_type* restrict_ data )
         : data_(data), references_(0)
      { 
#ifdef LTL_DEBUG_MEMORY_BLOCK
         cerr << "MemoryBlock " << this
              << "   Allocating : " << data_ << endl;
#endif
      }

      virtual ~MemoryBlock()
      {
#ifdef LTL_DEBUG_MEMORY_BLOCK
         cerr << "MemoryBlock " << this
              << "   Deleting : " << data_ << endl;
#endif
         if( data_ )
            delete [] data_;
      }

      //! Increment reference count.
      void addReference()
      {
         ++references_;
      }

      //! Return new reference to data memory.
      value_type* restrict_ data()
      {
         addReference();
         return data_;
      }

      value_type* restrict_ data() const
      {
         addReference();
         return data_;
      }

      //! Decrement refernce count and destroy if count==0.
      void removeReference()
      {
         --references_;
         if( !references_ )
         {
            delete this;
         }
      }

      //! Return actual reference count.
      int references() const
      {
         return references_;
      }

      void describeSelf() const
      {
         cout << "  MemBlock at         : " << this << endl;
         cout << "     References       : " << references_ << endl;
         cout << "     Data ptr         : " << data_ << endl;
      }

   protected:   // Data members
      value_type*   restrict_ data_;
      int           references_;
};


/*!
  Provides a reference-counted block of mapped memory. This block
  may be referred to by multiple array objects. The memory
  is automatically deallocated when the last referring object is destructed.
*/
template<class T>
class MemoryMapBlock : public MemoryBlock<T>
{

   public:

      MemoryMapBlock( const int items, const char * filename = NULL )
         : MemoryBlock<T>(),
         length_(items * sizeof(T)), savemmap_(filename != NULL)
      {
         if(savemmap_)
            strcpy(filename_, filename);
         else
            strcpy(filename_, "./ltl.XXXXXX");
         const int tmp_fd = savemmap_ ?
            open(filename_, O_RDWR|O_CREAT,
                 S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP|S_IROTH|S_IWOTH)
            : mkstemp(filename_);
         if(tmp_fd < 0)
            throw RangeException("Cannot open memory map file.");
         // set length of file
         if(length_ > 0)
         {
            if( ftruncate(tmp_fd, length_) < 0)
               throw RangeException("Cannot set file size for map file");
            if(fsync(tmp_fd) < 0)
               throw RangeException("Cannot sync map file");
            // map file
            this->data_ = (T *) mmap(NULL, length_,
                                     PROT_READ|PROT_WRITE, MAP_SHARED,
                                     tmp_fd, 0);
            if((unsigned char *)(this->data_) == (unsigned char *) MAP_FAILED)
               throw RangeException("Cannot map mapfile");
         }         
         close(tmp_fd);
      }

      virtual ~MemoryMapBlock()
      {
         if( this->data_ )
         {
            munmap((char *)(this->data_), length_);
            this->data_ = NULL;
            if(!savemmap_)
               remove(filename_);
         }
      }

   private:   // Data members
      const size_t  length_;
      bool          savemmap_;
      char          filename_[16];      
};

LTL_END_NAMESPACE

#endif // __LTL_MEMBLOCK__
