/* -*- C++ -*-
 *
 * ---------------------------------------------------------------------
 * $Id: option_parser.h,v 1.4.4.5 2005/01/29 19:25:00 drory Exp $
 * ---------------------------------------------------------------------
 *
 * Copyright (C) 2000-2001 Niv Drory <drory@usm.uni-muenchen.de>
 *                         Claus A. Goessl <cag@usm.uni-muenchen.de>
 *                         Arno Riffeser <arri@usm.uni-muenchen.de>
 *                         Jan Snigula <snigula@usm.uni-muenchen.de>
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA 
 *
 * ---------------------------------------------------------------------
 *
 */


#ifndef __UTIL_OPTION_PARSER__
#define __UTIL_OPTION_PARSER__

#include <ltl/config.h>

#include <map>
#include <string>
#include <iostream>
#include <fstream>

#include <cstring>

#include <ltl/util/u_exception.h>
#include <ltl/util/option.h>

#ifdef UTIL_USING_NAMESPACE
using std::string;
using std::ostream;
using std::ofstream;
using std::map;
using std::endl;
#endif

UTIL_BEGIN_NAMESPACE

/*! \addtogroup util_options
*/

//@{

class OptionReader;

//! Generic parser object for options.
/*! Uses an object of type util::OptionReader
  to retrieve the options from whereever (config file, command-line, etc.)
*/
class OptionParser
{
   protected:
      friend ostream& operator<<( ostream& os, const OptionParser& op );
      typedef map<string,Option*> omap;
      typedef map<int,Option*>    onmap;
      
      OptionReader *reader_;

      omap  options_;      // holds all options we know of indexed by name
      omap  cmd_options_;  // index of options by command line character
      onmap n_options_;    // holds options indexed by order of definition
      int   nopts_;        // number of options
   public:
      OptionParser( OptionReader* reader );
      virtual ~OptionParser();

      OptionParser( const OptionParser& other );

      void deleteOptions( void );

      void changeReader( OptionReader* reader );

      void addOption( Option* option );
      Option* getOption( const string& name );
      void parseOptions() throw( UException );
      void printUsage( ostream& os ) const;
      void writeConfig( const string& filename, 
                        const bool withComment = true, const bool order_by_n=true ) const throw( UException );
      string toString( const bool withComment = false ) const throw( UException );
};

/*! \relates util::OptionParser

  Write options to \e op.
*/
ostream& operator<<( ostream& os, const OptionParser& op );

//! An \b abstract base class for sources feeding into util::OptionParser .
/*! It defines the following interface
  supplying exactly three methods, namely
  \li string nextOptionName(),
  \li string nextOptionValue( const Option* op ),
  \li bool   done().
  These are then converted into objects of type Option.
*/
class OptionReader
{
   public:
      OptionReader()
      { }
      ;

      virtual ~OptionReader()
      { }
      ;

      virtual string nextOptionName()  = 0;
      virtual string nextOptionValue( const Option* op ) = 0;
      virtual bool   done() = 0;
};

//@}

UTIL_END_NAMESPACE

#endif
