/* -*- C++ -*-
*
* ---------------------------------------------------------------------
* $Id: testfitsio.cpp,v 1.5.4.4 2006/03/10 19:26:50 cag Exp $
* ---------------------------------------------------------------------
*
* Copyright (C) 2000-2002 Niv Drory <drory@usm.uni-muenchen.de>
*                         Claus A. Goessl <cag@usm.uni-muenchen.de>
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2, or (at your option)
* any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
*
* ---------------------------------------------------------------------
*
*/

#define LTL_RANGE_CHECKING

#include <fitsio.h>
#include <statistics.h>
#include <marray_io.h>

#include <iostream>
#include <cstdio>

using namespace ltl;

using std::cout;
using std::endl;

int main(int argc, char **argv);
void testfloat( void );
void testdouble( void );
void testbyte( void );
void testshort( void );
void testint( void );
void testcopy( void );
void testreadline( void );
void testwriteregion( void );

void testfloat( void )
{
   cerr << "  BITPIX -32 ..." << endl;   

   MArray<float,2> A(512,512);

   A = indexPos(A,1) + 100*indexPos(A,2);
   
   FitsOut fo("test-32.fits");
   
   fo.addValueCard("TESTTEST", "This is a test");
   LTL_ASSERT_( fo.getString( "TESTTEST" ) == "This is a test",
                "FITS AddValueCard( string ) failed" );
   fo.addValueCard("TEST1234", 1.234);
   LTL_ASSERT_( fo.getFloat( "TEST1234" ) == 1.234,
                "FITS AddValueCard( float ) failed" ); 
   fo << A;

   MArray<float,2> B;
   FitsIn fi("test-32.fits");
   fi >> B;

   LTL_ASSERT_( allof(A == B),
                "FITS -32 read or write failed");

   LTL_ASSERT_( fi.getString( "TESTTEST" ) == "This is a test",
                "FITS getString() failed" );
   LTL_ASSERT_( fi.getFloat( "TEST1234" ) == 1.234,
                "FITS getFloat() failed" );
}

void testdouble( void )
{
   cerr << "  BITPIX -64 ..." << endl;   
   MArray<double,2> A(512,512);

   A = indexPos(A,1) + 25*indexPos(A,2);
   FitsOut fo("test-64.fits");
   
   fo << A;

   MArray<double,2> B;
   FitsIn fi("test-64.fits");
   fi >> B;

   LTL_ASSERT_( allof(A == B),
                "FITS -64 read or write failed");

   fi.freeData();
   remove("test-64.fits");
}

void testint( void )
{
   cerr << "  BITPIX 32 ..." << endl;   
   MArray<int,2> A(512,512);

   A = indexPos(A,1) + 25*indexPos(A,2);
   FitsOut fo("test32.fits");
   
   fo << A;

   MArray<int,2> B;
   FitsIn fi("test32.fits");
   fi >> B;

   LTL_ASSERT_( allof(A == B),
                "FITS 32 read or write failed");

   fi.freeData();
   remove("test32.fits");
}

void testshort( void )
{
   cerr << "  BITPIX 16 ..." << endl;   
   MArray<short,2> A(512,512);

   A = indexPos(A,1) + 25*indexPos(A,2);
   FitsOut fo("test16.fits");
   
   fo << A;

   MArray<short,2> B;
   FitsIn fi("test16.fits");
   fi >> B;

   LTL_ASSERT_( allof(A == B),
                "FITS 16 read or write failed");

   fi.freeData();
   remove("test16.fits");
}

void testbyte( void )
{
   cerr << "  BITPIX 8 ..." << endl;   
   MArray<char,2> A(16,16);

   A = indexPos(A,1) + 25*indexPos(A,2);
   FitsOut fo("test8.fits");
   
   fo << A;

   MArray<char,2> B;
   FitsIn fi("test8.fits");
   fi >> B;

   LTL_ASSERT_( allof(A == B),
                "FITS 8 read or write failed");

   fi.freeData();
   remove("test8.fits");
}

void testcopy( void )
{
   cerr << "  BITPIX -32 copy ..." << endl;   

   FitsIn fi1("test-32.fits");
   FitsOut fo("testc.fits", fi1);
   fi1 >> fo;

   MArray<float,2> A, B;
   fi1 >> A;
   FitsIn fi3("testc.fits");
   fi3 >> B;

   LTL_ASSERT_( allof(A == B),
                "FITS -32 copy failed");

   LTL_ASSERT_( fi3.getString( "TESTTEST" ) == "This is a test",
                "FITS getString() failed" );
   LTL_ASSERT_( fi3.getFloat( "TEST1234" ) == 1.234,
                "FITS getFloat() failed" ); 

   fi3.freeData();
   remove("testc.fits");
}

void testreadline( void )
{
   cerr << "  Reading file per line ..." << endl;
   
   MArray<float, 2> A;
   MArray<float, 1> B(512);

   FitsIn fi("test-32.fits");
   fi >> A;

   for(size_t i = 1; i <= size_t(fi.getNaxis(2)); ++i)
   {
      fi >> B;
      LTL_ASSERT_( allof(A(Range::all(), i) == B),
                   "FITS reading per line failed");
   }

   fi.freeData();
   remove("test-32.fits");
}

void testwriteregion( void )
{
   cerr << "  Writing region of file ..." << endl;   

   MArray<float,2> A(500,500);

   A = indexPos(A,1) + 100*indexPos(A,2);
   
   FitsOut fo("test-reg.fits");
   Region region(2);
   region.setRange(1, 1, 1000);
   region.setRange(2, 1, 1000);
   // creating file
   fo.setGeometry(-32, region);

   region.setRange(1, 251, 750);
   region.setRange(2, 251, 750);
   fo.setRegion(region);
   fo << A;

   MArray<float,2> B;
   FitsIn fi("test-reg.fits", region);
   fi >> B;

   LTL_ASSERT_( allof(A == B),
                "FITS file region read or write failed (written values)");
   B.free();

   region.setRange(1, 751, 1000);
   region.setRange(2, 1, 1000);
   fi.setRegion(region);
   fi >> B;

   LTL_ASSERT_( allof(B == 0.0f),
                "FITS file region read or write failed (truncate with zeroes)");   

   fi.freeData();
   remove("test-reg.fits");
}

void testwriteconstarray( void )
{
   MArray<float,2> A(500,500);
   A = indexPos(A,1) + 100*indexPos(A,2);
   const MArray<float,2>B(A);
   FitsOut fo("testconst.fits");
   fo << B;
   remove("testconst.fits");
}

int main(int argc, char **argv)
{
   cerr << "Testing MArray FITS I/O ..." << endl;   
   const int offtsize = sizeof(off_t);
   if(sizeof(off_t)!=8){
     cerr << "Wrong offset type off_t: Must be 64 bit, but is "
	  << offtsize * 8 << " bit."<< endl;
     throw FitsException("");
   }
   try
   {
      testfloat();
      testdouble();
      testbyte();
      testshort();
      testint();
      testcopy();
      testreadline();
      testwriteregion();
      testwriteconstarray();
   }
   catch (exception &e){
      cerr << e.what() << endl;
      throw;
   }
}
