/*******************************************************************************
 * POP3 Library.
 * COPYRIGHT(C) 1999-2001 TKS
 * 
 * $Revision: 1.8 $
 * $Date: 2001/02/07 14:34:41 $
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 ******************************************************************************/

// #include <config.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <unistd.h>
#include <errno.h>
#include <strings.h>
#include <stdio.h>

#include "libpop.h"
#include "main.h"

extern typMascot *Mascot;

/*==========================================================================*/
/* Function Prototype                                                       */
/*==========================================================================*/
int popReadLine(char *ptr, int size);
int popWriteLine(char *ptr);

/*==========================================================================*/
/* Internal Function Prototype                                              */
/*==========================================================================*/
static int popIsOK(char *ptr);
static int popLineCommand(char *ptr);
static int popApopGetKey(char *ptr, char *apop_key);
static int popApop(char *user, char *passwd, char *apop_key);

extern  char * MD5Digest (unsigned char *s);

/*==========================================================================*/
/* Global Variables                                                         */
/* Can not use application.                                                 */
/*==========================================================================*/
static int  Sockfd;                                   /* POP3 server socket */
static FILE *Rfp;                                     /* Read file pointer  */
static FILE *Wfp;                                     /* Write file pointer */



/*==========================================================================*/
/* Read data (one line) from POP3 server                                    */
/*==========================================================================*/
int popReadLine(char *ptr, int size)
{
  char *funcret;

  funcret = fgets(ptr, size, Rfp);
  if( funcret == NULL ){
    return(errno);
  }
  return(0);
}



/*============================================================================*/
/* Send data (one line) to POP3 server                                        */
/*============================================================================*/
int popWriteLine(char *ptr)
{
  int funcret;

  funcret = fputs(ptr, Wfp);
  if( funcret < 0 ){
    return(errno);
  }
  if(!strstr(ptr, "\r\n")){
    funcret = fputs("\r\n", Wfp);
    if( funcret < 0 ){
      return(errno);
    }
  }
  fflush(Wfp);
  return(0);
}



/*============================================================================*/
/* POP3 server reply check                                                    */
/* +OK   : POP_OK returned                                                    */
/* +ERR  : POP_ERROR returned                                                 */
/* other : POP_INVALID returned                                               */
/*============================================================================*/
static int
popIsOK(char *ptr)
{
  int ret = POP_INVALID;

  if( strncmp(ptr, POP_OK_STR, strlen(POP_OK_STR)) == 0 ){
    ret = POP_OK;
  }
  else if( strncmp(ptr, POP_ERR_STR, strlen(POP_ERR_STR)) == 0 ){
    ret = POP_ERROR;
  }

  return(ret);
}



/*============================================================================*/
/* Get APOP Key                                                               */
/*============================================================================*/
static int 
popApopGetKey(char *ptr, char *apop_key)
{

  int  i;
  char *pos;

  if ((pos = (char *)strrchr(ptr, '<'))==NULL) {
    return 1;
  } 

  for (i = 0; i < 128; i++) {
    apop_key[i] = *pos;
    if (*pos == '>') break;
    pos++;
  }

  return 0;
}



/*============================================================================*/
/* POP Line Command                                                           */
/*============================================================================*/
static int
popLineCommand(char *ptr)
{
  int funcret;
  char buffer[POP_MAX_LINE];
  int status;

  funcret = popWriteLine(ptr);
  if( funcret != 0 ){
    return(1);
  }

  funcret = popReadLine(buffer, POP_MAX_LINE);
  if( funcret != 0 ){
    return(2);
  }

  status = popIsOK(buffer);
  if( status != POP_OK ){
    return(3);
  }

  return(0);
}


/*============================================================================*/
/* APOP command                                                               */
/*============================================================================*/
static int
popApop(char *user, char *passwd, char *apop_key)
{
  char buffer[POP_MAX_CMD];
  char md5_passwd[MD5_MAX_LEN];

  strcat(apop_key, passwd);

  /*MD5Data(apop_key, strlen(apop_key), md5_passwd);*/
  strcpy(md5_passwd, MD5Digest((unsigned char *)apop_key));

  sprintf(buffer, "APOP %s %s\r\n", user, md5_passwd);
  return( popLineCommand(buffer) );
}




/*============================================================================*/
/* POP USER command                                                           */
/*============================================================================*/
int
popUser(char *user)
{
  char buffer[POP_MAX_CMD];

  sprintf(buffer, "USER %s\r\n", user);
  return( popLineCommand(buffer) );
}



/*============================================================================*/
/* POP PASS command                                                           */
/*============================================================================*/
int
popPasswd(char *passwd)
{
  char buffer[POP_MAX_CMD];

  sprintf(buffer, "PASS %s\r\n", passwd);
  return( popLineCommand(buffer) );
}



/*============================================================================*/
/* POP DELE command                                                           */
/*============================================================================*/
int
popDelete(int no)
{
  int funcret;
  char buffer[POP_MAX_CMD];

  sprintf(buffer, "DELE %d\r\n", no);
  return( popLineCommand(buffer) );
}



/*============================================================================*/
/* POP RSET command                                                           */
/*============================================================================*/
int
popRset()
{
  int funcret;

  funcret = popLineCommand("RSET\r\n");
  return(funcret);
}



/*============================================================================*/
/* POP STAT command                                                           */
/*============================================================================*/
int
popStat(int *mail, long *size)
{
  int funcret;
  char buffer[POP_MAX_LINE];
  char p[32];
  char dummy[32];
  char dummy_mail[32];
  char dummy_size[32];
  int status;

  funcret = popWriteLine("STAT\r\n");
  if( funcret != 0 ){
    return(1);
  }

  funcret = popReadLine(buffer, POP_MAX_LINE);
  if( funcret != 0 ){
    return(2);
  }

  status = popIsOK(buffer);
  if( status != POP_OK ){
    return(3);
  }

  /* "+OK <Number> <Size>" */
  sscanf(buffer, "%s %s %s", dummy, dummy_mail, dummy_size);
  *mail = atoi(dummy_mail);
  *size = atol(dummy_size);

  return(0);
}



/*============================================================================*/
/* POP UIDL command                                                           */
/*============================================================================*/
int
popUidl(int no, char *uniqid, int size)
{
  int funcret;
  char buffer[POP_MAX_LINE];
  char dummy[32];
  int  dummyno;
  int status;

  if( size >= POP_MAX_UIDL+1 ){
    return(4);
  }

  sprintf(buffer, "UIDL %d\r\n", no);
  funcret = popWriteLine(buffer);
  if( funcret != 0 ){
    return(1);
  }

  funcret = popReadLine(buffer, POP_MAX_LINE);
  if( funcret != 0 ){
    return(2);
  }

  status = popIsOK(buffer);
  if( status != POP_OK ){
    return(3);
  }

  /* "+OK <Number> <uniq-id(32byte)>" */
  sscanf(buffer, "%s %d %s", dummy, &dummyno, uniqid);

  return(0);
}



/*============================================================================*/
/* POP LIST command                                                           */
/*============================================================================*/
int
popList(PopList **list, int *size)
{
  int funcret;
  char buffer[POP_MAX_LINE];
  int status;
  int cnt;

  funcret = popWriteLine("LIST\r\n");
  if( funcret != 0 ){
    return(funcret);
  }

  memset(list, 0, sizeof(PopList)*(*size));
  memset(buffer, 0, POP_MAX_LINE);
  cnt = 0;
  while( 1 ){
    funcret = popReadLine(buffer, POP_MAX_LINE);
    if( funcret != 0 ){
      return(funcret);
    }
    if( cnt >= *size ){
      break;
    }
    if( strcmp(buffer, ".\r\n") != 0 ){
      break;
    }
    sscanf(buffer, "%d %d", &((*list+cnt)->no), &((*list+cnt)->size));
    cnt++;
  }

  *size = cnt;

  return(0);
}



/*============================================================================*/
/* POP RETR command                                                           */
/*============================================================================*/
int
popRetr(int no, FILE *fp)
{
  int funcret;
  char buffer[POP_MAX_LINE];
  int status;
  int cnt;

  sprintf(buffer, "RETR %d\r\n", no);
  funcret = popWriteLine(buffer);
  if( funcret != 0 ){
    return(funcret);
  }

  while( 1 ){
    funcret = popReadLine(buffer, POP_MAX_LINE);
    if( funcret != 0 ){
      return(funcret);
    }
    if( strcmp(buffer, ".\r\n") == 0 ){
      break;
    }
    if( fputs(buffer, fp) == EOF ){
      return(4);
    }
    fflush(fp);
  }

  return(0);
}



/*============================================================================*/
/* POP TOP command                                                            */
/*============================================================================*/
int
popTop(int no, int line,  FILE *fp)
{
  int funcret;
  char buffer[POP_MAX_LINE];
  int status;
  int cnt;

  sprintf(buffer, "TOP %d %d\r\n", no, line);
  funcret = popWriteLine(buffer);
  if( funcret != 0 ){
    return(funcret);
  }

  while( 1 ){
    funcret = popReadLine(buffer, POP_MAX_LINE);
    if( funcret != 0 ){
      return(funcret);
    }
    if( strcmp(buffer, ".\r\n") == 0 ){
      break;
    }
    if( fputs(buffer, fp) == EOF ){
      return(4);
    }
  }

  return(0);
}

/*============================================================================*/
/* POP NOOP command                                                           */
/*============================================================================*/
int
popNoop()
{
  int funcret;

  funcret = popLineCommand("NOOP\r\n");
  return(funcret);
}



/*============================================================================*/
/* POP QUIT command                                                           */
/*============================================================================*/
int
popQuit()
{
  int funcret;

  funcret = popWriteLine("QUIT\r\n");
  return(funcret);
}


/*============================================================================*/
/* Connect POP server                                                         */
/*============================================================================*/
int
popConnect(char *server, int port, char *apop_key)
{
  int    funcret;
  struct hostent *host;
  struct sockaddr_in pop_addr;
  char   buffer[POP_MAX_LINE];
  int    status;
  int    true;

  /*--------------------------------------------------------------------------*/
  /* Get hostent data from server name                                        */
  /*--------------------------------------------------------------------------*/
  host = gethostbyname(server);
  if( host == NULL ){
    return(1);
  }

  /*--------------------------------------------------------------------------*/
  /* Create socket                                                            */
  /*--------------------------------------------------------------------------*/
  Sockfd = socket(AF_INET, SOCK_STREAM, 0);
  if( Sockfd == -1 ){
    return(2);
  }

  /*--------------------------------------------------------------------------*/
  /* Connect POP server                                                       */
  /*--------------------------------------------------------------------------*/
  pop_addr.sin_family = AF_INET;
  pop_addr.sin_port = htons(port);
  pop_addr.sin_addr.s_addr = *(long *)(host->h_addr_list[0]);

  funcret = connect(Sockfd, (struct sockaddr *)&pop_addr, sizeof(struct sockaddr_in));
  if( funcret == -1 ){
    return(3);
  }

  /*--------------------------------------------------------------------------*/
  /* Associate FILE pointer                                                   */
  /*--------------------------------------------------------------------------*/
  Rfp = (FILE *)fdopen(Sockfd, "r");
  Wfp = (FILE *)fdopen(Sockfd, "w");
  if( Rfp == NULL || Wfp == NULL ){
    popClose();
    return(4);
  }

  /*--------------------------------------------------------------------------*/
  /* Read initial message from pop server.                                    */
  /*--------------------------------------------------------------------------*/
  funcret = popReadLine(buffer, POP_MAX_LINE);
  if( funcret != 0 ){
    return(5);
  }

  /*--------------------------------------------------------------------------*/
  /* Confirm initial message.                                                 */
  /*--------------------------------------------------------------------------*/
  status = popIsOK(buffer);
  if( status != POP_OK ){
    return(6);
  }

  /*--------------------------------------------------------------------------*/
  /* Get apop auth key.                                                       */
  /*--------------------------------------------------------------------------*/
  if(Mascot->mail.type==MAIL_APOP){
    funcret = popApopGetKey(buffer, apop_key);
    if( funcret != 0 ){
      return(7);
    }
  }

 END_FUNC:
  return(0);
}



/*============================================================================*/
/* Login pop server                                                           */
/*============================================================================*/
int
popLogin(char *user, char *pass, char *apop_key)
{
  int funcret;

  if(Mascot->mail.type==MAIL_APOP){
  /*------------------------------------------------------------------------*/
  /* Try to APOP login                                                      */
  /*------------------------------------------------------------------------*/
    funcret = popApop(user, pass, apop_key);
    if( funcret == 0 ){
      return(0);
    }
  }
  else if(Mascot->mail.type==MAIL_POP3){
  /*------------------------------------------------------------------------*/
  /* Try to Normal Login                                                    */
  /*------------------------------------------------------------------------*/
    funcret = popUser(user);
    if( funcret != 0 ){
      return(funcret+10);
    }
    funcret = popPasswd(pass);
    if( funcret != 0 ){
      return(funcret+20);
    }
  }

  return(0);
}



/*============================================================================*/
/* Check mail number                                                          */
/*============================================================================*/
int
popCheckMailNum(int *number, char *uidl, int *num)
{
  int  funcret;
  int  s, e;
  long size;
  char chk_uidl[POP_MAX_UIDL+1];

  /*------------------------------------------------------------------------*/
  /* Get Total Mail Number                                                  */
  /*------------------------------------------------------------------------*/
  funcret = popStat(&e, &size);
  if( funcret != 0 ){
    return(funcret+10);
  }

  if( e == 0 || uidl == NULL ){
    *number = *num = e;
    return(0);
  }

  /*------------------------------------------------------------------------*/
  /* Check UIDL                                                             */
  /*------------------------------------------------------------------------*/
  for( s = e; s > 0; s-- ){
    funcret = popUidl(s, chk_uidl, POP_MAX_UIDL);
    if( funcret == 2 ){ /* -ERR Humm. May be not support UIDL. */
      s = e;
      break;
    }
    else if( funcret != 0 ){
      return(funcret+20);
    }
    if( strcmp(uidl, chk_uidl) == 0 ){
      break;
    }
  }
  if( s == 0 ){
    *number = e;
  }
  else {
    *number = e - s;
  }

  *num = e;
  funcret = popUidl(e, uidl, POP_MAX_UIDL);

  return(0);
}



/*============================================================================*/
/* Get mail                                                                   */
/*============================================================================*/
int
popGetMail(int number, FILE *fp, char *uidl)
{
  int funcret;
  char chk_uidl[POP_MAX_UIDL+1];

  funcret = popRetr(number, fp);
  if( funcret != 0 ){
    return(funcret + 10);
  }

  if( uidl != NULL ){
    funcret = popUidl(number, chk_uidl, POP_MAX_UIDL);
    if( funcret == 2 ){ /* -ERR Humm. May be not support UIDL. */
      uidl[0] = '\0';
    }
    else if( funcret != 0 ){
      return(funcret+20);
    }
    else {
      strcpy(uidl, chk_uidl);
    }
  }

  return(0);
}



/*============================================================================*/
/* Disconnect POP3 server                                                     */
/*============================================================================*/
int
popClose()
{
  fclose(Rfp);
  fclose(Wfp);
  shutdown(Sockfd, 0);
  close(Sockfd);
  return(0);
} 



/*============================================================================*/
/* Disconnect POP server                                                      */
/*============================================================================*/
int
popDisconnect()
{
  (void)popQuit();
  (void)popClose();
  return(0);
}
