//  MaCoPiX = Mascot Construnctive Pilot for X
//                                (ActX / Gtk+ Evolution)
//    A unified desktop mascot program
//    for UNIX / X Window System with Gdk Environment
//
//  mail.c  : Biff functions for MaCoPiX
//       (POP3 /APOP and local spool)
//                                copyright K.Chimari 2002-4
//
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
// 
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
// 
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//

#include "main.h"
#include <dirent.h>
#include <sys/wait.h>
#include <signal.h>

#undef POP_DEBUG

#define BUFFSIZE 200
#define BUFFSIZE5 1000
#define HEADER_MAX 256


enum{ NODAT, STRING, ID, SYMBOL, RESERVE } TokenType;

void biff_init();

gint SetMailChecker();
gint MailChecker();

GtkWidget * make_biff_pix();
void display_biff_balloon();

void mail_check();
void mail_check_mbox();
void mail_check_qmail();
void mail_check_pop3();
int get_pop3();
gchar * fs_get_mbox();
gchar * fs_get_procmail();
gchar * fs_get_qmail();
void fs_get_pop3();
void pop3_signal();
void pop3_data_read();

char* strbuf();

static char* code_conv();
void  set_biff_filter();
void  set_token();
int get_token2();

void make_fs_max();
void create_biff_dialog();
static void close_biff();
static void mailer_start();

gchar* set_mhdir();


//
extern void ext_play();
extern void sound_play();

extern GtkWidget *biff_pix;
extern typMascot *Mascot;
extern gboolean flag_balloon;


// global argument
int pop3_fd[2];
gchar *pop_froms = NULL;
gchar *biff_filter[12];
char token_buf[BUFFSIZE+1];
static char *token_p;
time_t former_newest=0;


void biff_init(typMascot *mascot)
{
#ifdef POP_DEBUG
  fprintf(stderr,"\nBiff init\n");
#endif
  mascot->mail.size=0;
  mascot->mail.last_check=0;
  mascot->mail.count=0;
  mascot->mail.fetched_count = 0;    // POP3 fetched fs this access
  mascot->mail.displayed_count = 0;  // POP3 displayed fs
  mascot->mail.status = NO_MAIL;
  mascot->mail.pop3_fs_status = POP3_OK_NORMAL;
  g_free(pop_froms);
  pop_froms=strbuf(NULL);
}    


gint SetMailChecker(gpointer gdata){
  typMascot *mascot;

  mascot=(typMascot *)gdata;

  if(mascot->mail.proc_id>0){
    gtk_timeout_remove((guint)mascot->mail.proc_id);
    fflush(stdin);
    fflush(stdout);
  }
  gtk_widget_unmap(biff_pix);

  if(mascot->mail.flag){
    biff_init(mascot);
    MailChecker(mascot);
    mascot->mail.proc_id
      =(gint)gtk_timeout_add((guint32)(mascot->mail.interval*1000),MailChecker,
    		       (gpointer)mascot);
#ifdef POP_DEBUG
    fprintf(stderr,"proc_id=%d\n",mascot->mail.proc_id);
#endif
  }
}


gint MailChecker(gpointer gdata){
  typMascot *mascot;

  mascot=(typMascot *)gdata;

  if(mascot->flag_menu) return;

  gdk_flush();

  ext_play(mascot,mascot->mail.polling);  
  mail_check(mascot);

  gdk_flush();

  display_biff_balloon(mascot);

  gdk_flush();

}


GtkWidget * make_biff_pix(){
  GtkWidget *biff_vbox;
  GtkWidget *biff_drawing_area;

#ifdef USE_GTK2
  biff_pix = gtk_window_new(GTK_WINDOW_TOPLEVEL);
#else
  biff_pix = gtk_window_new(GTK_WINDOW_DIALOG);
#endif
  gtk_widget_set_app_paintable(biff_pix, TRUE);
  gtk_widget_set_events(GTK_WIDGET (biff_pix), 
			GDK_FOCUS_CHANGE_MASK | 
			GDK_BUTTON_MOTION_MASK | 
			GDK_BUTTON_RELEASE_MASK | 
			GDK_BUTTON_PRESS_MASK | 
			GDK_EXPOSURE_MASK);
  gtk_widget_realize(biff_pix);
  gdk_window_set_decorations(biff_pix->window, 0);
  gdk_window_set_override_redirect(biff_pix->window,TRUE);

  biff_vbox = gtk_hbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER (biff_pix), biff_vbox);
  gtk_widget_show(biff_vbox);

  biff_drawing_area = gtk_drawing_area_new();
  gtk_drawing_area_size (GTK_DRAWING_AREA(biff_drawing_area), 1, 1);
  gtk_box_pack_start(GTK_BOX(biff_vbox), biff_drawing_area,
		     TRUE, TRUE, 0);
  gtk_widget_show(biff_drawing_area);

  return(biff_drawing_area);
}


void display_biff_balloon(typMascot *mascot)
{
  if((mascot->mail.status != NO_MAIL)
     &&(mascot->mail.status != POP3_ERROR)){
    MoveBiffPix(mascot,mascot->x,mascot->y);
    gtk_widget_map(biff_pix);
  }
  else if(mascot->mail.status == POP3_ERROR){
    gtk_widget_unmap(biff_pix);

    if(flag_balloon==FALSE){
      mascot->balseq=0;
      mascot->bal_mode=BALLOON_POPERROR;
      DoBalloon(mascot); 
      flag_balloon=TRUE;
    }
  }
  else{
    gtk_widget_unmap(biff_pix);
  }
  if(mascot->mail.status==NEW_MAIL){
    sound_play(mascot,mascot->mail.sound);

    if((mascot->mail.word)&&(flag_balloon==FALSE)) {
      mascot->balseq=0;
      mascot->bal_mode=BALLOON_MAIL;
      DoBalloon(mascot); 
      flag_balloon=TRUE;
    }
  }
}


void mail_check(typMascot *mascot){
  static struct sigaction act;

  switch(mascot->mail.type){     
  case MAIL_LOCAL:
  case MAIL_PROCMAIL:
    mail_check_mbox(mascot);
    break;
  case MAIL_POP3: 
  case MAIL_APOP: 
    act.sa_handler=pop3_signal;
    sigemptyset(&act.sa_mask);
    act.sa_flags=0;
    sigaction(SIGUSR1, &act,0);
    mail_check_pop3(mascot);
    break;
  case MAIL_QMAIL:
    mail_check_qmail(mascot);
    break;
  }

}


void mail_check_mbox(typMascot *mascot){
    struct stat t;
    DIR *dp;
    struct dirent *entry;
    time_t newest_time=0;
    int filenum=0;
    int mc=0;
    FILE *fp;
    char buf[BUFFSIZE];

    if (!stat(mascot->mail.file, &t)){
      if(t.st_size == 0){ // ᥤե륵
	mascot->mail.status = NO_MAIL;
	mascot->mail.count = 0;
      }
      else if (t.st_size < mascot->mail.size){
	// mailbox smaller in size; some mails have been deleted
	mascot->mail.status = OLD_MAIL;
      }
      else if (t.st_atime > t.st_mtime){ 
	// mailbox read after most recent write
	mascot->mail.status = OLD_MAIL;
      }
      else if (t.st_size > mascot->mail.size){
	// mailbox modified after most recent read, and larger in size
	// this implies the arrival of some new mails
	mascot->mail.status = NEW_MAIL;
	// get mail count
	if((fp=fopen(mascot->mail.file,"r"))!= NULL){
	  do{
	    if(fgets(buf,BUFFSIZE-1,fp)==NULL) break;
	    if(strncmp(buf,"From ",5)==0){
	      mc++;
	    }
	  } while (1);
	  fclose(fp);
	  mascot->mail.count = mc;
	}
      }
      else if ((t.st_size == mascot->mail.size)&&
	       ((mascot->mail.status == NEW_MAIL)||
		(mascot->mail.status ==KEEP_NEW_MAIL))){
	mascot->mail.status=KEEP_NEW_MAIL;
      }
      else{
	mascot->mail.status = NO_MAIL;          // no such mailbox
	mascot->mail.count = 0;
      }
    }
    else{ // ᥤե뤬¸ߤʤȤ
      mascot->mail.status = NO_MAIL;  
      mascot->mail.count = 0;
    }
    mascot->mail.size = t.st_size;
}


void mail_check_qmail(typMascot *mascot){
    struct stat t;
    DIR *dp;
    struct dirent *entry;
    char tmp[256];
    time_t newest_time=0;
    int filenum=0;

    if ((dp=opendir(mascot->mail.file))==NULL){
      mascot->mail.status = NO_MAIL;  
      mascot->mail.count = 0;
      return;
    }
    
    while((entry=readdir(dp))!=NULL){
      if(entry->d_name[0]!='.'){
	sprintf(tmp,"%s/%s",mascot->mail.file,entry->d_name);
	if (!stat(tmp, &t)){
	  filenum++;
	  if (t.st_mtime>newest_time){ 
	    newest_time=t.st_mtime;
	  }
	}
      }
    }
	
    closedir(dp);
	
    if(filenum==0){
      mascot->mail.status = NO_MAIL;
      mascot->mail.count = 0;
    }
    else{
      if(newest_time==former_newest){
	mascot->mail.status = KEEP_NEW_MAIL;  
      }
      else if(newest_time>former_newest){
	mascot->mail.status = NEW_MAIL;  
      }
      else{
	mascot->mail.status = OLD_MAIL;  
      }
    }
    former_newest=newest_time;
    mascot->mail.count = filenum;
}


void mail_check_pop3(typMascot *mascot){

  int status = 0;
  static int pid;
  char *buff = NULL;
  char buf[BUFFSIZE];

  if(mascot->mail.pop_child_fl){
    return ;
  }
  mascot->mail.pop_child_fl=TRUE;

  if(pipe(pop3_fd)==-1) {
    fprintf(stderr,"pipe open error\n");
  }
  waitpid(pid,0,WNOHANG);
 
  if( (pid = fork()) <0){
    fprintf(stderr,"fork error\n");
  }
  else if(pid ==0) {
#ifdef POP_DEBUG
    fprintf(stderr,"child\n");
#endif
    status = get_pop3(mascot);
#ifdef POP_DEBUG
    fprintf(stderr,"get pop3 end\n");
    fprintf(stderr,"pop3 status = %d\n", mascot->mail.pop3_fs_status);
#endif
    g_free(buff);
    buff = strbuf(NULL);  // stand by buff
    sprintf(buf,"%d\n",mascot->mail.status); // Write mail status
    buff = strbuf(buf);
    if(status != POP3_ERROR){                  // POP status is GOOD?
      sprintf(buf,"%d\n",mascot->mail.count);  // write total mail count
      buff = strbuf(buf);
      sprintf(buf,"%d\n",mascot->mail.fetched_count);  // write fetched mail count
      buff = strbuf(buf);
      sprintf(buf,"%d\n",mascot->mail.pop3_fs_status);  // write POP status
      buff = strbuf(buf);
      buff = strbuf(pop_froms);                // write fs data
    }
    if(close(STDOUT_FILENO)==-1) fprintf(stderr,"pipe close error\n");
    if(dup2(pop3_fd[1],STDOUT_FILENO)==-1)
       fprintf(stderr,"pipe duplicate error\n");
    if(close(pop3_fd[0])==-1) fprintf(stderr,"pipe close error\n");

#ifdef POP_DEBUG
    fprintf(stderr,"pipe open/close end\n");
    //    fprintf(stderr,"%s",buff);
    fprintf(stderr,"buff size = %d\n", strlen(buff));
#endif
    printf ("%s",buff);
#ifdef POP_DEBUG
    fprintf(stderr, "child end\n");
#endif
    kill(getppid(), SIGUSR1);
    fflush(stdout);
    fflush(stdin);
     _exit(-1);
  }
  else{
#ifdef POP_DEBUG
    fprintf(stderr, "Parent: --------------------------------\n");
    fprintf(stderr, "Parent: forked a child with pid = %d\n", (int)pid);
    fprintf(stderr, "Parent: mail status = %d\n", mascot->mail.status);
#endif
  }
}


int get_pop3(typMascot *mascot)
{
  signed int ret;
  int num;
  long size;
  char apop_key[BUFFSIZE];

#ifdef POP_DEBUG
  fprintf(stderr,"Conncet %s : %d\n", mascot->mail.pop_server, mascot->mail.pop_port);
#endif
  memset(apop_key, '\0', BUFFSIZE);
  ret = popConnect(mascot->mail.pop_server, mascot->mail.pop_port, apop_key);
  if( ret != 0 ){
    fprintf(stderr, "ERR: popConnect() [ret=%d]\n", ret);
    mascot->mail.status = POP3_ERROR;
    return(POP3_ERROR);
  }

#ifdef POP_DEBUG
  fprintf(stderr,"POP LOGIN %s %s %s\n",mascot->mail.pop_id, mascot->mail.pop_pass,
	 apop_key);
#endif
  ret = popLogin(mascot->mail.pop_id, mascot->mail.pop_pass, apop_key);
  if( ret != 0 ){
    fprintf(stderr, "ERR: popLogin() [ret=%d]\n", ret);
    mascot->mail.status = POP3_ERROR;
    popQuit();
    popClose();
    return(POP3_ERROR);
  }

#ifdef POP_DEBUG
  fprintf(stderr,"POP STAT\n");
#endif
  ret = popStat(&num, &size);
  if( ret != 0 ){
    fprintf(stderr, "ERR: popStat() [ret=%d]\n", ret);
    mascot->mail.status = POP3_ERROR;
    popQuit();
    popClose();
    return(POP3_ERROR);
  }
#ifdef POP_DEBUG
  fprintf(stderr,"mail: %d, size: %ld\n", num, size);
#endif
  if(num == 0){  // ᡼̵
    mascot->mail.status = NO_MAIL;
    mascot->mail.fetched_count = 0;
    mascot->mail.pop3_fs_status = POP3_OK_NORMAL;
    g_free(pop_froms);
    pop_froms = strbuf(NULL);
  }
  else{
    if(num == mascot->mail.count){  // ᡼ʤ
      mascot->mail.status = KEEP_NEW_MAIL;
      mascot->mail.fetched_count = 0;
      mascot->mail.pop3_fs_status = POP3_OK_NORMAL;
      g_free(pop_froms);  //  fs 򥯥ꥢ
      pop_froms = strbuf(NULL);
    }
    else if(num > mascot->mail.count){  // ᡼
      mascot->mail.status = NEW_MAIL;
      fs_get_pop3(num,mascot);      
    }
    else{  // ᡼äƤ餹٤ƿ᡼ȸʤ
      mascot->mail.status = NEW_MAIL;      
      pop_froms = strbuf(NULL);
      mascot->mail.count = 0;
      fs_get_pop3(num,mascot);
    }
  }
#ifdef POP_DEBUG
  fprintf(stderr,"data read end\n");
#endif
  popQuit();
  popClose();
#ifdef POP_DEBUG
  fprintf(stderr,"pop quit\n");
  //  fprintf(stderr,"data3 = %s\n",pop_froms);
#endif
  mascot->mail.count = num;
  return(mascot->mail.status);
}


void pop3_signal(int sig){
#ifdef POP_DEBUG
  fprintf(stderr, "get child end\n");
#endif
  pop3_data_read(Mascot);
}


void pop3_data_read(typMascot *mascot)
{
  FILE *fp;
  char buf[BUFFSIZE];
  int pre_mail_count;
  
  if(mascot->mail.pop_child_fl){
    pre_mail_count = mascot->mail.count;  // store mail count
#ifdef POP_DEBUG
    fprintf(stderr,"Parent: pop3 data read\n");
#endif
    close(pop3_fd[1]);
    if( (fp = fdopen( pop3_fd[0], "r" )) == NULL ){
      fprintf(stderr,"pipe open error\n");    
    }
    else{
      fgets( buf,BUFFSIZE-1,fp );  // read mail status
      mascot->mail.status = atoi(buf);
      if( mascot->mail.status < 0){  // POP3 status is ERR
	fprintf(stderr,"POP3 error\n");
	mascot->mail.count = 0;
	mascot->mail.spam_count = 0;
	mascot->mail.fetched_count = 0;
	mascot->mail.displayed_count = 0;
	mascot->mail.status = POP3_ERROR;
	g_free(pop_froms);
	pop_froms = strbuf(NULL);
      }
      else{
	fgets( buf,BUFFSIZE-1,fp );  // read total mail count
	mascot->mail.count = atoi(buf);
	fgets( buf,BUFFSIZE-1,fp );  // read fetched mail count
	mascot->mail.fetched_count = atoi(buf);
	fgets( buf,BUFFSIZE-1,fp );  // read pop status
	mascot->mail.pop3_fs_status = atoi(buf);

	if(mascot->mail.count == 0){
	  g_free(pop_froms);  // fs data clear
	  pop_froms = strbuf(NULL);
	  mascot->mail.displayed_count = 0;
	}
	else {
	  if(mascot->mail.pop3_fs_status == POP3_OK_NORMAL){
	    if(pre_mail_count > mascot->mail.count){  //᡼븺ä
	      // ᡼븺äľʤΤ fs clear
	      mascot->mail.displayed_count = mascot->mail.fetched_count; 
	      g_free(pop_froms);
	      pop_froms = strbuf(NULL);
	    } 
	    else {
	      mascot->mail.displayed_count += mascot->mail.fetched_count; 
	    }
	    while( fgets( buf,BUFFSIZE-1,fp ) != NULL ){
	      pop_froms = strbuf( buf );
	    }
	  }
	  else if(mascot->mail.pop3_fs_status == POP3_OK_FS_OVER){
	    //̵ fs clear Ƥ褤
	    mascot->mail.displayed_count = mascot->mail.fetched_count; 
	    g_free(pop_froms);    // fs data clear
	    pop_froms = strbuf(NULL);
	    sprintf(buf,"\n     ***** %d mails are skipped *****\n\n",
		    mascot->mail.count - mascot->mail.displayed_count);
	    // ܸˤ euc ʳlocaleǤνʤ
	    pop_froms = strbuf( buf );
	    while( fgets( buf,BUFFSIZE-1,fp ) != NULL ){
	      pop_froms = strbuf( buf );
	    }
	  }
	}
      }
      fclose( fp );
    }
    close(pop3_fd[0]);

    mascot->mail.pop_child_fl=FALSE;
#ifdef POP_DEBUG
    fprintf(stderr,"Parent: status = %d\n",mascot->mail.status);
    fprintf(stderr,"Parent: pop3 status = %d\n",mascot->mail.pop3_fs_status);
    fprintf(stderr,"Parent: count = %d\n",mascot->mail.count);
    fprintf(stderr,"Parent: fetched count = %d\n",mascot->mail.fetched_count);
    fprintf(stderr,"Parent: disped count = %d\n",mascot->mail.displayed_count);
    //    fprintf(stderr,"data2 = %s\n",pop_froms);
#endif
  }
}


// ᡼ܥå From: Subject: 
gchar * fs_get_mbox(typMascot *mascot){
  FILE *fp;
  char buf[BUFFSIZE],bufs[BUFFSIZE5];
  gchar *froms=NULL;
  char *p;
  gboolean ed_fl=FALSE, spam_flag;
  fpos_t pos;
  int i;
  
  char *f, *s, *sdel;
  
  froms=strbuf(NULL);
  
  if((fp=fopen(mascot->mail.file,"r"))!= NULL){
    do{
      if(fgets(buf,BUFFSIZE-1,fp)==NULL) break;
      if(strncmp(buf,"From ",5)==0){  // ĤΥ᡼λϤޤ
	f=s=NULL;
	spam_flag = FALSE;
	for(;;){
	  if((p=fgets(buf,BUFFSIZE-1,fp))==NULL
	     ||buf[0]=='\n'||buf[0]=='\0')
	    break;
	  if((strncmp(buf,"From:",5)==0)||
	     (strncmp(buf,"FROM:",5)==0))
	    {
	      strcpy(bufs,buf);
	      for(i=0;i<5;i++){
		fgetpos(fp,&pos);
		if((p=fgets(buf,BUFFSIZE-1,fp))==NULL
		   ||buf[0]=='\n'||buf[0]=='\0'){
		  ed_fl=TRUE;
/* tnaka start */
if( strlen(bufs)) {
	f = g_strdup(bufs);
	fsetpos(fp,&pos);
}
/* tnaka end */
		  break;
		}
		if(bufs[strlen(bufs)-1]!='\n'){
		  strcat(bufs,buf); /* Ԥ200byteʾ */
		}
		else if((buf[0]==' ') || (buf[0]=='\t')){
		  bufs[strlen(bufs)-1]='\0';
		  sdel = &buf[1];     /* \t  */
		  strcat(bufs,sdel); /* ʣ */
		}
		else{
		  f=g_strdup(bufs);
		  fsetpos(fp,&pos);
		  break;
		}
	      }
	      if(ed_fl) break;
	    }
	  else if((strncmp(buf,"Subject:",8)==0)||
		  (strncmp(buf,"SUBJECT:",8)==0))
	    {
	      strcpy(bufs,buf);
	      for(i=0;i<5;i++){
		fgetpos(fp,&pos);
		if((p=fgets(buf,BUFFSIZE-1,fp))==NULL
		   ||buf[0]=='\n'||buf[0]=='\0'){
		  ed_fl=TRUE;
/* tnaka start */
if( strlen(bufs)) {
	s = g_strdup(bufs);
	fsetpos(fp,&pos);
}
/* tnaka end */
		  break;
		}
		if(bufs[strlen(bufs)-1]!='\n'){
		  strcat(bufs,buf); /* Ԥ200byteʾ */
		}
		else if((buf[0]==' ') || (buf[0]=='\t')){
		  bufs[strlen(bufs)-1]='\0';
		  sdel = &buf[1];     /* \t  */
		  strcat(bufs,sdel); /* ʣ */
		}
		else{
		  s=g_strdup(bufs);
		  fsetpos(fp,&pos);
		  break;
		}
	      }
	      if(ed_fl) break;
	    } 
	  else if(strncmp(buf, mascot->mail.spam_mark, 
			  strlen(mascot->mail.spam_mark))==0){
	    spam_flag = TRUE;
	    mascot->mail.spam_count++;
	  }
	}
	
	//
	// From: Subject:
	// νɽ
	//
	if ( (!spam_flag) || (!mascot->mail.spam_check) ){
	  if (f) {
	    froms=strbuf(f);
	    g_free(f);
	  }
	  else{
	    froms=strbuf("(no From: in original)\n");
	  }
	  if (s) {
	    froms=strbuf(" ");
	    froms=strbuf(s);
	    g_free(s);
	  }
	  else{
	    froms=strbuf(" (no Subject: in original)\n");
	  }
	}
ed_fl = FALSE;  /* tnaka */
      }
      while (p != NULL && buf[0] != '\n' && buf[0] != '\0')
	p = fgets(buf, BUFFSIZE - 1, fp);
    } while (1);
    fclose(fp);
  }
  return(froms);
}


// MH + Procmail 
gchar *  fs_get_procmail(typMascot  *mascot){
    FILE *fp,*fp_folder;
    gchar buf[BUFFSIZE],tmp[10],bufs[BUFFSIZE5];
    gchar *folder_file,folder_tmp[BUFFSIZE];
    gchar *froms=NULL, *p;
    gboolean ed_fl=FALSE;
    fpos_t pos;
    int i;
    gchar *f, *s, *sdel;


    froms=strbuf(NULL);

    if((fp=fopen(mascot->mail.file,"r"))==NULL){
	return;
    }
    
    while(!feof(fp)){
	f=s=NULL;
	if(fgets(buf,BUFFSIZE-1,fp)==NULL) break;
	if(strncmp(buf,"  Folder:",9)==0){
	    sscanf(buf,"%s%s",tmp,folder_tmp);
	    folder_file=g_strconcat(set_mhdir(),"/",
				    folder_tmp,NULL);

	    if((fp_folder=fopen(folder_file,"r"))!=NULL){
		while(!feof(fp_folder)){
		    if((p=fgets(buf,BUFFSIZE-1,fp_folder))==NULL
		       ||buf[0]=='\n'||buf[0]=='\0')
			break;
		    if((strncmp(buf,"From:",5)==0)||
		       (strncmp(buf,"FROM:",5)==0))
		    {
			strcpy(bufs,buf);
			for(i=0;i<5;i++){
			    fgetpos(fp_folder,&pos);
			    if((p=fgets(buf,BUFFSIZE-1,fp_folder))==NULL
			       ||buf[0]=='\n'||buf[0]=='\0'){
				ed_fl=TRUE;
				break;
			    }
			    if(bufs[strlen(bufs)-1]!='\n'){
				strcat(bufs,buf); /* Ԥ200byteʾ */
			    }
			    else if((buf[0]==' ') || (buf[0]=='\t')){
				bufs[strlen(bufs)-1]='\0';
				sdel = &buf[1];     /* \t  */
				strcat(bufs,sdel); /* ʣ */
			    }
			    else{
			        f=g_strdup(bufs);
				fsetpos(fp_folder,&pos);
				break;
			    }
			}
			if(ed_fl) break;
		    }
		    else if((strncmp(buf,"Subject:",8)==0)||
			    (strncmp(buf,"SUBJECT:",8)==0))
		    {
			strcpy(bufs,buf);
			for(i=0;i<5;i++){
			    fgetpos(fp_folder,&pos);
			    if((p=fgets(buf,BUFFSIZE-1,fp_folder))==NULL
			       ||buf[0]=='\n'||buf[0]=='\0'){
				ed_fl=TRUE;
				break;
			    }
			    if(bufs[strlen(bufs)-1]!='\n'){
				strcat(bufs,buf); /* Ԥ200byteʾ */
			    }
			    else if((buf[0]==' ') || (buf[0]=='\t')){
				bufs[strlen(bufs)-1]='\0';
				sdel = &buf[1];     /* \t  */
				strcat(bufs,sdel); /* ʣ */
			    }
			    else{
			        s=g_strdup(bufs);
				fsetpos(fp_folder,&pos);
				break;
			    }
			}
			if(ed_fl) break;
		    } 
		}
		fclose(fp_folder);
		//
		// From: Subject:
		// νɽ
		//
		if (f) {
		    froms=strbuf(f);
		    g_free(f);
		}
		else{
		    froms=strbuf(" (no From: in original)\n");
		}
		if (s) {
		    froms=strbuf(" ");
		    froms=strbuf(s);
		    g_free(s);
		}
		else{
		    froms=strbuf(" (no Subject: in original)\n");
		}
	    }
	}
	
    }
    fclose(fp);

    g_free(folder_file);

    return(froms);
}	


// Qmail 
gchar * fs_get_qmail(typMascot *mascot){
    FILE *fp_folder;
    gchar buf[BUFFSIZE],tmp[10],bufs[BUFFSIZE5];
    gchar folder_file[BUFFSIZE],folder_tmp[BUFFSIZE];
    gchar *froms=NULL, *p;
    gboolean ed_fl=FALSE, spam_flag;
    fpos_t pos;
    int i;
    DIR *dp;
    struct dirent *entry;

    gchar *f, *s, *sdel;

    froms=strbuf(NULL);

    if ((dp=opendir(mascot->mail.file))==NULL){
	return;
    }	
    

    while((entry=readdir(dp))!=NULL){
	f=s=NULL;
	if(entry->d_name[0]!='.'){
	    sprintf(folder_file,"%s/%s",mascot->mail.file,entry->d_name);

	    if((fp_folder=fopen(folder_file,"r"))!=NULL){
	        spam_flag = FALSE;
                while(!feof(fp_folder)){
		  if((p=fgets(buf,BUFFSIZE-1,fp_folder))==NULL
		     ||buf[0]=='\n'||buf[0]=='\0')
		    break;

		  if(strncmp(buf, mascot->mail.spam_mark, 
			     strlen(mascot->mail.spam_mark))==0){
		    spam_flag = TRUE;
		    mascot->mail.spam_count++;
		  }
		}
		if( (spam_flag) && (mascot->mail.spam_check) ){
		  fclose(fp_folder);
		  continue;
		}

		rewind(fp_folder);
		while(!feof(fp_folder)){
		    if((p=fgets(buf,BUFFSIZE-1,fp_folder))==NULL
		       ||buf[0]=='\n'||buf[0]=='\0')
			break;
		    if((strncmp(buf,"From:",5)==0)||
                       (strncmp(buf,"FROM:",5)==0))
                    {
			strcpy(bufs,buf);
			for(i=0;i<5;i++){
			    fgetpos(fp_folder,&pos);
			    if((p=fgets(buf,BUFFSIZE-1,fp_folder))==NULL
			       ||buf[0]=='\n'||buf[0]=='\0'){
				ed_fl=TRUE;
				break;
			    }
			    if(bufs[strlen(bufs)-1]!='\n'){
			      strcat(bufs,buf); // Ԥ200byteʾ
			    }
			    else if((buf[0]==' ') || (buf[0]=='\t')){
				bufs[strlen(bufs)-1]='\0';
				sdel = &buf[1];     // \t 
				strcat(bufs,sdel); // ʣ
			    }
			    else{
			        f=g_strdup(bufs);
				fsetpos(fp_folder,&pos);
				break;
			    }
			}
			if(ed_fl) break;
		    }
		    else if((strncmp(buf,"Subject:",8)==0)||
			    (strncmp(buf,"SUBJECT:",8)==0))
                    {
			strcpy(bufs,buf);
			for(i=0;i<5;i++){
			    fgetpos(fp_folder,&pos);
			    if((p=fgets(buf,BUFFSIZE-1,fp_folder))==NULL
			       ||buf[0]=='\n'||buf[0]=='\0'){
				ed_fl=TRUE;
				break;
			    }
			    if(bufs[strlen(bufs)-1]!='\n'){
				strcat(bufs,buf); /* Ԥ200byteʾ */
			    }
			    else if((buf[0]==' ') || (buf[0]=='\t')){
				bufs[strlen(bufs)-1]='\0';
				sdel = &buf[1];     /* \t  */
				strcat(bufs,sdel); /* ʣ */
			    }
			    else{
				s=g_strdup(bufs);
				fsetpos(fp_folder,&pos);
				break;
			    }
			}
			if(ed_fl) break;
		    }
		}
		fclose(fp_folder);
		//
		// From: Subject:
		// νɽ
		//
		if (f) {
		    froms=strbuf(f);
		    g_free(f);
		}
		else{
		    froms=strbuf(" (no From: in original)\n");
		}
		if (s) {
		    froms=strbuf(" ");
		    froms=strbuf(s);
		    g_free(s);
		}
		else{
		    froms=strbuf(" (no Subject: in original)\n");
		}

	    }
	}
	
    }
    closedir(dp);

    return(froms);
}


// pop3 server  From: Subject: 
void fs_get_pop3(int num, typMascot *mascot){

  int funcret;
  char buffer_header[HEADER_MAX][POP_MAX_LINE], bufs[BUFFSIZE5];
  char buffer[POP_MAX_LINE];
  int mail_cnt, header, header_line, i;
  int mail_cnt_start;
  char *f, *s, *sdel, *tmp_froms=NULL, tmp_fs[256];
  int disp=0;
  gboolean spam_flag;
  
  tmp_froms=strbuf(NULL); 

#ifdef POP_DEBUG
  fprintf(stderr,"fs read num = %d mail_count = %d\n", 
	  num, mascot->mail.count); 
#endif

  if((num - mascot->mail.count) > POP3_MAX_FS){  //overflow
    mail_cnt_start = num - POP3_MAX_FS +1;
    mascot->mail.fetched_count = POP3_MAX_FS;
    mascot->mail.pop3_fs_status = POP3_OK_FS_OVER;
  }
  else {    // under
    mail_cnt_start = mascot->mail.count +1;
    mascot->mail.fetched_count = num - mascot->mail.count;
    mascot->mail.pop3_fs_status = POP3_OK_NORMAL;
  }

#ifdef POP_DEBUG
  fprintf(stderr,"fs read num = %d mail_cnt_start = %d\n", 
	  num, mail_cnt_start); 
  fprintf(stderr,"fs mail_fetched = %d\n", mascot->mail.fetched_count); 
#endif
  for(mail_cnt = mail_cnt_start; mail_cnt <= num; mail_cnt++){

    sprintf(buffer, "TOP %d %d\r\n", mail_cnt, 0);  // get header only
#ifdef POP_DEBUG
    //    fprintf(stderr,"TOP %d %d\n", mail_cnt, 0);  // get header only
#endif
    funcret = popWriteLine(buffer);
    if( funcret != 0 ){
      fprintf(stderr,"write err = %d\n",funcret);
      return;
    }
    for(header=0; header < HEADER_MAX; header++){
      funcret = popReadLine(buffer_header[header], POP_MAX_LINE);
      if( funcret != 0 ){
	fprintf(stderr,"read err\n");
	return;
      }
      if( strcmp(buffer_header[header], ".\r\n") == 0 )      break;
      buffer_header[header][strlen(buffer_header[header])-2]=' ';  // \r delete
    }
    header_line = header-1;

    spam_flag = FALSE;
    for(header=0; header < header_line; header++){
      if(strncmp(buffer_header[header], mascot->mail.spam_mark, 
		 strlen(mascot->mail.spam_mark))==0){
	spam_flag = TRUE;
	mascot->mail.spam_count++;
      }
    }
    if( (spam_flag) && (mascot->mail.spam_check) )  continue;

    f=s=NULL;
    for(header=0; header < header_line; header++){
      if((strncmp(buffer_header[header],"From:",5)==0)||
	 (strncmp(buffer_header[header],"FROM:",5)==0))
      {
	strcpy(bufs, buffer_header[header]);
	//bufs=g_strconcat(buffer_header[header],NULL);
	for(i=1;i<5;i++){
	  if((i+header) > header_line)  break;
	  if((buffer_header[i+header][0]==' ') || (buffer_header[i+header][0]=='\t')){
	    bufs[strlen(bufs)-2]='\0';
	    sdel = &buffer_header[i+header][1];     /* \t  */
	    strcat(bufs,sdel); /* ʣ */
	    //bufs=g_strconcat(bufs,sdel,NULL); /* ʣ */
	  }
	  else{
	    f=g_strdup(bufs);
	    break;
	  }
	}
      }
      else if((strncmp(buffer_header[header],"Subject:",8)==0)||
	      (strncmp(buffer_header[header],"SUBJECT:",8)==0))
      {
	strcpy(bufs, buffer_header[header]);
	//bufs=g_strconcat(buffer_header[header],NULL);
	for(i=1;i<5;i++){
	  if((i+header) > header_line)  break;
	  if((buffer_header[i+header][0]==' ') || (buffer_header[i+header][0]=='\t')){
	    bufs[strlen(bufs)-2]='\0';
	    sdel = &buffer_header[i+header][1];     /* \t  */
	    strcat(bufs,sdel); /* ʣ */
	    //bufs=g_strconcat(bufs,sdel,NULL); /* ʣ */
	  }
	  else{
	    s=g_strdup(bufs);
	    break;
	  }
	}
      }
    }
    
    if (f) {
      tmp_froms=strbuf(f);
      g_free(f);
    }
    else{
      tmp_froms=strbuf("(no From: in original)\n");
    }
    if (s) {
      tmp_froms=strbuf(" ");
      tmp_froms=strbuf(s);
      g_free(s);
    }
    else{
      tmp_froms=strbuf("(no Subject: in original)\n");
    }
  }
  g_free(pop_froms);
  pop_froms=strbuf(NULL);
  pop_froms=strbuf(tmp_froms);
  g_free(tmp_froms);
#ifdef POP_DEBUG
  //  printf("data = %s\n",pop_froms);
  fprintf(stderr,"fs_get_pop3 end\n");
#endif
}

//
char* strbuf(char *p)
{
  static char    *start = NULL;
  static char    *next;
  static int      len, len2, l;
  
  if (p == NULL) {
    len = len2 = 4096;
    start = next = (char *) g_malloc(len + 1);
    start[0] = '\0';
  } else if (start != NULL) {
    l = strlen(p);
    while (len - l < 0) {
      char           *p;
      len2 += 256;
      len += 256;
      p = (char *) g_realloc(start, len2 + 1);
      start = p;
      next = p + len2 - len;
                }
    strcpy(next,p);
    len -= l;
    next += l;
  }
  return start;
}


// ե륿ޥɸƤӤ
static char* code_conv(typMascot *mascot, char *data )
{
  FILE *fp;
  char buf[BUFFSIZE];
  int fd1[2],fd2[2];
  static pid_t pid1,pid2;
  char *p;
  
  pipe(fd1);
  pipe(fd2);

  waitpid(pid1,0,WNOHANG);
  waitpid(pid2,0,WNOHANG);
  if( (pid1 = fork()) < 0 ){
    perror("fork");	
    exit(1);
  }
  
  // ҥץ
  if( pid1 == 0 ){
    close( fd1[1] );
    close( fd2[0] );
    close(0);
    dup(fd1[0]);
    close(fd1[0]);
    close(1);
    dup(fd2[1]);
    close(fd2[1]);
    execvp( biff_filter[0], &biff_filter[1] );
    perror("execvp");
    fflush(stdin);
    fflush(stdout);
    _exit(1);
  }
  
  if( (pid2 = fork()) < 0 ){
    perror("fork");
    exit(1);
  }
  
  // ƤΣ
  if( pid2 == 0 ){
    close( fd2[1] );
    close( fd2[0] );
    close( fd1[0] );
    
    if( (fp = fdopen( fd1[1], "w" )) != NULL ){
      fputs( data, fp );
      fclose( fp );
    }
    fflush(stdin);
    fflush(stdout);
    _exit(1);
  }
  
  
  // ƤΣ
  close( fd1[1] );
  close( fd1[0] );
  close( fd2[1] );
  
  if( (p = strbuf(NULL)) != NULL ){
    if( (fp = fdopen( fd2[0], "r" )) != NULL ){
      while( fgets( buf,BUFFSIZE-1,fp ) != NULL )
	p = strbuf( buf );
      fclose( fp );
    }
  }
  return p;
}


void set_biff_filter(gchar* com){
  int             t, i;
  char           *p;
  static int imax;
  
  biff_filter[0] = NULL;
  
  for(i=0;i<12;i++){
    g_free(biff_filter[i]);
  }
  if (strcmp(com,"\0")!=0) {
    set_token(com);
    if ((t = get_token2()) != NODAT) {
      biff_filter[0]=g_strdup(token_buf);

      // 0x2F='/'
      if ((p = strrchr(biff_filter[0], 0x2F)))
	p++;
      else
	p = biff_filter[0];
      biff_filter[1] = p;
      
      for (i = 2; i < 12;) {
	if ((t = get_token2()) == NODAT) {
	  biff_filter[i] = (char *) NULL;
	  break;
	} 
	else{
	  biff_filter[i++]=g_strdup(token_buf);
	}
      }
      imax=i;;
    }
  }
}


void  set_token(char *p){
    token_p = p;
}

int get_token2(void)
{
    char            ch;
    
    while ((ch = *token_p++) == ' ' || ch == '\t' || ch == '\n');
    
    if (ch == '\0')
	return NODAT;
    
    if (ch == '"') {
	char           *p = token_buf;
	for (;;) {
	    ch = *token_p++;
	    if (ch == '"')
		break;
	    else if (ch == '\\') {
		ch = *token_p++;
	    }
	    if (ch == '\n' || '\0')
		return NODAT;
	    *p++ = ch;
	}
	*p = '\0';
	return STRING;
    } else {
	char           *p = token_buf;
	*p++ = ch;
	while (!((ch = *token_p) == ' ' || ch == '\t' || ch == '\n' || ch == '\0')) {
	    *p++ = ch;
	    token_p++;
		}
	*p = '\0';
	return ID;
    }
}

void make_fs_max(GtkWidget *widget, typMascot *mascot){
  GtkWidget *label;
  gchar tmp_fs_max[256];
  gchar tmp_fs[8];

  if(mascot->mail.displayed_count != mascot->mail.count){
    if(mascot->mail.spam_count!=0){
      sprintf(tmp_fs_max,
	      _("Displaying the newest %d mails out of %ds. [%d SPAMs are excluded.]"),
	      mascot->mail.displayed_count,
	      mascot->mail.count, mascot->mail.spam_count);
    }
    else{
      sprintf(tmp_fs_max,
	      _("Displaying the newest %d mails out of %ds."),
	      mascot->mail.displayed_count,
	      mascot->mail.count);
    }
  }
  else{
    if(mascot->mail.spam_count!=0){
      sprintf(tmp_fs_max,_("You have %d new mails.  [%d SPAMs are excluded.]"),
	      mascot->mail.count, mascot->mail.spam_count);
    }
    else{
      sprintf(tmp_fs_max,_("You have %d new mails."),
	      mascot->mail.count);
    }
  }
  label=gtk_label_new(tmp_fs_max);
  gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0);
  gtk_table_attach (GTK_TABLE (widget), label, 0, 5, 1, 2,
		    GTK_FILL, GTK_SHRINK , 0, 0);
}


// ᥤ忮ꥹȤ
void create_biff_dialog(typMascot *mascot)
{
  GtkWidget *biff_main;
  GtkWidget *biff_tbl;
  GtkWidget *biff_text;
  GtkWidget *button;
  GtkWidget *biff_scroll;
  gchar *tmp_froms=NULL;
  gchar *fp_1, *fp_2;
#ifdef USE_GTK2
  GtkTextBuffer *text_buffer;
  GtkTextIter start_iter, end_iter;
  GtkTextMark *end_mark;
#endif

  // WinۤϽŤΤExpose٥򤹤٤ƽƤ
  while (g_main_iteration(FALSE));

  Mascot->flag_menu=TRUE;

  
#ifdef USE_GTK2
  biff_main = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  text_buffer = gtk_text_buffer_new(NULL);
#else
  biff_main = gtk_window_new(GTK_WINDOW_DIALOG);
#endif

  gtk_widget_set_usize (biff_main, Mascot->mail.win_width, 
  			Mascot->mail.win_height);
  gtk_window_set_title(GTK_WINDOW(biff_main), 
		       _("MaCoPiX : Arrived mail list"));
  gtk_widget_realize(biff_main);
  gtk_signal_connect(GTK_OBJECT(biff_main),"destroy",
		     GTK_SIGNAL_FUNC(close_biff), 
		     GTK_WIDGET(biff_main));
  gtk_container_set_border_width (GTK_CONTAINER (biff_main), 5);
  
  // 6x3Υơ֥
  biff_tbl = gtk_table_new (6, 3, FALSE);
  gtk_container_add (GTK_CONTAINER (biff_main), biff_tbl);

#ifdef USE_GTK2
  biff_scroll = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(biff_scroll),
                                 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

  biff_text = gtk_text_view_new_with_buffer (text_buffer);
  gtk_text_view_set_editable (GTK_TEXT_VIEW (biff_text), FALSE);
  gtk_text_view_set_cursor_visible (GTK_TEXT_VIEW (biff_text), FALSE);

  gtk_container_add(GTK_CONTAINER(biff_scroll), biff_text);

  gtk_table_attach_defaults (GTK_TABLE (biff_tbl), biff_scroll, 0, 5, 0, 1);
#else
  biff_text = gtk_text_new (NULL, NULL);
  gtk_text_set_editable (GTK_TEXT (biff_text), FALSE);
  gtk_table_attach_defaults (GTK_TABLE(biff_tbl), biff_text, 0, 5, 0, 1);

  biff_scroll = gtk_vscrollbar_new (GTK_TEXT (biff_text)->vadj);
  gtk_table_attach (GTK_TABLE (biff_tbl), biff_scroll, 5, 6, 0, 1,
		    GTK_FILL, 
		    GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);
#endif


  switch(mascot->mail.type){
  case MAIL_LOCAL: 
    tmp_froms=fs_get_mbox(mascot);
    break;
  case MAIL_POP3: 
  case MAIL_APOP: 
    tmp_froms=g_strdup(pop_froms);
    make_fs_max(biff_tbl, mascot);
    break;
  case MAIL_QMAIL:
    tmp_froms=fs_get_qmail(mascot);
    break;
  case MAIL_PROCMAIL:
    tmp_froms=fs_get_procmail(mascot);
    break;
  }

  if(mascot->mail.filter){
    set_biff_filter(mascot->mail.filter);
    tmp_froms=code_conv(mascot, tmp_froms);
    gdk_flush();
  }

#ifdef USE_GTK2
  gtk_text_buffer_get_start_iter(text_buffer, &start_iter);
  gtk_text_buffer_insert (text_buffer, &start_iter, tmp_froms, -1);
  gtk_text_buffer_get_end_iter(text_buffer, &end_iter);
  gtk_text_buffer_place_cursor(text_buffer, &end_iter);
  end_mark= gtk_text_buffer_create_mark(text_buffer, "end", &end_iter, FALSE);
  gtk_text_view_scroll_to_mark(GTK_TEXT_VIEW(biff_text),
			       end_mark, 0.0, FALSE, 0.0, 0.0);
  gtk_text_view_scroll_to_iter(GTK_TEXT_VIEW(biff_text),
			       &end_iter,0.0, FALSE,0.0, 0.0); 
#else
  gtk_text_insert (GTK_TEXT (biff_text), NULL, NULL, NULL,
		   tmp_froms, -1);
  gtk_adjustment_set_value(GTK_TEXT (biff_text)->vadj,
    			   (gfloat)(GTK_TEXT(biff_text)->text_len));
#endif
  
  button=gtk_button_new_with_label(_("Start Mailer"));
  gtk_table_attach(GTK_TABLE(biff_tbl), button, 0, 1, 2, 3,
		   GTK_FILL,GTK_SHRINK,0,0);
  gtk_signal_connect(GTK_OBJECT(button),"pressed",
		     GTK_SIGNAL_FUNC(mailer_start), 
		     GTK_WIDGET(biff_main));


  button=gtk_button_new_with_label(_("OK"));
  gtk_table_attach(GTK_TABLE(biff_tbl), button, 4, 5, 2, 3,
		   GTK_FILL,GTK_SHRINK,0,0);
  gtk_signal_connect(GTK_OBJECT(button),"pressed",
		     GTK_SIGNAL_FUNC(close_biff), 
		     GTK_WIDGET(biff_main));
  
  gtk_widget_show_all(biff_main);
  
  gdk_flush();
}


static void close_biff(GtkWidget *w, GtkWidget *dialog)
{
  //gdk_pointer_ungrab(GDK_CURRENT_TIME);

  //Mascot->mail.win_width=dialog->allocation.width;
  //Mascot->mail.win_height=dialog->allocation.height;

  while (g_main_iteration(FALSE));
  gtk_widget_destroy(GTK_WIDGET(dialog));
  while (g_main_iteration(FALSE));
 
  Mascot->flag_menu=FALSE;
  gdk_flush();
}


// Biff winΥᥤ鵯ư
static void mailer_start(GtkWidget *w, GtkWidget *dialog)
{
  while (g_main_iteration(FALSE));
  gtk_widget_destroy(GTK_WIDGET(dialog));
  while (g_main_iteration(FALSE));
 
  Mascot->flag_menu=FALSE;

  ext_play(Mascot,Mascot->mail.mailer);
  gdk_flush();
}


gchar* set_mhdir(){
  FILE *fp;
  gchar *c=NULL,buf[256],*mhd=NULL, *tmp;
    
  c=g_strconcat(g_get_home_dir(),PROCMAILRC,NULL);

  if((fp=fopen(c,"r"))!=NULL){
    while(!feof(fp)){
      if(fgets(buf,256-1,fp)==NULL) break;
      if(strncmp(buf,"MAILDIR=",8)==0){
	tmp=buf+8;
	if(strncmp(tmp,"$HOME",5)==0){
	  mhd=g_strconcat(g_get_home_dir(),tmp+5,NULL);
	}
	else{
	  mhd=tmp;
	}
	break;
      }
    }
  }
    
  if(mhd==NULL){
    mhd=g_strconcat(g_get_home_dir(),MH_MAIL_DIR,NULL);
  }
  if(mhd[strlen(mhd)-1]=='\n') mhd[strlen(mhd)-1]='\0';

  g_free(c);
  
  return mhd;
}

