///////////////////////////////////////////////////////////////////////////////
// Project:     M - cross platform e-mail GUI client
// File name:   ViewFilter.h: declaration of ViewFilter ABC
// Purpose:     ViewFilter allows plugging in filters between MessageView and
//              the MessageViewer used by it
// Author:      Vadim Zeitlin
// Modified by:
// Created:     28.11.02
// CVS-ID:      $Id: ViewFilter.h,v 1.6 2003/07/22 22:01:38 vadz Exp $
// Copyright:   (c) 2002 Vadim Zeitlin <vadim@wxwindows.org>
// Licence:     M license
///////////////////////////////////////////////////////////////////////////////

#ifndef _VIEWFILTER_H_
#define _VIEWFILTER_H_

#include "MModule.h"

class MessageView;
class MessageViewer;
class MTextStyle;

// the viewer filter module interface name
#define VIEW_FILTER_INTERFACE _T("ViewFilter")

/**
  ViewFilter
 */

class ViewFilter
{
public:
   /**
     Filters are applied in order of their priorities and this enum defines
     some typical values of them.

     The values Priority_Last and Priority_First are reserved for
     MessageView itself and shouldn't be used by the other filters.

     Finally note that all filters with equal priority are applied in some
     random (i.e. undefined) order
    */
   enum
   {
      /// the filter with this priority is applied after all the others
      Priority_Lowest = -100,

      /// low priority filter: applied after the default ones
      Priority_Low = -50,

      /// this is the priority of the URL detection/colorization default filter
      Priority_Default = 0,

      /// high priority filter: applied before the default ones
      Priority_High = 50,

      /// the filter with this priority is applied before all the others
      Priority_Highest = 100,

      /// invalid value for filter priority
      Priority_Max
   };


   /**
      @name Creation and destruction
    */
   //@{

   /**
      Filters are stored in a linked list ordered by their priorities and the
      ctor sets the internal next poitner to the next object in this list.

      The next filter is never NULL except for the special last filter (i.e.
      the one with Priority_Last) installed by MessageView itself.

      @param msgView the message view we're associated with
      @param next the next filter in the chain
      @param enable true to enable the filter initially, false to disable it
    */
   ViewFilter(MessageView *msgView, ViewFilter *next, bool enable)
      { m_msgView = msgView; m_next = next; m_active = enable; }

   /// virtual dtor for any base class
   virtual ~ViewFilter() { }

   //@}

   /**
     Pass the text through the filter.

     This is where the filter does its work: it analyzes the message text and
     outputs it using the given viewer and the filter, if any. In other words,
     all the text (as opposed to anything special) generated by this filter
     must be passed to Process() method of the next filter.

     @param text which may be modified in place by the filter
     @param viewer the viewer to use for the real output
     @param style the current style used for the output
    */
   void Process(String& text,
                MessageViewer *viewer,
                MTextStyle& style)
   {
      if ( IsEnabled() )
         DoProcess(text, viewer, style);
      else if ( m_next )
         m_next->Process(text, viewer, style);
   }

   /**
     @name State

     A filter can be enabled or (temporarily) disabled.
    */
   //@{

   /// activate the filter
   void Enable(bool enable) { m_active = enable; }

   /// return true if the filter is enabled
   bool IsEnabled() const { return m_active; }

   //@}

protected:
   /// the function to implement in the derived classes, called by Process()
   virtual void DoProcess(String& text,
                          MessageViewer *viewer,
                          MTextStyle& style) = 0;

   /// the message view we're associated with
   MessageView *m_msgView;

   /// pointer to the next filter or NULL if this is the last one
   ViewFilter *m_next;

   /// true if we're enabled
   bool m_active;

   friend class ViewFilterNode;
};

// ----------------------------------------------------------------------------
// Loading the filters from the modules
// ----------------------------------------------------------------------------

class ViewFilterFactory : public MModule
{
public:
   /// returns the priority of the filter object of this class
   virtual int GetPriority() const = 0;

   /// return the default filter state: active or not?
   virtual bool GetDefaultState() const = 0;

   /// creates the new filter object, to be deleted by the caller
   virtual ViewFilter *Create(MessageView *msgView, ViewFilter *next) const = 0;
};

/**
  This macro must be used in the implementation part of all filter classes.

  @param cname    the name of the class (derived from ViewFilter)
  @param prio     the filter priority (see ViewFilter::Priority_XXX enum)
  @param state    the initial state (enabled or disabled)
  @param desc     the short description shown in the filters dialog
  @param cpyright the module author/copyright string
 */
#define IMPLEMENT_VIEWER_FILTER(cname, prio, state, desc, cpyright)        \
   class cname##Factory : public ViewFilterFactory                         \
   {                                                                       \
   public:                                                                 \
      virtual int GetPriority() const { return prio; }                     \
      virtual bool GetDefaultState() const { return state; }               \
      virtual ViewFilter *Create(MessageView *msgView,                     \
                                 ViewFilter *next) const                   \
         { return new cname(msgView, next, state); }                       \
                                                                           \
      MMODULE_DEFINE();                                                    \
      DEFAULT_ENTRY_FUNC;                                                  \
   };                                                                      \
   MMODULE_BEGIN_IMPLEMENT(cname##Factory, #cname,                         \
                           VIEW_FILTER_INTERFACE, desc, "1.00")            \
      MMODULE_PROP("author", cpyright)                                     \
   MMODULE_END_IMPLEMENT(cname##Factory)                                   \
   MModule *cname##Factory::Init(int /* version_major */,                  \
                                 int /* version_minor */,                  \
                                 int /* version_release */,                \
                                 MInterface * /* minterface */,            \
                                 int * /* errorCode */)                    \
   {                                                                       \
      return new cname##Factory();                                         \
   }

#endif // _VIEWFILTER_H_

