// //// //// //// //// //// //// //// //// //// //// //// //// //// //// //////
// Project:     M - cross platform e-mail GUI client
// File name:   adb/AdbDataProvider.h - data provider public interface
// Purpose:     a data provider is a class which creates/deletes the address
//              books of given type
// Author:      Vadim Zeitlin
// Modified by:
// Created:     09.08.98
// CVS-ID:      $Id: AdbDataProvider.h,v 1.10 2002/12/28 02:28:33 nerijus Exp $
// Copyright:   (c) 1998 Vadim Zeitlin <zeitlin@dptmaths.ens-cachan.fr>
// Licence:     M license
// //// //// //// //// //// //// //// //// //// //// //// //// //// //// //////

#ifndef   _ADBDATAPROVIDER_H
#define   _ADBDATAPROVIDER_H

#include "MObject.h"    // the base class declaration

// forward declaration for classes we use
class AdbBook;
class wxArrayString;

/**
  The object of this class are created by AdbManager when it need to manipulate
  the address books. As there can be an arbitrary number of (simultaneously
  used) implementations of AdbDataProvider, there is a special mechanism for
  creation of objects of this class: in declaration of each AdbDataProvider
  derived class the macro DECLARE_ADB_PROVIDER() should be used and the macro
  IMPLEMENET_ADB_PROVIDER() must be put in the implementation file. These
  macros add the static object describing this provider to the linked list of
  all data providers managed by AdbManager.

  This class derives from MObjectRC and uses reference counting, see
  the comments in MObject.h for more details about it.
*/

class AdbDataProvider : public MObjectRC
{
public:
  typedef AdbDataProvider *(*Constructor)();

  // the syntax for the address book name
  enum AdbNameFormat
  {
    Name_No,      // this provider supports only one address book
    Name_File,    // address books are files (the most common case)
    Name_String   // anything else
  };

  // the flags for TestBookAccess (not a bit flag)
  enum AdbTests
  {
    Test_Create,        // may create book with this name?
    Test_Open,          // does the book with this name already exists?
    Test_OpenReadOnly   // currently unused
  };

  // list holding descriptions of all data providers
  static struct AdbProviderInfo
  {
    const wxChar *szName;     // internal name

    bool bSupportsCreation;   // do we support creating new address books?
    AdbNameFormat nameFormat; // to know what ask the user for...
    const wxChar *szFmtName;    // NB: this name is shown to the user

    Constructor CreateProvider;

    AdbProviderInfo *pNext;

    AdbProviderInfo(const wxChar *name, Constructor ctor,
                    bool canCreate, const wxChar *formatName,
                    AdbNameFormat adbFormat);
  } *ms_listProviders;

  // returns provider by name: as with any function returning new pointer, you
  // must DecRef() it when you don't need it any more
  static AdbDataProvider *GetProviderByName(const String& name);

  // returns the native provider: the one which reads ADBs in our native
  // format. This function always succeeds, i.e. never returns NULL.
  static AdbDataProvider *GetNativeProvider();

  // create the book by "name" (in fact, it can be anything at all: a file name
  // or an URL or ...); if the book doesn't exist it's created and NULL is
  // returned if the function fails.
  virtual AdbBook *CreateBook(const String& name) = 0;

  // enumerate all books known to this provider; may be unsupported in which
  // case it just returns FALSE, otherwise it fills the array passed in with
  // the names of all known address books
  virtual bool EnumBooks(wxArrayString& aNames) = 0;

  // delete the physical support of the book (i.e. the file in the most common
  // case); if not supported (can't delete www.yahoo.com...) just return FALSE
  virtual bool DeleteBook(AdbBook *book) = 0;

  // this function is used for testing for different things:
  //  * if test is Test_Open, it asks whether the book specified by 'name'
  //    exists and, in this case, whether it's in the format supported by this
  //    provider.
  //  * if test is Test_Create it asks whether we can create a book by such
  //    name (it's not an error if it already exists). Note that even if we
  //    return TRUE from here, CreateBook() may still fail, so the return code
  //    must still be tested!
  //
  // in any case, the function must return TRUE if the operation may succeed
  // and FALSE if it will not.
  virtual bool TestBookAccess(const String& name, AdbTests test) = 0;

  // get the name of this provider (this function is generated by
  // IMPLEMENT_ADB_PROVIDER macro expansion)
  virtual String GetProviderName() const = 0;
};

/// dynamic object creation helpers
#define DECLARE_ADB_PROVIDER(name)                                            \
  String GetProviderName() const;                                             \
  static AdbProviderInfo ms_info
#define IMPLEMENT_ADB_PROVIDER(name, bCanCreate, userName, fmt)               \
  String name::GetProviderName() const { return _T(#name); }                      \
  AdbDataProvider *ConstructorFor##name() { return new name; }                \
  AdbDataProvider::AdbProviderInfo name::ms_info(_T(#name), ConstructorFor##name, \
                                                 bCanCreate, userName,        \
                                                 AdbDataProvider::fmt)

#endif  //_ADBDATAPROVIDER_H
