///////////////////////////////////////////////////////////////////////////////
// Project:     M - cross platform e-mail GUI client
// File name:   class/ComposeTemplate.cpp - template parser for composer
// Purpose:     used to initialize composer with the text from template
//              expansion
// Author:      Vadim Zeitlin
// Modified by:
// Created:     13.04.01 (extracted from src/gui/wxComposeView.cpp)
// CVS-ID:      $Id: ComposeTemplate.cpp,v 1.48 2003/07/22 22:01:39 vadz Exp $
// Copyright:   (c) 2001 Vadim Zeitlin <vadim@wxwindows.org>
// Licence:     M license
///////////////////////////////////////////////////////////////////////////////

// ============================================================================
// declarations
// ============================================================================

// ----------------------------------------------------------------------------
// headers
// ----------------------------------------------------------------------------

#include "Mpch.h"

#ifndef USE_PCH
#  include <wx/log.h>

#  include "sysutil.h"
#  include "strutil.h"
#endif

#include "TemplateDialog.h"
#include "MApplication.h"

#include "Composer.h"
#include "MDialogs.h"
#include "Mdefaults.h"

#include "Address.h"

#include "MessageView.h"

#include "Mpers.h"

#include <wx/confbase.h>      // for wxExpandEnvVars()
#include <wx/file.h>
#include <wx/ffile.h>
#include <wx/textfile.h>
#include <wx/tokenzr.h>

#include <wx/regex.h>

#include "wx/persctrl.h"

// ----------------------------------------------------------------------------
// options we use here
// ----------------------------------------------------------------------------

extern const MOption MP_AUTOMATIC_WORDWRAP;
extern const MOption MP_COMPOSETEMPLATEPATH_GLOBAL;
extern const MOption MP_COMPOSETEMPLATEPATH_USER;
extern const MOption MP_COMPOSE_SIGNATURE;
extern const MOption MP_COMPOSE_USE_SIGNATURE;
extern const MOption MP_COMPOSE_USE_SIGNATURE_SEPARATOR;
extern const MOption MP_DATE_FMT;
extern const MOption MP_REPLY_DETECT_SIG;
extern const MOption MP_REPLY_MSGPREFIX;
extern const MOption MP_REPLY_MSGPREFIX_FROM_SENDER;
extern const MOption MP_REPLY_QUOTE_EMPTY;
extern const MOption MP_REPLY_QUOTE_SELECTION;
extern const MOption MP_REPLY_SIG_SEPARATOR;
extern const MOption MP_WRAPMARGIN;

// ----------------------------------------------------------------------------
// persistent msgboxes we use here
// ----------------------------------------------------------------------------

extern const MPersMsgBox *M_MSGBOX_SIGNATURE_LENGTH;
extern const MPersMsgBox *M_MSGBOX_ASK_FOR_SIG;

// ----------------------------------------------------------------------------
// the classes which are used together with compose view - we have here a
// derivation of variable expander and expansion sink which are used for parsing
// the templates, but are completely hidden from the outside world because we
// provide just a single public function, ExpandTemplate()
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
// this struct and array are used by ExpansionSink only - they contain the
// information about the attachments
// ----------------------------------------------------------------------------

struct AttachmentInfo
{
   AttachmentInfo(void *d,
                  size_t l,
                  const String& m,
                  const String& f) : mimetype(m), filename(f)
      { data = d; len = l; }

   void   *data;
   size_t  len;
   String  mimetype,
           filename;
};

WX_DECLARE_OBJARRAY(AttachmentInfo, ArrayAttachmentInfo);

// ----------------------------------------------------------------------------
// implementation of template sink which stores the data internally and then
// puts it into the composer when InsertTextInto() is called
// ----------------------------------------------------------------------------

class ExpansionSink : public MessageTemplateSink
{
public:
   // ctor
   ExpansionSink() { m_hasCursorPosition = FALSE; m_x = m_y = 0; }

   // called after successful parsing of the template to insert the resulting
   // text into the compose view
   void InsertTextInto(Composer& cv) const;

   // implement base class pure virtual function
   virtual bool Output(const String& text);

   // TODO the functions below should be in the base class (as pure virtuals)
   //      somehow, not here

   // called by VarExpander to remember the current position as the initial
   // cursor position
   void RememberCursorPosition() { m_hasCursorPosition = TRUE; }

   // called by VarExpander to insert an attachment
   void InsertAttachment(void *data, size_t len,
                         const String& mimetype,
                         const String& filename);

private:
   // as soon as m_hasCursorPosition becomes TRUE we stop to update the current
   // cursor position which we normally keep track of in m_x and m_y - so that
   // we'll have the position of the cursor when RememberCursorPosition() in
   // them at the end
   bool m_hasCursorPosition;
   int  m_x, m_y;

   // before each m_attachments[n] there is text from m_texts[n] - and after
   // the last attachment there is m_text
   wxString m_text;
   wxArrayString m_texts;
   ArrayAttachmentInfo m_attachments;
};

// ----------------------------------------------------------------------------
// VarExpander is the implementation of MessageTemplateVarExpander used here
// ----------------------------------------------------------------------------

class VarExpander : public MessageTemplateVarExpander
{
public:
   // all categories
   enum Category
   {
      Category_Misc,       // empty category - misc variables
      Category_File,       // insert file
      Category_Attach,     // attach file
      Category_Command,    // execute external command
#ifdef USE_PYTHON
      Category_Python,     // execute Python script
#endif // USE_PYTHON
      Category_Message,    // access the headers of the message being written
      Category_Original,   // variables pertaining to the original message
      Category_Headers,    // set the headers of the message being written
      Category_Invalid,    // unknown/invalid category
      Category_Max = Category_Invalid
   };

   // the variables in "misc" category
   enum Variable
   {
      MiscVar_Date,        // insert the date in the default format
      MiscVar_Cursor,      // position the cursor here after template expansion
      MiscVar_To,          // the recipient name
      MiscVar_Cc,          // the copied-to recipient name
      MiscVar_Subject,     // the message subject (without any "Re"s)

      // all entries from here only apply to the reply/forward/followup
      // templates because they work with the original message
      MiscVar_Quote,       // quote the original text
      MiscVar_Quote822,    // include the original msg as a RFC822 attachment
      MiscVar_Text,        // include the original text as is
      MiscVar_Sender,      // the fullname of the sender
      MiscVar_Signature,   // the signature

      MiscVar_Invalid,
      MiscVar_Max = MiscVar_Invalid
   };

   // the variables in "message" category
   //
   // NB: the values should be identical to wxComposeView::RecipientType enum
   //     (or the code in ExpandMessage should be changed)
   enum MessageHeader
   {
      MessageHeader_To,
      MessageHeader_Cc,
      MessageHeader_Bcc,
      MessageHeader_LastControl = MessageHeader_Bcc,
      MessageHeader_Subject,
      MessageHeader_FirstName,
      MessageHeader_LastName,
      MessageHeader_Invalid,
      MessageHeader_Max = MessageHeader_Invalid
   };

   // the variables from "original" category which map to headers
   enum OriginalHeader
   {
      OriginalHeader_Date,
      OriginalHeader_From,
      OriginalHeader_Subject,
      OriginalHeader_PersonalName,
      OriginalHeader_FirstName,
      OriginalHeader_LastName,
      OriginalHeader_To,
      OriginalHeader_Cc,
      OriginalHeader_ReplyTo,
      OriginalHeader_Newsgroups,
      OriginalHeader_Domain,
      OriginalHeader_Invalid,
      OriginalHeader_Max = OriginalHeader_Invalid
   };

   // get the category from the category string (may return Category_Invalid)
   static Category GetCategory(const String& category)
   {
      return (Category)FindStringInArray(ms_templateVarCategories,
                                         Category_Max, category);
   }

   // get the variable id from the string (works for misc variables only, will
   // return MiscVar_Invalid if variable is unknown)
   static Variable GetVariable(const String& variable)
   {
      return (Variable)FindStringInArray(ms_templateMiscVars,
                                         MiscVar_Max, variable);
   }

   // get the header corresponding to the variable of "message" category
   static MessageHeader GetMessageHeader(const String& variable)
   {
      return (MessageHeader)FindStringInArray(ms_templateMessageVars,
                                              MessageHeader_Max, variable);
   }

   // get the header corresponding to the variable of "original" category
   // (will return OriginalHeader_Invalid if there is no corresponding header -
   // note that this doesn't mean that the variable is invalid because, for
   // example, "quote" doesn't correspond to any header, yet $(original:quote)
   // is perfectly valid)
   static OriginalHeader GetOriginalHeader(const String& variable)
   {
      return (OriginalHeader)FindStringInArray(ms_templateOriginalVars,
                                               OriginalHeader_Max, variable);
   }

   // ctor takes the sink (we need it to implement some pseudo macros such as
   // "$CURSOR") and also a pointer to message for things like $QUOTE - this
   // may (and in fact should) be NULL for new messages, in this case using the
   // macros which require it will result in an error.
   //
   // And we also need the compose view to expand the macros in the "message"
   // category.
   VarExpander(ExpansionSink& sink,
               Composer& cv,
               Profile *profile = NULL,
               Message *msg = NULL,
               const MessageView *msgview = NULL)
      : m_sink(sink), m_cv(cv)
   {
      m_profile = profile ? profile : mApplication->GetProfile();
      m_profile->IncRef();

      m_msg = msg;
      SafeIncRef(m_msg);

      m_msgview = msgview;
   }

   virtual ~VarExpander()
   {
      m_profile->DecRef();
      SafeDecRef(m_msg);
   }

   // implement base class pure virtual function
   virtual bool Expand(const String& category,
                       const String& name,
                       const wxArrayString& arguments,
                       String *value) const;

protected:
   // read the file into the string, return TRUE on success
   static bool SlurpFile(const String& filename, String *value);

   // try to find the filename the user wants if only the name was specified
   // (look in the standard locations...)
   static String GetAbsFilename(const String& name);

   // Expand determines the category and then calls one of these functions
   bool ExpandMisc(const String& name,
                   const wxArrayString& arguments,
                   String *value) const;
   bool ExpandFile(const String& name,
                   const wxArrayString& arguments,
                   String *value) const;
   bool ExpandAttach(const String& name,
                     const wxArrayString& arguments,
                     String *value) const;
   bool ExpandCommand(const String& name,
                      const wxArrayString& arguments,
                      String *value) const;
#ifdef USE_PYTHON
   bool ExpandPython(const String& name, String *value) const;
#endif // USE_PYTHON
   bool ExpandMessage(const String& name, String *value) const;
   bool ExpandOriginal(const String& name, String *value) const;

   bool SetHeaderValue(const String& name,
                       const wxArrayString& arguments,
                       String *value) const;

   // handle quoting the original message
   void DoQuoteOriginal(bool isQuote, String *value) const;

   // quote a single MimePart, return true if we quoted it or false if we
   // ignored it (because we can't put such data in the composer...)
   bool DoQuotePart(const MimePart *mimePart,
                    const String& prefix,
                    String *value) const;

   // get the signature to use (including the signature separator, if any)
   String GetSignature() const;

   // return the reply prefix to use for message quoting when replying
   String GetReplyPrefix() const;

   // put the text quoted according to our current quoting options with the
   // given reply prefix into value
   enum
   {
      NoDetectSig = 0,
      DetectSig = 1
   };
   void ExpandOriginalText(const String& text,
                           const String& prefix,
                           String *value,
                           int flags = DetectSig) const;

private:
   // helper used by GetCategory and GetVariable
   static int FindStringInArray(const wxChar *strs[], int max, const String& s);

   // helper used by ExpandOriginal(): return the personal name part of the
   // first address of the given type
   //
   // FIXME: this function is bad as it completely ignores the other addresses
   void GetNameForAddress(String *value, MessageAddressType type) const;

   // the sink we use when expanding pseudo variables
   ExpansionSink& m_sink;

   // the compose view is used for expansion of the variables in "message"
   // category
   Composer& m_cv;

   // the message used for expansion of variables pertaining to the original
   // message (may be NULL for new messages)
   Message *m_msg;

   // the message viewer we use for querying the selection if necessary
   const MessageView *m_msgview;

   // the profile to use for everything (global one by default)
   Profile *m_profile;

   // this array contains the list of all categories
   static const wxChar *ms_templateVarCategories[Category_Max];

   // this array contains the list of all variables without category
   static const wxChar *ms_templateMiscVars[MiscVar_Max];

   // this array contains the variable names from "message" category
   static const wxChar *ms_templateMessageVars[MessageHeader_Max];

   // this array contains all the variables in the "original" category which
   // map to the headers of the original message (there are other variables in
   // this category as well)
   static const wxChar *ms_templateOriginalVars[OriginalHeader_Max];

   DECLARE_NO_COPY_CLASS(VarExpander)
};

// ----------------------------------------------------------------------------
// global data: the definitions of the popum menu for the template editing
// dialog.
// ----------------------------------------------------------------------------

// NB: These menus should be kept in sync (if possible) with the variable names

// the misc submenu
static TemplatePopupMenuItem gs_popupSubmenuMisc[] =
{
   TemplatePopupMenuItem(gettext_noop("Put &cursor here"), _T("$cursor")),
   TemplatePopupMenuItem(gettext_noop("Insert &signature"), _T("$signature")),
   TemplatePopupMenuItem(),
   TemplatePopupMenuItem(gettext_noop("Insert current &date"), _T("$date")),
   TemplatePopupMenuItem(),
   TemplatePopupMenuItem(gettext_noop("Insert &quoted text"), _T("$quote")),
   TemplatePopupMenuItem(gettext_noop("&Attach original text"), _T("$quote822")),
};

// the file insert/attach sub menu
static TemplatePopupMenuItem gs_popupSubmenuFile[] =
{
   TemplatePopupMenuItem(gettext_noop("&Insert file..."), _T("${file:%s}"), TRUE),
   TemplatePopupMenuItem(gettext_noop("Insert &any file..."), _T("${file:%s?ask"), TRUE),
   TemplatePopupMenuItem(gettext_noop("Insert &quoted file..."), _T("${file:%s?quote}"), TRUE),
   TemplatePopupMenuItem(gettext_noop("&Attach file..."), _T("${attach:%s}"), TRUE),
};

// the message submenu
static TemplatePopupMenuItem gs_popupSubmenuMessage[] =
{
   TemplatePopupMenuItem(gettext_noop("&To"), _T("${message:to}")),
   TemplatePopupMenuItem(gettext_noop("&First name"), _T("${message:firstname}")),
   TemplatePopupMenuItem(gettext_noop("&Last name"), _T("${message:lastname}")),
   TemplatePopupMenuItem(gettext_noop("&Subject"), _T("${message:subject}")),
   TemplatePopupMenuItem(gettext_noop("&CC"), _T("${message:cc}")),
   TemplatePopupMenuItem(gettext_noop("&BCC"), _T("${message:bcc}")),
};

// the original message submenu
static TemplatePopupMenuItem gs_popupSubmenuOriginal[] =
{
   TemplatePopupMenuItem(gettext_noop("&Date"), _T("${original:date}")),
   TemplatePopupMenuItem(gettext_noop("&From"), _T("${original:from}")),
   TemplatePopupMenuItem(gettext_noop("&Subject"), _T("${original:subject}")),
   TemplatePopupMenuItem(gettext_noop("Full &name"), _T("${original:fullname}")),
   TemplatePopupMenuItem(gettext_noop("F&irst name"), _T("${original:firstname}")),
   TemplatePopupMenuItem(gettext_noop("&Last name"), _T("${original:lastname}")),
   TemplatePopupMenuItem(gettext_noop("&To"), _T("${original:to}")),
   TemplatePopupMenuItem(gettext_noop("&Reply to"), _T("${original:replyto}")),
   TemplatePopupMenuItem(gettext_noop("&Newsgroups"), _T("${original:newsgroups}")),
   TemplatePopupMenuItem(),
   TemplatePopupMenuItem(gettext_noop("Insert &quoted text"), _T("$quote")),
   TemplatePopupMenuItem(gettext_noop("&Attach original text"), _T("$quote822")),
   TemplatePopupMenuItem(gettext_noop("Insert &unquoted Text"), _T("$text")),
};

// the whole menu
static TemplatePopupMenuItem gs_popupMenu[] =
{
   TemplatePopupMenuItem(gettext_noop("&Miscellaneous"),
                         gs_popupSubmenuMisc,
                         WXSIZEOF(gs_popupSubmenuMisc)),
   TemplatePopupMenuItem(gettext_noop("Message &headers"),
                         gs_popupSubmenuMessage,
                         WXSIZEOF(gs_popupSubmenuMessage)),
   TemplatePopupMenuItem(gettext_noop("&Original message"),
                         gs_popupSubmenuOriginal,
                         WXSIZEOF(gs_popupSubmenuOriginal)),
   TemplatePopupMenuItem(gettext_noop("Insert or attach a &file"),
                         gs_popupSubmenuFile,
                         WXSIZEOF(gs_popupSubmenuFile)),
   TemplatePopupMenuItem(),
   TemplatePopupMenuItem(gettext_noop("&Execute command..."), _T("${cmd:%s}"), FALSE),
};

const TemplatePopupMenuItem& g_ComposeViewTemplatePopupMenu =
   TemplatePopupMenuItem(_T(""), gs_popupMenu, WXSIZEOF(gs_popupMenu));

// ============================================================================
// implementation
// ============================================================================

// ----------------------------------------------------------------------------
// ExpansionSink - the sink used with wxComposeView
// ----------------------------------------------------------------------------

#include <wx/arrimpl.cpp>
WX_DEFINE_OBJARRAY(ArrayAttachmentInfo);

bool
ExpansionSink::Output(const String& text)
{
   if ( !m_hasCursorPosition )
   {
      // update the current x and y position
      //
      // TODO: this supposes that there is no autowrap, to be changed if/when
      //       it appears
      int deltaX = 0, deltaY = 0;
      for ( const wxChar *pc = text.c_str(); *pc; pc++ )
      {
         if ( *pc == '\n' )
         {
            deltaX = -m_x;
            deltaY++;
         }
         else
         {
            deltaX++;
         }
      }

      m_x += deltaX;
      m_y += deltaY;
   }
   //else: we don't have to count anything, we already have the cursor position
   //      and this is all we want

   // we always add this text to the last "component" of text (i.e. the one
   // after the last attachment)
   m_text += text;

   return TRUE;
}

void
ExpansionSink::InsertAttachment(void *data,
                                size_t len,
                                const String& mimetype,
                                const String& filename)
{
   if ( !m_hasCursorPosition )
   {
      // an attachment count as one cursor position
      m_x++;
   }

   // create a new attachment info object (NB: it will be deleted by the array
   // automatically because we pass it by pointer and not by reference)
   m_attachments.Add(new AttachmentInfo(data, len, mimetype, filename));

   // the last component of text becomes the text before this attachment
   m_texts.Add(m_text);

   // and the new last component is empty
   m_text.Empty();
}

void
ExpansionSink::InsertTextInto(Composer& cv) const
{
   size_t nCount = m_texts.GetCount();
   ASSERT_MSG( m_attachments.GetCount() == nCount,
               _T("something is very wrong in template expansion sink") );

   for ( size_t n = 0; n < nCount; n++ )
   {
      cv.InsertText(m_texts[n]);

      AttachmentInfo& attInfo = m_attachments[n];
      cv.InsertData(attInfo.data, attInfo.len,
                    attInfo.mimetype,
                    attInfo.filename);
   }

   cv.InsertText(m_text);

   // position the cursor - if RememberCursorPosition() hadn't been called, it
   // will be put in (0, 0)
   cv.MoveCursorBy(m_x, m_y);

   // as the inserted text comes from the program, not from the user, don't
   // mark the composer contents as dirty
   cv.ResetDirty();
}

// ----------------------------------------------------------------------------
// VarExpander - used by wxComposeView
// ----------------------------------------------------------------------------

const wxChar *VarExpander::ms_templateVarCategories[] =
{
   _T(""),
   _T("file"),
   _T("attach"),
   _T("cmd"),
#ifdef USE_PYTHON
   _T("python"),
#endif // USE_PYTHON
   _T("message"),
   _T("original"),
   _T("header"),
};

const wxChar *VarExpander::ms_templateMiscVars[] =
{
   _T("date"),
   _T("cursor"),
   _T("to"),
   _T("cc"),
   _T("subject"),
   _T("quote"),
   _T("quote822"),
   _T("text"),
   _T("sender"),
   _T("signature"),
};

const wxChar *VarExpander::ms_templateMessageVars[] =
{
   _T("to"),
   _T("cc"),
   _T("bcc"),
   _T("subject"),
   _T("firstname"),
   _T("lastname"),
};

const wxChar *VarExpander::ms_templateOriginalVars[] =
{
   _T("date"),
   _T("from"),
   _T("subject"),
   _T("fullname"),
   _T("firstname"),
   _T("lastname"),
   _T("to"),
   _T("cc"),
   _T("replyto"),
   _T("newsgroups"),
   _T("domain"),
};

int
VarExpander::FindStringInArray(const wxChar *strings[],
                               int max,
                               const String& s)
{
   int n;
   for ( n = 0; n < max; n++ )
   {
      if ( strings[n] == s )
         break;
   }

   return n;
}

bool
VarExpander::SlurpFile(const String& filename, String *value)
{
   // it's important that value is not empty even if we return FALSE because if
   // it's empty when Expand() returns the template parser will log a
   // misleading error message about "unknown variable"
   *value = _T('?');

   wxFFile file(filename);

   return file.IsOpened() && file.ReadAll(value);
}

String
VarExpander::GetAbsFilename(const String& name)
{
   String filename = wxExpandEnvVars(name);
   if ( !wxIsAbsolutePath(filename) )
   {
      Profile *profile = mApplication->GetProfile();
      String path = READ_CONFIG(profile, MP_COMPOSETEMPLATEPATH_USER);
      if ( path.empty() )
         path = mApplication->GetLocalDir();
      if ( !path.empty() || path.Last() != '/' )
      {
         path += '/';
      }
      String filename2 = path + filename;

      if ( wxFile::Exists(filename2) )
      {
         // ok, found
         filename = filename2;
      }
      else
      {
         // try the global dir
         String path = READ_CONFIG(profile, MP_COMPOSETEMPLATEPATH_GLOBAL);
         if ( path.empty() )
            path = mApplication->GetGlobalDir();
         if ( !path.empty() || path.Last() != '/' )
         {
            path += '/';
         }

         filename.Prepend(path);
      }
   }
   //else: absolute filename given, don't look anywhere else

   return filename;
}

bool
VarExpander::Expand(const String& category,
                    const String& name,
                    const wxArrayString& arguments,
                    String *value) const
{
   value->Empty();

   // comparison is case insensitive
   switch ( GetCategory(category.Lower()) )
   {
      case Category_File:
         return ExpandFile(name, arguments, value);

      case Category_Attach:
         return ExpandAttach(name, arguments, value);

      case Category_Command:
         return ExpandCommand(name, arguments, value);

#ifdef USE_PYTHON
      case Category_Python:
         return ExpandPython(name, value);
#endif // USE_PYTHON

      case Category_Message:
         return ExpandMessage(name, value);

      case Category_Original:
         return ExpandOriginal(name, value);

      case Category_Misc:
         return ExpandMisc(name, arguments, value);

      case Category_Headers:
         return SetHeaderValue(name, arguments, value);

      default:
         // unknown category
         return FALSE;
   }
}

bool
VarExpander::ExpandMisc(const String& name,
                        const wxArrayString& /* arguments */,
                        String *value) const
{
   // deal with all special cases
   switch ( GetVariable(name.Lower()) )
   {
      case MiscVar_Date:
         {
            time_t ltime;
            (void)time(&ltime);

#ifdef OS_WIN
            // MP_DATE_FMT contains '%' which are being (mis)interpreted as
            // env var expansion characters under Windows
            ProfileEnvVarSave noEnvVars(m_profile);
#endif // OS_WIN

            *value = strutil_ftime(ltime, READ_CONFIG(m_profile, MP_DATE_FMT));
         }
         break;

      case MiscVar_Cursor:
         m_sink.RememberCursorPosition();
         break;

         // some shortcuts for the values of the "original:" category
      case MiscVar_To:
      case MiscVar_Cc:
      case MiscVar_Subject:
      case MiscVar_Quote:
      case MiscVar_Quote822:
      case MiscVar_Text:
         return ExpandOriginal(name, value);

      case MiscVar_Sender:
         return ExpandOriginal(_T("from"), value);

      case MiscVar_Signature:
         *value = GetSignature();
         break;

      default:
         // unknown name
         return FALSE;
   }

   return TRUE;
}

bool
VarExpander::ExpandFile(const String& name,
                        const wxArrayString& arguments,
                        String *value) const
{
   // first check if we don't want to ask user
   String filename = GetAbsFilename(name);
   if ( arguments.Index(_T("ask"), FALSE /* no case */) != wxNOT_FOUND )
   {
      filename = MDialog_FileRequester(_("Select the file to insert"),
                                       m_cv.GetFrame(),
                                       filename);
   }

   if ( !!filename )
   {
      // insert the contents of a file
      if ( !SlurpFile(filename, value) )
      {
         wxLogError(_("Failed to insert file '%s' into the message."),
                    name.c_str());

         return FALSE;
      }

      // do we want to quote the files contents before inserting?
      if ( arguments.Index(_T("quote"), FALSE /* no case */) != wxNOT_FOUND )
      {
         String prefix = READ_CONFIG(m_profile, MP_REPLY_MSGPREFIX);
         String quotedValue;
         quotedValue.Alloc(value->length());

         const wxChar *cptr = value->c_str();
         quotedValue = prefix;
         while ( *cptr )
         {
            if ( *cptr == '\r' )
            {
               cptr++;
               continue;
            }

            quotedValue += *cptr;
            if( *cptr++ == '\n' && *cptr )
            {
               quotedValue += prefix;
            }
         }

         *value = quotedValue;
      }
   }
   //else: no file, nothing to insert

   return TRUE;
}

bool
VarExpander::ExpandAttach(const String& name,
                          const wxArrayString& arguments,
                          String *value) const
{
   String filename = GetAbsFilename(name);
   if ( arguments.Index(_T("ask"), FALSE /* no case */) != wxNOT_FOUND )
   {
      filename = MDialog_FileRequester(_("Select the file to attach"),
                                       m_cv.GetFrame(),
                                       filename);
   }

   if ( !!filename )
   {
      if ( !SlurpFile(filename, value) )
      {
         wxLogError(_("Failed to attach file '%s' to the message."),
                    name.c_str());

         return FALSE;
      }

      // guess MIME type from extension
      m_sink.InsertAttachment(wxStrdup(value->c_str()),
                              value->length(),
                              _T(""), // will be determined from filename laer
                              filename);

      // avoid inserting file as text additionally
      value->Empty();
   }
   //else: no file, nothing to attach

   return TRUE;
}

bool
VarExpander::ExpandCommand(const String& name,
                           const wxArrayString& arguments,
                           String *value) const
{
   // execute a command
   MTempFileName temp;

   bool ok = temp.IsOk();
   wxString filename = temp.GetName();

   if ( ok )
   {
      wxString command = name;

      // although the arguments may be included directly in the template,
      // passing them via "?" argument mechanism allows to calculate them
      // during run-time, i.e. it makes it possible to call an external command
      // using the result of application of another template
      size_t count = arguments.GetCount();
      for ( size_t n = 0; n < count; n++ )
      {
         // forbid further expansion in the arguments by quoting them
         wxString arg = arguments[n];
         arg.Replace(_T("'"), _T("\\'"));

         command << _T(" '") << arg << '\'';
      }

      command << _T(" > ") << filename;

      ok = wxSystem(command) == 0;
   }

   if ( ok )
      ok = SlurpFile(filename, value);

   if ( !ok )
   {
      wxLogSysError(_("Failed to execute the command '%s'"), name.c_str());

      // make sure the value isn't empty to avoid message about unknown
      // variable from the parser
      *value = _T('?');

      return FALSE;
   }

   return TRUE;
}

bool
VarExpander::SetHeaderValue(const String& name,
                            const wxArrayString& arguments,
                            String *value) const
{
   if ( arguments.GetCount() != 1 )
   {
      wxLogError(_("${header:%s} requires exactly one argument."),
                 name.c_str());

      *value = _T('?');

      return FALSE;
   }

   String headerValue = arguments[0];

   // is it one of the standard headers or some other one?
   String headerName = name.Lower();
   if ( headerName == _T("subject") )
      m_cv.SetSubject(headerValue);
   else if ( headerName == _T("from") )
      m_cv.SetFrom(headerValue);
   else if ( headerName == _T("to") )
      m_cv.AddTo(headerValue);
   else if ( headerName == _T("cc") )
      m_cv.AddCc(headerValue);
   else if ( headerName == _T("bcc") )
      m_cv.AddBcc(headerValue);
   else if ( headerName == _T("fcc") )
      m_cv.AddFcc(headerValue);
   else // some other header
      m_cv.AddHeaderEntry(headerName, headerValue);

   return TRUE;
}

#ifdef USE_PYTHON
bool
VarExpander::ExpandPython(const String& name, String *value) const
{
   // TODO
   return FALSE;
}
#endif // USE_PYTHON

bool
VarExpander::ExpandMessage(const String& name, String *value) const
{
   MessageHeader header = GetMessageHeader(name.Lower());
   if ( header == MessageHeader_Invalid )
   {
      // unknown variable
      return FALSE;
   }

   switch ( header )
   {
      case MessageHeader_Subject:
         *value = m_cv.GetSubject();
         break;

      case MessageHeader_FirstName:
      case MessageHeader_LastName:
         {
            // FIXME: this won't work if there are several addresses!

            wxString to = m_cv.GetRecipients(Composer::Recipient_To);
            if ( header == MessageHeader_FirstName )
               *value = Message::GetFirstNameFromAddress(to);
            else
               *value = Message::GetLastNameFromAddress(to);
         }
         break;

      default:
         CHECK( header <= MessageHeader_LastControl, FALSE,
                _T("unexpected macro in message category") );

         // the MessageHeader enum values are the same as RecipientType ones,
         // so no translation is needed
         *value = m_cv.GetRecipients((Composer::RecipientType)header);
   }

   return TRUE;
}

void
VarExpander::GetNameForAddress(String *value, MessageAddressType type) const
{
   AddressList_obj addrList = m_msg->GetAddressList(type);
   Address *addr = addrList ? addrList->GetFirst() : NULL;
   if ( addr )
      *value = addr->GetName();
}

bool
VarExpander::ExpandOriginal(const String& Name, String *value) const
{
   // insert the quoted text of the original message - of course, this
   // only works if we have this original message
   if ( !m_msg )
   {
      // unfortunately we don't have the real name of the variable
      // here
      wxLogWarning(_("The variables using the original message cannot "
                     "be used in this template; variable ignored."));
   }
   else
   {
      // headers need to be decoded before showing them to the user
      bool isHeader = true;

      String name = Name.Lower();
      switch ( GetOriginalHeader(name) )
      {
         case OriginalHeader_Date:
            *value = m_msg->Date();
            break;

         case OriginalHeader_From:
            *value = m_msg->From();
            break;

         case OriginalHeader_Subject:
            *value = m_msg->Subject();
            break;

         case OriginalHeader_ReplyTo:
            GetNameForAddress(value, MAT_REPLYTO);
            break;

         case OriginalHeader_To:
            *value = m_msg->GetAddressesString(MAT_TO);
            break;

         case OriginalHeader_Cc:
            *value = m_msg->GetAddressesString(MAT_CC);
            break;

         case OriginalHeader_PersonalName:
            GetNameForAddress(value, MAT_FROM);
            break;

         case OriginalHeader_FirstName:
            GetNameForAddress(value, MAT_FROM);
            *value = Message::GetFirstNameFromAddress(*value);
            break;

         case OriginalHeader_LastName:
            GetNameForAddress(value, MAT_FROM);
            *value = Message::GetLastNameFromAddress(*value);
            break;

         case OriginalHeader_Newsgroups:
            m_msg->GetHeaderLine(_T("Newsgroups"), *value);
            break;

         case OriginalHeader_Domain:
            {
               AddressList_obj addrList(m_msg->From());
               Address *addr = addrList->GetFirst();
               if ( addr )
               {
                  *value = addr->GetDomain();
               }
            }
            break;

         default:
            isHeader = false;

            // it isn't a variable which maps directly onto header, check the
            // others
            bool isQuote = name == _T("quote");
            if ( isQuote || name == _T("text") )
            {
               DoQuoteOriginal(isQuote, value);
            }
            else if ( name == _T("quote822") )
            {
               // insert the original message as RFC822 attachment
               String str;
               m_msg->WriteToString(str);
               m_sink.InsertAttachment(wxStrdup(str), str.Length(),
                                       _T("message/rfc822"), _T(""));
            }
            else
            {
               return FALSE;
            }
      }

      if ( isHeader )
      {
         // we show the decoded headers to the user and then encode them back
         // when sending the messages
         //
         // FIXME: of course, this means that we lose the additional encoding
         //        info
         *value = MailFolder::DecodeHeader(*value);
      }
   }

   return TRUE;
}

// ----------------------------------------------------------------------------
// ExpandMisc("signature") helper
// ----------------------------------------------------------------------------

String VarExpander::GetSignature() const
{
   String signature;

   // first check if we want to insert it at all: this setting overrides the
   // $signature in the template because the latter is there by default and
   // it's simpler (especially for a novice user who might not know about the
   // templates at all) to just uncheck the checkbox "Use signature" in the
   // options dialog instead of editing all templates
   if ( READ_CONFIG(m_profile, MP_COMPOSE_USE_SIGNATURE) )
   {
      wxTextFile fileSig;

      // loop until we have a valid file to read the signature from
      bool hasSign = false;
      while ( !hasSign )
      {
         String strSignFile = READ_CONFIG(m_profile, MP_COMPOSE_SIGNATURE);
         if ( !strSignFile.empty() )
            hasSign = fileSig.Open(strSignFile);

         if ( !hasSign )
         {
            // no signature at all or sig file not found, propose to choose or
            // change it now
            wxString msg;
            if ( strSignFile.empty() )
            {
               msg = _("You haven't configured your signature file.");
            }
            else
            {
               // to show message from wxTextFile::Open()
               wxLog *log = wxLog::GetActiveTarget();
               if ( log )
                  log->Flush();

               msg.Printf(_("Signature file '%s' couldn't be opened."),
                          strSignFile.c_str());
            }

            msg += _("\n\nWould you like to choose your signature "
                     "right now (otherwise no signature will be used)?");
            if ( MDialog_YesNoDialog(msg, m_cv.GetFrame(), MDIALOG_YESNOTITLE,
                                     M_DLG_YES_DEFAULT,
                                     M_MSGBOX_ASK_FOR_SIG) )
            {
               strSignFile = wxPFileSelector(_T("sig"),
                                             _("Choose signature file"),
                                             NULL, _T(".signature"), NULL,
                                             wxALL_FILES,
                                             0, m_cv.GetFrame());
            }
            else
            {
               // user doesn't want to use signature file
               break;
            }

            if ( strSignFile.empty() )
            {
               // user canceled "choose signature" dialog
               break;
            }

            m_profile->writeEntry(MP_COMPOSE_SIGNATURE, strSignFile);
         }
      }

      if ( hasSign )
      {
         // insert separator optionally
         if ( READ_CONFIG(m_profile, MP_COMPOSE_USE_SIGNATURE_SEPARATOR) )
         {
            signature += _T("-- \n");
         }

         // read the whole file
         size_t nLineCount = fileSig.GetLineCount();
         for ( size_t nLine = 0; nLine < nLineCount; nLine++ )
         {
            if ( nLine )
               signature += '\n';

            signature += fileSig[nLine];
         }

         // let's respect the netiquette
         static const size_t nMaxSigLines = 4;
         if ( nLineCount > nMaxSigLines )
         {
            wxString msg;
            if ( nLineCount > 10 )
            {
               // *really* insult the user -- [s]he merits it
               msg.Printf(_("Your signature is waaaaay too long: "
                            "it should be no more than %d lines, "
                            "please trim it as it it risks to be "
                            "unreadable for the others."),
                          nMaxSigLines);
            }
            else // too long but not incredibly too long
            {
               msg.Printf(_("Your signature is too long: it should "
                            "not be more than %d lines."),
                            nMaxSigLines);
            }

            MDialog_Message(msg, m_cv.GetFrame(),
                            _("Signature is too long"),
                            GetPersMsgBoxName(M_MSGBOX_SIGNATURE_LENGTH));

         }
      }
      else
      {
         // don't ask the next time
         m_profile->writeEntry(MP_COMPOSE_USE_SIGNATURE, false);
      }
   }

   return signature;
}

// ----------------------------------------------------------------------------
// Quoting helpers
// ----------------------------------------------------------------------------

bool
VarExpander::DoQuotePart(const MimePart *mimePart,
                         const String& prefix,
                         String *value) const
{
   if ( !mimePart )
   {
      // this can only happen if the top level MIME part is NULL (when we call
      // ourselves recursively the pointer is never NULL) and in this case we
      // must let the user know that something is wrong
      wxLogError(_("Failed to quote the original message."));

      return false;
   }

   bool quoted = false;

   const MimeType mimeType = mimePart->GetType();
   switch ( mimeType.GetPrimary() )
   {
      case MimeType::TEXT:
         ExpandOriginalText(mimePart->GetTextContent(), prefix, value);

         quoted = true;
         break;

      case MimeType::MULTIPART:
         // to process multipart/alternative correctly we should really iterate
         // from the end to the beginning so that we could take the best
         // representation of the data but knowing that currently we only
         // really handle plain/text anyhow, it doesn't matter -- but it will
         // if/when we extend the composer to deal with other kinds of data
         {
            for ( MimePart *mimePartNested = mimePart->GetNested();
                  mimePartNested;
                  mimePartNested = mimePartNested->GetNext() )
            {
               if ( DoQuotePart(mimePartNested, prefix, value) )
               {
                  quoted = true;

                  if ( mimeType.GetSubType() == _T("ALTERNATIVE") )
                  {
                     // only one of the alternative parts should be used
                     break;
                  }
               }
            }
         }
         break;

      default:
         // ignore all the others -- we don't want to quote pictures, do we?
         ;
   }

   return quoted;
}

void
VarExpander::DoQuoteOriginal(bool isQuote, String *value) const
{
   // insert the original text (optionally prefixed by reply
   // string)
   String prefix;
   if ( isQuote )
   {
      prefix = GetReplyPrefix();
   }
   //else: template "text", so no reply prefix at all


   // do we include everything or just the selection?

   // first: can we get the selection?
   bool justSelection = m_msgview != NULL;

   // second: should we use the selection?
   if ( justSelection && !READ_CONFIG(m_profile, MP_REPLY_QUOTE_SELECTION) )
   {
      justSelection = false;
   }

   // third: do we have any selection?
   if ( justSelection )
   {
      String selection = m_msgview->GetSelection();
      if ( selection.empty() )
      {
         // take everything if no selection
         justSelection = false;
      }
      else
      {
         // include the selection only in the template expansion
         ExpandOriginalText(selection, prefix, value, NoDetectSig);
      }
   }


   // quote everything
   if ( !justSelection )
   {
      DoQuotePart(m_msg->GetTopMimePart(), prefix, value);
   }
}

String VarExpander::GetReplyPrefix() const
{
   String prefix;

   // prepend the senders initials to the reply prefix (this
   // will make reply prefix like "VZ>")
   if ( READ_CONFIG(m_profile, MP_REPLY_MSGPREFIX_FROM_SENDER) )
   {
      // take from address, not reply-to which can be set to
      // reply to a mailing list, for example
      String name;
      GetNameForAddress(&name, MAT_FROM);
      if ( name.empty() )
      {
         // no from address? try to find anything else
         GetNameForAddress(&name, MAT_REPLYTO);
      }

      name = MailFolder::DecodeHeader(name);

      // it's (quite) common to have quotes around the personal
      // part of the address, remove them if so

      // remove spaces
      name.Trim(TRUE);
      name.Trim(FALSE);
      if ( !name.empty() )
      {
         if ( name[0u] == '"' && name.Last() == '"' )
         {
            name = name.Mid(1, name.length() - 2);
         }

         // take the first letter of each word
         wxStringTokenizer tk(name);
         while ( tk.HasMoreTokens() )
         {
            char chInitial = tk.GetNextToken()[0u];

            if ( chInitial == '<' )
            {
               // this must be the start of embedded "<...>"
               // address, skip it completely
               break;
            }

            // only take letters as initials
            if ( isalpha(chInitial) )
            {
               prefix += chInitial;
            }
         }
      }
   }

   // and then the standard reply prefix too
   prefix += READ_CONFIG(m_profile, MP_REPLY_MSGPREFIX);

   return prefix;
}

// return the length of the line terminator if we're at the end of line or 0
// otherwise
static inline size_t IsEndOfLine(const wxChar *p)
{
   // although the text of the mail message itself has "\r\n" at the end of
   // each line, when we quote the selection only (which we got from the text
   // control) it has just "\n"s, so we should really understand both of them
   if ( p[0] == '\n' )
      return 1;
   else if ( p[0] == '\r' && p[1] == '\n' )
      return 2;

   return 0;
}

void
VarExpander::ExpandOriginalText(const String& text,
                                const String& prefix,
                                String *value,
                                int flags) const
{
   // should we quote the empty lines?
   //
   // this option is ignored when we're inserting text verbatim (hence without
   // reply prefix) and not quoting it
   bool quoteEmpty = !prefix.empty() &&
                        READ_CONFIG(m_profile, MP_REPLY_QUOTE_EMPTY);

   // where to break lines (if at all)?
   size_t wrapMargin;
   if ( READ_CONFIG(m_profile, MP_AUTOMATIC_WORDWRAP) )
   {
      wrapMargin = READ_CONFIG(m_profile, MP_WRAPMARGIN);
      if ( wrapMargin <= prefix.length() )
      {
         wxLogError(_("The configured automatic wrap margin (%u) is too "
                      "small, please increase it.\n"
                      "\n"
                      "Disabling automatic wrapping for now."), wrapMargin);

         m_profile->writeEntry(MP_AUTOMATIC_WORDWRAP, false);
         wrapMargin = 0;
      }
   }
   else
   {
      // don't wrap
      wrapMargin = 0;
   }

   // should we detect the signature and discard it?
   bool detectSig = (flags & DetectSig) &&
                        READ_CONFIG_BOOL(m_profile, MP_REPLY_DETECT_SIG);

#if wxUSE_REGEX
   // don't use RE at all by default because the manual code is faster
   bool useRE = false;

   // a RE to detect the start of the signature
   wxRegEx reSig;
   if ( detectSig )
   {
      String sig = READ_CONFIG(m_profile, MP_REPLY_SIG_SEPARATOR);

      if ( sig != GetStringDefault(MP_REPLY_SIG_SEPARATOR) )
      {
         // we have no choice but to use the user-supplied RE
         useRE = true;

         // we implicitly anchor the RE at start/end of line
         //
         // VZ: couldn't we just use wxRE_NEWLINE in Compile() instead of "\r\n"?
         String sigRE;
         sigRE << '^' << sig << _T("\r\n");

         if ( !reSig.Compile(sigRE, wxRE_NOSUB) )
         {
            wxLogError(_("Regular expression '%s' used for detecting the "
                         "signature start is invalid, please modify it.\n"
                         "\n"
                         "Disabling sinature stripping for now."),
                       sigRE.c_str());

            m_profile->writeEntry(MP_REPLY_DETECT_SIG, false);
            detectSig = false;
         }
      }
   }
#endif // wxUSE_REGEX

   // if != 0, then we're at the end of the current line
   size_t lenEOL = 0;

   // the current line
   String lineCur;

   // the last detected signature start
   int posSig = -1;

   for ( const wxChar *cptr = text.c_str(); ; cptr++ )
   {
      // start of [real] new line?
      if ( lineCur.empty() )
      {
         if ( detectSig )
         {
            bool isSig = false;

#if wxUSE_REGEX
            if ( useRE )
            {
               isSig = reSig.Matches(cptr);
            }
            else
#endif // wxUSE_REGEX
            {
               // hard coded detection for standard signature separator "--"
               // and the mailing list trailer "____...___"
               if ( cptr[0] == '-' && cptr[1] == '-' )
               {
                  // there may be an optional space after "--" (in fact the
                  // space should be there but some people don't put it)
                  const wxChar *p = cptr + 2;
                  if ( IsEndOfLine(p) || (*p == ' ' && IsEndOfLine(p + 1)) )
                  {
                     // looks like the start of the sig
                     isSig = true;
                  }
               }
               else if ( cptr[0] == '_' )
               {
                  const wxChar *p = cptr + 1;
                  while ( *p == '_' )
                     p++;

                  // consider that there should be at least 5 underscores...
                  if ( IsEndOfLine(p) && p - cptr >= 5 )
                  {
                     // looks like the mailing list trailer
                     isSig = true;
                  }
               }
            }

            if ( isSig )
            {
               // remember that the sig apparently starts here
               posSig = value->length();
            }
         }

         if ( !quoteEmpty && (lenEOL = IsEndOfLine(cptr)) != 0 )
         {
            // this line is empty, skip it entirely (i.e. don't output the
            // prefix for it)
            cptr += lenEOL - 1;

            *value += '\n';

            continue;
         }

         lineCur += prefix;
      }

      if ( !*cptr || (lenEOL = IsEndOfLine(cptr)) != 0 )
      {
         // sanity test
         ASSERT_MSG( !wrapMargin || lineCur.length() <= wrapMargin,
                     _T("logic error in auto wrap code") );

         *value += lineCur;

         if ( !*cptr )
         {
            // end of text
            break;
         }

         // put just '\n' in output, we don't need "\r\n"
         *value += '\n';

         lineCur.clear();

         // -1 to compensate for ++ in the loop
         cptr += lenEOL - 1;
      }
      else // !EOL
      {
         lineCur += *cptr;

         // we don't need to wrap a line if it is its last character anyhow
         if ( wrapMargin && lineCur.length() >= wrapMargin
               && !IsEndOfLine(cptr + 1) )
         {
            // break the line before the last word
            size_t n = wrapMargin - 1;
            while ( n > prefix.length() )
            {
               if ( isspace(lineCur[n]) )
                  break;

               n--;
            }

            if ( n == prefix.length() )
            {
               // no space found in the line or it is in prefix which
               // we don't want to wrap - so just cut the line right here
               n = wrapMargin;
            }

            value->append(lineCur, n);
            *value += '\n';

            // we don't need to start the new line with spaces so remove them
            // from the tail
            while ( n < lineCur.length() && isspace(lineCur[n]) )
            {
               n++;
            }

            lineCur.erase(0, n);
            lineCur.Prepend(prefix);
         }
      }
   }

   // if we had a sig, truncate it now: we have to do it like this because
   // otherwise we risk discarding a too big part of the message, e.g. if it
   // contains a quoted message with a sig inside it so we want to discard
   // everything after the last sig detected
   if ( posSig != -1 )
   {
      value->erase(posSig);
   }
}

// ----------------------------------------------------------------------------
// public API
// ----------------------------------------------------------------------------

extern bool TemplateNeedsHeaders(const String& templateValue)
{
   // check if there are any occurences of "${message:xxx}" in the template
   //
   // TODO: really parse it using a specialized expanded and without any
   //       sink, just checking if message category appears in it
   return templateValue.Lower().Find(_T("message:")) != wxNOT_FOUND;
}

extern bool ExpandTemplate(Composer& cv,
                           Profile *profile,
                           const String& templateValue,
                           Message *msg,
                           const MessageView *msgview)
{
   ExpansionSink sink;
   VarExpander expander(sink, cv, profile, msg, msgview);
   MessageTemplateParser parser(templateValue, _("template"), &expander);
   if ( !parser.Parse(sink) )
   {
      return false;
   }

   sink.InsertTextInto(cv);

   return true;
}

