/*
 * mdh (MailDooHicky), a GTK+-based toolbar.
 *
 * Copyright (c) 2003-2005 by Mike Hokenson <mdh at gozer dot org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <unistd.h>

#include <gtk/gtk.h>

#include "toolbar.h"

#include "button.h"

#include "widget.h"
#include "util.h"

MdhButton *mdh_button_new(gboolean enabled,
                          gboolean can_toggle,
                          const gchar *tooltip,
                          MdhButtonFunc func,
                          const gchar *image,
                          gint type)
{
    GtkWidget *icon = NULL;

    MdhButton *obj;

    g_return_val_if_fail(func != NULL, FALSE);
    g_return_val_if_fail(image != NULL, FALSE);

    obj             = g_new0(MdhButton, 1);

    obj->tooltip    = tooltip;
    obj->can_toggle = can_toggle;
    obj->enabled    = enabled;
    obj->callback   = func;
    obj->hidden     = TRUE;

    switch(type) {
        case MDH_IMAGE_TYPE_STOCK:
            icon = gtk_image_new_from_stock(image, GTK_ICON_SIZE_MENU);
            break;
        case MDH_IMAGE_TYPE_FILE:
            if(IS_FILE(image))
                icon = gtk_image_new_from_file(image);
            else
                icon = gtk_image_new_from_stock(GTK_STOCK_MISSING_IMAGE,
                                                GTK_ICON_SIZE_MENU);
            break;
        default:
            g_assert_not_reached();
    }

    obj->widgets.button = gtk_button_new();
    obj->widgets.icon   = icon;

    return(obj);
}

void mdh_button_free(MdhButton *obj)
{
    g_return_if_fail(obj != NULL);

    g_free(obj);
}

void mdh_button_toggle(MdhButton *obj)
{
    g_return_if_fail(obj != NULL);

    if(!obj->enabled || !obj->can_toggle)
        return;

    if(obj->hidden)
        mdh_button_show(obj);
    else
        mdh_button_hide(obj);
}

void mdh_button_show(MdhButton *obj)
{
    g_return_if_fail(obj != NULL);

    gtk_widget_show_all(obj->widgets.button);

    obj->hidden = FALSE;
}

void mdh_button_hide(MdhButton *obj)
{
    g_return_if_fail(obj != NULL);
        
    gtk_widget_hide_all(obj->widgets.button);
    
    obj->hidden = TRUE;
}
        
void mdh_button_reset(MdhButton *obj)
{
    g_return_if_fail(obj != NULL);

    if(obj->enabled)
        mdh_button_show(obj);
    else
        mdh_button_hide(obj);
}

void mdh_button_foreach(gpointer toolbar, GFunc func, gpointer data)
{
    g_return_if_fail(toolbar != NULL);
    g_return_if_fail(func != NULL);

    g_slist_foreach(MDH_TOOLBAR(toolbar)->buttons, func, data);
}

gboolean mdh_button_hidden(MdhButton *obj)
{
    g_return_val_if_fail(obj != NULL, FALSE);

    return(obj->hidden);
}

gboolean mdh_button_execute_callback(MdhButton *obj, GError **err)
{
    g_return_val_if_fail(obj != NULL, FALSE);
    g_return_val_if_fail(obj->callback != NULL, FALSE);

    obj->callback(Toolbar);

    return(TRUE);
}

void mdh_button_set_relief(MdhButton *obj, GtkReliefStyle relief)
{
    MdhButtonWidgets *widgets;

    g_return_if_fail(obj != NULL);

    widgets = &obj->widgets;

    gtk_button_set_relief(GTK_BUTTON(widgets->button), relief);
}

void mdh_button_set_tip(MdhButton *obj, const gchar *text)
{
    GtkTooltips *tips;

    g_return_if_fail(obj != NULL);
    g_return_if_fail(obj->toolbar != NULL);

    tips = MDH_TOOLBAR(obj->toolbar)->widgets.tips;

    g_return_if_fail(tips != NULL);

    mdh_widget_set_tip(tips, obj->widgets.button, text);
}
