/*
 * mdh (MailDooHicky), a GTK+-based toolbar.
 *
 * Copyright (c) 2003-2005 by Mike Hokenson <mdh at gozer dot org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <sys/types.h>
#include <signal.h>
#include <unistd.h>

#include <gtk/gtk.h>

#ifdef HAVE_GINI_H
#include <gini.h>
#endif

#include "mdh.h"

#include "toolbar.h"

#include "button.h"
#include "custom.h"
#include "panel.h"

#include "panel_time.h"
#include "panel_mail.h"
#include "panel_cpu.h"
#include "panel_mem.h"
#include "panel_disk.h"
#include "panel_net.h"
#include "panel_weather.h"
#include "panel_uptime.h"
#include "panel_custom.h"

#include "config.h"
#include "error.h"
#include "util.h"

#if (!defined(GINI_CHECK_VERSION) || !GINI_CHECK_VERSION(0, 4, 0))
# error "libgini version is too old"
#endif

static GIni *ini;

void mdh_config_load(const gchar *file)
{
    g_return_if_fail(file != NULL);

    gini_init();

    if(!ini)
        if(IS_FILE(file)) {
            GError *err = NULL;

            if(!(ini = gini_load(file, &err))) {
                ADD_LOG("%s", err->message);
                g_error_free(err);
            }
        }

    if(!ini)
        ini = gini_new(file);
}

void mdh_config_apply(void)
{
    MdhPanelFlags flags;

    const gchar *sval;
    gboolean     bval;
    gint         ival;

    gint thresh, i;

    g_return_if_fail(ini != NULL);

    /* general */

    if(gini_has_section(ini, "toolbar")) {
        if(gini_entry_get_int(ini, "toolbar", "geom_x", &ival) && ival > -100)
            Toolbar->geom_x = ival;

        if(gini_entry_get_int(ini, "toolbar", "geom_y", &ival) && ival > -100)
            Toolbar->geom_y = ival;

        if(gini_entry_get_string(ini, "toolbar", "layer", &sval)) {
            if(!g_ascii_strcasecmp(sval, "above"))
                Toolbar->layer = MDH_WINDOW_LAYER_ABOVE;
            else if(!g_ascii_strcasecmp(sval, "below"))
                Toolbar->layer = MDH_WINDOW_LAYER_BELOW;
            else
                Toolbar->layer = MDH_WINDOW_LAYER_NORMAL;
        }

        if(gini_entry_get_boolean(ini, "toolbar", "sticky", &bval))
            Toolbar->sticky = bval;

        if(gini_entry_get_boolean(ini, "toolbar", "tooltips", &bval))
            Toolbar->tooltips = bval;

        if(gini_entry_get_string(ini, "toolbar", "font", &sval)) {
            g_free(Toolbar->font);
            Toolbar->font = g_strdup(sval);
        }

        if(gini_entry_get_string(ini, "toolbar", "button_relief", &sval)) {
            if(!g_ascii_strcasecmp(sval, "half"))
                Toolbar->button_relief = GTK_RELIEF_HALF;
            else if(!g_ascii_strcasecmp(sval, "none"))
                Toolbar->button_relief = GTK_RELIEF_NONE;
            else  /* normal */
                Toolbar->button_relief = GTK_RELIEF_NORMAL;
        }

        if(gini_entry_get_string(ini, "toolbar", "panel_shadow", &sval)) {
            if(!g_ascii_strcasecmp(sval, "none"))
                Toolbar->panel_shadow = GTK_SHADOW_NONE;
            else if(!g_ascii_strcasecmp(sval, "etched_in"))
                Toolbar->panel_shadow = GTK_SHADOW_ETCHED_IN;
            else if(!g_ascii_strcasecmp(sval, "etched_out"))
                Toolbar->panel_shadow = GTK_SHADOW_ETCHED_OUT;
            else if(!g_ascii_strcasecmp(sval, "out"))
                Toolbar->panel_shadow = GTK_SHADOW_OUT;
            else  /* in */
                Toolbar->panel_shadow = GTK_SHADOW_IN;
        }
    }

    /* general */

    if(!gini_entry_get_string(ini, "general", "browser", &sval))
        gini_entry_set_string(ini, "general", "browser", mdh_get_browser());

    /* run-related */

    if(!gini_entry_get_boolean(ini, "run", "save", &bval))
        gini_entry_set_boolean(ini, "run", "save", FALSE);

    if(!gini_entry_get_boolean(ini, "run", "close", &bval))
        gini_entry_set_boolean(ini, "run", "close", TRUE);

    if(!gini_entry_get_string(ini, "run", "xterm", &sval))
        gini_entry_set_string(ini, "run", "xterm", mdh_get_xterm());

    /* scratch-pad-related */

    if(!gini_entry_get_boolean(ini, "scratch_pad", "save", &bval))
        gini_entry_set_boolean(ini, "scratch_pad", "save", TRUE);

    /* mixer */

#if _MDH_HAS_MIXER
    if(!gini_entry_get_string(ini, "mixer", "device", &sval))
        gini_entry_set_string(ini, "mixer", "device", "/dev/mixer");
#endif

    /* buttons */

    if(gini_has_section(ini, "buttons")) {
        if(gini_entry_get_boolean(ini, "buttons", "menu", &bval))
            B_Menu->enabled = bval;

        if(gini_entry_get_boolean(ini, "buttons", "toggle", &bval))
            B_Toggle->enabled = bval;

        if(gini_entry_get_boolean(ini, "buttons", "run", &bval))
            B_Run->enabled = bval;

        if(gini_entry_get_boolean(ini, "buttons", "pad", &bval))
            B_Pad->enabled = bval;

#if _MDH_HAS_MIXER
        if(gini_entry_get_boolean(ini, "buttons", "volume", &bval))
            B_Volume->enabled = bval;
#endif
    }

    /* custom buttons */

    if(gini_has_section(ini, "button_custom")) {
        /* free defaults or previous values */
        for(i = 0; i < _MDH_BUTTON_COUNT_CUSTOM_MAX; i++)
            mdh_custom_free_values(B_Cust[i]);

        mdh_custom_count_set(0);

        if(gini_entry_get_int(ini, "button_custom", "count", &ival))
            if(ival >= 0 && ival <= _MDH_BUTTON_COUNT_CUSTOM_MAX)
                mdh_custom_count_set(ival);

        for(i = 0; i < mdh_custom_count(); i++) {
            const gchar *value   = NULL, *command = NULL,
                        *tooltip = NULL, *icon    = NULL;
            gboolean enabled = FALSE;

            gchar *p = g_strdup_printf("button_custom_%d", i);

            if(gini_has_section(ini, p)) {
                gini_entry_get_boolean(ini, p, "enabled", &enabled);
                gini_entry_get_string(ini,  p, "value",   &value);
                gini_entry_get_string(ini,  p, "command", &command);
                gini_entry_get_string(ini,  p, "tooltip", &tooltip);
                gini_entry_get_string(ini,  p, "icon",    &icon);
            }

            g_free(p);

            mdh_custom_set_values(B_Cust[i],
                                  enabled,
                                  value,
                                  command,
                                  tooltip,
                                  icon);
        }

        /* setup the remainder */
        for(i = mdh_custom_count(); i < _MDH_BUTTON_COUNT_CUSTOM_MAX; i++)
            mdh_custom_set_values(B_Cust[i], FALSE, NULL, NULL, NULL, NULL);
    }

    /* panels */

#if _MDH_HAS_TIME
    if(gini_has_section(ini, "panel_time")) {
        if(gini_entry_get_boolean(ini, "panel_time", "enabled", &bval))
            P_Time->enabled = bval;

        if(gini_entry_get_string(ini, "panel_time", "display", &sval)) {
            g_free(P_Time->display);
            P_Time->display = g_strdup(sval);
        }

        if(gini_entry_get_string(ini, "panel_time", "command", &sval)) {
            g_free(P_Time->command);
            P_Time->command = mdh_expand_tilde(sval);
        }

        if(!*P_Time->display) {
            g_free(P_Time->display);
            P_Time->display = g_strdup(_TIME_DIS);

            ADD_LOG("Invalid display format, defaulting to %s",
                    P_Time->display);
        }
    }
#endif /* _MDH_HAS_TIME */

    if(gini_has_section(ini, "panel_mail")) {
        flags = mdh_panel_flags(P_Mail[0]);

        if(gini_entry_get_boolean(ini, "panel_mail", "bold", &bval)) {
            if(bval)
                flags |=  MDH_PANEL_FLAG_BOLD;
            else
                flags &= ~MDH_PANEL_FLAG_BOLD;
        }

        thresh = mdh_panel_mail_threshold(P_Mail[0]);

        if(gini_entry_get_int(ini, "panel_mail", "threshold", &ival))
            if(ival >= _MAIL_THRESH_MIN && ival <= _MAIL_THRESH_MAX)
                thresh = ival;

        if(gini_entry_get_int(ini, "panel_mail", "count", &ival))
            if(ival >= _MDH_PANEL_COUNT_MAIL_MIN &&
               ival <= _MDH_PANEL_COUNT_MAIL_MAX)
                mdh_panel_mail_count_set(P_Mail[0], ival);

        for(i = 0; i < mdh_panel_mail_count(P_Mail[0]); i++) {
            mdh_panel_flags_set(P_Mail[i], flags);
            mdh_panel_mail_threshold_set(P_Mail[i], thresh);
        }
    }

    for(i = 0; i < mdh_panel_mail_count(P_Mail[0]); i++) {
        gchar *p = g_strdup_printf("panel_mail_%d", i);

        if(gini_has_section(ini, p)) {
            if(gini_entry_get_boolean(ini, p, "enabled", &bval))
                P_Mail[i]->enabled = bval;

            if(gini_entry_get_string(ini, p, "value", &sval)) {
                g_free(P_Mail[i]->value);
                P_Mail[i]->value = mdh_expand_tilde(sval);
            }

            if(gini_entry_get_string(ini, p, "display", &sval)) {
                g_free(P_Mail[i]->display);
                P_Mail[i]->display = g_strdup(sval);
            }

            if(gini_entry_get_string(ini, p, "command", &sval)) {
                g_free(P_Mail[i]->command);
                P_Mail[i]->command = mdh_expand_tilde(sval);
            }

            if(gini_entry_get_int(ini, p, "interval", &ival))
                if(ival >= _MAIL_MIN && ival <= _MAIL_MAX)
                    P_Mail[i]->interval = ival;

            g_free(P_Mail[i]->tooltip);
            P_Mail[i]->tooltip = g_strdup_printf("Mail: %s", P_Mail[i]->value);

            if(!*P_Mail[i]->display) {
                g_free(P_Mail[i]->display);
                P_Mail[i]->display = g_strdup(_MAIL_DIS);

                ADD_LOG("Invalid display format, defaulting to %s",
                        P_Mail[i]->display);
            }
        }

        g_free(p);
    }

#if _MDH_HAS_CPU
    if(gini_has_section(ini, "panel_cpu")) {
        flags = mdh_panel_flags(P_Cpu);

        if(gini_entry_get_boolean(ini, "panel_cpu", "bold", &bval)) {
            if(bval)
                flags |=  MDH_PANEL_FLAG_BOLD;
            else
                flags &= ~MDH_PANEL_FLAG_BOLD;
        }

        mdh_panel_flags_set(P_Cpu, flags);

        if(gini_entry_get_boolean(ini, "panel_cpu", "enabled", &bval))
            P_Cpu->enabled = bval;

        if(gini_entry_get_int(ini, "panel_cpu", "threshold", &ival))
            if(ival >= _CPU_THRESH_MIN && ival <= _CPU_THRESH_MAX)
                mdh_panel_cpu_threshold_set(P_Cpu, ival);

        if(gini_entry_get_string(ini, "panel_cpu", "display", &sval)) {
            g_free(P_Cpu->display);
            P_Cpu->display = g_strdup(sval);
        }
        
        if(gini_entry_get_string(ini, "panel_cpu", "command", &sval)) {
            g_free(P_Cpu->command);
            P_Cpu->command = mdh_expand_tilde(sval);
        }

        if(gini_entry_get_int(ini, "panel_cpu", "interval", &ival))
            if(ival >= _CPU_MIN && ival <= _CPU_MAX)
                P_Cpu->interval = ival;

        if(!*P_Cpu->display) {
            g_free(P_Cpu->display);
            P_Cpu->display = g_strdup(_CPU_DIS);

            ADD_LOG("Invalid display format, defaulting to %s", P_Cpu->display);
        }
    }
#endif /* _MDH_HAS_CPU */

#if _MDH_HAS_MEM
    if(gini_has_section(ini, "panel_mem")) {
        flags = mdh_panel_flags(P_Mem);

        if(gini_entry_get_boolean(ini, "panel_mem", "bold", &bval)) {
            if(bval)
                flags |=  MDH_PANEL_FLAG_BOLD;
            else
                flags &= ~MDH_PANEL_FLAG_BOLD;
        }

        mdh_panel_flags_set(P_Mem, flags);

        if(gini_entry_get_int(ini, "panel_mem", "threshold", &ival))
            if(ival >= _MEM_THRESH_MIN && ival <= _MEM_THRESH_MAX)
                mdh_panel_mem_threshold_set(P_Mem, ival);

        if(gini_entry_get_boolean(ini, "panel_mem", "enabled", &bval))
            P_Mem->enabled = bval;

        if(gini_entry_get_string(ini, "panel_mem", "display", &sval)) {
            g_free(P_Mem->display);
            P_Mem->display = g_strdup(sval);
        }

        if(gini_entry_get_string(ini, "panel_mem", "command", &sval)) {
            g_free(P_Mem->command);
            P_Mem->command = mdh_expand_tilde(sval);
        }

        if(gini_entry_get_int(ini, "panel_mem", "interval", &ival))
            if(ival >= _MEM_MIN && ival <= _MEM_MAX)
                P_Mem->interval = ival;

        if(!*P_Mem->display) {
            g_free(P_Mem->display);
            P_Mem->display = g_strdup(_MEM_DIS);

            ADD_LOG("Invalid display format, defaulting to %s", P_Mem->display);
        }
    }
#endif /* _MDH_HAS_MEM */

#if _MDH_HAS_DISK
    if(gini_has_section(ini, "panel_disk")) {
        flags = mdh_panel_flags(P_Disk[0]);

        if(gini_entry_get_boolean(ini, "panel_disk", "bold", &bval)) {
            if(bval)
                flags |=  MDH_PANEL_FLAG_BOLD;
            else
                flags &= ~MDH_PANEL_FLAG_BOLD;
        }

        thresh = mdh_panel_disk_threshold(P_Disk[0]);

        if(gini_entry_get_int(ini, "panel_disk", "threshold", &ival))
            if(ival >= _DISK_THRESH_MIN && ival <= _DISK_THRESH_MAX)
                thresh = ival;

        if(gini_entry_get_int(ini, "panel_disk", "count", &ival))
            if(ival >= _MDH_PANEL_COUNT_DISK_MIN &&
               ival <= _MDH_PANEL_COUNT_DISK_MAX)
                mdh_panel_disk_count_set(P_Disk[0], ival);

        for(i = 0; i < mdh_panel_disk_count(P_Disk[0]); i++) {
            mdh_panel_flags_set(P_Disk[i], flags);
            mdh_panel_disk_threshold_set(P_Disk[i], thresh);
        }
    }

    for(i = 0; i < mdh_panel_disk_count(P_Disk[0]); i++) {
        gchar *p = g_strdup_printf("panel_disk_%d", i);

        if(gini_has_section(ini, p)) {
            if(gini_entry_get_boolean(ini, p, "enabled", &bval))
                P_Disk[i]->enabled = bval;

            if(gini_entry_get_string(ini, p, "value", &sval)) {
                g_free(P_Disk[i]->value);
                P_Disk[i]->value = g_strdup(sval);
            }

            if(gini_entry_get_string(ini, p, "display", &sval)) {
                g_free(P_Disk[i]->display);
                P_Disk[i]->display = g_strdup(sval);
            }

            if(gini_entry_get_string(ini, p, "command", &sval)) {
                g_free(P_Disk[i]->command);
                P_Disk[i]->command = mdh_expand_tilde(sval);
            }

            if(gini_entry_get_int(ini, p, "interval", &ival))
                if(ival >= _DISK_MIN && ival <= _DISK_MAX)
                    P_Disk[i]->interval = ival;

            g_free(P_Disk[i]->tooltip);
            P_Disk[i]->tooltip = g_strdup_printf("Disk: %s", P_Disk[i]->value);

            if(!*P_Disk[i]->display) {
                g_free(P_Disk[i]->display);
                P_Disk[i]->display = g_strdup(_DISK_DIS);

                ADD_LOG("Invalid display format, defaulting to %s",
                        P_Disk[i]->display);
            }
        }

        g_free(p);
    }
#endif /* _MDH_HAS_DISK */

#if _MDH_HAS_NET
    if(gini_has_section(ini, "panel_net"))
        if(gini_entry_get_int(ini, "panel_net", "count", &ival))
            if(ival >= _MDH_PANEL_COUNT_NET_MIN &&
               ival <= _MDH_PANEL_COUNT_NET_MAX)
                mdh_panel_net_count_set(P_Net[0], ival);

    for(i = 0; i < mdh_panel_net_count(P_Net[0]); i++) {
        gchar *p = g_strdup_printf("panel_net_%d", i);

        if(gini_has_section(ini, p)) {
            if(gini_entry_get_boolean(ini, p, "enabled", &bval))
                P_Net[i]->enabled = bval;

#ifndef __sun__
            if(gini_entry_get_string(ini, p, "value", &sval)) {
                g_free(P_Net[i]->value);
                P_Net[i]->value = g_strdup(sval);
            }
#endif

            if(gini_entry_get_string(ini, p, "display", &sval)) {
                g_free(P_Net[i]->display);
                P_Net[i]->display = g_strdup(sval);
            }

            if(gini_entry_get_string(ini, p, "command", &sval)) {
                g_free(P_Net[i]->command);
                P_Net[i]->command = mdh_expand_tilde(sval);
            }

            if(gini_entry_get_int(ini, p, "interval", &ival))
                if(ival >= _NET_MIN && ival <= _NET_MAX)
                    P_Net[i]->interval = ival;

#ifndef __sun__
            g_free(P_Net[i]->tooltip);
            P_Net[i]->tooltip = g_strdup_printf("Net: %s", P_Net[i]->value);
#endif

            if(!*P_Net[i]->display) {
                g_free(P_Net[i]->display);
                P_Net[i]->display = g_strdup(_NET_DIS);

                ADD_LOG("Invalid display format, defaulting to %s",
                        P_Net[i]->display);
            }
        }

        g_free(p);
    }
#endif /* _MDH_HAS_NET */

#if _MDH_HAS_WEATHER
    if(gini_has_section(ini, "panel_weather")) {
       flags = mdh_panel_flags(P_Weather);
        
       if(gini_entry_get_boolean(ini, "panel_weather", "icon", &bval)) {
           if(bval)
               flags |=  MDH_PANEL_FLAG_ICON;
           else
               flags &= ~MDH_PANEL_FLAG_ICON;
       }
            
       mdh_panel_flags_set(P_Weather, flags);

       if(gini_entry_get_boolean(ini, "panel_weather", "enabled", &bval))
           P_Weather->enabled = bval;

       if(gini_entry_get_string(ini, "panel_weather", "value", &sval)) {
           g_free(P_Weather->value);
           P_Weather->value = g_strdup(sval);
       }

       if(gini_entry_get_string(ini, "panel_weather", "display", &sval)) {
           g_free(P_Weather->display);
           P_Weather->display = g_strdup(sval);
       }

       if(gini_entry_get_string(ini, "panel_weather", "command", &sval)) {
           g_free(P_Weather->command);
           P_Weather->command = mdh_expand_tilde(sval);
       }

       if(gini_entry_get_int(ini, "panel_weather", "interval", &ival))
           if(ival >= _WEATHER_MIN && ival <= _WEATHER_MAX)
               P_Weather->interval = ival;

       if(gini_entry_get_string(ini, "panel_weather", "unit", &sval))
           if(!g_ascii_strcasecmp(sval, _WEATHER_UNIT_M))
               mdh_panel_weather_unit_metric_set(P_Weather, TRUE);

       if(!_WEATHER_ID_VALID(P_Weather->value)) {
           g_free(P_Weather->value);
           P_Weather->value = g_strdup(_WEATHER_LOC);

           ADD_LOG("Invalid LocID, defaulting to %s", P_Weather->value);
       }

       if(!*P_Weather->display) {
           g_free(P_Weather->display);
           P_Weather->display = g_strdup(_WEATHER_DIS);

           ADD_LOG("Invalid display format, defaulting to %s",
                   P_Weather->display);
       }

       g_free(P_Weather->tooltip);
       P_Weather->tooltip = g_strdup_printf("Weather: %s", P_Weather->value);
    }
#endif /* _MDH_HAS_WEATHER */

#if _MDH_HAS_UPTIME
    if(gini_has_section(ini, "panel_uptime")) {
        if(gini_entry_get_boolean(ini, "panel_uptime", "enabled", &bval))
            P_Uptime->enabled = bval;

        if(gini_entry_get_string(ini, "panel_uptime", "display", &sval)) {
            g_free(P_Uptime->display);
            P_Uptime->display = g_strdup(sval);
        }

        if(gini_entry_get_string(ini, "panel_uptime", "command", &sval)) {
            g_free(P_Uptime->command);
            P_Uptime->command = mdh_expand_tilde(sval);
        }

        if(gini_entry_get_int(ini, "panel_uptime", "interval", &ival))
            if(ival >= _UPTIME_MIN && ival <= _UPTIME_MAX)
                P_Uptime->interval = ival;

        if(!*P_Uptime->display) {
            g_free(P_Uptime->display);
            P_Uptime->display = g_strdup(_UPTIME_DIS);

            ADD_LOG("Invalid display format, defaulting to %s",
                    P_Uptime->display);
        }
    }
#endif /* _MDH_HAS_UPTIME */

    if(gini_has_section(ini, "panel_custom"))
        if(gini_entry_get_int(ini, "panel_custom", "count", &ival))
            if(ival >= _MDH_PANEL_COUNT_CUSTOM_MIN &&
               ival <= _MDH_PANEL_COUNT_CUSTOM_MAX)
                mdh_panel_custom_count_set(P_Custom[0], ival);

    for(i = 0; i < mdh_panel_custom_count(P_Custom[0]); i++) {
        gchar *p = g_strdup_printf("panel_custom_%d", i);

        if(gini_has_section(ini, p)) {
            if(gini_entry_get_boolean(ini, p, "enabled", &bval))
                P_Custom[i]->enabled = bval;

            if(gini_entry_get_string(ini, p, "value", &sval)) {
                g_free(P_Custom[i]->value);
                P_Custom[i]->value = mdh_expand_tilde(sval);
            }

            if(gini_entry_get_string(ini, p, "command", &sval)) {
                g_free(P_Custom[i]->command);
                P_Custom[i]->command = mdh_expand_tilde(sval);
            }

            if(gini_entry_get_int(ini, p, "interval", &ival))
                if(ival >= _CUSTOM_MIN && ival <= _CUSTOM_MAX)
                    P_Custom[i]->interval = ival;

            g_free(P_Custom[i]->tooltip);
            P_Custom[i]->tooltip = g_strdup_printf("Custom: %s",
                                                   P_Custom[i]->value);
        }

        g_free(p);
    }
}

void mdh_config_save(void)
{
    gint i;

    g_return_if_fail(ini != NULL);

    /* general */

    gini_entry_set_int(ini, "toolbar", "geom_x", Toolbar->geom_x);
    gini_entry_set_int(ini, "toolbar", "geom_y", Toolbar->geom_y);

    {
        const gchar *layer;

        switch(Toolbar->layer) {
            case MDH_WINDOW_LAYER_ABOVE:
                layer = "above";
                break;
            case MDH_WINDOW_LAYER_BELOW:
                layer = "below";
                break;
            case MDH_WINDOW_LAYER_NORMAL:
            default:
                layer = "normal";
                break;
        }

        gini_entry_set_string(ini, "toolbar", "layer", layer);
    }

    gini_entry_set_boolean(ini, "toolbar", "sticky", Toolbar->sticky);

    gini_entry_set_boolean(ini, "toolbar", "tooltips", Toolbar->tooltips);

    gini_entry_set_string(ini, "toolbar", "font", Toolbar->font);

    {
        const gchar *button_relief;

        switch(Toolbar->button_relief) {
            case GTK_RELIEF_HALF:
                button_relief = "half";
                break;
            case GTK_RELIEF_NONE:
                button_relief = "none";
                break;
            case GTK_RELIEF_NORMAL:
            default:
                button_relief = "normal";
                break;
        }

        gini_entry_set_string(ini, "toolbar", "button_relief", button_relief);
    }

    {
        const gchar *panel_shadow;

        switch(Toolbar->panel_shadow) {
            case GTK_SHADOW_NONE:
                panel_shadow = "none";
                break;
            case GTK_SHADOW_ETCHED_IN:
                panel_shadow = "etched_in";
                break;
            case GTK_SHADOW_ETCHED_OUT:
                panel_shadow = "etched_out";
                break;
            case GTK_SHADOW_OUT:
                panel_shadow = "out";
                break;
            case GTK_SHADOW_IN:
            default:
                panel_shadow = "in";
                break;
        }

        gini_entry_set_string(ini, "toolbar", "panel_shadow", panel_shadow);
    }

    /* buttons */

    gini_entry_set_boolean(ini, "buttons", "menu", B_Menu->enabled);
    gini_entry_set_boolean(ini, "buttons", "toggle", B_Toggle->enabled);
    gini_entry_set_boolean(ini, "buttons", "run", B_Run->enabled);
    gini_entry_set_boolean(ini, "buttons", "pad", B_Pad->enabled);

#if _MDH_HAS_MIXER
    gini_entry_set_boolean(ini, "buttons", "volume", B_Volume->enabled);
#endif

    /* mixer */

#if _MDH_HAS_MIXER
    gini_entry_set_string(ini, "mixer", "device", "/dev/mixer");
#endif

    /* custom buttons */

    gini_entry_set_int(ini, "button_custom", "count", mdh_custom_count());

    for(i = 0; i < mdh_custom_count(); i++) {
        gchar *p = g_strdup_printf("button_custom_%d", i);

        gini_entry_set_boolean(ini, p, "enabled", B_Cust[i]->enabled);
        gini_entry_set_string(ini, p, "value", B_Cust[i]->value);
        gini_entry_set_string(ini, p, "command", B_Cust[i]->command);
        gini_entry_set_string(ini, p, "icon", B_Cust[i]->icon);
        gini_entry_set_string(ini, p, "tooltip", B_Cust[i]->tooltip);

        g_free(p);
    }

    /* panels */

#if _MDH_HAS_TIME
    gini_entry_set_boolean(ini, "panel_time", "enabled", P_Time->enabled);
    gini_entry_set_string(ini, "panel_time", "display", P_Time->display);
    gini_entry_set_string(ini, "panel_time", "command", P_Time->command);
#endif /* _MDH_HAS_TIME */

    gini_entry_set_int(ini, "panel_mail", "count",
                       mdh_panel_mail_count(P_Mail[0]));
    gini_entry_set_boolean(ini, "panel_mail", "bold",
                        (mdh_panel_flags(P_Mail[0]) & MDH_PANEL_FLAG_BOLD));
    gini_entry_set_int(ini, "panel_mail", "threshold",
                       mdh_panel_mail_threshold(P_Mail[0]));

    for(i = 0; i < mdh_panel_mail_count(P_Mail[0]); i++) {
        gchar *p = g_strdup_printf("panel_mail_%d", i);

        gini_entry_set_boolean(ini, p, "enabled", P_Mail[i]->enabled);
        gini_entry_set_string(ini, p, "value", P_Mail[i]->value);
        gini_entry_set_string(ini, p, "display", P_Mail[i]->display);
        gini_entry_set_string(ini, p, "command", P_Mail[i]->command);
        gini_entry_set_int(ini, p, "interval", P_Mail[i]->interval);
    }

#if _MDH_HAS_CPU
    gini_entry_set_boolean(ini, "panel_cpu", "bold",
                        (mdh_panel_flags(P_Cpu) & MDH_PANEL_FLAG_BOLD));
    gini_entry_set_int(ini, "panel_cpu", "threshold",
                       mdh_panel_cpu_threshold(P_Cpu));

    gini_entry_set_boolean(ini, "panel_cpu", "enabled", P_Cpu->enabled);
    gini_entry_set_string(ini, "panel_cpu", "display", P_Cpu->display);
    gini_entry_set_string(ini, "panel_cpu", "command", P_Cpu->command);
    gini_entry_set_int(ini, "panel_cpu", "interval", P_Cpu->interval);
#endif /* _MDH_HAS_CPU */

#if _MDH_HAS_MEM
    gini_entry_set_boolean(ini, "panel_mem", "bold",
                        (mdh_panel_flags(P_Mem) & MDH_PANEL_FLAG_BOLD));
    gini_entry_set_int(ini, "panel_mem", "threshold",
                       mdh_panel_mem_threshold(P_Mem));

    gini_entry_set_boolean(ini, "panel_mem", "enabled", P_Mem->enabled);
    gini_entry_set_string(ini, "panel_mem", "display", P_Mem->display);
    gini_entry_set_string(ini, "panel_mem", "command", P_Mem->command);
    gini_entry_set_int(ini, "panel_mem", "interval", P_Mem->interval);
#endif /* _MDH_HAS_MEM */

#if _MDH_HAS_DISK
    gini_entry_set_int(ini, "panel_disk", "count", mdh_panel_disk_count(P_Disk[0]));
    gini_entry_set_boolean(ini, "panel_disk", "bold",
                        (mdh_panel_flags(P_Disk[0]) & MDH_PANEL_FLAG_BOLD));
    gini_entry_set_int(ini, "panel_disk", "threshold",
                       mdh_panel_disk_threshold(P_Disk[0]));

    for(i = 0; i < mdh_panel_disk_count(P_Disk[0]); i++) {
        gchar *p = g_strdup_printf("panel_disk_%d", i);

        gini_entry_set_boolean(ini, p, "enabled", P_Disk[i]->enabled);
        gini_entry_set_string(ini, p, "value", P_Disk[i]->value);
        gini_entry_set_string(ini, p, "display", P_Disk[i]->display);
        gini_entry_set_string(ini, p, "command", P_Disk[i]->command);
        gini_entry_set_int(ini, p, "interval", P_Disk[i]->interval);
    }
#endif /* _MDH_HAS_DISK */

#if _MDH_HAS_NET
    gini_entry_set_int(ini, "panel_net", "count",
                       mdh_panel_net_count(P_Net[0]));

    for(i = 0; i < mdh_panel_net_count(P_Net[0]); i++) {
        gchar *p = g_strdup_printf("panel_net_%d", i);

        gini_entry_set_boolean(ini, p, "enabled", P_Net[i]->enabled);
#ifndef __sun__
        gini_entry_set_string(ini, p, "value", P_Net[i]->value);
#endif
        gini_entry_set_string(ini, p, "display", P_Net[i]->display);
        gini_entry_set_string(ini, p, "command", P_Net[i]->command);
        gini_entry_set_int(ini, p, "interval", P_Net[i]->interval);
    }
#endif /* _MDH_HAS_NET */

#if _MDH_HAS_WEATHER
    gini_entry_set_boolean(ini, "panel_weather", "icon",
                        (mdh_panel_flags(P_Weather) & MDH_PANEL_FLAG_ICON));

    gini_entry_set_boolean(ini, "panel_weather", "enabled", P_Weather->enabled);
    gini_entry_set_string(ini, "panel_weather", "value", P_Weather->value);
    gini_entry_set_string(ini, "panel_weather", "display", P_Weather->display);
    gini_entry_set_string(ini, "panel_weather", "command", P_Weather->command);
    gini_entry_set_int(ini, "panel_weather", "interval", P_Weather->interval);
    gini_entry_set_string(ini, "panel_weather", "unit",
                       mdh_panel_weather_unit(P_Weather));
#endif /* _MDH_HAS_WEATHER */

#if _MDH_HAS_UPTIME
    gini_entry_set_boolean(ini, "panel_uptime", "enabled", P_Uptime->enabled);
    gini_entry_set_string(ini, "panel_uptime", "display", P_Uptime->display);
    gini_entry_set_string(ini, "panel_uptime", "command", P_Uptime->command);
    gini_entry_set_int(ini, "panel_uptime", "interval", P_Uptime->interval);
#endif /* _MDH_HAS_UPTIME */

    gini_entry_set_int(ini, "panel_custom", "count",
                                           mdh_panel_custom_count(P_Custom[0]));

    for(i = 0; i < mdh_panel_custom_count(P_Custom[0]); i++) {
        gchar *p = g_strdup_printf("panel_custom_%d", i);

        gini_entry_set_boolean(ini, p, "enabled", P_Custom[i]->enabled);
        gini_entry_set_string(ini, p, "value", P_Custom[i]->value);
        gini_entry_set_string(ini, p, "command", P_Custom[i]->command);
        gini_entry_set_int(ini, p, "interval", P_Custom[i]->interval);
    }

    {
        GError *err = NULL;

        if(!gini_save(ini, NULL, &err)) {
            ADD_LOG("%s", err->message);
            g_error_free(err);
        }
    }
}

void mdh_config_free(void)
{
    gini_unref(ini);
}

#define _LOCK ".lock"

gboolean mdh_config_locked(const gchar *file, gchar **dpy, pid_t *pid)
{
#ifndef G_OS_WIN32
    gchar path[PATH_MAX], *lock;
    
    pid_t p;

    g_return_val_if_fail(file != NULL, FALSE);

    g_strlcpy(path, file,  sizeof(path));
    g_strlcat(path, _LOCK, sizeof(path));

    if(!IS_LINK(path))
        return(FALSE);

    {
        GError *err = NULL;

        if(!(lock = g_file_read_link(path, &err))) {
            fprintf(stderr, "%s\n", err->message);
            g_error_free(err);
            return(FALSE);
        }
    }

    if(sscanf(lock, ":%*d.%*d#%d", &p) != 1 || p < 100 || kill(p, 0)) {
        g_free(lock);
        return(FALSE);
    }

    {
        gchar *cp;

        if((cp = strrchr(lock, '#')))
            *cp = 0;
    }

    if(dpy)
        *dpy = lock;
    else
        g_free(lock);

    if(pid)
        *pid = p;
#endif /* G_OS_WIN32 */

    return(TRUE);
}

gboolean mdh_config_lock(const gchar *file)
{
#ifndef G_OS_WIN32
    gchar path[PATH_MAX], lock[PATH_MAX];

    const gchar *dpy;

    g_return_val_if_fail(file != NULL, FALSE);

    g_strlcpy(path, file,  sizeof(path));
    g_strlcat(path, _LOCK, sizeof(path));

    if(!(dpy = g_getenv("DISPLAY")) || !*dpy)
        return(FALSE);

    g_snprintf(lock, sizeof(lock), "%s#%d", dpy, getpid());

    if(symlink(lock, path) == -1) {
        perror("symlink");
        return(FALSE);
    }
#endif /* G_OS_WIN32 */

    return(TRUE);
}

gboolean mdh_config_unlock(const gchar *file)
{
#ifndef G_OS_WIN32
    gchar path[PATH_MAX];

    g_return_val_if_fail(file != NULL, FALSE);

    g_strlcpy(path, file,  sizeof(path));
    g_strlcat(path, _LOCK, sizeof(path));

    if(IS_LINK(path))
        if(unlink(path) == -1) {
            perror("unlink");
            return(FALSE);
        }
#endif /* G_OS_WIN32 */

    return(TRUE);
}

gboolean mdh_config_get_pointer(const gchar *section,
                                const gchar *variable,
                                gpointer *value)
{
    g_return_val_if_fail(ini != NULL, FALSE);

    return(gini_entry_get_pointer(ini, section, variable, value));
}

void mdh_config_set_pointer(const gchar *section,
                            const gchar *variable,
                            gpointer value)
{
    g_return_if_fail(ini != NULL);

    gini_entry_set_pointer(ini, section, variable, value);
}

gboolean mdh_config_get_string(const gchar *section,
                               const gchar *variable,
                               const gchar **value)
{
    g_return_val_if_fail(ini != NULL, FALSE);

    return(gini_entry_get_string(ini, section, variable, value));
}

void mdh_config_set_string(const gchar *section,
                           const gchar *variable,
                           const gchar *value)
{
    g_return_if_fail(ini != NULL);

    gini_entry_set_string(ini, section, variable, value);
}

gboolean mdh_config_get_boolean(const gchar *section,
                                const gchar *variable,
                                gboolean *value)
{
    g_return_val_if_fail(ini != NULL, FALSE);

    return(gini_entry_get_boolean(ini, section, variable, value));
}

void mdh_config_set_boolean(const gchar *section,
                            const gchar *variable,
                            gboolean value)
{
    g_return_if_fail(ini != NULL);

    gini_entry_set_boolean(ini, section, variable, value);
}

gboolean mdh_config_get_int(const gchar *section,
                            const gchar *variable,
                            gint *value)
{
    g_return_val_if_fail(ini != NULL, FALSE);

    return(gini_entry_get_int(ini, section, variable, value));
}

void mdh_config_set_int(const gchar *section,
                        const gchar *variable,
                        gint value)
{
    g_return_if_fail(ini != NULL);

    gini_entry_set_int(ini, section, variable, value);
}
