/*
 * mdh (MailDooHicky), a GTK+-based toolbar.
 *
 * Copyright (c) 2003-2005 by Mike Hokenson <mdh at gozer dot org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <unistd.h>

#include <gtk/gtk.h>

#include "toolbar.h"

#include "custom.h"

#include "widget.h"
#include "window.h"
#include "util.h"

static gint custom_count;

const gchar *mdh_custom_icon_default(void)
{
    return(mdh_icon_path("default"));
}

MdhCustom *mdh_custom_new(gboolean enabled,
                          const gchar *value,
                          const gchar *command,
                          const gchar *tooltip,
                          const gchar *icon)
{
    MdhCustom *obj;

    obj          = g_new0(MdhCustom, 1);

    obj->toolbar = NULL;

    obj->hidden  = TRUE;

    mdh_custom_set_values(obj, enabled, value, command, tooltip, icon);

    obj->widgets.button = gtk_button_new();
    obj->widgets.label  = gtk_label_new(NULL);
    obj->widgets.icon   = gtk_image_new();
                          
    return(obj);          
}
                          
void mdh_custom_set_values(MdhCustom *obj,
                           gboolean enabled,
                           const gchar *value,
                           const gchar *command,
                           const gchar *tooltip,
                           const gchar *icon)
{
    g_return_if_fail(obj != NULL);

    obj->enabled = (command) ? enabled : FALSE;
    obj->value   = (value)   ? g_strdup(value)   : g_strdup("");
    obj->command = (command) ? g_strdup(command) : g_strdup("");
    obj->tooltip = (tooltip) ? g_strdup(tooltip) : g_strdup("");
    obj->icon    = (icon)    ? g_strdup(icon)    :
                               g_strdup(mdh_custom_icon_default());
}

void mdh_custom_free_values(MdhCustom *obj)
{
    g_return_if_fail(obj != NULL);
    
    g_free(obj->value);
    g_free(obj->icon);
    g_free(obj->command);    
    g_free(obj->tooltip);      
}

void mdh_custom_free(MdhCustom *obj)
{
    g_return_if_fail(obj != NULL);

    mdh_custom_free_values(obj);

    g_free(obj);
}

void mdh_custom_toggle(MdhCustom *obj)
{
    g_return_if_fail(obj != NULL);

    if(!obj->enabled)
        return;

    if(obj->hidden)
        mdh_custom_show(obj);
    else
        mdh_custom_hide(obj);
}

void mdh_custom_show(MdhCustom *obj)
{
    g_return_if_fail(obj != NULL);

    gtk_widget_show_all(obj->widgets.button);

    if(!obj->icon || !*obj->icon)
        gtk_widget_hide(obj->widgets.icon);
    
    obj->hidden = FALSE;
}
        
void mdh_custom_hide(MdhCustom *obj)
{
    g_return_if_fail(obj != NULL);
    
    gtk_widget_hide_all(obj->widgets.button);

    obj->hidden = TRUE;
}

void mdh_custom_reset(MdhCustom *obj)
{
    g_return_if_fail(obj != NULL);

    gtk_label_set_text(GTK_LABEL(obj->widgets.label), obj->value);
    
    if(obj->icon && *obj->icon && IS_FILE(obj->icon))
        gtk_image_set_from_file(GTK_IMAGE(obj->widgets.icon), obj->icon);

    if(obj->enabled)
        mdh_custom_show(obj);
    else
        mdh_custom_hide(obj);
}

void mdh_custom_foreach(gpointer toolbar, GFunc func, gpointer data)
{
    g_return_if_fail(toolbar != NULL);
    g_return_if_fail(func != NULL);

    g_slist_foreach(MDH_TOOLBAR(toolbar)->custom, func, data);
}

gboolean mdh_custom_hidden(MdhCustom *obj)
{
    g_return_val_if_fail(obj != NULL, FALSE);

    return(obj->hidden);
}

gboolean mdh_custom_execute_callback(MdhCustom *obj, GError **err)
{
    g_return_val_if_fail(obj != NULL, FALSE);
    g_return_val_if_fail(obj->command != NULL, FALSE);

    return(g_spawn_command_line_async(obj->command, err));
}

void mdh_custom_set_relief(MdhCustom *obj, GtkReliefStyle relief)
{
    MdhCustomWidgets *widgets;

    g_return_if_fail(obj != NULL);

    widgets = &obj->widgets;

    gtk_button_set_relief(GTK_BUTTON(widgets->button), relief);
}

void mdh_custom_set_tip(MdhCustom *obj, const gchar *text)
{
    GtkTooltips *tips;

    g_return_if_fail(obj != NULL);
    g_return_if_fail(obj->toolbar != NULL);

    tips = MDH_TOOLBAR(obj->toolbar)->widgets.tips;

    g_return_if_fail(tips != NULL);

    mdh_widget_set_tip(tips, obj->widgets.button, text);
}

void mdh_custom_set_font_desc(MdhCustom *obj, PangoFontDescription *font_desc)
{
    g_return_if_fail(obj != NULL);
    g_return_if_fail(font_desc != NULL);

    mdh_widget_set_font_desc(obj->widgets.label, font_desc);
}

void mdh_custom_set_font(MdhCustom *obj, const gchar *font)
{
    g_return_if_fail(obj != NULL);
    g_return_if_fail(font != NULL);

    mdh_widget_set_font(obj->widgets.label, font);
}

gint mdh_custom_count(void)
{
    return(custom_count);
}

void mdh_custom_count_set(gint count)
{
    g_return_if_fail(count >= 0);
    g_return_if_fail(count <= _MDH_BUTTON_COUNT_CUSTOM_MAX);

    custom_count = count;
}
