/*
 * mdh (MailDooHicky), a GTK+-based toolbar.
 *
 * Copyright (c) 2003-2005 by Mike Hokenson <mdh at gozer dot org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <sys/types.h>
#include <signal.h>
#include <unistd.h>

#include <gtk/gtk.h>

#include "getopt.h"

#include "mdh.h"

#include "toolbar.h"

#include "button.h"
#include "custom.h"
#include "panel.h"

#include "panel_time.h"
#include "panel_mail.h"
#include "panel_cpu.h"
#include "panel_mem.h"
#include "panel_disk.h"
#include "panel_net.h"
#include "panel_weather.h"
#include "panel_uptime.h"
#include "panel_custom.h"

#include "ui_run.h"
#include "ui_scratch_pad.h"
#include "ui_mixer.h"

#include "config.h"
#include "error.h"
#include "run_history.h"
#include "scratch_pad.h"
#include "threads.h"
#include "widget.h"
#include "window.h"
#include "util.h"
#include "x11.h"

static gchar *mdh_home,
             *mdh_cache,
             *mdh_config,
             *mdh_hist,
             *mdh_scratch,
             *mdh_browser,
             *mdh_xterm;

typedef void (*sig_handler)(gint);

static void set_signal(gint sig, sig_handler handler)
{
#ifdef HAVE_SIGACTION
    struct sigaction s;

    s.sa_handler = handler;
    s.sa_flags   = 0;

#ifdef SA_RESTART
    s.sa_flags  |= SA_RESTART;
#endif

    sigemptyset(&s.sa_mask);

    sigaction(sig, &s, 0);
#else /* HAVE_SIGACTION */
    signal(sig, handler);
#endif
}

RETSIGTYPE signal_handler(gint sig)
{
    switch(sig) {
        case SIGTERM:
        case SIGQUIT:
        case SIGINT:
        case SIGHUP:
            exit(0);

#if 0 /* XXX: not safe, probably not real useful either... */
        case SIGUSR1:
            mdh_config_load(mdh_config);
            mdh_toolbar_reset(Toolbar);
            break;
#endif

        default:
            g_critical("Caught unhandled signal (%d), exiting...", sig);
            exit(1);
    }

    set_signal(sig, signal_handler);
}

static void mdh_shutdown(void)
{
    gboolean bval;

    mdh_panel_foreach(Toolbar, (GFunc) mdh_panel_stop, NULL);

    mdh_config_save();

    if(mdh_config_get_boolean("run", "save", &bval) && bval) {
        GError *err = NULL;

        if(!mdh_run_history_save(mdh_hist, &err)) {
            g_warning("%s", err->message);
            g_error_free(err);
        }
    }

    if(mdh_config_get_boolean("scratch_pad", "save", &bval) && bval) {
        GError *err = NULL;

        if(!mdh_scratch_pad_save(&err)) {
            g_warning("%s", err->message);
            g_error_free(err);
        }
    }

    mdh_button_foreach(Toolbar, (GFunc) mdh_button_free, NULL);
    mdh_custom_foreach(Toolbar, (GFunc) mdh_custom_free, NULL);
    mdh_panel_foreach(Toolbar,  (GFunc) mdh_panel_free,  NULL);

    mdh_toolbar_free(Toolbar);

    mdh_error_free_all();

    mdh_run_history_free();
    mdh_scratch_pad_free();

    mdh_config_unlock(mdh_config);

    mdh_config_free();

    g_free(mdh_home);
    g_free(mdh_cache);
    g_free(mdh_config);
    g_free(mdh_hist);
    g_free(mdh_scratch);
    g_free(mdh_browser);
    g_free(mdh_xterm);
}

typedef struct {
    const gchar *value;
    const gchar *command;
    const gchar *tooltip;
} MdhButtonCustomValues;

static const MdhButtonCustomValues button_custom_value_list[] = {
        { "Web",  "firefox",  "Firefox"               },
        { "Mail", "balsa",    "Balsa"                 },
        { "News", "pan",      "Pan"                   },
        { "OO",   "ooffice",  "Open Office"           },
        { "File", "nautilus", "Nautilus File Browser" },
        { "FTP",  "gftp",     "gFTP"                  },
        { "Term", "xterm",    "X Terminal"            },
        { "XQF",  "xqf",      "Quake Server Browser"  },
        { "Mix",  "aumix",    "Audio Mixer"           },
        { "Lock", "xlock",    "Lock Workstation"      }
    };

static gint button_custom_value_size = G_N_ELEMENTS(button_custom_value_list);

static const gchar *browser_list[] = {
        "x-www-browser",
        "firefox",
        "mozilla",
        "opera",
        "konqueror",
        "galeon",
        "epiphany",
        "netscape",
        NULL
    };

static const gchar *xterm_list[] = {
        "x-terminal-emulator",
        "xterm",
        "rxvt",
        "Eterm",
        "aterm",
        NULL
    };

static gchar *mdh_find_program_in_path(const gchar *program_list[])
{
    gint i;

    for(i = 0; program_list[i]; i++) {
        gchar *path;

        if((path = g_find_program_in_path(program_list[i])))
            return(path);

        g_free(path);
    }

    return(NULL);
}

void version(void)
{
    fprintf(stderr, "%s | Version %s | Copyright (c) 2003-2005 Mike Hokenson\n",
                    PACKAGE_NAME, PACKAGE_VERSION);

    fprintf(stderr, "This is free software with ABSOLUTELY NO WARRANTY. See ");
    fprintf(stderr, "COPYING for details.\n");

    fprintf(stderr, "Send bug reports and questions to <%s>\n",
                    PACKAGE_BUGREPORT);
}

static void usage(void)
{
    fprintf(stderr, "Usage: %s [options] [rc]\n", g_get_prgname());
    fprintf(stderr, "  options:\n");
    fprintf(stderr, "    -d/--disable-netwm  disable NetWM support\n");
    fprintf(stderr, "    -f/--enable-netwm   enable NetWM support\n");
    fprintf(stderr, "    -D/--disable-gnome  disable Gnome support\n");
    fprintf(stderr, "    -F/--enable-gnome   enable Gnome support\n");
    fprintf(stderr, "    -h/--help           help (this message)\n");
    fprintf(stderr, "    -v/--version        display version\n");
}

static struct option main_options[] = {
    { "disable-netwm", 0, NULL, 'd' },
    { "enable-netwm",  0, NULL, 'f' },
    { "disable-gnome", 0, NULL, 'D' },
    { "enable-gnome",  0, NULL, 'F' },
    { "help",          0, NULL, 'h' },
    { "version",       0, NULL, 'v' },
    { NULL,            0, NULL,  0  }
};

gint main(gint argc, gchar **argv)
{
    const gchar *config = NULL;

    gint i;

    g_set_prgname(argv[0]);

#if 0 /* XXX: mdh_x11_wait_for_window_manager() */
    mdh_x11_enable_netwm(TRUE);
#endif

    while(1) {
        gint c;

        if((c = getopt_long(argc, argv, "dfDFhv", main_options, NULL)) == -1)
            break;

        switch(c) {
            case 'd':
                mdh_x11_enable_netwm(FALSE);
                break;
            case 'f':
                mdh_x11_enable_netwm(TRUE);
                break;
            case 'D':
                mdh_x11_enable_gnome(FALSE);
                break;
            case 'F':
                mdh_x11_enable_gnome(TRUE);
                break;
            case 'h':
                usage();
                exit(0);
            case 'v':
                version();
                exit(0);
            default:
                fprintf(stderr, "See `%s --help' for usage\n", argv[0]);
                exit(1);
        }
    }

    if(optind < argc)
        config = argv[optind++];

    /* silently discard the rest of the command line */

    {
        const gchar *home  = g_getenv("XDG_CONFIG_HOME"),
                    *cache = g_getenv("XDG_CACHE_HOME");

        if(home && *home)
            mdh_home = g_build_filename(home, "mdh", NULL);
        else
            mdh_home = g_build_filename(g_get_home_dir(),
                                        ".config", "mdh", NULL);

        mdh_mkdir_path(mdh_home, 0755);

        if(cache && *cache)
            mdh_cache = g_build_filename(cache, "mdh", NULL);
        else
            mdh_cache = g_build_filename(g_get_home_dir(),
                                        ".cache", "mdh", NULL);

        mdh_mkdir_path(mdh_cache, 0755);

        if(!config)
            mdh_config = g_build_filename(mdh_home, "config", NULL);
        else
            mdh_config = g_strdup(config);

        mdh_hist = g_build_filename(mdh_cache, "run_history",  NULL);

        mdh_scratch = g_build_filename(mdh_cache, "scratch_pad",  NULL);
    }

    mdh_browser = mdh_find_program_in_path(browser_list);
    mdh_xterm   = mdh_find_program_in_path(xterm_list);

    mdh_threads_init();

    mdh_threads_enter();

    gtk_init(&argc, &argv);

#if 0 /* XXX: mdh_x11_wait_for_window_manager() */
    if(!mdh_x11_wait_for_window_manager()) {
        g_warning("No window manager found, skipping feature detection.");
        mdh_x11_enable_netwm(FALSE);
    } else
#endif
        mdh_x11_get_features();

    {
        gchar *dpy = NULL;
        pid_t pid;

        if(mdh_config_locked(mdh_config, &dpy, &pid)) {
            gchar *text = g_strdup_printf("mdh lock found on \"%s\" (pid %d).\nContinue anyway?", dpy, pid);

            if(!mdh_window_yes_no_prompt(text))
                exit(1);

            g_free(text);
        }

        g_free(dpy);
    }

    mdh_config_unlock(mdh_config);
    mdh_config_lock(mdh_config);

    /* load the config file ahead of time to get the 'use_dock' variable */
    mdh_config_load(mdh_config);

    {
        gboolean use_dock = FALSE;

        if(!mdh_config_get_boolean("toolbar", "use_dock", &use_dock))
            mdh_config_set_boolean("toolbar", "use_dock", use_dock); /* save */

        Toolbar = mdh_toolbar_new(use_dock);
    }

    /* buttons */

    B_Menu = mdh_button_new(TRUE, FALSE, "MailDooHicky",
                            (MdhButtonFunc) mdh_toolbar_menu_show,
                            GTK_STOCK_HOME,
                            MDH_IMAGE_TYPE_STOCK);

    B_Toggle = mdh_button_new(TRUE, FALSE, "Toggle Toolbar",
                              (MdhButtonFunc) mdh_toolbar_toggle,
                              GTK_STOCK_REFRESH,
                              MDH_IMAGE_TYPE_STOCK);

    B_Run = mdh_button_new(FALSE, TRUE, "Run...",
                           (MdhButtonFunc) mdh_ui_run,
                           GTK_STOCK_EXECUTE,
                           MDH_IMAGE_TYPE_STOCK);

    B_Pad = mdh_button_new(FALSE, TRUE, "Scratch-Pad",
                           (MdhButtonFunc) mdh_ui_scratch_pad,
                           GTK_STOCK_EDIT,
                            MDH_IMAGE_TYPE_STOCK);

#if _MDH_HAS_MIXER
    B_Volume = mdh_button_new(FALSE, TRUE, "Volume",
                              (MdhButtonFunc) mdh_ui_mixer_volume_popup,
                              mdh_icon_path("mixer"),
                              MDH_IMAGE_TYPE_FILE);
#endif

    /* custom buttons */

    B_Cust = g_new(MdhCustom *, _MDH_BUTTON_COUNT_CUSTOM_MAX);

    mdh_custom_count_set(_MDH_BUTTON_COUNT_CUSTOM_DEF);

    for(i = 0; i < button_custom_value_size; i++) {
        const MdhButtonCustomValues *v = &button_custom_value_list[i];

        B_Cust[i] = mdh_custom_new(FALSE,
                                   v->value,
                                   v->command,
                                   v->tooltip,
                                   NULL);
    }

    /* allocate the remainder */
    for(i = button_custom_value_size; i < _MDH_BUTTON_COUNT_CUSTOM_MAX; i++)
        B_Cust[i] = mdh_custom_new(FALSE, NULL, NULL, NULL, NULL);

    /* panels */

#if _MDH_HAS_TIME
    P_Time = mdh_panel_time_new(NULL, NULL, TRUE);
#endif

    P_Mail = g_new(MdhPanel *, _MDH_PANEL_COUNT_MAIL_MAX);

    P_Mail[0] = mdh_panel_mail_new(g_getenv("MAIL"),    NULL, NULL, FALSE);
    P_Mail[1] = mdh_panel_mail_new(g_getenv("MAILDIR"), NULL, NULL, FALSE);

    for(i = 2; i < _MDH_PANEL_COUNT_MAIL_MAX; i++)
        P_Mail[i] = mdh_panel_mail_new(NULL, NULL, NULL, FALSE);

    for(i = 0; i < _MDH_PANEL_COUNT_MAIL_MAX; i++) {
        mdh_panel_flags_set(P_Mail[i], MDH_PANEL_FLAG_BOLD);
        mdh_panel_mail_threshold_set(P_Mail[i], _MAIL_THRESH_DEF);
    }

    mdh_panel_mail_count_set(P_Mail[0], _MDH_PANEL_COUNT_MAIL_MIN);

#if _MDH_HAS_CPU
    P_Cpu = mdh_panel_cpu_new(NULL, NULL, TRUE);

    mdh_panel_flags_set(P_Cpu, MDH_PANEL_FLAG_BOLD);
    mdh_panel_cpu_threshold_set(P_Cpu, _CPU_THRESH_DEF);
#endif

#if _MDH_HAS_MEM
    P_Mem = mdh_panel_mem_new(NULL, NULL, TRUE);

    mdh_panel_flags_set(P_Mem, MDH_PANEL_FLAG_BOLD);
    mdh_panel_mem_threshold_set(P_Mem, _MEM_THRESH_DEF);
#endif

#if _MDH_HAS_DISK
    P_Disk = g_new(MdhPanel *, _MDH_PANEL_COUNT_DISK_MAX);

    for(i = 0; i < _MDH_PANEL_COUNT_DISK_MAX; i++)
        P_Disk[i] = mdh_panel_new();

    P_Disk[0] = mdh_panel_disk_new("/",    NULL, NULL, FALSE);
    P_Disk[1] = mdh_panel_disk_new("/usr", NULL, NULL, FALSE);
    P_Disk[2] = mdh_panel_disk_new("/var", NULL, NULL, FALSE);

    /* setup defaults for the rest */
    for(i = 3; i < _MDH_PANEL_COUNT_DISK_MAX; i++)
        P_Disk[i] = mdh_panel_disk_new(NULL, NULL, NULL, FALSE);

    for(i = 0; i < _MDH_PANEL_COUNT_DISK_MAX; i++) {
        mdh_panel_flags_set(P_Disk[i], MDH_PANEL_FLAG_BOLD);
        mdh_panel_disk_threshold_set(P_Disk[i], _DISK_THRESH_DEF);
    }

    mdh_panel_disk_count_set(P_Disk[0], _MDH_PANEL_COUNT_DISK_MIN);
#endif

#if _MDH_HAS_NET
    P_Net = g_new(MdhPanel *, _MDH_PANEL_COUNT_NET_MAX);

    for(i = 0; i < _MDH_PANEL_COUNT_NET_MAX; i++) {
        gchar *p = g_strdup_printf("eth%d", i);
        P_Net[i] = mdh_panel_net_new(p, NULL, NULL, (i == 0) ? TRUE : FALSE);
        g_free(p);
    }

    mdh_panel_net_count_set(P_Net[0], _MDH_PANEL_COUNT_NET_MIN);
#endif

#if _MDH_HAS_WEATHER
    P_Weather = mdh_panel_weather_new(NULL, NULL, NULL, FALSE);

    mdh_panel_flags_set(P_Weather, MDH_PANEL_FLAG_ICON);
    mdh_panel_weather_unit_metric_set(P_Weather, FALSE); /* standard */
#endif

#if _MDH_HAS_UPTIME
    P_Uptime = mdh_panel_uptime_new(NULL, NULL, TRUE);
#endif

    P_Custom = g_new(MdhPanel *, _MDH_PANEL_COUNT_CUSTOM_MAX);

    P_Custom[0] = mdh_panel_custom_new("hostname",  NULL, FALSE);
    P_Custom[1] = mdh_panel_custom_new("uname -sr", NULL, FALSE);
    P_Custom[2] = mdh_panel_custom_new("date +%D", NULL, FALSE);

    for(i = 3; i < _MDH_PANEL_COUNT_CUSTOM_MAX; i++)
        P_Custom[i] = mdh_panel_custom_new(NULL, NULL, FALSE);

    mdh_panel_custom_count_set(P_Custom[0], _MDH_PANEL_COUNT_CUSTOM_MIN);

    /* toolbar object initialization completed */

    g_atexit(mdh_shutdown);

    mdh_config_apply();

    {
        GError *err = NULL;

        if(!mdh_run_history_load(mdh_hist, &err)) {
            g_warning("%s", err->message);
            g_error_free(err);
        }
    }

    {
        GError *err = NULL;

        if(!mdh_scratch_pad_load(mdh_scratch, &err)) {
            g_warning("%s", err->message);
            g_error_free(err);
        }
    }

    /* add toolbar objects */

    mdh_toolbar_button_add(Toolbar, B_Menu);
    mdh_toolbar_button_add(Toolbar, B_Toggle);
    mdh_toolbar_button_add(Toolbar, B_Run);
    mdh_toolbar_button_add(Toolbar, B_Pad);

#if _MDH_HAS_MIXER
    mdh_toolbar_button_add(Toolbar, B_Volume);
#endif

    for(i = 0; i < _MDH_BUTTON_COUNT_CUSTOM_MAX; i++)
        mdh_toolbar_custom_add(Toolbar, B_Cust[i]);

#if _MDH_HAS_TIME
    mdh_toolbar_panel_add(Toolbar, P_Time);
#endif

    for(i = 0; i < _MDH_PANEL_COUNT_MAIL_MAX; i++)
        mdh_toolbar_panel_add(Toolbar, P_Mail[i]);

#if _MDH_HAS_CPU
    mdh_toolbar_panel_add(Toolbar, P_Cpu);
#endif

#if _MDH_HAS_MEM
    mdh_toolbar_panel_add(Toolbar, P_Mem);
#endif

#if _MDH_HAS_DISK
    for(i = 0; i < _MDH_PANEL_COUNT_DISK_MAX; i++)
        mdh_toolbar_panel_add(Toolbar, P_Disk[i]);
#endif

#if _MDH_HAS_NET
    for(i = 0; i < _MDH_PANEL_COUNT_NET_MAX; i++)
        mdh_toolbar_panel_add(Toolbar, P_Net[i]);
#endif

#if _MDH_HAS_WEATHER
    mdh_toolbar_panel_add(Toolbar, P_Weather);
#endif

#if _MDH_HAS_UPTIME
    mdh_toolbar_panel_add(Toolbar, P_Uptime);
#endif

    for(i = 0; i < _MDH_PANEL_COUNT_CUSTOM_MAX; i++)
        mdh_toolbar_panel_add(Toolbar, P_Custom[i]);

    /* everything added */

    ADD_LOG("MailDooHicky v%s started", PACKAGE_VERSION);

    /* setup signal handlers */

    set_signal(SIGTERM, signal_handler);
    set_signal(SIGQUIT, signal_handler);
    set_signal(SIGINT,  signal_handler);
    set_signal(SIGHUP,  signal_handler);
    set_signal(SIGUSR1, signal_handler);
    set_signal(SIGUSR2, signal_handler);

    mdh_threads_leave();

    /* initialization completed */

    mdh_toolbar_reset(Toolbar);  /* startup the panels */

    mdh_threads_enter();
    mdh_toolbar_show(Toolbar);   /* show the toolbar */
    mdh_toolbar_set_sticky(Toolbar, Toolbar->sticky);  /* XXX: for DOCK hint */
    mdh_threads_leave();

    mdh_threads_enter();
    gtk_main();
    mdh_threads_leave();

    exit(0);
}

const gchar *mdh_get_home_dir(void)
{
    return(mdh_home);
}

const gchar *mdh_get_cache_dir(void)
{
    return(mdh_cache);
}

const gchar *mdh_get_config_file(void)
{
    return(mdh_config);
}

const gchar *mdh_get_browser(void)
{
    static const gchar *browser;

    if(!browser)
        browser = (mdh_browser) ? mdh_basename(mdh_browser) : NULL;

    return(browser);
}

const gchar *mdh_get_xterm(void)
{
    static const gchar *xterm;

    if(!xterm)
        xterm = (mdh_xterm) ? mdh_basename(mdh_xterm) : NULL;

    return(xterm);
}
