/*
 * mdh (MailDooHicky), a GTK+-based toolbar.
 *
 * Copyright (c) 2003-2005 by Mike Hokenson <mdh at gozer dot org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <unistd.h>

#include <gtk/gtk.h>

#include "mdh.h"

#include "panel.h"

#include "panel_net.h"

#include "util.h"

#include "sysdeps.h"

#if _MDH_HAS_NET

typedef struct {
    const gchar *net_dev;

    guint64      net_rx;
    guint64      net_rx_last;

    guint64      net_tx;
    guint64      net_tx_last;

    gint         net_interval;
} MdhDataCache;

typedef struct {
    gint         count;

    MdhDataCache cache;
} MdhPanelData;

static gchar *panel_subst(MdhDataCache *cache, const gchar *fmt);

static gboolean panel_startup(gpointer data, GError **err)
{
    MdhPanel *obj = data;

    g_return_val_if_fail(obj != NULL, FALSE);
    g_return_val_if_fail(obj->display != NULL, FALSE);
    g_return_val_if_fail(obj->interval >= _NET_MIN, FALSE);
    g_return_val_if_fail(obj->interval <= _NET_MAX, FALSE);

    if(obj->init)
        return(TRUE);

    debug("panel_startup(): %d: %s", obj->handler, obj->value);

    if(obj->value && !*obj->value) {
        g_set_error(err, 0, 0, "Failed to process 'net': Empty value.");
        goto panel_startup_fail;
    }

    {
        MdhPanelData *pdata = obj->data;
        MdhDataCache *cache = &pdata->cache;

        cache->net_dev      = obj->value;
        cache->net_rx       = 0;
        cache->net_tx       = 0;
        cache->net_rx_last  = 0;
        cache->net_tx_last  = 0;
        cache->net_interval = obj->interval;
    }

    if(!mdh_sys_init(err))
        goto panel_startup_fail;

    obj->init = TRUE;

    return(TRUE);

panel_startup_fail:
    mdh_panel_stop(obj);

    mdh_panel_set_text_r(obj, FALSE, "Error");

    return(FALSE);
}

static gboolean panel_shutdown(gpointer data, GError **err)
{
    MdhPanel *obj = data;

    g_return_val_if_fail(obj != NULL, FALSE);

    if(!obj->init)
        return(FALSE);

    debug("panel_shutdown(): %d: %s", obj->handler, obj->value);

    mdh_sys_close();

    obj->init = FALSE;

    return(FALSE);
}

static gboolean panel_main(gpointer data, GError **err)
{
    MdhPanel *obj = data;

    MdhPanelData *pdata;

    guint64  rx, tx, rx_last, tx_last;
    gboolean up;

    g_return_val_if_fail(obj != NULL, FALSE);

    debug("panel_main(): %d: %s", obj->handler, obj->value);

    if(!panel_startup(obj, err))
        return(FALSE);

    if(!mdh_sys_get_net(obj->value, &rx, &tx, &up, err)) {
        mdh_panel_stop(obj);
        mdh_panel_set_text_r(obj, FALSE, "N/A");
        return(FALSE);
    }

    debug("panel_main(): %d: %s: rx = %llu, tx = %llu",
          obj->handler, obj->value, rx, tx);

    pdata = obj->data;

    rx_last = ((MdhDataCache)pdata->cache).net_rx;
    tx_last = ((MdhDataCache)pdata->cache).net_tx;

    if(!rx_last || !tx_last) {
        rx_last = rx;
        tx_last = tx;
    }

    {
      MdhDataCache *cache = &pdata->cache;

      cache->net_rx      = (up) ? rx      : 0;
      cache->net_tx      = (up) ? tx      : 0;
      cache->net_rx_last = (up) ? rx_last : 0;
      cache->net_tx_last = (up) ? tx_last : 0;
    }

    /* handle counter wrap */
    if(rx < rx_last || tx < tx_last)
        return(TRUE);

    {
        gchar *p = panel_subst(&pdata->cache, obj->display);
        mdh_panel_set_text_r(obj, FALSE, "%s", p);
        g_free(p);
    }

    return(TRUE);
}

/*
 * %i: interface: eth0
 * %r: rx (KB/sec): 32.4
 * %t: tx (KB/sec): 5.1
 */

static gchar *panel_subst(MdhDataCache *cache, const gchar *fmt)
{
    const gchar *p;

    GString *string;

    gdouble net_rx, net_tx;

    g_return_val_if_fail(cache != NULL, NULL);
    g_return_val_if_fail(fmt != NULL, NULL);

    string = g_string_new(NULL);

    net_rx  = (gdouble) ((cache->net_rx - cache->net_rx_last) / 1024);
    net_rx /= cache->net_interval;

    net_tx  = (gdouble) ((cache->net_tx - cache->net_tx_last) / 1024);
    net_tx /= cache->net_interval;

    for(p = fmt; *p; p++) {
        if(*p == '%' && *(p + 1) && p++) {
            switch(*p) {
                case 'i': /* interface: eth0 */
                    if(cache->net_dev)
                        g_string_append(string, cache->net_dev);
                    else
                        g_string_append(string, "N/A");
                    break;

                case 'r': /* rx (KB/sec): 32.4 */
                    g_string_append_printf(string, "%5.1f", net_rx);
                    break;

                case 't': /* tx (KB/sec): 5.1 */
                    g_string_append_printf(string, "%5.1f", net_tx);
                    break;

                default:
                    g_string_append_c(string, *p);
                    break;
            }
        } else
            g_string_append_c(string, *p);
    }

    return(g_string_free(string, FALSE));
}

gint mdh_panel_net_count(MdhPanel *obj)
{
    g_return_val_if_fail(obj != NULL, 0);
    g_return_val_if_fail(obj->data != NULL, 0);

    return(((MdhPanelData *)obj->data)->count);
}

void mdh_panel_net_count_set(MdhPanel *obj, gint count)
{
    g_return_if_fail(obj != NULL);
    g_return_if_fail(obj->data != NULL);
    
    ((MdhPanelData *)obj->data)->count = count;
}

static gboolean panel_callback(gpointer data, GError **err)
{
    MdhPanel *obj = data;

    gchar *command;

    gboolean ret;

    g_return_val_if_fail(obj != NULL, FALSE);
    g_return_val_if_fail(obj->command != NULL, FALSE);

    if(strstr(obj->command, " %s")) {
        command = g_strdup_printf(obj->command, obj->value);
    } else
        command = obj->command;

    ret = g_spawn_command_line_async(command, err);

    if(command != obj->command)
        g_free(command);

    return(ret);
}

static MdhPanelFuncs funcs = {
    panel_main,
    panel_startup,
    panel_shutdown,
    panel_callback,
    NULL
};

MdhPanel *mdh_panel_net_new(const gchar *value,
                            const gchar *display,
                            const gchar *command,
                            gboolean enabled)
{
    MdhPanel *obj = mdh_panel_new();

    g_return_val_if_fail(obj != NULL, NULL);

#ifndef __sun__
    obj->value    = (value)   ? g_strdup(value)   : g_strdup("");
#endif
    obj->display  = (display) ? g_strdup(display) : g_strdup(_NET_DIS);
    obj->command  = (command) ? g_strdup(command) : g_strdup("ethereal %s");
#ifndef __sun__
    obj->tooltip  = g_strdup_printf("Net: %s", obj->value);
#endif
    obj->interval = _NET_DEF;
    obj->enabled  = enabled;
    obj->funcs    = funcs;

    obj->data     = g_new(MdhPanelData, 1);

    ((MdhPanelData *) obj->data)->count = 0;

    return(obj);
}

#endif /* _MDH_HAS_NET */
