/*
 * mdh (MailDooHicky), a GTK+-based toolbar.
 *
 * Copyright (c) 2003-2005 by Mike Hokenson <mdh at gozer dot org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include <gtk/gtk.h>

#if TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif

#ifdef HAVE_LIBXML_PARSER_H
# include <libxml/parser.h>
#endif

#include "mdh.h"

#include "toolbar.h"

#include "panel.h"

#include "panel_weather.h"

#include "curl.h"
#include "util.h"

#if _MDH_HAS_WEATHER

#define _WEATHER_URI_BASE ".weather.com/weather/local/%s"
#define _WEATHER_URI_QSTR "&cc=*&unit=%c&prod=xoap"_WEATHER_URI_KEY
#define _WEATHER_URI_XOAP "http://xoap"_WEATHER_URI_BASE"?"_WEATHER_URI_QSTR
#define _WEATHER_URI_HOME "http://www"_WEATHER_URI_BASE"?"_WEATHER_URI_KEY

#define _WEATHER_XML_OK      -999  /* ok */
#define _WEATHER_XML_ENOFILE -998  /* no cache file */
#define _WEATHER_XML_EPARSE  -997  /* parse error */
#define _WEATHER_XML_EDATA   -996  /* missing or invalid temp data */
#define _WEATHER_XML_ELOCID  -995  /* invalid LocID */
#define _WEATHER_XML_ERROR   -994  /* other error */

typedef struct {
    gchar *LocID;

    gchar *head_ut;
    gchar *head_ud;
    gchar *head_us;
    gchar *head_up;
    gchar *head_ur;

    gchar *loc_dnam;
    gchar *loc_tm;
    gchar *loc_sunr;
    gchar *loc_suns;

    gchar *cc_lsup;
    gchar *cc_tmp;
    gchar *cc_flik;
    gchar *cc_t;
    gchar *cc_icon;
    gchar *cc_bar_r;
    gchar *cc_bar_d;
    gchar *cc_wind_s;
    gchar *cc_wind_t;
    gchar *cc_hmid;
    gchar *cc_vis;
    gchar *cc_uv_i;
    gchar *cc_uv_t;
    gchar *cc_dewp;
} MdhDataCache;

typedef enum {
    UNIT_STANDARD,
    UNIT_METRIC
} MdhPanelUnit;

typedef struct {
    MdhDataCache *cache;
    gint          last;

    MdhPanelUnit  unit;
} MdhPanelData;

static gint cache_parse(const gchar *file,
                        const gchar *LocID,
                        MdhDataCache **cache,
                        time_t *last,
                        GError **err);

static void cache_free(MdhDataCache *cache);

static gboolean get_http(const gchar *file,
                         const gchar *LocID,
                         gboolean metric,
                         GError **err);

static gchar *panel_subst(MdhDataCache *cache, const gchar *fmt);

static gboolean panel_startup(gpointer data, GError **err)
{
    MdhPanel *obj = data;

    g_return_val_if_fail(obj != NULL, FALSE);
    g_return_val_if_fail(obj->value != NULL, FALSE);
    g_return_val_if_fail(obj->display != NULL, FALSE);
    g_return_val_if_fail(obj->interval >= _WEATHER_MIN, FALSE);
    g_return_val_if_fail(obj->interval <= _WEATHER_MAX, FALSE);

    if(obj->init)
        return(TRUE);

    debug("panel_startup(): %d: %s", obj->handler, obj->value);

    if(!_WEATHER_ID_VALID(obj->value)) {
        g_set_error(err, 0, 0, "Failed to process 'temp': Invalid LocID.");
        mdh_panel_stop(obj);
        mdh_panel_set_text_r(obj, FALSE, "Error");
        return(FALSE);
    }

    {
        MdhPanelData *pdata = obj->data;

        pdata->cache = NULL;
        pdata->last  = 0;
    }

    obj->init = TRUE;

    return(TRUE);
}

static gboolean panel_shutdown(gpointer data, GError **err)
{
    MdhPanel *obj = data;

    g_return_val_if_fail(obj != NULL, FALSE);

    if(!obj->init)
        return(FALSE);

    debug("panel_shutdown(): %d: %s", obj->handler, obj->value);

    {
        MdhPanelData *pdata = obj->data;
        cache_free(pdata->cache);
    }

    obj->init = FALSE;

    return(FALSE);
}

static gboolean panel_main(gpointer data, GError **err)
{
    MdhPanel *obj = data;

    MdhPanelData *pdata;

    time_t c_time;

    gboolean error = FALSE;

    g_return_val_if_fail(obj != NULL, FALSE);

    debug("panel_main(): %d: %s", obj->handler, obj->value);

    if(!panel_startup(obj, err))
        return(FALSE);

    pdata = obj->data;

    c_time = time(NULL);

    /* check if memory cache is old */
    if((pdata->last - 300) <= (c_time - obj->interval)) {
        const gchar *LocID = obj->value;

        gchar file[PATH_MAX];

        MdhDataCache *cache = NULL;

        time_t last = 0;
        gint ret;

        debug("panel_main(): %d: %s: memory cache is old",
              obj->handler, obj->value);

        g_snprintf(file, sizeof(file), "%s%c%s",
                                       mdh_get_cache_dir(),
                                       G_DIR_SEPARATOR,
                                       LocID);

        ret = cache_parse(file, LocID, &cache, &last, err);

        /* now disk cache */
        if((last - 300) <= (c_time - obj->interval)) {
            GError *e = NULL;

            debug("panel_main(): %d: %s: disk cache is old",
                  obj->handler, obj->value);

            if(get_http(file, LocID, (pdata->unit == UNIT_METRIC), &e)) {
                cache_free(cache);
                cache = NULL;

                ret = cache_parse(file, LocID, &cache, &last, &e);

                if(ret == _WEATHER_XML_OK)
                    pdata->last = last;
            } else
                ret = _WEATHER_XML_ERROR;

            /* don't overwrite previous errors, use most recent */
            if(e) {
                if(err && *err)
                    g_error_free(*err);

                g_set_error(err, 0, 0, e->message);

                g_error_free(e);
            }
        }

        /* display message only if no prior successful request has been made */
        if(pdata->last == 0)
            switch(ret) {
                case _WEATHER_XML_ENOFILE:
                case _WEATHER_XML_EPARSE:
                case _WEATHER_XML_EDATA:
                case _WEATHER_XML_ERROR:
                case _WEATHER_XML_ELOCID:
                    cache_free(cache);

                    mdh_panel_set_icon_r(obj, "weather/na");

                    if(ret == _WEATHER_XML_ELOCID)
                        mdh_panel_set_text_r(obj, FALSE, "Invalid ID");
                    else
                        mdh_panel_set_text_r(obj, FALSE, "Error");

                    return(FALSE);
                default:
                    break;
            }

        if(cache && ret == _WEATHER_XML_OK) {
            if(cache->loc_dnam) {  /* update tooltip with location name */
                g_free(obj->tooltip);
                obj->tooltip = g_strdup_printf("Weather: %s", cache->loc_dnam);

                mdh_panel_set_tip_r(obj, obj->tooltip);
            }

            if(cache->cc_icon) {   /* update current condition icon */
                gchar *icon = g_strdup_printf("weather/%s", cache->cc_icon);
                mdh_panel_set_icon_r(obj, icon);
                g_free(icon);
            }

            if(pdata->cache)
               cache_free(pdata->cache);

            pdata->cache = cache;
        } else
            error = TRUE;
    }

    g_return_val_if_fail(pdata->cache != NULL, FALSE);

    {
        gchar *p = panel_subst(pdata->cache, obj->display);
        mdh_panel_set_text_r(obj, FALSE, "%s%s", p, (error) ? " *": "");
        g_free(p);
    }

    return(TRUE);
}

#define _LIKELY_NUMERIC(c) \
            (g_ascii_isdigit(c) || g_ascii_ispunct(c) || c == '-')

static gboolean str_to_int(const gchar *str, gint *n)
{
    const gchar *p;

    g_return_val_if_fail(str != NULL, FALSE);

    for(p = str; *p; p++)
        if(!_LIKELY_NUMERIC(*p))
            return(FALSE);

    if(n)
        *n = atoi(str);

    return(TRUE);
}

static gboolean str_to_dec(const gchar *str, gdouble *n)
{
    const gchar *p;

    g_return_val_if_fail(str != NULL, FALSE);

    for(p = str; *p; p++)
        if(!_LIKELY_NUMERIC(*p))
            return(FALSE);

    if(n)
        *n = atof(str);

    return(TRUE);
}

/*
 * <error>
 *   <err type="@num">@message</type>
 * </error>
 */

#define _WEATHER_XML_DATA_UNKNOWN_ERROR 0
#define _WEATHER_XML_DATA_MISSING_LOCID 1
#define _WEATHER_XML_DATA_INVALID_LOCID 2
#define _WEATHER_XML_DATA_INVALID_PARID 100
#define _WEATHER_XML_DATA_INVALID_PRDID 101
#define _WEATHER_XML_DATA_INVALID_KEYID 102

static MdhDataCache *cache_new(const gchar *LocID)
{
    MdhDataCache *cache;

    g_return_val_if_fail(LocID != NULL, NULL);

    cache = g_new(MdhDataCache, 1);

    g_return_val_if_fail(cache != NULL, NULL);

    cache->LocID     = g_strdup(LocID);

    cache->head_ut   = NULL;
    cache->head_ud   = NULL;
    cache->head_us   = NULL;
    cache->head_up   = NULL;
    cache->head_ur   = NULL;

    cache->loc_dnam  = NULL;
    cache->loc_tm    = NULL;
    cache->loc_sunr  = NULL;
    cache->loc_suns  = NULL;

    cache->cc_lsup   = NULL;
    cache->cc_tmp    = NULL;
    cache->cc_flik   = NULL;
    cache->cc_t      = NULL;
    cache->cc_icon   = NULL;
    cache->cc_bar_r  = NULL;
    cache->cc_wind_s = NULL;
    cache->cc_wind_t = NULL;
    cache->cc_hmid   = NULL;
    cache->cc_vis    = NULL;
    cache->cc_uv_i   = NULL;
    cache->cc_uv_t   = NULL;
    cache->cc_dewp   = NULL;

    return(cache);
}

static void cache_free(MdhDataCache *cache)
{
    if(!cache)
        return;

    g_free(cache->LocID);

    g_free(cache->head_ut);
    g_free(cache->head_ud);
    g_free(cache->head_us);
    g_free(cache->head_up);
    g_free(cache->head_ur);

    g_free(cache->loc_dnam);
    g_free(cache->loc_tm);
    g_free(cache->loc_sunr);
    g_free(cache->loc_suns);

    g_free(cache->cc_lsup);
    g_free(cache->cc_tmp);
    g_free(cache->cc_flik);
    g_free(cache->cc_t);
    g_free(cache->cc_icon);
    g_free(cache->cc_bar_r);
    g_free(cache->cc_wind_s);
    g_free(cache->cc_wind_t);
    g_free(cache->cc_hmid);
    g_free(cache->cc_vis);
    g_free(cache->cc_uv_i);
    g_free(cache->cc_uv_t);
    g_free(cache->cc_dewp);

    g_free(cache);
}

xmlNodePtr xml_node_find(xmlNodePtr node, const gchar *name)
{
    xmlNodePtr n = node;

    g_return_val_if_fail(node != NULL, NULL);
    g_return_val_if_fail(name != NULL, NULL);

    while(n) {
        if(!strcmp((const gchar *) n->name, name))
            return(n);

        n = n->next;
    }

    return(NULL);
}

gchar *xml_node_attr(xmlNodePtr node, const gchar *name)
{
    xmlNodePtr n;

    g_return_val_if_fail(node != NULL, NULL);
    g_return_val_if_fail(name != NULL, NULL);

    if((n = xml_node_find(node, name)))
        return((gchar *) xmlNodeGetContent(n));

    return(NULL);
}

static gint cache_parse(const gchar *file,
                        const gchar *LocID,
                        MdhDataCache **cache,
                        time_t *last,
                        GError **err)
{
    MdhDataCache *c;

    xmlDocPtr doc;
    xmlNodePtr root, node;

    gint ret;

    g_return_val_if_fail(file != NULL, -1);
    g_return_val_if_fail(LocID != NULL, -1);
    g_return_val_if_fail(cache != NULL, -1);

    debug("cache_parse(): %s", LocID);

    {
        struct stat st;

        if(stat(file, &st) || !S_ISREG(st.st_mode))
            return(_WEATHER_XML_ENOFILE);

        if(last)
            *last = st.st_mtime;
    }

    if(!(doc = xmlParseFile(file))) {
        xmlErrorPtr e = xmlGetLastError();
        g_set_error(err, 0, 0, "Failed to parse '%s': %s.", file,
                               (e) ? e->message : "Unknown error");
        return(_WEATHER_XML_EPARSE);
    }

    ret = _WEATHER_XML_EDATA;

    if(!(root = xmlDocGetRootElement(doc))) {
        g_set_error(err, 0, 0, "Failed to load '%s': Empty document.", file);
        xmlFreeDoc(doc);
        return(ret);
    }

    if(!strcmp((const gchar *) root->name, "error")) {
        xmlNodePtr n;
        xmlChar *p;

        ret = _WEATHER_XML_ERROR;

        if((n = xml_node_find(root->children, "err")))
            if((p = xmlGetProp(n, (const xmlChar *) "type"))) {
                gint s;

                if(str_to_int((const gchar *) p, &s))
                    if(s == _WEATHER_XML_DATA_INVALID_LOCID)
                        ret = _WEATHER_XML_ELOCID;

                xmlFree(p);
            }

        p = xmlNodeGetContent(n);
        g_set_error(err, 0, 0, "Failed to load '%s': %s.", file, p);
        xmlFree(p);

        xmlFreeDoc(doc);

        return(ret);
    }

    if(strcmp((const gchar *) root->name, "weather")) {
        g_set_error(err, 0, 0, "Failed to load '%s': Invalid root: '%s'.",
                               file, root->name);
        xmlFreeDoc(doc);
        return(ret);
    }

    c = cache_new(LocID);

    g_return_val_if_fail(cache != NULL, FALSE);

    /*
     * <head>                    block of document header data
     *   <locale>en_US</locale>  text localization of the document
     *   <form>MEDIUM</form>     text format of the document
     *   <ut>F</ut>              text unit of temp
     *   <ud>mi</ud>             text unit of distance
     *   <us>mph</us>            text unit of speed
     *   <up>in</up>             text unit of precipitation
     *   <ur>in</ur>             text unit of presure
     * </head>
     */

    if((node = xml_node_find(root->children, "head"))) {
        c->head_ut = xml_node_attr(node->children, "ut");
        c->head_ud = xml_node_attr(node->children, "ud");
        c->head_us = xml_node_attr(node->children, "us");
        c->head_up = xml_node_attr(node->children, "up");
        c->head_ur = xml_node_attr(node->children, "ur");
    }

    /*
     * <loc id="54311">                      block of location data
     *   <dnam>Green Bay, WI (54311)</dnam>  text display name of the location
     *   <tm>10:05 PM</tm>
     *   <lat>44.49</lat>                    decimal latitude
     *   <lon>-87.92</lon>                   decimal longitude
     *   <sunr>6:51 AM</sunr>                time of sunrise
     *   <suns>5:21 PM</suns>                time of sunset
     *   <zone>-6</zone>                     integer time zone (GMT Offset)
     * </loc>
     */

    if((node = xml_node_find(root->children, "loc"))) {
        c->loc_dnam = xml_node_attr(node->children, "dnam");
        c->loc_tm   = xml_node_attr(node->children, "tm");
        c->loc_sunr = xml_node_attr(node->children, "sunr");
        c->loc_suns = xml_node_attr(node->children, "suns");
    }

    /*
     * <cc>                                   block of current conditions
     *   <lsup>2/15/05 9:53 PM CST</lsup>     Date/Time of last update
     *   <obst>Green Bay, WI</obst>           text name of observation station
     *   <tmp>26</tmp>                        integer temp
     *   <flik>18</flik>                      integer feels like temp
     *   <t>Fair</t>                          text description of condition
     *   <icon>33</icon>                      integer icon code for display
     *   <bar>                                block of barometric pressure data
     *     <r>30.03</r>                       decimal current pressure
     *     <d>steady</d>                      text description of raise or fall
     *   </bar>
     *   <wind>                               block of wind data
     *     <s>8</s>                           integer wind speed
     *     <gust>N/A</gust>                   integer maximum wind gust speed
     *     <d>340</d>                         integer direction in degrees
     *     <t>NNW</t>                         text description of direction
     *   </wind>
     *   <hmid>56</hmid>                      integer relative humidity
     *   <vis>10.0</vis>                      decimal visibility
     *   <uv>                                 block of uv index data
     *     <i>0</i>                           integer index value
     *     <t>Low</t>                         text description of condition
     *   </uv>
     *   <dewp>15</dewp>                      integer dew point
     *   <moon>
     *     <icon>7</icon>                     integer icon code for display
     *     <t>Waxing Half, First Quarter</t>  text description of condition
     *   </moon>
     * </cc>
     */

    if((node = xml_node_find(root->children, "cc"))) {
        xmlNodePtr n;

        c->cc_lsup = xml_node_attr(node->children, "lsup");
        c->cc_tmp  = xml_node_attr(node->children, "tmp");
        c->cc_flik = xml_node_attr(node->children, "flik");
        c->cc_t    = xml_node_attr(node->children, "t");
        c->cc_icon = xml_node_attr(node->children, "icon");

        if((n = xml_node_find(node->children, "bar"))) {
            c->cc_bar_r = xml_node_attr(n->children, "r");
            c->cc_bar_d = xml_node_attr(n->children, "d");
        }

        if((n = xml_node_find(node->children, "wind"))) {
            c->cc_wind_s = xml_node_attr(n->children, "s");
            c->cc_wind_t = xml_node_attr(n->children, "t");
        }

        c->cc_hmid = xml_node_attr(node->children, "hmid");
        c->cc_vis  = xml_node_attr(node->children, "vis");

        if((n = xml_node_find(node->children, "uv"))) {
            c->cc_uv_i = xml_node_attr(n->children, "i");
            c->cc_uv_t = xml_node_attr(n->children, "t");
        }

        c->cc_dewp = xml_node_attr(node->children, "dewp");
    }

    xmlFreeDoc(doc);

    if(!c->cc_tmp || !*c->cc_tmp) {
        g_set_error(err, 0, 0, "Failed to load '%s': %s temperature.",
                               file, (c->cc_tmp) ? "Empty" : "Missing");
        cache_free(c);
        return(ret);
    }

    *cache = c;

    return(_WEATHER_XML_OK);
}

static gboolean get_http(const gchar *file,
                         const gchar *LocID,
                         gboolean metric,
                         GError **err)
{
    gchar *res;
    gint res_len;

    FILE *fp;

    g_return_val_if_fail(file != NULL, FALSE);
    g_return_val_if_fail(LocID != NULL, FALSE);

    debug("get_http(): %s", LocID);

    {
        gchar uri[PATH_MAX];

        g_snprintf(uri, sizeof(uri), _WEATHER_URI_XOAP,
                                     LocID, (metric) ? 'm' : 's');

        if(!mdh_curl_get(uri, 0, &res, &res_len, NULL, err))
            return(FALSE);
    }

    if(!(fp = fopen(file, "w"))) {
        g_set_error(err, 0, 0, "Failed to open '%s': %s.", file,
                                g_strerror(errno));
        g_free(res);
        return(FALSE);
    }

    fwrite(res, res_len, 1, fp);

    g_free(res);

    fclose(fp);

    return(TRUE);
}

#define SVAL(v) ((v) ? v : "N/A")

/*
 * %l: location id: 54311
 * %L: location string: Green Bay, WI (54311)
 * %i: time: 10:05 PM
 * %s: sunrise: 6:51 AM
 * %S: sunset: 5:21 PM
 * %u: last update: 2/15/05 9:53 PM CST
 * %t: temperature: 26F
 * %f: feels like: 18F
 * %c: condition: Fair
 * %b: barometer pressure: 30.03in
 * %B: barometer pressure direction: steady
 * %w: wind speed and direction: 8mph NNW
 * %h: humidity: 56% 
 * %v: visibility: 10.0mi
 * %I: UV index, rating: 0 (Low)
 * %d: dew point: 15F
 */

static gchar *panel_subst(MdhDataCache *cache, const gchar *fmt)
{
    const gchar *p;

    gchar buf[25];

    GString *string;

    g_return_val_if_fail(cache != NULL, NULL);
    g_return_val_if_fail(cache->LocID != NULL, NULL);
    g_return_val_if_fail(cache->cc_tmp != NULL, NULL);
    g_return_val_if_fail(fmt != NULL, NULL);

    string = g_string_new(NULL);

    for(p = fmt; *p; p++) {
        if(*p == '%' && *(p + 1) && p++) {
            const gchar *v = NULL;

            switch(*p) {
                case 'l': /* location id: 54311 */
                    v = cache->LocID;
                    break;

                case 'L': /* location string: Green Bay, WI (54311) */
                    v = SVAL(cache->loc_dnam);
                    break;

                case 'i': /* time: 10:05 PM */
                    v = SVAL(cache->loc_tm);
                    break;

                case 's': /* sunrise: 6:51 AM */
                    v = SVAL(cache->loc_sunr);
                    break;
                case 'S': /* sunset: 5:21 PM */
                    v = SVAL(cache->loc_suns);
                    break;

                case 'u': /* last update: 2/15/05 9:53 PM CST */
                    v = SVAL(cache->cc_lsup);
                    break;

                /*
                 * Celcius    = (5 / 9) * (Fahrenheit - 32)
                 * Fahrenheit = (9 / 5) * (Celcius    + 32)
                 */

                case 't': /* temperature: 26F */
                    v = SVAL(cache->cc_tmp);

                    if(str_to_int(v, NULL) && cache->head_ut) {
                        g_snprintf(buf, sizeof(buf), "%s%s", v, cache->head_ut);
                        v = buf;
                    }

                    break;

                case 'f': /* feels like: 18F */
                    v = SVAL(cache->cc_flik);

                    if(str_to_int(v, NULL) && cache->head_ut) {
                        g_snprintf(buf, sizeof(buf), "%s%s", v, cache->head_ut);
                        v = buf;
                    }

                    break;

                case 'c': /* condition: Fair */
                    v = SVAL(cache->cc_t);
                    break;

                case 'b': /* barometer pressure: 30.03in */
                    v = SVAL(cache->cc_bar_r);

                    if(str_to_dec(v, NULL) && cache->head_ur) {
                        g_snprintf(buf, sizeof(buf), "%s%s", v, cache->head_ur);
                        v = buf;
                    }

                    break;

                case 'B': /* barometer pressure direction: steady */
                    v = SVAL(cache->cc_bar_d);
                    break;

                case 'w': /* wind speed and direction: 8mph NNW */
                    if(cache->cc_wind_s && cache->head_us && cache->cc_wind_t) {
                        g_snprintf(buf, sizeof(buf), "%s%s %s",
                                                     cache->cc_wind_s,
                                                     cache->head_us,
                                                     cache->cc_wind_t);
                        v = buf;
                    } else
                        v = SVAL(cache->cc_wind_s);

                    break;

                case 'h': /* humidity: 56% */
                    v = SVAL(cache->cc_hmid);

                    if(str_to_int(v, NULL)) {
                        g_snprintf(buf, sizeof(buf), "%s%%", v);
                        v = buf;
                    }

                    break;

                case 'v': /* visibility: 10.0mi */
                    v = SVAL(cache->cc_vis);

                    if(str_to_dec(v, NULL) && cache->head_ud) {
                        g_snprintf(buf, sizeof(buf), "%s%s", v, cache->head_ud);
                        v = buf;
                    }

                    break;

                case 'I': /* UV index, rating: 0 (Low) */
                    v = SVAL(cache->cc_uv_i);

                    if(str_to_int(v, NULL) && cache->cc_uv_t) {
                        g_snprintf(buf, sizeof(buf), "%s (%s)", v,
                                                     cache->cc_uv_t);
                        v = buf;
                    }

                    break;

                case 'd': /* dew point: 15F */
                    v = SVAL(cache->cc_dewp);

                    if(str_to_int(v, NULL) && cache->head_ut) {
                        g_snprintf(buf, sizeof(buf), "%s%s", v, cache->head_ut);
                        v = buf;
                    }

                    break;

                default:
                {
                    gchar buf[2]; buf[0] = *p; buf[1] = 0;
                    v = buf;
                    break;
                }
            }

            g_return_val_if_fail(v != NULL, NULL);

            g_string_append(string, v);
        } else
            g_string_append_c(string, *p);
    }

    return(g_string_free(string, FALSE));
}

const gchar *mdh_panel_weather_unit(MdhPanel *obj)
{
    g_return_val_if_fail(obj != NULL, NULL);
    g_return_val_if_fail(obj->data != NULL, NULL);

    if(((MdhPanelData *)obj->data)->unit == UNIT_METRIC)
        return(_WEATHER_UNIT_M);

    return(_WEATHER_UNIT_S);
}

void mdh_panel_weather_unit_metric_set(MdhPanel *obj, gboolean metric)
{
    g_return_if_fail(obj != NULL);
    g_return_if_fail(obj->data != NULL);

    if(metric)
        ((MdhPanelData *)obj->data)->unit = UNIT_METRIC;
    else
        ((MdhPanelData *)obj->data)->unit = UNIT_STANDARD;
}

static gboolean panel_callback(gpointer data, GError **err)
{
    MdhPanel *obj = data;

    gchar *command;

    gboolean ret;

    g_return_val_if_fail(obj != NULL, FALSE);
    g_return_val_if_fail(obj->command != NULL, FALSE);

    if(strstr(obj->command, " %s")) {
        gchar *s = g_strdup_printf(_WEATHER_URI_HOME, obj->value);
        command = g_strdup_printf(obj->command, s);
        g_free(s);
    } else
        command = obj->command;

    ret = g_spawn_command_line_async(command, err);

    if(command != obj->command)
        g_free(command);

    return(ret);
}

static void panel_config(const gchar *v_in, gchar **v_out,
                         const gchar *d_in, gchar **d_out,
                         const gchar *c_in, gchar **c_out,
                         const gint   i_in, gint   *i_out)
{
    g_return_if_fail(v_in != NULL);
    g_return_if_fail(v_out != NULL);
    g_return_if_fail(d_in != NULL);
    g_return_if_fail(d_out != NULL);
    g_return_if_fail(c_in != NULL);
    g_return_if_fail(c_out != NULL);
    g_return_if_fail(i_out != NULL);

    *v_out = g_ascii_strup(v_in, -1);
    *d_out = g_strdup(d_in);
    *c_out = mdh_expand_tilde(c_in);
    *i_out = i_in * 3600;
}

static MdhPanelFuncs funcs = {
    panel_main,
    panel_startup,
    panel_shutdown,
    panel_callback,
    panel_config
};

MdhPanel *mdh_panel_weather_new(const gchar *value,
                                const gchar *display,
                                const gchar *command,
                                gboolean enabled)
{
    const gchar *browser = mdh_get_browser();

    MdhPanel *obj = mdh_panel_new();

    g_return_val_if_fail(obj != NULL, NULL);

    obj->value    = (value)   ? g_strdup(value)   : g_strdup(_WEATHER_LOC);
    obj->display  = (display) ? g_strdup(display) : g_strdup(_WEATHER_DIS);
    obj->command  = (command) ? g_strdup(command) : g_strdup_printf("%s %%s",
                                                                    browser);
    obj->tooltip  = g_strdup_printf("Weather: %s", obj->value);
    obj->interval = _WEATHER_DEF;
    obj->enabled  = enabled;
    obj->funcs    = funcs;

    obj->data     = g_new(MdhPanelData, 1);

    ((MdhPanelData *) obj->data)->unit = UNIT_STANDARD;

    return(obj);
}

#endif /* _MDH_HAS_WEATHER */
