/*
 * mdh (MailDooHicky), a GTK+-based toolbar.
 *
 * Copyright (c) 2003-2005 by Mike Hokenson <mdh at gozer dot org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <sys/sysctl.h>

#if __FreeBSD_version >= 500101
# include <sys/resource.h>
# include <sys/user.h>
#else
# include <kvm.h>
#endif

#if __FreeBSD_version < 500101
static kvm_t *kd;
#endif

static gboolean sys_init = FALSE;

gboolean mdh_sys_init(GError **err)
{
    if(sys_init)
        return(TRUE);

#if __FreeBSD_version < 500101
    if(!(kd = kvm_openfiles(NULL, NULL, NULL, O_RDONLY, NULL))) {
        g_set_error(err, 0, 0, "kvm_openfiles: %s", g_strerror(errno));
        return(FALSE);
    }

    /* drop group (kmem) privs */
    if(setgid(getgid()) == 1) {
        g_set_error(err, 0, 0, "setgid: %s", g_strerror(errno));
        return(FALSE);
    }
#endif

    sys_init = TRUE;

    return(TRUE);
}

void mdh_sys_close(void)
{
    if(!sys_init)
        return;

#if __FreeBSD_version < 500101
    kvm_close(kd);
#endif

    sys_init = FALSE;
}

gboolean mdh_sys_get_mem(guint64 *total, guint64 *free, GError **err)
{
    gulong swap_total, swap_free;

    gulong physmem;
    size_t size;
    guint free;
    guint inactive;
    gulong pagesize;

#if __FreeBSD_version >= 500101
    gulong swap_used;

    gint mib[16], n;
    size_t mibsize;
#else
    gint mib[2];
    struct kvm_swap swapinfo;
#endif

    g_return_val_if_fail(total != NULL, FALSE);
    g_return_val_if_fail(free != NULL, FALSE);

    pagesize = getpagesize();

    mib[0] = CTL_HW;
    mib[1] = HW_PHYSMEM;

    size = sizeof(physmem);

    if(sysctl(mib, 2, &physmem, &size, NULL, 0) == -1) {
        g_set_error(err, 0, 0, "sysctl: %s", g_strerror(errno));
        return(FALSE);
    }

    size = sizeof(free);

    if(sysctlbyname("vm.stats.vm.v_free_count", &free, &size, NULL, 0) == -1) {
        g_set_error(err, 0, 0, "sysctlbyname: %s", g_strerror(errno));
        return(FALSE);
    }

    size = sizeof(inactive);

    if(sysctlbyname("vm.stats.vm.v_inactive_count",
                    &inactive, &size, NULL, 0) == -1) {
        g_set_error(err, 0, 0, "sysctlbyname: %s", g_strerror(errno));
        return(FALSE);
    }

#if __FreeBSD_version >= 500101
    swap_total = 0;
    swap_used  = 0;

    mibsize = sizeof(mib) / sizeof(mib[0]);

    if(sysctlnametomib("vm.swap_info", mib, &mibsize) == -1) {
        g_set_error(err, 0, 0, "sysctlnametomib: %s", g_strerror(errno));
        return(FALSE);
    }

    for(n = 0; ; ++n) {
        struct xswdev xsw;

        mib[mibsize] = n;
        size = sizeof(xsw);

        if(sysctl(mib, mibsize + 1, &xsw, &size, NULL, NULL) == -1)
            break;

        if(xsw.xsw_version != XSWDEV_VERSION)
            return(FALSE);

        if(xsw.xsw_version != XSWDEV_VERSION)
            return(FALSE);

        swap_total += (long long) xsw.xsw_nblks;
        swap_used  += (long long) xsw.xsw_used;
    }

    if(errno != ENOENT)
        return(FALSE);

    swap_free = swap_total - swap_used;
#else
    if(!mdh_sys_init(err))
        return(FALSE);

    if((kvm_getswapinfo(kd, &swapinfo, 1, 0)) == -1) {
        g_set_error(err, 0, 0, "kvm_getswapinfo: %s", g_strerror(errno));
        return(FALSE);
    }

    swap_total = (long long)swapinfo.ksw_total;
    swap_free  = swap_total - (long long)swapinfo.ksw_used;
#endif

    *total = (guint64) physmem + (swap_total * pagesize);
    *free  = (guint64) (free * pagesize) + (inactive * pagesize) + swap_free;

    return(TRUE);
}
