/*
 * mdh (MailDooHicky), a GTK+-based toolbar.
 *
 * Copyright (c) 2003-2005 by Mike Hokenson <mdh at gozer dot org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <unistd.h>

#include <gtk/gtk.h>

#include "mdh.h"

#include "toolbar.h"

#include "button.h"
#include "custom.h"
#include "panel.h"

#include "ui_about.h"
#include "ui_config.h"
#include "ui_custom.h"
#include "ui_error.h"
#include "ui_run.h"
#include "ui_scratch_pad.h"
#include "ui_mixer.h"

#include "error.h"
#include "widget.h"
#include "window.h"
#include "util.h"
#include "x11.h"

static gboolean toolbar_configure_event(GtkWidget *widget,
                                        GdkEventConfigure *event,
                                        gpointer data)
{
    MdhToolbar *toolbar = data;

    g_return_val_if_fail(toolbar != NULL, FALSE);

    toolbar->geom_x = event->x;
    toolbar->geom_y = event->y;

    return(FALSE);
}

/* setup MWM hints for the window managers that still look for it */
static gboolean toolbar_expose_event(GtkWidget *widget,
                                     GdkEventExpose *event,
                                     gpointer data)
{
    gdk_window_set_decorations(widget->window, 0);
    gdk_window_set_functions(widget->window, 0);

    return(FALSE);
}

MdhToolbar *mdh_toolbar_new(gboolean type_dock)
{
    MdhToolbar *toolbar;

    GtkWindow *window;

    toolbar                  = g_new(MdhToolbar, 1);

    toolbar->geom_x          = 0;
    toolbar->geom_y          = 0;

    toolbar->layer           = MDH_WINDOW_LAYER_NORMAL;
    toolbar->sticky          = TRUE;

    toolbar->tooltips        = TRUE;

    toolbar->font            = g_strdup("");

    toolbar->button_relief   = GTK_RELIEF_NORMAL;

    toolbar->panel_shadow    = GTK_SHADOW_IN;

    toolbar->hidden          = FALSE;

    toolbar->buttons         = NULL;
    toolbar->custom          = NULL;
    toolbar->panels          = NULL;

    if(mdh_x11_has_netwm())
        toolbar->widgets.window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    else
        toolbar->widgets.window = gtk_window_new(GTK_WINDOW_POPUP);

    toolbar->widgets.hbox = gtk_hbox_new(FALSE, 0);

    toolbar->widgets.tips = gtk_tooltips_new();

    toolbar->widgets.menu = NULL;

    window = GTK_WINDOW(toolbar->widgets.window);

    gtk_widget_set_name(GTK_WIDGET(window), PACKAGE_NAME);

    /*
     * Openbox 2.x: shows titlebar without
     * Openbox 3.x: mostly unaffected, no longer movable with alt+mouse
     */
    if(type_dock)
      gtk_window_set_type_hint(window, GDK_WINDOW_TYPE_HINT_DOCK);

    if(mdh_x11_has_netwm_state_skip_taskbar())
        gtk_window_set_skip_taskbar_hint(window, TRUE);

    if(mdh_x11_has_netwm_state_skip_pager())
        gtk_window_set_skip_pager_hint(window, TRUE);

    gtk_window_set_resizable(window, FALSE);
    gtk_window_set_decorated(window, FALSE);
    gtk_window_set_accept_focus(window, FALSE);

    gtk_container_add(GTK_CONTAINER(window), toolbar->widgets.hbox);

    gtk_container_set_border_width(GTK_CONTAINER(window), 1);

    g_signal_connect(G_OBJECT(window), "configure_event",
                     G_CALLBACK(toolbar_configure_event), toolbar);

    g_signal_connect(G_OBJECT(window), "expose_event",
                     G_CALLBACK(toolbar_expose_event), NULL);

    /* for mdh_x11_set_gnome_layer() */
    gtk_widget_realize(toolbar->widgets.window);

    return(toolbar);
}

void mdh_toolbar_free(MdhToolbar *toolbar)
{
    g_return_if_fail(toolbar != NULL);

    g_free(toolbar->font);

    g_slist_free(toolbar->buttons);
    g_slist_free(toolbar->custom);
    g_slist_free(toolbar->panels);

    g_free(toolbar);
}

void mdh_toolbar_show(MdhToolbar *toolbar)
{
    g_return_if_fail(toolbar != NULL);
    g_return_if_fail(toolbar->widgets.window != NULL);
    g_return_if_fail(toolbar->widgets.hbox != NULL);

    gtk_widget_show(toolbar->widgets.hbox);
    gtk_widget_show(toolbar->widgets.window);
}

static void toolbar_quit(MdhToolbar *toolbar)
{
  /*if(mdh_window_yes_no_prompt("Quit MailDooHicky. Are you sure?", NULL))*/
        gtk_main_quit();
}

static void toolbar_menu_custom_activate(MdhCustom *obj)
{
    GError *err = NULL;

    g_return_if_fail(obj != NULL);
    g_return_if_fail(obj->command != NULL);

    if(!mdh_custom_execute_callback(obj, &err)) {
        mdh_window_error("Execution failed.", err->message);
        g_error_free(err);
    }
}

static GtkWidget *toolbar_menu_create(void)
{
    GtkWidget *menu;
    GtkWidget *menu_sub;
    GtkWidget *menu_item;

    debug("menu_create()");

    menu = gtk_menu_new();
    gtk_widget_show(menu);

    menu_sub = gtk_menu_new();
    gtk_widget_show(menu_sub);

    menu_item = mdh_image_menu_item_new_from_stock("MailDooHicky",
                                                   GTK_STOCK_HOME);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), menu_item);
    gtk_widget_show(menu_item);

    gtk_menu_item_set_submenu(GTK_MENU_ITEM(menu_item), menu_sub);

    menu_item = gtk_menu_item_new();
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), menu_item);
    gtk_widget_show(menu_item);

    /* MailDooHicky sub-menu */

        menu_item = mdh_image_menu_item_new_from_stock("About",
                                                       GTK_STOCK_ABOUT);
        gtk_menu_shell_append(GTK_MENU_SHELL(menu_sub), menu_item);
        gtk_widget_show(menu_item);

        if(mdh_window_item_open(&wi_about))
            gtk_widget_set_sensitive(GTK_WIDGET(menu_item), FALSE);

        g_signal_connect(G_OBJECT(menu_item), "activate",
                         G_CALLBACK(mdh_ui_about), NULL);

        wi_about.menu_item = menu_item;

        menu_item = gtk_menu_item_new();
        gtk_menu_shell_append(GTK_MENU_SHELL(menu_sub), menu_item);
        gtk_widget_show(menu_item);

        menu_item = mdh_image_menu_item_new_from_stock("Preferences",
                                                       GTK_STOCK_PREFERENCES);
        gtk_menu_shell_append(GTK_MENU_SHELL(menu_sub), menu_item);
        gtk_widget_show(menu_item);

        if(mdh_window_item_open(&wi_config))
            gtk_widget_set_sensitive(GTK_WIDGET(menu_item), FALSE);

        g_signal_connect(G_OBJECT(menu_item), "activate",
                         G_CALLBACK(mdh_ui_config), NULL);

        wi_config.menu_item = menu_item;

        menu_item = mdh_image_menu_item_new_from_stock("Custom Buttons",
                                                       GTK_STOCK_PROPERTIES);

        gtk_menu_shell_append(GTK_MENU_SHELL(menu_sub), menu_item);
        gtk_widget_show(menu_item);

        if(mdh_window_item_open(&wi_custom))
            gtk_widget_set_sensitive(GTK_WIDGET(menu_item), FALSE);

        g_signal_connect(G_OBJECT(menu_item), "activate",
                         G_CALLBACK(mdh_ui_custom), NULL);

        wi_custom.menu_item = menu_item;

        menu_item = mdh_image_menu_item_new_from_stock("Toggle Toolbar",
                                                       GTK_STOCK_REFRESH);
        gtk_menu_shell_append(GTK_MENU_SHELL(menu_sub), menu_item);
        gtk_widget_show(menu_item);

        /* Menu and Hide buttons are the only two visible when toggled */
        if(!B_Menu->enabled && !B_Toggle->enabled)
            gtk_widget_set_sensitive(GTK_WIDGET(menu_item), FALSE);

        g_signal_connect_swapped(G_OBJECT(menu_item), "activate",
                                 G_CALLBACK(mdh_toolbar_toggle), Toolbar);

        menu_item = gtk_menu_item_new();
        gtk_menu_shell_append(GTK_MENU_SHELL(menu_sub), menu_item);
        gtk_widget_show(menu_item);

        menu_item = mdh_image_menu_item_new_from_stock("Error Log",
                                                       GTK_STOCK_DIALOG_ERROR);
        gtk_menu_shell_append(GTK_MENU_SHELL(menu_sub), menu_item);
        gtk_widget_show(menu_item);

        if(mdh_window_item_open(&wi_error))
            gtk_widget_set_sensitive(GTK_WIDGET(menu_item), FALSE);

        g_signal_connect(G_OBJECT(menu_item), "activate",
                         G_CALLBACK(mdh_ui_error), NULL);

        wi_error.menu_item = menu_item;

    menu_sub = gtk_menu_new();
    gtk_widget_show(menu_sub);

    {
        const gchar *file = mdh_custom_icon_default();
        menu_item = mdh_image_menu_item_new_from_file("Custom Buttons", file);
    }

    gtk_menu_shell_append(GTK_MENU_SHELL(menu), menu_item);
    gtk_widget_show(menu_item);

    gtk_menu_item_set_submenu(GTK_MENU_ITEM(menu_item), menu_sub);

    /* Custom Buttons sub-menu */

    {
        gint i;

        for(i = 0; i < mdh_custom_count(); i++) {
            MdhCustom *obj = B_Cust[i];

            const gchar *title = (obj->tooltip) ? obj->tooltip : obj->value;

            menu_item = gtk_image_menu_item_new_with_label(title);
            gtk_menu_shell_append(GTK_MENU_SHELL(menu_sub), menu_item);
            gtk_widget_show(menu_item);

            if(obj->icon && *obj->icon && IS_FILE(obj->icon)) {
                GtkWidget *icon = gtk_image_new_from_file(obj->icon);
                gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(menu_item),
                                              icon);
            }

            g_signal_connect_swapped(G_OBJECT(menu_item), "activate",
                                G_CALLBACK(toolbar_menu_custom_activate), obj);
        }
    }

    menu_item = gtk_menu_item_new();
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), menu_item);
    gtk_widget_show(menu_item);

    menu_item = mdh_image_menu_item_new_from_stock("Scratch-Pad",
                                                   GTK_STOCK_EDIT);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), menu_item);
    gtk_widget_show(menu_item);

    if(mdh_window_item_open(&wi_scratch_pad))
        gtk_widget_set_sensitive(GTK_WIDGET(menu_item), FALSE);

    g_signal_connect(G_OBJECT(menu_item), "activate",
                     G_CALLBACK(mdh_ui_scratch_pad), NULL);

    wi_scratch_pad.menu_item = menu_item;

#if _MDH_HAS_MIXER
    menu_item = mdh_image_menu_item_new_from_file("Mixer",
                                                  mdh_icon_path("mixer"));
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), menu_item);
    gtk_widget_show(menu_item);

    if(mdh_window_item_open(&wi_mixer))
        gtk_widget_set_sensitive(GTK_WIDGET(menu_item), FALSE);

    g_signal_connect(G_OBJECT(menu_item), "activate",
                     G_CALLBACK(mdh_ui_mixer), NULL);

    wi_mixer.menu_item = menu_item;
#endif

    menu_item = mdh_image_menu_item_new_from_stock("Run...", GTK_STOCK_EXECUTE);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), menu_item);
    gtk_widget_show(menu_item);

    if(mdh_window_item_open(&wi_run))
        gtk_widget_set_sensitive(GTK_WIDGET(menu_item), FALSE);

    g_signal_connect(G_OBJECT(menu_item), "activate",
                     G_CALLBACK(mdh_ui_run), NULL);

    wi_run.menu_item = menu_item;

    menu_item = gtk_menu_item_new();
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), menu_item);
    gtk_widget_show(menu_item);

    menu_item = mdh_image_menu_item_new_from_stock("Quit", GTK_STOCK_QUIT);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), menu_item);
    gtk_widget_show(menu_item);

    g_signal_connect_swapped(G_OBJECT(menu_item), "activate",
                             G_CALLBACK(toolbar_quit), Toolbar);

    return(menu);
}

void mdh_toolbar_menu_destroy(MdhToolbar *toolbar)
{
    MdhToolbarWidgets *widgets;

    g_return_if_fail(toolbar != NULL);

    widgets = &toolbar->widgets;

    g_return_if_fail(widgets != NULL);

    debug("mdh_toolbar_menu_destroy()");

    if(widgets->menu && GTK_IS_MENU(widgets->menu)) {
        gtk_widget_unparent(GTK_WIDGET(widgets->menu));
        gtk_widget_destroy(GTK_WIDGET(widgets->menu));
        gtk_object_sink(GTK_OBJECT(widgets->menu));
    }
    
    widgets->menu = NULL;

    wi_about.menu_item       = NULL;
    wi_config.menu_item      = NULL;
    wi_custom.menu_item      = NULL;
    wi_error.menu_item       = NULL;
    wi_run.menu_item         = NULL;
    wi_scratch_pad.menu_item = NULL;
#if _MDH_HAS_MIXER
    wi_mixer.menu_item       = NULL;
#endif
}

gboolean mdh_toolbar_menu_show(MdhToolbar *toolbar)
{
    MdhToolbarWidgets *widgets;

    g_return_val_if_fail(toolbar != NULL, FALSE);

    widgets = &toolbar->widgets;

    g_return_val_if_fail(widgets != NULL, FALSE);

    if(!widgets->menu || GTK_IS_MENU(widgets->menu))
        widgets->menu = toolbar_menu_create();

    gtk_menu_popup(GTK_MENU(widgets->menu), NULL, NULL, NULL, NULL, 0, 0);

    return(FALSE);
}

static gint toolbar_handle_event(GdkEvent *event,
                                 MdhPanelFunc func,
                                 gpointer data)
{
    g_return_val_if_fail(event != NULL, 0);
    g_return_val_if_fail(func != NULL, 0);

    if(event->type == GDK_2BUTTON_PRESS || event->type == GDK_3BUTTON_PRESS)
        return(0);

    if(event->type == GDK_BUTTON_RELEASE) {
        GdkEventButton *bevent = (GdkEventButton *) event;

        switch(bevent->button) {
            case 1:
          /*case 2:*/
            {
                GError *err = NULL;

                if(!func(data, &err)) {
                    mdh_window_error("Execution failed.", err->message);
                    g_error_free(err);
                }

                break;
            }

            case 3:
                mdh_toolbar_menu_show(Toolbar);
                break;

            default:
                break;
        }
    }

    return(0);
}

static gint toolbar_button_event(GtkWidget *w, GdkEvent *e, MdhButton *o)
{
    MdhPanelFunc func = (MdhPanelFunc) mdh_button_execute_callback;
    return(toolbar_handle_event(e, func, o));
}

static gint toolbar_custom_event(GtkWidget *w, GdkEvent *e, MdhCustom *o)
{
    MdhPanelFunc func = (MdhPanelFunc) mdh_custom_execute_callback;
    return(toolbar_handle_event(e, func, o));
}

static gint toolbar_panel_event(GtkWidget *w, GdkEvent *e, MdhPanel *o)
{
    MdhPanelFunc func = (MdhPanelFunc) mdh_panel_execute_callback;
    return(toolbar_handle_event(e, func, o));
}

void mdh_toolbar_set_layer(MdhToolbar *toolbar, MdhWindowLayer layer)
{
    MdhToolbarWidgets *widgets;

    gboolean above = FALSE,
             below = FALSE;

    g_return_if_fail(toolbar != NULL);

    widgets = &toolbar->widgets;

    g_return_if_fail(widgets != NULL);
    g_return_if_fail(widgets->window != NULL);

    switch(layer) {
        case MDH_WINDOW_LAYER_NORMAL:
            break;
        case MDH_WINDOW_LAYER_BELOW:
            below = TRUE;
            break;
        case MDH_WINDOW_LAYER_ABOVE:
            above = TRUE;
            break;
        default:
            g_assert_not_reached();
    }

    if(mdh_x11_has_netwm_layers()) {
        gtk_window_set_keep_above(GTK_WINDOW(widgets->window), above);
        gtk_window_set_keep_below(GTK_WINDOW(widgets->window), below);
    } else if(mdh_x11_has_gnome_layers()) {
        mdh_x11_set_gnome_layer_above(GTK_WIDGET(widgets->window), above);
        mdh_x11_set_gnome_layer_below(GTK_WIDGET(widgets->window), below);
    }
}

void mdh_toolbar_set_sticky(MdhToolbar *toolbar, gboolean sticky)
{
    MdhToolbarWidgets *widgets;

    g_return_if_fail(toolbar != NULL);

    /* GTK_WINDOW_POPUP should already keep the window stuck */
    if(!mdh_x11_has_netwm())
        return;

    widgets = &toolbar->widgets;

    g_return_if_fail(widgets != NULL);
    g_return_if_fail(widgets->window != NULL);

    if(sticky)
        gtk_window_stick(GTK_WINDOW(widgets->window));
    else
        gtk_window_unstick(GTK_WINDOW(widgets->window));
}

void mdh_toolbar_move(MdhToolbar *toolbar, gint x, gint y)
{
    g_return_if_fail(toolbar != NULL);
    g_return_if_fail(toolbar->widgets.window != NULL);

    gtk_window_move(GTK_WINDOW(toolbar->widgets.window), x, y);
}

gboolean mdh_toolbar_hidden(MdhToolbar *toolbar)
{
    g_return_val_if_fail(toolbar != NULL, FALSE);

    return(toolbar->hidden);
}

/* make sure one button or panel is displayed. if none, show menu button */
static void toolbar_check_visibility(MdhToolbar *toolbar)
{
    GSList *p;

    g_return_if_fail(toolbar != NULL);

    debug("toolbar_check_visibility()");

    for(p = toolbar->buttons; p; p = g_slist_next(p))
        if(!mdh_button_hidden(p->data))
            return;

    for(p = toolbar->custom; p;  p = g_slist_next(p))
        if(!mdh_custom_hidden(p->data))
            return;

    for(p = toolbar->panels; p;  p = g_slist_next(p))
        if(!mdh_panel_hidden(p->data))
            return;

    debug("toolbar_check_visibility(): all items disabled");

    mdh_button_show(B_Menu);
}

void mdh_toolbar_toggle(MdhToolbar *toolbar)
{
    g_return_if_fail(toolbar != NULL);

    debug("mdh_toolbar_toggle()");

    /* hiding the panel while the config window is open can cause problems */
    if(mdh_window_item_open(&wi_config) || mdh_window_item_open(&wi_custom))
        return;

    toolbar->hidden = !toolbar->hidden;

    mdh_button_foreach(toolbar, (GFunc) mdh_button_toggle, NULL);
    mdh_custom_foreach(toolbar, (GFunc) mdh_custom_toggle, NULL);
    mdh_panel_foreach(toolbar,  (GFunc) mdh_panel_toggle,  NULL);
}

void mdh_toolbar_reset(MdhToolbar *toolbar)
{
    PangoFontDescription *font_desc;

    GSList *p;

    g_return_if_fail(toolbar != NULL);

    debug("mdh_toolbar_reset()");
    
    mdh_toolbar_move(toolbar, toolbar->geom_x, toolbar->geom_y);
    
    mdh_toolbar_set_layer(toolbar, toolbar->layer);
    mdh_toolbar_set_sticky(toolbar, toolbar->sticky);
    
    if(toolbar->tooltips)
        gtk_tooltips_enable(toolbar->widgets.tips);
    else
        gtk_tooltips_disable(toolbar->widgets.tips);

    if(toolbar->hidden)
        mdh_toolbar_toggle(toolbar);

    font_desc = pango_font_description_from_string(toolbar->font);

    for(p = toolbar->buttons; p; p = g_slist_next(p)) {
        MdhButton *obj = p->data;

        mdh_button_set_relief(obj, toolbar->button_relief);
        mdh_button_set_tip(obj, obj->tooltip);
        mdh_button_reset(obj);
    }

    for(p = toolbar->custom; p; p = g_slist_next(p)) {
        MdhCustom *obj = p->data;

        mdh_custom_set_relief(obj, toolbar->button_relief);
        mdh_custom_set_tip(obj, obj->tooltip);
        mdh_custom_set_font_desc(obj, font_desc);
        mdh_custom_reset(obj);
    }

    for(p = toolbar->panels; p; p = g_slist_next(p)) {
        MdhPanel *obj = p->data;

        mdh_panel_set_shadow_type(obj, toolbar->panel_shadow);
        mdh_panel_set_tip(obj, obj->tooltip);
        mdh_panel_set_font_desc(obj, font_desc);
        mdh_panel_reset(obj);
    }

    pango_font_description_free(font_desc);

    toolbar_check_visibility(toolbar);
}

void mdh_toolbar_reset_custom(MdhToolbar *toolbar)
{
    GSList *p;

    g_return_if_fail(toolbar != NULL);

    debug("mdh_toolbar_reset_custom()");

    for(p = toolbar->custom; p; p = g_slist_next(p)) {
        MdhCustom *obj = p->data;

        mdh_custom_set_tip(obj, obj->tooltip);
        mdh_custom_reset(obj);
    }

    toolbar_check_visibility(toolbar);
}

void mdh_toolbar_button_add(MdhToolbar *toolbar, gpointer data)
{
    MdhButton *obj = data;

    GtkWidget *vbox;

    MdhButtonWidgets *widgets;

    g_return_if_fail(toolbar != NULL);
    g_return_if_fail(toolbar->widgets.hbox != NULL);
    g_return_if_fail(obj != NULL);

    widgets = &obj->widgets;

    g_return_if_fail(widgets != NULL);
    g_return_if_fail(widgets->button != NULL);
    g_return_if_fail(widgets->icon != NULL);

    vbox = gtk_vbox_new(TRUE, 0);

    gtk_box_pack_start(GTK_BOX(vbox), widgets->icon, TRUE, TRUE, 0);

    gtk_box_pack_start(GTK_BOX(toolbar->widgets.hbox), widgets->button,
                       FALSE, FALSE, 0);

    GTK_WIDGET_UNSET_FLAGS(GTK_WIDGET(widgets->button), GTK_CAN_FOCUS);

    gtk_container_add(GTK_CONTAINER(widgets->button), vbox);

    g_signal_connect(G_OBJECT(widgets->button), "button_release_event",
                     G_CALLBACK(toolbar_button_event), obj);

    toolbar->buttons = g_slist_append(toolbar->buttons, obj);

    obj->toolbar = toolbar;
}

void mdh_toolbar_custom_add(MdhToolbar *toolbar, gpointer data)
{
    MdhCustom *obj = data;

    GtkWidget *vbox,
              *hbox;

    MdhCustomWidgets *widgets;

    g_return_if_fail(toolbar != NULL);
    g_return_if_fail(toolbar->widgets.hbox != NULL);
    g_return_if_fail(obj != NULL);

    widgets = &obj->widgets;

    g_return_if_fail(widgets != NULL);
    g_return_if_fail(widgets->button != NULL);
    g_return_if_fail(widgets->label != NULL);
    g_return_if_fail(widgets->icon != NULL);

    vbox = gtk_vbox_new(TRUE, 0);

    hbox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

    gtk_box_pack_start(GTK_BOX(hbox), widgets->icon, TRUE, TRUE, 0);

    if(obj->icon && obj->icon[0] && IS_FILE(obj->icon))
        gtk_image_set_from_file(GTK_IMAGE(widgets->icon), obj->icon);

    gtk_box_pack_start(GTK_BOX(hbox), widgets->label, TRUE, TRUE, 0);

    gtk_box_pack_start(GTK_BOX(toolbar->widgets.hbox), widgets->button,
                       FALSE, FALSE, 0);

    GTK_WIDGET_UNSET_FLAGS(GTK_WIDGET(widgets->button), GTK_CAN_FOCUS);

    gtk_container_add(GTK_CONTAINER(widgets->button), vbox);

    g_signal_connect(G_OBJECT(widgets->button), "button_release_event",
                     G_CALLBACK(toolbar_custom_event), obj);

    toolbar->custom = g_slist_append(toolbar->custom, obj);

    obj->toolbar = toolbar;
}

void mdh_toolbar_panel_add(MdhToolbar *toolbar, gpointer data)
{
    MdhPanel *obj = data;

    GtkWidget *hbox;

    MdhPanelWidgets *widgets;

    g_return_if_fail(toolbar != NULL);
    g_return_if_fail(toolbar->widgets.hbox != NULL);
    g_return_if_fail(obj != NULL);

    widgets = &obj->widgets;

    g_return_if_fail(widgets != NULL);
    g_return_if_fail(widgets->event_box != NULL);
    g_return_if_fail(widgets->label != NULL);
    g_return_if_fail(widgets->icon != NULL);

    gtk_box_pack_start(GTK_BOX(toolbar->widgets.hbox), widgets->event_box,
                       FALSE, FALSE, 0);

    g_signal_connect(G_OBJECT(widgets->event_box), "button_release_event",
                     G_CALLBACK(toolbar_panel_event), obj);

    gtk_container_add(GTK_CONTAINER(widgets->event_box), widgets->frame);

    hbox = gtk_hbox_new(FALSE, 0);

    gtk_box_pack_start(GTK_BOX(hbox), widgets->icon, FALSE, FALSE, 0);

    gtk_misc_set_alignment(GTK_MISC(widgets->label), 0.5, 0.5);
    gtk_misc_set_padding(GTK_MISC(widgets->label), 4, 3);
    gtk_box_pack_start(GTK_BOX(hbox), widgets->label, FALSE, FALSE, 0);

    gtk_container_add(GTK_CONTAINER(widgets->frame), hbox);

    toolbar->panels = g_slist_append(toolbar->panels, obj);

    obj->toolbar = toolbar;
}
