/*
 * mdh (MailDooHicky), a GTK+-based toolbar.
 *
 * Copyright (c) 2003-2005 by Mike Hokenson <mdh at gozer dot org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <unistd.h>

#include <gtk/gtk.h>

#include "toolbar.h"

#include "custom.h"

#include "error.h"
#include "widget.h"
#include "window.h"
#include "util.h"

static GtkWidget *custom_window;
static GtkWidget *custom_view;
static GtkWidget *custom_fs;

typedef enum {
    USER_ENABLED,
    USER_NAME,
    USER_PIXBUF,
    USER_ICON,
    USER_TOOLTIP,
    USER_COMMAND,
    USER_EDITABLE,
    USER_N_COLUMNS
} MdhCustomColumns;

static void ui_custom_cell_edit(GtkCellRendererText *cell,
                                gchar *path_string,
                                gchar *new_text,
                                gpointer data)
{
    GtkTreeModel *model = data;
    GtkTreePath  *path  = gtk_tree_path_new_from_string(path_string);

    gint *column = g_object_get_data(G_OBJECT(cell), "column");

    gchar *text = new_text;

    GtkTreeIter iter;

    gtk_tree_model_get_iter(model, &iter, path);

    switch(GPOINTER_TO_INT(column)) {
        case USER_COMMAND:
            if(!*new_text) {
                gtk_tree_model_get(model, &iter, column, &text, -1);

                mdh_window_notice("Command is required.",
                    "This field cannot be empty, reverting to previous.");

                break;
            }

            break;

        default:
            break;
    }

    gtk_list_store_set(GTK_LIST_STORE(model), &iter, column, text, -1);

    if(text != new_text)
        g_free(text);

    gtk_tree_path_free(path);
}

static void ui_custom_toggled(GtkCellRendererToggle *cell,
                              gchar *path_str,
                              gpointer data)
{
    GtkTreeModel *model = data;
    GtkTreePath  *path  = gtk_tree_path_new_from_string(path_str);
    GtkTreeIter  iter;

    gboolean enabled;

    gtk_tree_model_get_iter(model, &iter, path);
    gtk_tree_model_get(model, &iter, USER_ENABLED, &enabled, -1);

    enabled ^= 1;

    gtk_list_store_set(GTK_LIST_STORE(model), &iter, USER_ENABLED, enabled, -1);

    gtk_tree_path_free(path);
}

static void ui_custom_apply(GtkTreeModel *model)
{
    GtkTreeIter iter;

    gint i, count = 0;

    if(gtk_tree_model_get_iter_first(model, &iter))
        do {
            if(count >= _MDH_BUTTON_COUNT_CUSTOM_MAX)
                break;

            mdh_custom_free_values(B_Cust[count]);

            gtk_tree_model_get(model, &iter,
                               USER_ENABLED, &B_Cust[count]->enabled,
                               USER_NAME,    &B_Cust[count]->value,
                               USER_ICON,    &B_Cust[count]->icon,
                               USER_TOOLTIP, &B_Cust[count]->tooltip,
                               USER_COMMAND, &B_Cust[count]->command,
                               -1);

            count++;
        } while(gtk_tree_model_iter_next(model, &iter));

    /* free and disable deleted */
    for(i = count; i < mdh_custom_count(); i++) {
        mdh_custom_free_values(B_Cust[i]);
        mdh_custom_set_values(B_Cust[i], FALSE, NULL, NULL, NULL, NULL);
    }

    mdh_custom_count_set(count);

    /* just refresh the custom buttons */
    mdh_toolbar_reset_custom(Toolbar);
}

static void ui_custom_close(GtkWidget *window)
{
    mdh_window_item_open_set(&wi_custom, FALSE);
    mdh_toolbar_menu_destroy(Toolbar);
}

static void ui_custom_move_up(GtkTreeView *view)
{
    GtkTreeSelection *selection;
    GtkTreeModel *model;
    GtkTreePath *path;
    GtkTreeIter iter, iter_dst;

    GList *list;

    g_return_if_fail(view != NULL);

    selection = gtk_tree_view_get_selection(view);

    /* disallow allow move if more than one is selected */
    if(gtk_tree_selection_count_selected_rows(selection) != 1)
        return;

    list = gtk_tree_selection_get_selected_rows(selection, &model);

    g_return_if_fail(list != NULL);

    path = gtk_tree_path_copy(list->data);

    g_list_foreach(list, (GFunc) gtk_tree_path_free, NULL);
    g_list_free(list);

    gtk_tree_model_get_iter(model, &iter, path);

    if(gtk_tree_path_prev(path)) {
        gtk_tree_model_get_iter(model, &iter_dst, path);
        gtk_list_store_swap(GTK_LIST_STORE(model), &iter, &iter_dst);
    }

    gtk_tree_path_free(path);
}

static void ui_custom_move_down(GtkTreeView *view)
{
    GtkTreeSelection *selection;
    GtkTreeModel *model;
    GtkTreePath *path;
    GtkTreeIter iter, iter_dst;

    GList *list;

    g_return_if_fail(view != NULL);

    selection = gtk_tree_view_get_selection(view);

    /* disallow allow move if more than one is selected */
    if(gtk_tree_selection_count_selected_rows(selection) != 1)
        return;

    list = gtk_tree_selection_get_selected_rows(selection, &model);

    g_return_if_fail(list != NULL);

    path = gtk_tree_path_copy(list->data);

    g_list_foreach(list, (GFunc) gtk_tree_path_free, NULL);
    g_list_free(list);

    gtk_tree_model_get_iter(model, &iter, path);

    gtk_tree_path_next(path);

    if(gtk_tree_model_get_iter(model, &iter_dst, path))
        gtk_list_store_swap(GTK_LIST_STORE(model), &iter, &iter_dst);

    gtk_tree_path_free(path);
}

static void ui_custom_delete(GtkTreeView *view)
{
    GtkTreeSelection *selection;
    GtkTreeModel *model;

    GList *list, *p;

    g_return_if_fail(view != NULL);

    selection = gtk_tree_view_get_selection(view);

    if(gtk_tree_selection_count_selected_rows(selection) > 1)
        if(!mdh_window_yes_no_prompt("Delete selected buttons. Are you sure?"))
            return;

    list = gtk_tree_selection_get_selected_rows(selection, &model);

    for(p = g_list_reverse(list); p; p = g_list_next(p)) {
        GtkTreeIter iter;

        gtk_tree_model_get_iter(model, &iter, p->data);
        gtk_list_store_remove(GTK_LIST_STORE(model), &iter);
    }

    g_list_foreach(list, (GFunc) gtk_tree_path_free, NULL);
    g_list_free(list);
}

static void ui_custom_append(GtkTreeView *view, MdhCustom *obj)
{
    GtkTreeModel *model;
    GtkTreeIter iter;

    const gchar *value, *command, *tooltip;
    gboolean enabled;

    const gchar *file;

    GdkPixbuf *icon = NULL;

    g_return_if_fail(view != NULL);

    enabled = (obj) ? obj->enabled : FALSE;
    value   = (obj) ? obj->value   : "Name";
    command = (obj) ? obj->command : "command";
    tooltip = (obj) ? obj->tooltip : "Tooltip";
    file    = (obj) ? obj->icon    : mdh_custom_icon_default();

    if(IS_FILE(file)) {
        GError *err = NULL;

        if(!(icon = gdk_pixbuf_new_from_file(file, &err))) {
            ADD_LOG("%s", err->message);
            g_error_free(err);
        }
    }

    model = gtk_tree_view_get_model(GTK_TREE_VIEW(view));

    gtk_list_store_append(GTK_LIST_STORE(model), &iter);

    gtk_list_store_set(GTK_LIST_STORE(model), &iter,
                       USER_ENABLED,  enabled,
                       USER_NAME,     value,
                       USER_PIXBUF,   icon,
                       USER_ICON,     file,
                       USER_TOOLTIP,  tooltip,
                       USER_COMMAND,  command,
                       USER_EDITABLE, TRUE, -1);

    if(icon)
        g_object_unref(G_OBJECT(icon));
}

static void ui_custom_add(GtkTreeView *view)
{
    GtkTreeModel *model;
    GtkTreeIter iter;

    gint count = 0;

    g_return_if_fail(view != NULL);

    model = gtk_tree_view_get_model(view);

    if(gtk_tree_model_get_iter_first(model, &iter))
        do {
            count++;
        } while(gtk_tree_model_iter_next(model, &iter));

    if(count < _MDH_BUTTON_COUNT_CUSTOM_MAX)
        ui_custom_append(view, NULL);
}

static void ui_custom_icon_select(GtkWidget *widget, GtkTreePath *path)
{
    GtkFileSelection *fs = GTK_FILE_SELECTION(custom_fs);

    const gchar *file = gtk_file_selection_get_filename(fs);

    GtkTreeModel *model = gtk_tree_view_get_model(GTK_TREE_VIEW(custom_view));

    GtkTreeIter iter;

    GdkPixbuf *icon = NULL;

    if(IS_FILE(file)) {
        GError *err = NULL;

        /* now display an error when selecting individual icons */
        if(!(icon = gdk_pixbuf_new_from_file(file, &err))) {
            mdh_window_error("Error selecting icon.", err->message);
            g_error_free(err);
            goto custom_icon_select_end;
        }
    } else
        file = "";

    gtk_tree_model_get_iter(model, &iter, path);

    gtk_list_store_set(GTK_LIST_STORE(model), &iter,
                       USER_PIXBUF, icon,
                       USER_ICON,   file,
                       -1);

    if(icon)
        g_object_unref(G_OBJECT(icon));

custom_icon_select_end:

    gtk_tree_path_free(path);
}

static void ui_custom_icon_selection_close(void)
{
    custom_fs = NULL;
}

static void ui_custom_icon_selection(GtkWidget *widget,
                                     GdkEvent *event,
                                     GtkTreeView *view)
{
    GdkEventButton *bevent = (GdkEventButton *) event;

    GtkWidget *fs;

    GtkTreeViewColumn *column;
    GtkTreePath *path;

    g_return_if_fail(event != NULL);
    g_return_if_fail(event->type == GDK_BUTTON_RELEASE);
    g_return_if_fail(view != NULL);

    if(bevent->button != 1)
        return;

    if(custom_fs) {
        mdh_window_notice("Put on pants Zoidberg!",
                          "Sorry, I'm a slacker. Only one at a time. :I");
        return;
    }

    if(!gtk_tree_view_get_path_at_pos(view,
                                      bevent->x, 
                                      bevent->y, 
                                      &path,
                                      &column,
                                      NULL,
                                      NULL))
        return;

    /* XXX: there must be a better way to do this */
    if(!column->title || strcmp(column->title, "Icon"))
        return;

    custom_fs = fs = gtk_file_selection_new("Icon selection");

    gtk_file_selection_set_select_multiple(GTK_FILE_SELECTION(fs), FALSE);
    gtk_file_selection_hide_fileop_buttons(GTK_FILE_SELECTION(fs));
    gtk_file_selection_complete(GTK_FILE_SELECTION(fs), "*.png");

    /* set current selection */
    {
        GtkTreeModel *model = gtk_tree_view_get_model(view);
        GtkTreeIter iter;

        gchar *text;

        gtk_tree_model_get_iter(model, &iter, path);
        gtk_tree_model_get(model, &iter, USER_ICON, &text, -1);

        gtk_file_selection_set_filename(GTK_FILE_SELECTION(fs), text);

        g_free(text);
    }

    g_signal_connect(G_OBJECT(fs), "destroy",
                     G_CALLBACK(ui_custom_icon_selection_close), NULL);

    g_signal_connect(GTK_FILE_SELECTION(fs)->ok_button, "clicked",
                     G_CALLBACK(ui_custom_icon_select), path);

    g_signal_connect_swapped(GTK_FILE_SELECTION(fs)->ok_button, "clicked",
                     G_CALLBACK(gtk_widget_destroy), fs);

    g_signal_connect_swapped(GTK_FILE_SELECTION(fs)->cancel_button, "clicked",
                     G_CALLBACK(gtk_widget_destroy), fs);

    gtk_widget_show(GTK_WIDGET(fs));
}

void mdh_ui_custom(void)
{
    GtkWidget *window;
    GtkWidget *vbox;
    GtkWidget *main_hbox, *hbox;
    GtkWidget *button;
    GtkWidget *vsep;
    GtkWidget *view;

    GtkWidget *scrolled_window;

    GtkCellRenderer *renderer;

    GtkListStore *store;
    GtkTreeViewColumn *column;
    GtkTreeModel *model;

    gint i;

    g_return_if_fail(mdh_window_item_open(&wi_custom) != TRUE);

    /* expand panel */
    if(mdh_toolbar_hidden(Toolbar))
        mdh_toolbar_toggle(Toolbar);

    /* set after toggle_all() otherwise the toolbar won't expand */
    mdh_window_item_open_set(&wi_custom, TRUE);

    custom_window = window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(window), "MailDooHicky: Custom Buttons");
    gtk_window_set_resizable(GTK_WINDOW(window), FALSE);

    gtk_widget_set_size_request(GTK_WIDGET(window), -1, 375);

    g_signal_connect(G_OBJECT(window), "delete_event",
                     G_CALLBACK(gtk_true), NULL);

    g_signal_connect(G_OBJECT(window), "destroy",
                     G_CALLBACK(ui_custom_close), NULL);

    g_signal_connect(G_OBJECT(window), "expose_event",
                     G_CALLBACK(mdh_window_expose_event), NULL);

    gtk_container_set_border_width(GTK_CONTAINER(window), 5);

    vbox = gtk_vbox_new(FALSE, 5);
    gtk_widget_show(vbox);

    gtk_container_add(GTK_CONTAINER(window), vbox);

    scrolled_window = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window),
                                   GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_widget_show(scrolled_window);

    gtk_box_pack_start(GTK_BOX(vbox), scrolled_window, TRUE, TRUE, 0);

    store = gtk_list_store_new(USER_N_COLUMNS,
                               G_TYPE_BOOLEAN,
                               G_TYPE_STRING,
                               GDK_TYPE_PIXBUF,
                               G_TYPE_STRING,
                               G_TYPE_STRING,
                               G_TYPE_STRING,
                               G_TYPE_BOOLEAN);

    /* Create a view */
    custom_view = view = gtk_tree_view_new_with_model(GTK_TREE_MODEL(store));
    gtk_widget_show(view);

    {
        GtkTreeSelection *s = gtk_tree_view_get_selection(GTK_TREE_VIEW(view));
        gtk_tree_selection_set_mode(s, GTK_SELECTION_MULTIPLE);
    }

    gtk_tree_view_set_rules_hint(GTK_TREE_VIEW(view), TRUE);

    gtk_container_add(GTK_CONTAINER(scrolled_window), view);

    model = gtk_tree_view_get_model(GTK_TREE_VIEW(view));

    renderer = gtk_cell_renderer_toggle_new();

    g_signal_connect(G_OBJECT(renderer), "toggled",
                     G_CALLBACK(ui_custom_toggled), model);

    column = gtk_tree_view_column_new_with_attributes(NULL,
                                                      renderer,
                                                      "active",
                                                      USER_ENABLED,
                                                      NULL);

    gtk_tree_view_column_set_resizable(column, FALSE);
    gtk_tree_view_column_set_min_width(column, 30);
    gtk_tree_view_append_column(GTK_TREE_VIEW(view), column);

    renderer = gtk_cell_renderer_text_new();

    g_signal_connect(G_OBJECT(renderer), "edited",
                     G_CALLBACK(ui_custom_cell_edit), model);

    g_object_set_data(G_OBJECT(renderer), "column", (gint *) USER_NAME);

    column = gtk_tree_view_column_new_with_attributes("Name",
                                                      renderer,
                                                      "text",
                                                      USER_NAME,
                                                      "editable",
                                                      USER_EDITABLE,
                                                      NULL);

    gtk_tree_view_column_set_resizable(column, TRUE);
    gtk_tree_view_column_set_min_width(column, 75);
    gtk_tree_view_append_column(GTK_TREE_VIEW(view), column);

    renderer = gtk_cell_renderer_pixbuf_new();

    g_object_set_data(G_OBJECT(renderer), "column", (gint *) USER_PIXBUF);

    column = gtk_tree_view_column_new_with_attributes("Icon",
                                                      renderer,
                                                      "pixbuf",
                                                      USER_PIXBUF,
                                                      NULL);

    gtk_tree_view_column_set_resizable(column, FALSE);
    gtk_tree_view_column_set_min_width(column, 30);
    gtk_tree_view_append_column(GTK_TREE_VIEW(view), column);

    renderer = gtk_cell_renderer_text_new();

    g_object_set_data(G_OBJECT(renderer), "column", (gint *) USER_ICON);

    column = gtk_tree_view_column_new_with_attributes(NULL,
                                                      renderer,
                                                      "text",
                                                      USER_ICON,
                                                      NULL);

    gtk_tree_view_column_set_visible(column, FALSE);
    gtk_tree_view_append_column(GTK_TREE_VIEW(view), column);

    renderer = gtk_cell_renderer_text_new();

    g_signal_connect(G_OBJECT(renderer), "edited",
                     G_CALLBACK(ui_custom_cell_edit), model);

    g_object_set_data(G_OBJECT(renderer), "column", (gint *) USER_TOOLTIP);

    column = gtk_tree_view_column_new_with_attributes("Tooltip",
                                                      renderer,
                                                      "text",
                                                      USER_TOOLTIP,
                                                      "editable",
                                                      USER_EDITABLE,
                                                      NULL);

    gtk_tree_view_column_set_resizable(column, TRUE);
    gtk_tree_view_column_set_min_width(column, 175);
    gtk_tree_view_append_column(GTK_TREE_VIEW(view), column);

    renderer = gtk_cell_renderer_text_new();

    g_signal_connect(G_OBJECT(renderer), "edited",
                     G_CALLBACK(ui_custom_cell_edit), model);

    g_object_set_data(G_OBJECT(renderer), "column", (gint *) USER_COMMAND);

    column = gtk_tree_view_column_new_with_attributes("Command",
                                                      renderer,
                                                      "text",
                                                      USER_COMMAND,
                                                      "editable",
                                                      USER_EDITABLE,
                                                      NULL);

    gtk_tree_view_column_set_resizable(column, TRUE);
    gtk_tree_view_append_column(GTK_TREE_VIEW(view), column);

    for(i = 0; i < mdh_custom_count(); i++)
        ui_custom_append(GTK_TREE_VIEW(view), B_Cust[i]);

    g_signal_connect(G_OBJECT(view), "button_release_event",
                     G_CALLBACK(ui_custom_icon_selection), view);

    g_object_unref(G_OBJECT(store));

    main_hbox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(vbox), main_hbox, FALSE, TRUE, 0);
    gtk_widget_show(main_hbox);

    hbox = gtk_hbox_new(TRUE, 3);
    gtk_box_pack_start(GTK_BOX(main_hbox), hbox, TRUE, TRUE, 0);
    gtk_widget_show(hbox);

    button = gtk_button_new_from_stock(GTK_STOCK_GO_UP);
    gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
    gtk_widget_show(button);

    mdh_widget_set_tip(NULL, button, "Move selected button up");

    g_signal_connect_swapped(G_OBJECT(button), "clicked",
                             G_CALLBACK(ui_custom_move_up), view);

    button = gtk_button_new_from_stock(GTK_STOCK_GO_DOWN);
    gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
    gtk_widget_show(button);

    mdh_widget_set_tip(NULL, button, "Move selected button down");

    g_signal_connect_swapped(G_OBJECT(button), "clicked",
                             G_CALLBACK(ui_custom_move_down), view);

    hbox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(main_hbox), hbox, FALSE, FALSE, 0);
    gtk_widget_show(hbox);

    vsep = gtk_vseparator_new();
    gtk_box_pack_start(GTK_BOX(hbox), vsep, FALSE, FALSE, 5);
    gtk_widget_show(vsep);

    hbox = gtk_hbox_new(TRUE, 3);
    gtk_box_pack_start(GTK_BOX(main_hbox), hbox, TRUE, TRUE, 0);
    gtk_widget_show(hbox);

    button = mdh_button_new_from_stock(GTK_STOCK_DELETE, "Delete");
    gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
    gtk_widget_show(button);

    mdh_widget_set_tip(NULL, button, "Delete selected custom button");

    g_signal_connect_swapped(G_OBJECT(button), "clicked",
                             G_CALLBACK(ui_custom_delete), view);

    button = mdh_button_new_from_stock(GTK_STOCK_ADD, "Add");
    gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
    gtk_widget_show(button);

    mdh_widget_set_tip(NULL, button, "Add custom button");

    g_signal_connect_swapped(G_OBJECT(button), "clicked",
                             G_CALLBACK(ui_custom_add), view);

    hbox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(main_hbox), hbox, FALSE, FALSE, 0);
    gtk_widget_show(hbox);

    vsep = gtk_vseparator_new();
    gtk_box_pack_start(GTK_BOX(hbox), vsep, FALSE, FALSE, 5);
    gtk_widget_show(vsep);

    hbox = gtk_hbox_new(TRUE, 3);
    gtk_box_pack_start(GTK_BOX(main_hbox), hbox, TRUE, TRUE, 0);
    gtk_widget_show(hbox);

    button = mdh_button_new_from_stock(GTK_STOCK_CANCEL, "Cancel");
    gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
    gtk_widget_show(button);

    mdh_widget_set_tip(NULL, button,
                       "Cancel new (or unapplied) button settings");

    g_signal_connect_swapped(G_OBJECT(button), "clicked",
                             G_CALLBACK(gtk_widget_destroy), window);

    button = mdh_button_new_from_stock(GTK_STOCK_APPLY, "Apply");
    gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
    gtk_widget_show(button);

    mdh_widget_set_tip(NULL, button, "Apply new button settings");

    g_signal_connect_swapped(G_OBJECT(button), "clicked",
                             G_CALLBACK(ui_custom_apply), model);

    button = mdh_button_new_from_stock(GTK_STOCK_OK, "OK");
    gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
    gtk_widget_show(button);

    mdh_widget_set_tip(NULL, button, "Close and Apply new button settings");

    g_signal_connect_swapped(G_OBJECT(button), "clicked",
                             G_CALLBACK(ui_custom_apply), model);

    g_signal_connect_swapped(G_OBJECT(button), "clicked",
                             G_CALLBACK(gtk_widget_destroy), window);

    gtk_widget_show(window);
}
