/*
 * mdh (MailDooHicky), a GTK+-based toolbar.
 *
 * Copyright (c) 2003-2005 by Mike Hokenson <mdh at gozer dot org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <errno.h>
#include <unistd.h>

#include <gtk/gtk.h>

#ifdef HAVE_FCNTL_H
# include <fcntl.h>
#endif

#ifdef HAVE_SYS_IOCTL_H
# include <sys/ioctl.h>
#endif

#ifdef HAVE_OSS
# include <sys/soundcard.h>
#endif

#include "mdh.h"

#include "button.h"

#include "ui_mixer.h"

#include "config.h"
#include "error.h"
#include "widget.h"
#include "window.h"

#if _MDH_HAS_MIXER

static gboolean     mixer_init;

static gint         mixer_fd;
static gint         mixer_devmask;
static gint         mixer_stereodevs;

static const gchar *sound_device_names[] = SOUND_DEVICE_NAMES;

static GSList      *mixer_devices;

typedef struct {
    const gchar *name;
    gint         device;
    gint         level;
    gboolean     muted;
    GtkWidget   *scale;
} MdhMixerDevice;

static MdhMixerDevice *mixer_vol;

typedef struct {
    GtkWidget *window;
    GtkWidget *scale;
} MdhMixerPopup;

static MdhMixerPopup *popup;

/*
 * this is intended to be a simple mixer, so only use a few common devices.
 *
 * mixer settings are saved between sessions to give the mixer a more
 * stateful feel. this silliness, as usual, has made things a bit touchy.
 * it'll probably go away later when I get frustrated. :P
 */
static MdhMixerDevice mixer_device_table[] = {
        { "vol",     0, -1, FALSE, NULL },
        { "bass",    0, -1, FALSE, NULL },
        { "treble",  0, -1, FALSE, NULL },
        { "synth",   0, -1, FALSE, NULL },
        { "pcm",     0, -1, FALSE, NULL },
        { "speaker", 0, -1, FALSE, NULL },
        { "line",    0, -1, FALSE, NULL },
        { "mic",     0, -1, FALSE, NULL },
        { "cd",      0, -1, FALSE, NULL },
        { NULL,      0, -1, FALSE, NULL }
    };

static gboolean mixer_startup(GError **err)
{
    const gchar *device;

    if(!mdh_config_get_string("mixer", "device", &device))
        device = "/dev/mixer";

    if((mixer_fd = open(device, O_RDONLY)) == -1) {
        g_set_error(err, 0, 0, "Failed to open '%s': %s", device,
                               g_strerror(errno));
        return(FALSE);
    }

    if(ioctl(mixer_fd, SOUND_MIXER_READ_DEVMASK, &mixer_devmask) == -1) {
        g_set_error(err, 0, 0, "Failed to read mixer devmask: %s",
                               g_strerror(errno));
        return(FALSE);
    }

    if(ioctl(mixer_fd, SOUND_MIXER_READ_STEREODEVS, &mixer_stereodevs) == -1) {
        g_set_error(err, 0, 0, "Failed to read mixer devices: %s",
                               g_strerror(errno));
        return(FALSE);
    }

    return(TRUE);
}

#if 0
static void mixer_shutdown(void)
{
    close(mixer_fd);
}
#endif

static gboolean mixer_get(gint device, gint *left, gint *right, GError **err)
{
    gint level;

    g_return_val_if_fail(device >= 0, FALSE);
    g_return_val_if_fail(device < SOUND_MIXER_NRDEVICES, FALSE);
    g_return_val_if_fail(left  != NULL, FALSE);
    g_return_val_if_fail(right != NULL, FALSE);

    if(ioctl(mixer_fd, MIXER_READ(device), &level) == -1) {
        g_set_error(err, 0, 0, "Failed to read mixer settings: %s.",
                               g_strerror(errno));
        return(FALSE);
    }

    *left  = level & 0xff;
    *right = ((1 << device) & mixer_stereodevs) ? (level & 0xff00) >> 8 : -1;

    return(TRUE);
}

static gboolean mixer_set(gint device, gint left, gint right, GError **err)
{
    gint level;

    g_return_val_if_fail(device >= 0, FALSE);
    g_return_val_if_fail(device < SOUND_MIXER_NRDEVICES, FALSE);

    if(left  <   0) left  =   0;
    if(left  > 100) left  = 100;

    if(right <   0) right =   0;
    if(right > 100) right = 100;

    level = (right << 8) + left;

    if(ioctl(mixer_fd, MIXER_WRITE(device), &level) == -1) {
        g_set_error(err, 0, 0, "Failed to update mixer settings: %s.",
                               g_strerror(errno));
        return(FALSE);
    }

    return(TRUE);
}

static gboolean mixer_level_get(MdhMixerDevice *m, gint *level, GError **err)
{
    gint left, right;

    g_return_val_if_fail(m != NULL, FALSE);

    if(!mixer_get(m->device, &left, &right, err))
        return(FALSE);

    if(level)
        *level = MAX(left, right);

    return(TRUE);
}

static gboolean mixer_level_set(MdhMixerDevice *m, gint level, GError **err)
{
    g_return_val_if_fail(m != NULL, FALSE);

    if(!mixer_set(m->device, level, level, err))
        return(FALSE);

    return(TRUE);
}

static void ui_mixer_adjust(GtkWidget *widget, MdhMixerDevice *m)
{
    gdouble level;

    g_return_if_fail(m != NULL);
    g_return_if_fail(m->name != NULL);

    level = gtk_range_get_value(GTK_RANGE(widget));

    if(level > 0) {
        m->level = level;
        m->muted = FALSE;
    } 

   {
        GError *err = NULL;

        if(!mixer_level_set(m, level, &err)) {
            ADD_LOG("%s", err->message);
            g_error_free(err);
            return;
        }
    }
}

static void ui_mixer_toggle(GtkWidget *widget, MdhMixerDevice *m)
{
    gint level;

    g_return_if_fail(m != NULL);
    g_return_if_fail(m->name != NULL);
    g_return_if_fail(m->scale != NULL);

    m->muted = !gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget));

    if(m->muted)
        level = 0;
    else
        level = m->level;

   {
        GError *err = NULL;

        if(!mixer_level_set(m, level, &err)) {
            ADD_LOG("%s", err->message);
            g_error_free(err);
            return;
        }
    }

  /*gtk_range_set_value(GTK_RANGE(m->scale), level);*/
    gtk_range_set_value(GTK_RANGE(m->scale), m->level);

    gtk_widget_set_sensitive(GTK_WIDGET(m->scale), !m->muted);
}

static void mixer_devices_setup(void)
{
    GError *err = NULL;

    gint i;

    if(mixer_init)
        return;

    if(!mixer_startup(&err)) {
        mdh_window_error("Unable to setup mixer.", err->message);
        ADD_LOG("%s", err->message);
        g_error_free(err);
        return;
    }

    for(i = 0; i < SOUND_MIXER_NRDEVICES; i++)
        if((1 << i) & mixer_devmask) {
            gint x;

            for(x = 0; mixer_device_table[x].name; x++) {
                MdhMixerDevice *m = &mixer_device_table[x];

                if(!strcmp(m->name, "vol"))
                    mixer_vol = m;

                if(!strcmp(m->name, sound_device_names[i])) {
                    m->device = i;
                    mixer_devices = g_slist_append(mixer_devices, m);
                    break;
                }
            }
        }

    mixer_init = TRUE;
}

static void ui_mixer_close(GtkWidget *widget)
{
    mdh_window_item_open_set(&wi_mixer, FALSE);
}

/* thanks to gnome-applets for the popup handling bits, helped much */

static void ui_mixer_handle_event(GtkWidget *window,
                                  GdkEvent *event,
                                  MdhMixerPopup *popup)
{
    if(popup->window && popup->scale) {
        gtk_grab_remove(popup->window);
        gtk_grab_remove(popup->scale);

        gtk_widget_destroy(popup->window);
    }

    gdk_keyboard_ungrab(GDK_CURRENT_TIME);
    gdk_pointer_ungrab(GDK_CURRENT_TIME);
}

void mdh_ui_mixer_volume_popup(void)
{
    GtkWidget *frame;

    GdkGrabStatus pointer,
                  keyboard;

    gint level;

    mixer_devices_setup();

    g_return_if_fail(mixer_vol != NULL);

    {
        GError *err = NULL;

        if(!mixer_level_get(mixer_vol, &level, &err)) {
            ADD_LOG("%s", err->message);
            g_error_free(err);
            return;
        }
    }

    if(!popup)
        popup = g_new(MdhMixerPopup, 1);

    popup->window = gtk_window_new(GTK_WINDOW_POPUP);

    gtk_widget_set_size_request(GTK_WIDGET(popup->window), -1, 100);

    /* align with mixer button */

    {
        GtkWidget *widget = B_Volume->widgets.button;

        gint x, y, w, h;

        gdk_window_get_origin(widget->window, &x, &y);

        x += widget->allocation.x;

        gdk_drawable_get_size(widget->window, &w, &h);

        if(y + 100 > gdk_screen_height())   /* flip direction */
            y -= widget->allocation.y - 1 + 100;
        else
            y += widget->allocation.y + h - 1;

        gtk_window_move(GTK_WINDOW(popup->window), x, y);
    }

    frame = gtk_frame_new(NULL);
    gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_OUT);
    gtk_widget_show(frame);

    gtk_container_set_border_width(GTK_CONTAINER(frame), 1);

    gtk_container_add(GTK_CONTAINER(popup->window), frame);

    popup->scale = gtk_vscale_new_with_range(0.0, 100.0, 1.0);
    gtk_widget_set_size_request(GTK_WIDGET(popup->scale), -1, 95);
    gtk_widget_show(popup->scale);

    gtk_container_add(GTK_CONTAINER(frame), popup->scale);

    gtk_range_set_value(GTK_RANGE(popup->scale), level);

    gtk_range_set_inverted(GTK_RANGE(popup->scale), TRUE);

    g_signal_connect(G_OBJECT(popup->scale), "value_changed",
                     G_CALLBACK(ui_mixer_adjust), mixer_vol);

    mixer_vol->level = level;
    mixer_vol->scale = popup->scale;

    /*
     * a press event would probably be best, but this has to be kept
     * the same as the main toolbar event handlers, otherwise you'll
     * end up with some pretty odd behavior. a workaround could be
     * added which keeps track of the previous event, but that's ugly
     * and definatley not worth the hassle, especially for something
     * as trivial as a volume control popup.
     */
    g_signal_connect_after(G_OBJECT(popup->window), "button_release_event",
                           G_CALLBACK(ui_mixer_handle_event), popup);

    g_signal_connect_after(G_OBJECT(popup->scale), "key_press_event", 
                           G_CALLBACK(ui_mixer_handle_event), popup);

    gtk_widget_show(popup->window);

    gtk_widget_grab_focus(popup->window);
    gtk_grab_add(popup->window);

    pointer = gdk_pointer_grab(popup->window->window,
                               TRUE,
                               (GDK_BUTTON_PRESS_MASK |
                                GDK_BUTTON_RELEASE_MASK |
                                GDK_POINTER_MOTION_MASK),
                               NULL, NULL, GDK_CURRENT_TIME);

    keyboard = gdk_keyboard_grab(popup->window->window,
                                 TRUE,
                                 GDK_CURRENT_TIME);

    if(pointer != GDK_GRAB_SUCCESS || keyboard != GDK_GRAB_SUCCESS) {
        gtk_grab_remove(popup->window);
        gtk_widget_destroy(popup->window);

        popup->window = NULL;
        popup->scale  = NULL;
                
        if(pointer == GDK_GRAB_SUCCESS)
            gdk_keyboard_ungrab(GDK_CURRENT_TIME);

        if(keyboard == GDK_GRAB_SUCCESS)
            gdk_pointer_ungrab(GDK_CURRENT_TIME);
    }
}

void mdh_ui_mixer(void)
{
    GtkWidget *window,
              *vbox,
              *table,
              *scale,
              *button;

    GSList *p;

    mixer_devices_setup();

    g_return_if_fail(mixer_devices != NULL);

    if(mdh_window_item_open(&wi_mixer))
        return;

    mdh_window_item_open_set(&wi_mixer, TRUE);

    window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(window), "MailDooHicky: Mixer");
    gtk_window_set_resizable(GTK_WINDOW(window), FALSE);

    g_signal_connect(G_OBJECT(window), "destroy",
                     G_CALLBACK(ui_mixer_close), NULL);

    g_signal_connect(G_OBJECT(window), "event",
                     G_CALLBACK(mdh_window_event), NULL);

    g_signal_connect(G_OBJECT(window), "expose_event",
                     G_CALLBACK(mdh_window_expose_event), NULL);

    gtk_container_set_border_width(GTK_CONTAINER(window), 5);

    vbox = gtk_vbox_new(FALSE, 5);
    gtk_widget_show(vbox);

    gtk_container_add(GTK_CONTAINER(window), vbox);

    table = gtk_table_new(g_slist_length(mixer_devices), 3, FALSE);
    gtk_box_pack_start(GTK_BOX(vbox), table, TRUE, TRUE, 0);
    gtk_widget_show(table);

    for(p = mixer_devices; p; p = g_slist_next(p)) {
        MdhMixerDevice *m = p->data;

        GtkWidget *label;

        static gint i = 0;

        gint level;

        {
            GError *err = NULL;

            if(!mixer_level_get(m, &level, &err)) {
                ADD_LOG("%s", err->message);
                g_error_free(err);
                continue;
            }
        }

        /* first run, setup defaults */
        if(m->level < 0) {
            m->level = level;

            if(level < 1)
                m->muted = TRUE;
        }

        /* look for external mixer changes */

        if(m->muted && level > 0)
            m->muted = FALSE;

        if(!m->muted && level != m->level)
            m->level = level;

        scale = gtk_vscale_new_with_range(0.0, 100.0, 1.0);
        gtk_table_attach(GTK_TABLE(table), scale, i + 1, i + 2, 0, 1,
                                                  GTK_EXPAND, GTK_EXPAND, 3, 0);
        gtk_widget_show(scale);

        gtk_range_set_inverted(GTK_RANGE(scale), TRUE);

        gtk_widget_set_size_request(GTK_WIDGET(scale), -1, 100);

      /*gtk_range_set_value(GTK_RANGE(scale), level);*/
        gtk_range_set_value(GTK_RANGE(scale), m->level);

        gtk_widget_set_sensitive(GTK_WIDGET(scale), !m->muted);

        g_signal_connect(G_OBJECT(scale), "value_changed",
                         G_CALLBACK(ui_mixer_adjust), m);

        m->scale = scale;

        label = gtk_label_new(m->name);
        gtk_table_attach(GTK_TABLE(table), label, i + 1, i + 2, 1, 2,
                                                  GTK_EXPAND, GTK_EXPAND, 3, 0);
        gtk_widget_show(label);

        button = gtk_check_button_new();
        gtk_table_attach(GTK_TABLE(table), button, i + 1, i + 2, 2, 3,
                                                  GTK_EXPAND, GTK_EXPAND, 3, 0);
        gtk_widget_show(button);

        if(!m->muted)
            gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

        g_signal_connect(G_OBJECT(button), "clicked",
                         G_CALLBACK(ui_mixer_toggle), m);

        i++;
    }

    button = gtk_button_new_from_stock(GTK_STOCK_OK);
    gtk_box_pack_start(GTK_BOX(vbox), button, FALSE, FALSE, 0);
    gtk_widget_show(button);
    
    g_signal_connect_swapped(G_OBJECT(button), "clicked",
                             G_CALLBACK(gtk_widget_destroy), window);

    mdh_widget_grab_default(GTK_WIDGET(button));

    gtk_widget_show(window);
}

#endif /* _MDH_HAS_MIXER */
