/*
 * mdh (MailDooHicky), a GTK+-based toolbar.
 *
 * Copyright (c) 2003-2005 by Mike Hokenson <mdh at gozer dot org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <unistd.h>

#include <gtk/gtk.h>

#include "widget.h"
#include "util.h"

static GtkTooltips *widget_tool_tips;     /*
                                           * builtin tooltips used if NULL is
                                           * passed to mdh_widget_set_tip().
                                           */

void mdh_widget_set_font_desc(GtkWidget *widget,
                              PangoFontDescription *font_desc)
{
    g_return_if_fail(widget != NULL);
    g_return_if_fail(font_desc != NULL);

    gtk_widget_modify_font(widget, font_desc);
}

void mdh_widget_set_font(GtkWidget *widget, const gchar *font)
{
    PangoFontDescription *font_desc;

    g_return_if_fail(widget != NULL);
    g_return_if_fail(font != NULL);

    font_desc = pango_font_description_from_string(font);
    gtk_widget_modify_font(widget, font_desc);
    pango_font_description_free(font_desc);
}

void mdh_widget_set_tip(GtkTooltips *tips, GtkWidget *widget, const gchar *tip)
{
    GtkTooltips *tool_tips = tips;

    g_return_if_fail(widget != NULL);

    if(!tool_tips) {
        if(!widget_tool_tips)
            widget_tool_tips = gtk_tooltips_new();

        tool_tips = widget_tool_tips;
    }

    gtk_tooltips_set_tip(tool_tips, widget, (tip && *tip) ? tip : NULL, NULL);
}

/* create a button with a stock icon and custom text */
GtkWidget *mdh_button_new_from_stock(const gchar *stock_id, const gchar *text)
{
    GtkWidget *vbox;
    GtkWidget *hbox;
    GtkWidget *button;

    GtkWidget *image;
    GtkWidget *label;

    g_return_val_if_fail(stock_id != NULL, NULL);
    g_return_val_if_fail(text != NULL, NULL);
    
    /* to center the image and text in the button */
    vbox = gtk_hbox_new(TRUE, 0);
    gtk_widget_show(vbox);
    
    hbox = gtk_hbox_new(FALSE, 2);
    gtk_widget_show(hbox);

    gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, FALSE, 0);
    image = gtk_image_new_from_stock(stock_id, GTK_ICON_SIZE_BUTTON);
    gtk_widget_show(image);
    
    gtk_box_pack_start(GTK_BOX(hbox), image, FALSE, FALSE, 0);

    label = gtk_label_new(text);
    gtk_widget_show(label);

    gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
    
    button = gtk_button_new();
    
    gtk_container_add(GTK_CONTAINER(button), vbox);
    
    return(button);
}

/* http://developer.gnome.org/doc/API/2.0/pango/PangoMarkupFormat.html */
void mdh_label_set_markup(GtkLabel *label,
                          MdhMarkupFlags flags,
                          const gchar *fmt,
                          ...)
{
    GString *markup;

    gchar *msg;

    va_list args;

    g_return_if_fail(label != NULL);
    g_return_if_fail(fmt != NULL);

    markup = g_string_new("<span");

    if(flags & MDH_MARKUP_SIZE_SMALL)
        g_string_append(markup, " size=\"small\"");
    else if(flags & MDH_MARKUP_SIZE_MEDIUM)
        g_string_append(markup, " size=\"medium\"");
    else if(flags & MDH_MARKUP_SIZE_LARGE)
        g_string_append(markup, " size=\"large\"");
    else if(flags & MDH_MARKUP_SIZE_SMALLER)
        g_string_append(markup, " size=\"smaller\"");
    else if(flags & MDH_MARKUP_SIZE_LARGER)
        g_string_append(markup, " size=\"larger\"");

    if(flags & MDH_MARKUP_WEIGHT_LIGHT)
        g_string_append(markup, " weight=\"light\"");
    else if(flags & MDH_MARKUP_WEIGHT_NORMAL)
        g_string_append(markup, " weight=\"normal\"");
    else if(flags & MDH_MARKUP_WEIGHT_BOLD)
        g_string_append(markup, " weight=\"bold\"");
    else if(flags & MDH_MARKUP_WEIGHT_HEAVY)
        g_string_append(markup, " weight=\"heavy\"");

    if(flags & MDH_MARKUP_UNDERLINE)
        g_string_append(markup, " underline=\"single\"");

    if(flags & MDH_MARKUP_STRIKETHROUGH)
        g_string_append(markup, " strikethrough=\"true\"");

    if(flags & MDH_MARKUP_STYLE_ITALIC)
        g_string_append(markup, " style=\"italic\"");
    else if(flags & MDH_MARKUP_STYLE_OBLIQUE)
        g_string_append(markup, " style=\"oblique\"");

    if(flags & MDH_MARKUP_FG_RED)
        g_string_append(markup, " foreground=\"darkred\"");
    else if(flags & MDH_MARKUP_FG_BLUE)
        g_string_append(markup, " foreground=\"darkblue\"");
    else if(flags & MDH_MARKUP_FG_GREEN)
        g_string_append(markup, " foreground=\"darkgreen\"");
    else if(flags & MDH_MARKUP_FG_YELLOW)
        g_string_append(markup, " foreground=\"darkyellow\"");

    if(flags & MDH_MARKUP_BG_RED)
        g_string_append(markup, " background=\"darkred\"");
    else if(flags & MDH_MARKUP_BG_BLUE)
        g_string_append(markup, " background=\"darkblue\"");
    else if(flags & MDH_MARKUP_BG_GREEN)
        g_string_append(markup, " background=\"darkgreen\"");
    else if(flags & MDH_MARKUP_BG_YELLOW)
        g_string_append(markup, " background=\"darkyellow\"");
    
    va_start(args, fmt);
    msg = g_strdup_vprintf(fmt, args);
    va_end(args);
        
    g_string_append_printf(markup, ">%s</span>", msg);

    g_free(msg);
        
    gtk_label_set_markup(label, markup->str);
        
    g_string_free(markup, TRUE);
}

static gboolean set_cursor(GtkWidget *widget,
                           GdkEventExpose *event,
                           gpointer data)
{
    GdkCursor     *cursor;
    GdkCursorType  cursor_type;

    g_return_val_if_fail(widget != NULL, FALSE);
    g_return_val_if_fail(event != NULL, FALSE);
    g_return_val_if_fail(event->window != NULL, FALSE);
    g_return_val_if_fail(data != NULL, FALSE);

    cursor_type = GPOINTER_TO_INT(data);

    g_return_val_if_fail(cursor_type > -1, FALSE);
    g_return_val_if_fail(cursor_type < GDK_LAST_CURSOR, FALSE);

    cursor = gdk_cursor_new_for_display(gtk_widget_get_display(widget),
                                        cursor_type);
    gdk_window_set_cursor(event->window, cursor);
    gdk_cursor_unref(cursor);

    return(FALSE);
}

GtkWidget *mdh_clickable_new(const gchar *text,
                             gpointer func,
                             gpointer data,
                             MdhMarkupFlags flags,
                             GdkCursorType cursor)
{
    GtkWidget *label,
              *button;

    g_return_val_if_fail(func != NULL, NULL);
    
    label = gtk_label_new(NULL);

    mdh_label_set_markup(GTK_LABEL(label), flags, "%s", (text) ? text : "");
    
    button = gtk_event_box_new();
    gtk_container_add(GTK_CONTAINER(button), label);
    
    g_signal_connect(G_OBJECT(button), "button_release_event",
                     G_CALLBACK(func), data);

    if(cursor > GDK_CURSOR_IS_PIXMAP && cursor < GDK_LAST_CURSOR)
        g_signal_connect(G_OBJECT(button), "expose_event",
                         G_CALLBACK(set_cursor), GINT_TO_POINTER(cursor));
    
    return(button);
}

GtkWidget *mdh_frame_new_with_title(const gchar *title)
{
    GtkWidget *frame;
    GtkWidget *label;

    frame = gtk_frame_new(NULL);
    gtk_frame_set_label_align(GTK_FRAME(frame), 0.0, 0.5);

    label = gtk_label_new(NULL);
    mdh_label_set_bold(GTK_LABEL(label), title);
    gtk_widget_show(label);

    gtk_frame_set_label_widget(GTK_FRAME(frame), label);
    
    gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_NONE);
    
    return(frame);
}

static GtkWidget *mdh_image_menu_item_new(const gchar *label,
                                          const gchar *image,
                                          MdhImageType type)
{
    GtkWidget *menu_item,
              *icon = NULL;

    g_return_val_if_fail(label != NULL, NULL);
    g_return_val_if_fail(image != NULL, NULL);

    switch(type) {
        case MDH_IMAGE_TYPE_STOCK:
            icon = gtk_image_new_from_stock(image, GTK_ICON_SIZE_MENU);
            break;
        case MDH_IMAGE_TYPE_FILE:
            icon = gtk_image_new_from_file(image);
            break;
        default:
            g_assert_not_reached();
    }

    menu_item = gtk_image_menu_item_new_with_label(label);
    gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(menu_item), icon);

    return(menu_item);
}

GtkWidget *mdh_image_menu_item_new_from_stock(const gchar *label,
                                              const gchar *stock_id)
{
    return(mdh_image_menu_item_new(label, stock_id, MDH_IMAGE_TYPE_STOCK));
}

GtkWidget *mdh_image_menu_item_new_from_file(const gchar *label,
                                             const gchar *file)
{
    return(mdh_image_menu_item_new(label, file, MDH_IMAGE_TYPE_FILE));
}

#if (!defined(GTK_CHECK_VERSION) || !GTK_CHECK_VERSION(2, 6, 0))
gchar *gtk_combo_box_get_active_text(GtkComboBox *combo_box)
{
    GtkTreeModel *model;
    GtkTreeIter iter;
    gchar *text = NULL;

    g_return_val_if_fail(GTK_IS_COMBO_BOX(combo_box), NULL);

    model = gtk_combo_box_get_model(combo_box);

    if(gtk_combo_box_get_active_iter(combo_box, &iter))
        gtk_tree_model_get(model, &iter, 0, &text, -1);

    return(text);
}
#endif /* GTK_CHECK_VERSION */
