/*
 * mdh (MailDooHicky), a GTK+-based toolbar.
 *
 * Copyright (c) 2003-2005 by Mike Hokenson <mdh at gozer dot org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <gtk/gtk.h>
#include <gdk/gdkx.h>

#include <X11/Xatom.h>

#include "util.h"
#include "x11.h"

#define _GDK_ROOT()  (gdk_get_default_root_window())
#define _GDK_ATOM(a) (gdk_atom_intern(a, FALSE))

typedef enum {
    NETWM_STATE_ENABLE,
    NETWM_STATE_DISABLE,
    NETWM_STATE_DETECT
} NetWMState;

typedef enum {
    NETWM_FLAG_SUPPORTED          = 1 << 0,
    NETWM_FLAG_STATE_ABOVE        = 1 << 1,
    NETWM_FLAG_STATE_BELOW        = 1 << 2,
    NETWM_FLAG_STATE_SKIP_TASKBAR = 1 << 3,
    NETWM_FLAG_STATE_SKIP_PAGER   = 1 << 4
} NetWMFlags;

static NetWMState netwm_state = NETWM_STATE_DETECT;
static NetWMFlags netwm_flags;

typedef enum {
    GNOME_STATE_ENABLE,
    GNOME_STATE_DISABLE,
    GNOME_STATE_DETECT
} GnomeState;

typedef enum {
    GNOME_FLAG_SUPPORTED   = 1 << 0,
    GNOME_FLAG_STATE_ABOVE = 1 << 1,
    GNOME_FLAG_STATE_BELOW = 1 << 2
} GnomeFlags;

static GnomeState gnome_state = GNOME_STATE_DETECT;
static GnomeFlags gnome_flags;

gboolean mdh_x11_has_property(GdkWindow *window,
                              const gchar *prop,
                              const gchar *atom)
{
    GdkWindow *win = (window) ? window : _GDK_ROOT();

    return(gdk_property_get(win, _GDK_ATOM(prop), _GDK_ATOM(atom),
                            0, 1, FALSE, NULL, 0, 0, NULL));
}

#if 0
#define _WM_WAIT_ATTEMPTS_MAX (50)
#define _WM_WAIT_TIME         (G_USEC_PER_SEC / 10)
#define _WM_POST_WAIT_TIME    (G_USEC_PER_SEC *  3)

/*
 * with detection enabled, wait for the window manager to start before
 * attempting to get it's features. If the window manager is already
 * running, but the startup delay and ultimately the "No window manager
 * found" message is seen, the check can be disabled by forcing NETWM.
 */

gboolean mdh_x11_wait_for_window_manager(void)
{
    GdkDisplay *display;
    GdkScreen *screen;

    Display *xdisplay;

    gchar *p; 
    Atom s;

    gint i = 0;

    /* don't bother to check if forced */
    if(netwm_state != NETWM_STATE_DETECT || gnome_state != GNOME_STATE_DETECT)
        return(TRUE);

    debug("mdh_x11_wait_for_window_manager(): starting checks...");

    display  = gdk_display_get_default();
    screen   = gdk_screen_get_default();

    xdisplay = GDK_DISPLAY_XDISPLAY(display);

    p = g_strdup_printf("WM_S%d", gdk_x11_screen_get_screen_number(screen));
    s = XInternAtom(xdisplay, p, False);
    g_free(p);

    /* not sure why gdk_selection_owner_get() doesn't work */
    while(XGetSelectionOwner(xdisplay, s) == None) {
        debug("mdh_x11_wait_for_window_manager(): attempt %02d", i + 1);

        g_usleep(_WM_WAIT_TIME);

        if(i++ > _WM_WAIT_ATTEMPTS_MAX)
            break;
    }

    debug("mdh_x11_wait_for_window_manager(): finished after %d attempt(s)", i);

    /*
     * three seconds should be enough for the window manager to
     * start and get everything setup.
     */
    if(i > 0 && i <= _WM_WAIT_ATTEMPTS_MAX) {
        debug("mdh_x11_wait_for_window_manager(): caught in mid-startup...");
        g_usleep(_WM_POST_WAIT_TIME);
    }

    if(i > _WM_WAIT_ATTEMPTS_MAX) {
        debug("mdh_x11_wait_for_window_manager(): not found");
        return(FALSE);
    }

    debug("mdh_x11_wait_for_window_manager(): found");

    return(TRUE);
}
#endif

/* forcibly enable/disable netwm support, call before detection */
void mdh_x11_enable_netwm(gboolean enable)
{
  /*g_return_if_fail(netwm_state == NETWM_STATE_DETECT);*/

    netwm_flags = 0;

    if(enable) {
        netwm_state  = NETWM_STATE_ENABLE;

        netwm_flags  = NETWM_FLAG_SUPPORTED;

        netwm_flags |= NETWM_FLAG_STATE_ABOVE;
        netwm_flags |= NETWM_FLAG_STATE_BELOW;

        netwm_flags |= NETWM_FLAG_STATE_SKIP_TASKBAR;
        netwm_flags |= NETWM_FLAG_STATE_SKIP_PAGER;
    } else
        netwm_state = NETWM_STATE_DISABLE;
}

/* forcibly enable/disable gnome support, call before detection */
void mdh_x11_enable_gnome(gboolean enable)
{
  /*g_return_if_fail(gnome_state == GNOME_STATE_DETECT);*/

    gnome_flags = 0;

    if(enable) {
        gnome_state  = GNOME_STATE_ENABLE;

        gnome_flags  = GNOME_FLAG_SUPPORTED;

        gnome_flags |= GNOME_FLAG_STATE_ABOVE;
        gnome_flags |= GNOME_FLAG_STATE_BELOW;
    } else
        gnome_state = GNOME_STATE_DISABLE;
}

void mdh_x11_get_features(void)
{
    debug("mdh_x11_get_features()");

    if(netwm_state == NETWM_STATE_DETECT) {
        debug("mdh_x11_get_features(): NetWM");

        if(mdh_x11_has_property(NULL, "_NET_SUPPORTING_WM_CHECK", "WINDOW")) {
            debug("mdh_x11_get_features(): _NET_SUPPORTING_WM_CHECK: ok");
            netwm_flags |= NETWM_FLAG_SUPPORTED;
        }

        if(netwm_flags & NETWM_FLAG_SUPPORTED) {
            if(gdk_net_wm_supports(_GDK_ATOM("_NET_WM_STATE_ABOVE"))) {
                debug("mdh_x11_get_features(): _NET_WM_STATE_ABOVE: ok");
                netwm_flags |= NETWM_FLAG_STATE_ABOVE;
            }

            if(gdk_net_wm_supports(_GDK_ATOM("_NET_WM_STATE_BELOW"))) {
                debug("mdh_x11_get_features(): _NET_WM_STATE_BELOW: ok");
                netwm_flags |= NETWM_FLAG_STATE_BELOW;
            }

            if(gdk_net_wm_supports(_GDK_ATOM("_NET_WM_STATE_SKIP_TASKBAR"))) {
                debug("mdh_x11_get_features(): _NET_WM_STATE_SKIP_TASKBAR: ok");
                netwm_flags |= NETWM_FLAG_STATE_SKIP_TASKBAR;
            }

            if(gdk_net_wm_supports(_GDK_ATOM("_NET_WM_STATE_SKIP_PAGER"))) {
                debug("mdh_x11_get_features(): _NET_WM_STATE_SKIP_PAGER: ok");
                netwm_flags |= NETWM_FLAG_STATE_SKIP_PAGER;
            }
        }

        if(netwm_flags & NETWM_FLAG_SUPPORTED)
            netwm_state = NETWM_STATE_ENABLE;
        else
            netwm_state = NETWM_STATE_DISABLE;

        debug("mdh_x11_get_features(): NetWM support is %s",
              (netwm_state == NETWM_STATE_ENABLE) ? "enabled" : "disabled");
    }

    if(gnome_state == GNOME_STATE_DETECT) {
        debug("mdh_x11_get_features(): Gnome");

        if(mdh_x11_has_property(NULL, "_WIN_SUPPORTING_WM_CHECK", "CARDINAL")) {
            debug("mdh_x11_get_features(): _WIN_SUPPORTING_WM_CHECK: ok");
            gnome_flags |= GNOME_FLAG_SUPPORTED;
        }

        if(gnome_flags & GNOME_FLAG_SUPPORTED) {
            GdkAtom win_protocols = _GDK_ATOM("_WIN_PROTOCOLS");
    
            guchar *data;
            gint n_atoms;
    
            if(gdk_property_get(_GDK_ROOT(), win_protocols, _GDK_ATOM("ATOM"),
                                0, G_MAXLONG, FALSE, NULL, 0,
                                &n_atoms, &data)) {
                GdkAtom win_layer = _GDK_ATOM("_WIN_LAYER"),
                        *atoms = (GdkAtom *) data;

                gint i; 

                debug("mdh_x11_get_features(): _WIN_PROTOCOLS: ok");
        
                for(i = 0; i < n_atoms; i++)
                    if(atoms[i] == win_layer) {
                        debug("mdh_x11_get_features(): _WIN_LAYER: ok");

                        gnome_flags |= GNOME_FLAG_STATE_ABOVE;
                        gnome_flags |= GNOME_FLAG_STATE_BELOW;

                        break;
                    }
            }
        }

        if(gnome_flags & GNOME_FLAG_SUPPORTED)
            gnome_state = GNOME_STATE_ENABLE;
        else
            gnome_state = GNOME_STATE_DISABLE;
            
        debug("mdh_x11_get_features(): Gnome support is %s",
              (gnome_state == GNOME_STATE_ENABLE) ? "enabled" : "disabled");
    }
}

gboolean mdh_x11_has_netwm(void)
{
    g_return_val_if_fail(netwm_state != NETWM_STATE_DETECT, FALSE);

    if(netwm_state == NETWM_STATE_ENABLE)
        return(netwm_flags & NETWM_FLAG_SUPPORTED);

    return(FALSE);
}

gboolean mdh_x11_has_netwm_state_above(void)
{
    g_return_val_if_fail(netwm_state != NETWM_STATE_DETECT, FALSE);

    if(netwm_state == NETWM_STATE_ENABLE)
        return(netwm_flags & NETWM_FLAG_STATE_ABOVE);

    return(FALSE);
}

gboolean mdh_x11_has_netwm_state_below(void)
{
    g_return_val_if_fail(netwm_state != NETWM_STATE_DETECT, FALSE);

    if(netwm_state == NETWM_STATE_ENABLE)
        return(netwm_flags & NETWM_FLAG_STATE_BELOW);

    return(FALSE);
}

gboolean mdh_x11_has_netwm_state_skip_taskbar(void)
{
    g_return_val_if_fail(netwm_state != NETWM_STATE_DETECT, FALSE);

    if(netwm_state == NETWM_STATE_ENABLE)
        return(netwm_flags & NETWM_FLAG_STATE_SKIP_TASKBAR);

    return(FALSE);
}

gboolean mdh_x11_has_netwm_state_skip_pager(void)
{
    g_return_val_if_fail(netwm_state != NETWM_STATE_DETECT, FALSE);

    if(netwm_state == NETWM_STATE_ENABLE)
        return(netwm_flags & NETWM_FLAG_STATE_SKIP_PAGER);

    return(FALSE);
}

gboolean mdh_x11_has_gnome(void)
{
    g_return_val_if_fail(gnome_state != GNOME_STATE_DETECT, FALSE);

    if(gnome_state == GNOME_STATE_ENABLE)
        return(gnome_flags & GNOME_FLAG_SUPPORTED);

    return(FALSE);
}

gboolean mdh_x11_has_gnome_state_above(void)
{
    g_return_val_if_fail(gnome_state != GNOME_STATE_DETECT, FALSE);

    if(gnome_state == GNOME_STATE_ENABLE)
        return(gnome_flags & GNOME_FLAG_STATE_ABOVE);

    return(FALSE);
}

gboolean mdh_x11_has_gnome_state_below(void)
{
    g_return_val_if_fail(gnome_state != GNOME_STATE_DETECT, FALSE);

    if(gnome_state == GNOME_STATE_ENABLE)
        return(gnome_flags & GNOME_FLAG_STATE_BELOW);

    return(FALSE);
}

/* flags for the window layer */
typedef enum {
    WIN_LAYER_DESKTOP    =  0,
    WIN_LAYER_BELOW      =  2,
    WIN_LAYER_NORMAL     =  4,
    WIN_LAYER_ONTOP      =  6,
    WIN_LAYER_DOCK       =  8,
    WIN_LAYER_ABOVE_DOCK = 10
} WinLayer;

static void mdh_x11_set_gnome_layer(GtkWidget *window, WinLayer layer)
{
    Atom win_layer;

    if(!mdh_x11_has_gnome_state_above() || !mdh_x11_has_gnome_state_below())
        return;

    g_return_if_fail(window != NULL);

    win_layer = gdk_x11_get_xatom_by_name("_WIN_LAYER");

    if(GTK_WIDGET_MAPPED(window)) {
        XEvent xev;

        xev.type = ClientMessage;
        xev.xclient.type = ClientMessage;
        xev.xclient.window = GDK_WINDOW_XWINDOW(window->window);
        xev.xclient.message_type = win_layer;
        xev.xclient.format = 32;
        xev.xclient.data.l[0] = layer;

        XSendEvent(GDK_DISPLAY(), GDK_ROOT_WINDOW(), False,
                   SubstructureNotifyMask, &xev);
    } else {
        glong data[1];

        data[0] = layer;

        XChangeProperty(GDK_DISPLAY(), GDK_WINDOW_XWINDOW(window->window),
                        win_layer, XA_CARDINAL, 32, PropModeReplace,
                        (guchar *) data, 1);
    }
}

void mdh_x11_set_gnome_layer_above(GtkWidget *window, gboolean above)
{
    mdh_x11_set_gnome_layer(window,
                            (above) ? WIN_LAYER_ONTOP : WIN_LAYER_NORMAL);
}

void mdh_x11_set_gnome_layer_below(GtkWidget *window, gboolean below)
{
    mdh_x11_set_gnome_layer(window,
                            (below) ? WIN_LAYER_BELOW : WIN_LAYER_NORMAL);
}
