/*
 * advanced.c:
 *   an advanced example that reads in the specified .ini file, maually adds
 *   a section and a few entries, displays the results, and saves a new file.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "gini.h"

#if (!defined(GINI_CHECK_VERSION) || !GINI_CHECK_VERSION(0, 4, 0))
# error "libgini version is too old"
#endif

gint main(void)
{
    GIni *ini;

    const gchar *host, *def;
    gint timeout, port, value;

    struct stat *statbuf;

    GError *err = NULL;

    gini_init();

    if(!(ini = gini_load("sample.ini", &err))) {
        g_critical("%s\n", err->message);
        g_error_free(err);
        exit(1);
    }

    /* change to the [global] section */
    if(!gini_section_lookup(ini, "global")) {
        g_warning("Section 'global' not found, adding");
        gini_section_add_value(ini, "global");
    }

    /* sample.ini has 'default' value in a continued [global] section */
    if(!gini_entry_get_string(ini, "global", "default", &def)) {
        g_warning("'default' not found, adding as 'no'");
        gini_entry_set_string(ini, "global", "default", (def = "no"));
    }

    /* get 'timeout' value from [global] */
    if(!gini_entry_get_int(ini, "global", "timeout", &timeout)) {
        g_warning("'timeout' not found, adding as '30'");
        gini_entry_set_int(ini, "global", "timeout", (timeout = 30));
    }

    /* get 'host' value from [global] */
    if(!gini_entry_get_string(ini, "global", "host", &host)) {
        g_warning("'host' not found, adding as 'somehost'");
        gini_entry_set_string(ini, "global", "host", (host = "somehost"));
    }

    g_print("SECTION 'global'\n");
    g_print("  default = %s\n", def);
    g_print("  timeout = %d\n", timeout);
    g_print("  host    = %s\n\n", host);

    if(!gini_section_lookup(ini, host)) {
        g_warning("failed to get section '%s', adding...", host);

        if(!gini_section_add_value(ini, host)) {
            g_critical("failed to add section '%s'", host);
            exit(1);
        }
    }

    /* get 'port' value from [host] */
    if(!gini_entry_get_int(ini, host, "port", &port)) {
        g_warning("'port' not found, adding as '80'");
        gini_entry_set_int(ini, host, "port", (port = 80));
    }

    g_print("SECTION '%s'\n", host);
    g_print("  port = %d\n\n", port);

    /* add section [new] if it does not exist, else return existing section */
    if(!gini_section_add_value(ini, "new")) {
        g_critical("failed to add section 'new'");
        exit(1);
    }

    /* multiple calls to gini_entry_set_*() replace existing values */
    gini_entry_set_int(ini, "new", "val", 30);
    gini_entry_set_int(ini, "new", "val", 50);
    gini_entry_set_int(ini, "new", "val", 10);

    gini_entry_get_int(ini, "new", "val", &value);

    g_print("SECTION 'new'\n");
    g_print("  val = %d\n", value);

    g_print(" * changing 'val' types: bool, float, int64 ...");

    /* now convert to other types */
    gini_entry_set_boolean(ini, "new", "val", TRUE);
    gini_entry_set_float(ini, "new", "val", -2.75);
    gini_entry_set_int64(ini, "new", "val", 2034LL);

    g_print(" done. last was int64(2034)\n");

    /* convert back to string */
    {
        const gchar *sval;

        gini_entry_get_string(ini, "new", "val", &sval);

        g_print("  val (as str) = %s\n", sval);
    }

    /* quick, one-time transform (original type unchanged) */
    {
        gint i;
        glong l;
        gchar *s;

        if(gini_entry_get_val_as_type(ini, "new", "val", G_TYPE_INT, &i))
            g_print("  val (transform: as int) = %d\n", i);

        if(gini_entry_get_val_as_type(ini, "new", "val", G_TYPE_LONG, &l))
            g_print("  val (transform: as long) = %ld\n", l);

        /* pointer values must be freed */
        if(gini_entry_get_val_as_type(ini, "new", "val", G_TYPE_STRING, &s)) {
            g_print("  val (transform: as str) = %s\n", s);
            g_free(s);
        }
    }

    g_print("\n");

    /* types of G_TYPE_POINTER will not be saved, use as you wish */

    statbuf = g_new(struct stat, 1);

    if(!stat("/bin/false", statbuf))
        gini_entry_set_pointer(ini, "_INTERNAL", "/bin/false", statbuf);
    else
        g_warning("Unable to stat '/bin/false', skipping pointer example");

    /* still debating the G_TYPE_POINTER empty section hack... */
    gini_section_remove(ini, "_INTERNAL");

    /* manually create and add a section/entry */
    {
        GIniSection *sec = gini_section_new_from_string("[testing]");

        if(!sec)
            g_assert_not_reached();

        if(gini_section_add(ini, sec)) {
            GIniEntry *e = gini_entry_new_from_string("giggity=105");

            if(gini_entry_add(ini, sec, e)) {
                gini_entry_get_int(ini, "testing", "giggity", &value);

                g_print("SECTION 'testing'\n");
                g_print("  giggity = %d\n", value);
            } else {
                gini_entry_free(e); /* add failed, entry must be freed */
                g_warning("Failed to manually add 'giggity'");
            }

            /* manually add an entry with specified values ... */
            if(gini_entry_add_values(ini, sec, "lee", "lemon"))
                gini_entry_del(ini, sec, "lee"); /* ... and now remove it */
        } else {
            gini_section_free(sec); /* add failed, section must be freed */
            g_warning("Failed to manually add section 'testing'");
        }
    }

    /* save the changes to a new .ini file */
    if(!gini_save(ini, "advanced.ini", &err)) {
        g_warning("%s", err->message);
        g_error_free(err);
    }

    gini_unref(ini);

    /* memory associated with GIniEntry's of G_TYPE_POINTER must be freed */
    g_free(statbuf);

    exit(0);
}
