/*
 * libgini: a thread-safe, GLib-based .ini parser that uses hash
 *          tables for quick value lookup.
 *
 * Copyright (c) 2003-2005 Mike Hokenson <logan at gozer dot org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <errno.h>

#include "gini.h"

#include <gobject/gvaluecollector.h>

static gboolean init;

#if   (defined(DEBUG) && defined(G_HAVE_ISO_VARARGS))
# define debug(...) \
             g_log(G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG, \
                   __FILE__ ":" G_STRINGIFY (__LINE__) ": " __VA_ARGS__);
#elif (defined(DEBUG) && defined(G_HAVE_GNUC_VARARGS))
# define debug(format...) \
             g_log(G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG, \
                   __FILE__ ":" G_STRINGIFY (__LINE__) ": " format);
#elif DEBUG
# define debug g_debug
#else
# define debug(...)
#endif

#ifdef G_THREADS_ENABLED
static GStaticRecMutex static_rec_mutex = G_STATIC_REC_MUTEX_INIT;
# define gini_mutex_lock() \
    G_STMT_START { \
        debug("gini_mutex_lock()"); \
        g_static_rec_mutex_lock(&static_rec_mutex); \
    } G_STMT_END
# define gini_mutex_unlock() \
    G_STMT_START { \
        debug("gini_mutex_unlock()"); \
        g_static_rec_mutex_unlock(&static_rec_mutex); \
    } G_STMT_END
#else
# define gini_mutex_lock()
# define gini_mutex_unlock()
#endif

#ifdef GINI_STRICT
# define GINI_IS_COMMENT(c) (c == ';')
#else
# define GINI_IS_COMMENT(c) (c == ';' || c == '#')
#endif

struct _GIniEntry {
    gchar       *key;
    GValue      *val;

    GIniSection *sec;  /* pointer to GIniSection where entry is stored */
};

struct _GIniSection {
    gchar      *name;

    GHashTable *entries;

    GSList     *entry_list;  /* used for ordering */

    GIni       *ini;         /* pointer to GIni where section is stored */
};

typedef void (*GIniListFunc) (gpointer a, gpointer b);

struct _GIni {
    gchar        *file;

    GHashTable   *sections;

    const gchar **allowed_sections,
                **allowed_keys;

    GSList       *section_list;  /* used for ordering */

    guint         ref_count;

    GIniListFunc  section_append,
                  entry_append;
};

/* like g_str_equal(), but case-insensitive */
static gboolean gini_str_equal(gconstpointer a, gconstpointer b)
{
    return(!g_ascii_strcasecmp(a, b));
}

/* like g_str_hash(), but case-insensitive */
static guint gini_str_hash(gconstpointer key)
{
    const char *p = key;
    guint h = 0;
    
    if(p) {
        h = g_ascii_tolower(*p++);

        for(; *p; p++)
            h = (h << 5) - h + g_ascii_tolower(*p);
    }

    return(h);
}

/* return TRUE if array is NULL or value is in the array */
static gboolean gini_value_allowed(GIni *ini,
                                   const gchar **array,
                                   const gchar *value)
{
    gint i;

    g_return_val_if_fail(ini != NULL, FALSE);
    g_return_val_if_fail(value != NULL, FALSE);

    /* allow anything if NULL */
    if(!array)
        return(TRUE);

    for(i = 0; array[i]; i++)
        if(gini_str_equal(array[i], value))
            return(TRUE);

    g_critical("Skipping forbidden value '%s'.", value);

    return(FALSE);
}

static gboolean gini_section_name_allowed(GIni *ini, const gchar *section_name)
{
    g_return_val_if_fail(ini != NULL, FALSE);

    return(gini_value_allowed(ini, ini->allowed_sections, section_name));
}

static gboolean gini_entry_key_allowed(GIni *ini, const gchar *key)
{
    g_return_val_if_fail(ini != NULL, FALSE);

    return(gini_value_allowed(ini, ini->allowed_keys, key));
}

static void value_transform_string_boolean(const GValue *src, GValue *dst)
{
    dst->data[0].v_int = gini_str_equal(src->data[0].v_pointer, "TRUE");
}

static void value_transform_string_int(const GValue *src, GValue *dst)
{
    dst->data[0].v_int = atoi(src->data[0].v_pointer);
}

static void value_transform_string_uint(const GValue *src, GValue *dst)
{
    dst->data[0].v_uint = strtoul(src->data[0].v_pointer, NULL, 10);
}

static void value_transform_string_long(const GValue *src, GValue *dst)
{
    dst->data[0].v_long = atol(src->data[0].v_pointer);
}

static void value_transform_string_ulong(const GValue *src, GValue *dst)
{
    dst->data[0].v_ulong = strtoul(src->data[0].v_pointer, NULL, 10);
}

static void value_transform_string_int64(const GValue *src, GValue *dst)
{
    dst->data[0].v_int64 = atoll(src->data[0].v_pointer);
}

static void value_transform_string_uint64(const GValue *src, GValue *dst)
{
    dst->data[0].v_uint64 = strtoull(src->data[0].v_pointer, NULL, 10);
}

static void value_transform_string_float(const GValue *src, GValue *dst)
{
    dst->data[0].v_float = g_strtod(src->data[0].v_pointer, NULL);
}

static void value_transform_string_double(const GValue *src, GValue *dst)
{
    dst->data[0].v_double = g_strtod(src->data[0].v_pointer, NULL);
}

void gini_init(void)
{
    if(init)
        return;

    g_type_init();

    g_value_register_transform_func(G_TYPE_STRING, G_TYPE_BOOLEAN,
                                    value_transform_string_boolean);

    g_value_register_transform_func(G_TYPE_STRING, G_TYPE_INT,
                                    value_transform_string_int);
    g_value_register_transform_func(G_TYPE_STRING, G_TYPE_UINT,
                                    value_transform_string_uint);

    g_value_register_transform_func(G_TYPE_STRING, G_TYPE_LONG,
                                    value_transform_string_long);
    g_value_register_transform_func(G_TYPE_STRING, G_TYPE_ULONG,
                                    value_transform_string_ulong);

    g_value_register_transform_func(G_TYPE_STRING, G_TYPE_INT64,
                                    value_transform_string_int64);
    g_value_register_transform_func(G_TYPE_STRING, G_TYPE_UINT64,
                                    value_transform_string_uint64);

    g_value_register_transform_func(G_TYPE_STRING, G_TYPE_FLOAT,
                                    value_transform_string_float);
    g_value_register_transform_func(G_TYPE_STRING, G_TYPE_DOUBLE,
                                    value_transform_string_double);

    init = TRUE;
}

static gint gini_section_append_alpha_cmp(gconstpointer a, gconstpointer b)
{
    const GIniSection *_a = a,
                      *_b = b;

    return(strcmp(_a->name, _b->name));
}

static void gini_section_append_alpha(GIni *ini, GIniSection *sec)
{
    g_return_if_fail(ini != NULL);
    g_return_if_fail(sec != NULL);

    ini->section_list = g_slist_insert_sorted(ini->section_list, sec,
                                              gini_section_append_alpha_cmp);
}

static void gini_section_append_added(GIni *ini, GIniSection *sec)
{
    g_return_if_fail(ini != NULL);
    g_return_if_fail(sec != NULL);

    ini->section_list = g_slist_append(ini->section_list, sec);
}

static gint gini_entry_append_alpha_cmp(gconstpointer a, gconstpointer b)
{
    const GIniEntry *_a = a,
                    *_b = b;

    return(strcmp(_a->key, _b->key));
}

static void gini_entry_append_alpha(GIniSection *sec, GIniEntry *e)
{
    g_return_if_fail(sec != NULL);
    g_return_if_fail(e != NULL);

    sec->entry_list = g_slist_insert_sorted(sec->entry_list, e,
                                            gini_entry_append_alpha_cmp);
}

static void gini_entry_append_added(GIniSection *sec, GIniEntry *e)
{
    g_return_if_fail(sec != NULL);
    g_return_if_fail(e != NULL);

    sec->entry_list = g_slist_append(sec->entry_list, e);
}

/*
 * file can be NULL, and is only used to store the source file of the gini.
 *
 * allowed_sections and allowed_keys can be NULL, or a NULL-terminated array
 * of valid section or key names.
 *
 * if NULL, any section or key names will be considered valid, otherwise a
 * case-insensitive string comparison is performed to see if the section or
 * key name passed to gini_(entry|section)_add() matches a string in their
 * respective allowed list.
 */
GIni *gini_new_full(const gchar *path,
                    const gchar **allowed_sections,
                    const gchar **allowed_keys,
                    GIniOrderType order_type,
                    GIniFlags flags)
{
    GHashFunc  hash_func;
    GEqualFunc key_equal_func;

    GIni *ini;

    g_return_val_if_fail(init != FALSE, NULL);

    if(flags & GINI_FLAG_CASE_SENSITIVE) {
        hash_func      = g_str_hash;
        key_equal_func = g_str_equal;
    } else {
        hash_func      = gini_str_hash;
        key_equal_func = gini_str_equal;
    }

    ini                   = g_new(GIni, 1);

    ini->file             = g_strdup(path);

    ini->sections         = g_hash_table_new_full(hash_func,
                                                  key_equal_func,
                                                  NULL,
                                                  (GDestroyNotify)
                                                  gini_section_free);

    ini->allowed_sections = allowed_sections;
    ini->allowed_keys     = allowed_keys;

    ini->section_list     = NULL;

    ini->ref_count        = 1;

    switch(order_type) {
        case GINI_ORDER_ALPHA:
            ini->section_append = (GIniListFunc) gini_section_append_alpha;
            ini->entry_append   = (GIniListFunc) gini_entry_append_alpha;
            break;
        case GINI_ORDER_ADDED:
            ini->section_append = (GIniListFunc) gini_section_append_added;
            ini->entry_append   = (GIniListFunc) gini_entry_append_added;
            break;
        default:
            break;
    }

    return(ini);
}

/* increments the reference count */
void gini_ref(GIni *ini)
{
    g_return_if_fail(ini != NULL);
    g_return_if_fail(ini->ref_count > 0);

    gini_mutex_lock();

    ini->ref_count += 1;

    gini_mutex_unlock();
}

/* decrements the reference count and frees if 0 */
void gini_unref(GIni *ini)
{
    g_return_if_fail(ini != NULL);
    g_return_if_fail(ini->ref_count > 0);

    gini_mutex_lock();

    ini->ref_count -= 1;

    if(ini->ref_count == 0) {
        g_free(ini->file);
        g_hash_table_destroy(ini->sections);
        g_slist_free(ini->section_list);
        g_free(ini);
    }

    gini_mutex_unlock();
}

/* return ini filename */
G_CONST_RETURN gchar *gini_get_file(GIni *ini)
{
    g_return_val_if_fail(ini != NULL, NULL);

    return(ini->file);
}

/* set the source/target filename for the current GIni */
void gini_set_file(GIni *ini, const gchar *file)
{
    gchar *p;

    g_return_if_fail(ini != NULL);
    g_return_if_fail(file != NULL);

    gini_mutex_lock();

    p = g_strdup(file);
    g_free(ini->file);
    ini->file = p;

    gini_mutex_unlock();
}

/* returns the number of sections in ini */
guint gini_get_section_count(GIni *ini)
{
    g_return_val_if_fail(ini != NULL, 0);

    return(g_hash_table_size(ini->sections));
}

/* ini the contents of path and return, set GError on NULL */
GIni *gini_load_full(const gchar *path,
                     const gchar **allowed_sections,
                     const gchar **allowed_keys,
                     GIniOrderType order_type,
                     GIniFlags flags,
                     GError **err)
{
    GIni *ini;

    gchar *data;

    g_return_val_if_fail(path != NULL, NULL);

    debug("gini_load(): '%s'", path);

    if(!g_file_get_contents(path, &data, 0, err))
        return(NULL);

    ini = gini_parse_data_full(path,
                               data,
                               allowed_sections,
                               allowed_keys,
                               order_type,
                               flags);

    g_return_val_if_fail(ini != NULL, NULL);

    g_free(data);

    return(ini);
}

/* ignore spacing characters in section names */
static gchar *gini_safe_section_name(const gchar *section_name)
{
#ifdef GINI_STRICT
    GString *string;

    const gchar *p;

    g_return_val_if_fail(section_name != NULL, NULL);

    string = g_string_new(NULL);

    for(p = section_name; *p; p++) {
        if(g_ascii_isspace(*p))
            continue;

        g_string_append_c(string, *p);
    }

    return(g_string_free(string, FALSE));
#else
    return(g_strdup(section_name));
#endif
}

/* return TRUE if section_name was successfully extracted from str */
static gboolean gini_parse_section_line(const gchar *str, gchar **section_name)
{
    gchar name[256];

    g_return_val_if_fail(str != NULL, FALSE);
    g_return_val_if_fail(section_name != NULL, FALSE);

    if(str[0] == '[' && sscanf(str, "[%255[^]]]", name) == 1) {
        *section_name = gini_safe_section_name(name);
        return(TRUE);
    }

    return(FALSE);
}

/*
 * parse the contents of data array, data may be modified (for whitespace).
 *
 * the path argument is only used to set the file for the returned GIni *.
 */
GIni *gini_parse_data_full(const gchar *path,
                           gchar *data,
                           const gchar **allowed_sections,
                           const gchar **allowed_keys,
                           GIniOrderType order_type,
                           GIniFlags flags)
{
    gchar **line;

    guint i;

    GIni *ini;
    GIniSection *sec = NULL;

    g_return_val_if_fail(data != NULL, NULL);

    if(!(line = g_strsplit(g_strdelimit(data, "\r\n", '\n'), "\n", -1)))
        return(NULL);

    ini = gini_new_full(path,
                        allowed_sections,
                        allowed_keys,
                        order_type,
                        flags);

    g_return_val_if_fail(ini != NULL, NULL);

    for(i = 0; line[i]; i++) {
        gchar *s;

        if(!(s = g_strchug(line[i])) || !s[0])
            continue;

        if(GINI_IS_COMMENT(s[0])) /* skip comments */
            continue;

        if(s[0] == '[') {      /* section */
            gchar *section_name;

            if(gini_parse_section_line(s, &section_name)) {
                sec = gini_section_add_value(ini, section_name);
                g_free(section_name);
            } else
                debug("gini_parse_data_full(): section parse failed '%s'", s);

            continue;
        }

        if(!sec) /* section not located yet */
            continue;

        if(strchr(s, '=')) {   /* entry */
            GIniEntry *e;

            if((e = gini_entry_new_from_string(s))) {
                if(!gini_entry_add(ini, sec, e))
                    gini_entry_free(e);
            } else
                debug("gini_parse_data_full(): entry parse failed '%s'", s);
        } else
            debug("gini_parse_data_full(): unhandled line '%s'", s);
    }

    g_strfreev(line);

    return(ini);
}

/* return TRUE if str should be quoted, starting/trailing spaces and ,; */
static gboolean gini_entry_val_should_quote(const gchar *str)
{
    gint len;

    g_return_val_if_fail(str != NULL, FALSE);

    if((len = (str) ? strlen(str) : 0) > 1) {
        const gchar *p;

        if(g_ascii_isspace(str[0]) || g_ascii_isspace(str[len - 1]))
            return(TRUE);

        for(p = str; *p; p++)
            if(GINI_IS_COMMENT(*p) || *p == ',')
                return(TRUE);
    }

    return(FALSE);
}

static void gini_entry_val_string_write(const gchar *str, FILE *fp)
{
    const guchar *p = (guchar *) str;

    gboolean quote;

    g_return_if_fail(fp != NULL);
    g_return_if_fail(str != NULL);

    quote = gini_entry_val_should_quote(str);

    if(quote)
        fprintf(fp, "\"");

    while(*p) {
        gint c = *p++;

        if(c == '\\') {
            fprintf(fp, "\\\\");
            continue;
        }

        switch(c) {
            case '\a':
                fprintf(fp, "\\a");
                break;
            case '\b':
                fprintf(fp, "\\b");
                break;
            case '\f':
                fprintf(fp, "\\f");
                break;
            case '\n':
                fprintf(fp, "\\n");
                break;
            case '\r':
                fprintf(fp, "\\r");
                break;
            case '\t':
                fprintf(fp, "\\t");
                break;
            case '\v':
                fprintf(fp, "\\v");
                break;
            case '\'':
                fprintf(fp, "\\'");
                break;
            case '"':
                fprintf(fp, "\\\"");
                break;
            case '?':
                fprintf(fp, "\\?");
                break;
            default:
                if((c >= 0 && c <= 31) || (c >= 127 && c <= 159))
                    fprintf(fp, "\\%03o", c);
                else
                    fprintf(fp, "%c", c);
                break;
        }
    }

    if(quote)
        fprintf(fp, "\"");
}

static void gini_entry_write(gpointer key, gpointer val, gpointer data)
{
    GIniEntry *e = val;

    FILE *fp = data;

    g_return_if_fail(e != NULL);
    g_return_if_fail(e->key != NULL);
    g_return_if_fail(e->sec != NULL);
    g_return_if_fail(e->sec->ini != NULL);
    g_return_if_fail(fp != NULL);

    /*
     * although G_TYPE_POINTER cannot be quantified when saving, it may
     * still be of use as an internal data store of sorts...
     */
    if(G_VALUE_TYPE(e->val) == G_TYPE_POINTER) {
        debug("gini_entry_write(): '%s' is G_TYPE_POINTER, skipping", e->key);
        return;
    }

#if DEBUG
    {
        gchar *val = g_strdup_value_contents(e->val);
        debug("gini_entry_write(): '%s' = %s", e->key, val);
        g_free(val);
    }
#endif

    /* g_strdup_value_contents() alters the output, can't be used here */

    fprintf(fp, "%s=", e->key);

    switch(G_VALUE_TYPE(e->val)) {
        case G_TYPE_STRING:
            gini_entry_val_string_write(g_value_get_string(e->val), fp);
            break;
        case G_TYPE_BOOLEAN:
            fprintf(fp, "%s", (g_value_get_boolean(e->val)) ? "TRUE" : "FALSE");
            break;
        case G_TYPE_INT:
            fprintf(fp, "%d", g_value_get_int(e->val));
            break;
        case G_TYPE_UINT:
            fprintf(fp, "%u", g_value_get_uint(e->val));
            break;
        case G_TYPE_LONG:
            fprintf(fp, "%ld", g_value_get_long(e->val));
            break;
        case G_TYPE_ULONG:
            fprintf(fp, "%lu", g_value_get_ulong(e->val));
            break;
        case G_TYPE_INT64:
#ifdef G_GINT64_FORMAT
            fprintf(fp, "%" G_GINT64_FORMAT, g_value_get_int64(e->val));
#else
            g_critical("Unable to print int64 on this system.");
            break;
#endif
        case G_TYPE_UINT64:
#ifdef G_GUINT64_FORMAT
            fprintf(fp, "%" G_GUINT64_FORMAT, g_value_get_uint64(e->val));
#else
            g_critical("Unable to print uint64 on this system.");
#endif
            break;
        case G_TYPE_FLOAT:
            fprintf(fp, "%g", g_value_get_float(e->val));
            break;
        case G_TYPE_DOUBLE:
            fprintf(fp, "%g", g_value_get_double(e->val));
            break;
        default:
            g_assert_not_reached();
    }

    fprintf(fp, "\n");
}

#if 0 /* XXX: G_TYPE_POINTER empty section hack */
static void gini_section_valid_entry_count(gpointer key,
                                           gpointer val,
                                           gpointer *data)
{
    GIniEntry *e = val;

    gint c;

    g_return_if_fail(e != NULL);
    g_return_if_fail(e->key != NULL);
    g_return_if_fail(data != NULL);

    c = GPOINTER_TO_INT(*data);

    if(G_VALUE_TYPE(e->val) != G_TYPE_POINTER)
        c++;

    *data = GINT_TO_POINTER(c);
}
#endif

static void gini_section_write(gpointer key, gpointer val, gpointer data)
{
    GIniSection *sec = val;

    FILE *fp = data;

    g_return_if_fail(sec != NULL);
    g_return_if_fail(sec->name != NULL);
    g_return_if_fail(sec->ini != NULL);
    g_return_if_fail(fp != NULL);

    debug("gini_section_write(): %s", sec->name);

  /*if(!g_hash_table_size(sec->entries))
        return;*/

#if 0 /* XXX: G_TYPE_POINTER empty section hack, negligible impact */
    {
        gpointer p = GINT_TO_POINTER(0);

        g_hash_table_foreach(sec->entries,
                             (GHFunc) gini_section_valid_entry_count,
                             &p);

        if(GPOINTER_TO_INT(p) < 1)
            return;
    }
#endif

    fprintf(fp, "[%s]\n", sec->name);

    if(sec->entry_list) {
        GSList *p;

        for(p = sec->entry_list; p; p = g_slist_next(p))
            gini_entry_write(NULL, p->data, fp);
    } else
        g_hash_table_foreach(sec->entries, gini_entry_write, fp);

    fprintf(fp, "\n");
}

/* write the stored ini data to a specified FILE */
void gini_write(GIni *ini, FILE *fp)
{
    g_return_if_fail(ini != NULL);
    g_return_if_fail(fp != NULL);

    if(ini->section_list) {
        GSList *p;

        for(p = ini->section_list; p; p = g_slist_next(p))
            gini_section_write(NULL, p->data, fp);
    } else
        g_hash_table_foreach(ini->sections, gini_section_write, fp);

}

/*
 * save the contnents of ini to path, return FALSE on error
 *
 * if path is NULL, the file set in GIni when created will be used.
 */
gboolean gini_save(GIni *ini, const gchar *path, GError **err)
{
    const gchar *file;

    FILE *fp;

    g_return_val_if_fail(ini != NULL, FALSE);

    file = (path) ? path : ini->file;

    if(!file) {
        g_critical("Unable to determine file to use.");
        return(FALSE);
    }

    debug("gini_save(): %s", file);

    if(!(fp = fopen(file, "w"))) {
        if(!err)
            g_critical("Failed to open '%s': %s.", file, g_strerror(errno));

        g_set_error(err, 0, 0, "Failed to open '%s': %s.", file,
                               g_strerror(errno));

        return(FALSE);
    }

    gini_write(ini, fp);

    fclose(fp);

    return(TRUE);
}

/* return TRUE if section_name exists in ini */
gboolean gini_has_section(GIni *ini, const gchar *section_name)
{
    g_return_val_if_fail(ini != NULL, FALSE);
    g_return_val_if_fail(section_name != NULL, FALSE);

    return((gini_section_lookup(ini, section_name)) ? TRUE : FALSE);
}

/* return TRUE if key exists in section_name */
gboolean gini_has_entry(GIni *ini, const gchar *section_name, const gchar *key)
{
    GIniSection *sec;

    g_return_val_if_fail(ini != NULL, FALSE);
    g_return_val_if_fail(section_name != NULL, FALSE);
    g_return_val_if_fail(key != NULL, FALSE);

    if((sec = gini_section_lookup(ini, section_name)))
        if(gini_entry_lookup(ini, sec, key))
            return(TRUE);

    return(FALSE);
}

/* return a new section, section_name should not be freed */
GIniSection *gini_section_new(const gchar *section_name)
{
    GIniSection *sec;

    g_return_val_if_fail(section_name != NULL, NULL);

    sec             = g_new(GIniSection, 1);

    sec->name       = g_strdup(section_name);

    sec->entries    = g_hash_table_new_full(gini_str_hash,
                                         gini_str_equal,
                                         NULL,
                                         (GDestroyNotify) gini_entry_free);

    sec->entry_list = NULL;

    sec->ini        = NULL;

    return(sec);
}

/* free a section and all sub-entries */
void gini_section_free(GIniSection *sec)
{
    if(!sec)
        return;

    debug("gini_section_free(): '%s'", sec->name);

    gini_mutex_lock();

    if(sec->ini && sec->ini->section_list)
        sec->ini->section_list = g_slist_remove(sec->ini->section_list, sec);

    g_free(sec->name);
    g_hash_table_destroy(sec->entries);
    g_slist_free(sec->entry_list);
    g_free(sec);

    gini_mutex_unlock();
}

/*
 * removes any unquoted (trailing) comments from str, str may be modified
 *
 * quote handling could use some work... if a string value with
 * an extra (unescaped) quote is encountered, and a trailing
 * comment is included, that comment will not be removed.
 *
 * Example:
 *     "val"additional_text_here" ;comment
 *         ^-- extra, unescaped quote
 *   results in:
 *     valadditional_text_here ;comment
 */
static gchar *gini_entry_val_clean(gchar *str)
{
    gint len;

    gboolean quoted = FALSE;

    g_return_val_if_fail(str != NULL, NULL);

    if(GINI_IS_COMMENT(str[0]))
        str[0] = '\0';

    len = strlen(str);

    while(len--) {
        if(str[len] == '"' && str[len-1] != '\\')
            quoted = !quoted;

        if(GINI_IS_COMMENT(str[len]) && g_ascii_isspace(str[len-1]) && !quoted)
            str[len-1] = '\0';
    }

    return(g_strstrip(str));
}

static gchar *gini_entry_val_parse(const gchar *str)
{
    const guchar *p = (guchar *) str;

    GString *string;

    gboolean quoted = FALSE;

    g_return_val_if_fail(str != NULL, NULL);

    string = g_string_new(NULL);

    while(*p) {
        gchar c = *p++;

        if(c == '\\' && *p && *p == '\\') {
            g_string_append(string, "\\"); (void) *p++;
            continue;
        }

        if(c == '\\') {
            c = *p++;

            /* octal: \NNN */
            if(g_ascii_isdigit(c) && c != '8' && c != '9') {
                guint code;

                p--;

                if(sscanf((const gchar *) p, "%3o", &code) == 1) {
                    p += 3;
                    g_string_append_c(string, code);
                }

                continue;
            }

            /* hex: \xNNN */
            if(c == 'x' && g_ascii_isdigit(*p) && *p != '8' && *p != '9') {
                guint code;

                if(sscanf((const gchar *) p, "%3x", &code) == 1) {
                    p += 3;
                    g_string_append_c(string, code);
                }

                continue;
            }

            switch(c) {
                case 'a':
                    g_string_append_c(string, '\a');
                    break;
                case 'b':
                    g_string_append_c(string, '\b');
                    break;
                case 'f':
                    g_string_append_c(string, '\f');
                    break;
                case 'n':
                    g_string_append_c(string, '\n');
                    break;
                case 'r':
                    g_string_append_c(string, '\r');
                    break;
                case 't':
                    g_string_append_c(string, '\t');
                    break;
                case 'v':
                    g_string_append_c(string, '\v');
                    break;
                default:
                    g_string_append_c(string, c);
                    break;
            }
        } else if(c == '"')
            quoted = !quoted;
        else
            g_string_append_c(string, c);
    }

    return(g_string_free(string, FALSE));
}

/* ignore spacing or equal characters in key names */
static gchar *gini_safe_key_name(const gchar *key)
{
    GString *string;

    const gchar *p;

    g_return_val_if_fail(key != NULL, NULL);

    string = g_string_new(NULL);

    for(p = key; *p; p++) {
#ifdef GINI_STRICT
        if(g_ascii_isspace(*p) || *p == '=')
#else
        if(*p == '=')
#endif
            continue;

        g_string_append_c(string, *p);
    }

    return(g_string_free(string, FALSE));
}

/* return TRUE if key and val were successfully extracted from str */
static gboolean gini_parse_entry_line(const gchar *str,
                                      gchar **key,
                                      gchar **val)
{
    gchar **p;

    g_return_val_if_fail(str != NULL, FALSE);
    g_return_val_if_fail(key != NULL, FALSE);
    g_return_val_if_fail(val != NULL, FALSE);

    if((p = g_strsplit(str, "=", 2))) {
        gboolean ret = FALSE;

        if(p[0] && p[1]) {
            const gchar *k = g_strstrip(p[0]);
            const gchar *v = gini_entry_val_clean(g_strstrip(p[1]));

            if(k[0] && v) {
                *key = gini_safe_key_name(k);
                *val = gini_entry_val_parse(v);

                ret = TRUE;
            }
        }

        g_strfreev(p);

        return(ret);
    }

    return(FALSE);
}

/*
 * return a new GIniSection from a string.
 *
 * GIniSection *sec = gini_section_new_from_string("[section_name]");
 *
 * the passed string is not validated for proper syntax and should be
 * g_strstrip()'d to remove surrounding whitespace
 */
GIniSection *gini_section_new_from_string(const gchar *str)
{
    GIniSection *sec = NULL;

    gchar *section_name;

    g_return_val_if_fail(str != NULL, NULL);

    if(gini_parse_section_line(str, &section_name)) {
        sec = gini_section_new(section_name);
        g_free(section_name);
    }

    return(sec);
}

/* return the section name */
G_CONST_RETURN gchar *gini_section_get_name(GIniSection *sec)
{
    g_return_val_if_fail(sec != NULL, NULL);

    return(sec->name);
}

/* add a section */
gboolean gini_section_add(GIni *ini, GIniSection *sec)
{
    g_return_val_if_fail(ini != NULL, FALSE);
    g_return_val_if_fail(sec != NULL, FALSE);
    g_return_val_if_fail(sec->name != NULL, FALSE);

    debug("gini_section_add(): '%s'", sec->name);

    if(!gini_section_name_allowed(ini, sec->name))
        return(FALSE);

    gini_mutex_lock();

    g_hash_table_replace(ini->sections, sec->name, sec);

    if(ini->section_append)
        ini->section_append(ini, sec);

    sec->ini = ini;

    gini_mutex_unlock();

    return(TRUE);
}

/* add a section by name */
GIniSection *gini_section_add_value(GIni *ini, const gchar *section_name)
{
    GIniSection *sec;

    g_return_val_if_fail(ini != NULL, NULL);
    g_return_val_if_fail(section_name != NULL, NULL);

    /* existing section found */
    if((sec = gini_section_lookup(ini, section_name)))
#ifdef GINI_STRICT  /* remove */
        gini_section_del(ini, section_name);
#else               /* combine */
        return(sec);
#endif

    /* add section */
    if(gini_section_add(ini, (sec = gini_section_new(section_name))))
        return(sec);

    gini_section_free(sec);

    return(NULL);
}

/* remove section */
gboolean gini_section_del(GIni *ini, const gchar *section_name)
{
    gboolean ret;

    g_return_val_if_fail(ini != NULL, FALSE);
    g_return_val_if_fail(section_name != NULL, FALSE);

    debug("gini_section_del(): '%s'", section_name);

    gini_mutex_lock();

    ret = g_hash_table_remove(ini->sections, section_name);

    gini_mutex_unlock();

    return(ret);
}

/* locate a section by name */
GIniSection *gini_section_lookup(GIni *ini, const gchar *section_name)
{
    g_return_val_if_fail(ini != NULL, NULL);
    g_return_val_if_fail(section_name != NULL, NULL);

    return(g_hash_table_lookup(ini->sections, section_name));
}

/*
 * return the GIni of a selected section. this will be NULL if the
 * section has not yet been added to a GIni.
 */
G_CONST_RETURN GIni *gini_section_get_ini(GIniSection *sec)
{
    g_return_val_if_fail(sec != NULL, NULL);

    return(sec->ini);
}

/* returns the number of entries in section_name */
guint gini_section_get_entry_count(GIni *ini, const gchar *section_name)
{
    GIniSection *sec;

    g_return_val_if_fail(ini != NULL, 0);
    g_return_val_if_fail(section_name != NULL, 0);

    if((sec = gini_section_lookup(ini, section_name)))
        return(g_hash_table_size(sec->entries));

    return(0);
}

GIniEntry *gini_entry_new_with_type(const gchar *key,
                                    const gpointer val,
                                    GType type)
{
    GIniEntry *e;

    g_return_val_if_fail(key != NULL, NULL);
    g_return_val_if_fail(type != G_TYPE_INVALID, NULL);

    e      = g_new(GIniEntry, 1);

    e->key = g_strdup(key);
    e->val = g_new0(GValue, 1);

    e->sec = NULL;

    g_value_init(e->val, type);

    switch(type) {
        case G_TYPE_STRING:
            g_value_set_string(e->val, val);
            break;
        case G_TYPE_POINTER:
            g_value_set_pointer(e->val, val);
            break;
        default:
            break;
    }

    return(e);
}

/* return a new section entry, which must be added with gini_entry_add() */
GIniEntry *gini_entry_new(const gchar *key, const gchar *val)
{
    return(gini_entry_new_with_type(key, (const gpointer) val, G_TYPE_STRING));
}

/* free a section entry */
void gini_entry_free(GIniEntry *e)
{
    if(!e)
        return;

    debug("gini_entry_free(): '%s'", e->key);

    gini_mutex_lock();

    if(e->sec && e->sec->entry_list)
        e->sec->entry_list = g_slist_remove(e->sec->entry_list, e);

    g_free(e->key);
    g_value_unset(e->val);
    g_free(e->val);
    g_free(e);

    gini_mutex_unlock();
}

/*
 * return a new GIniEntry from a string.
 *
 * GIniEntry *e = gini_entry_new_from_string("key=val");
 *
 * the passed string is not validated for proper syntax and should be
 * g_strstrip()'d to remove surrounding whitespace
 */
GIniEntry *gini_entry_new_from_string(const gchar *str)
{
    GIniEntry *e = NULL;

    gchar *key = NULL,
          *val = NULL;

    g_return_val_if_fail(str != NULL, NULL);

    if(gini_parse_entry_line(str, &key, &val)) {
        e = gini_entry_new(key, val);

        g_free(key);
        g_free(val);
    }

    return(e);
}

GIniEntry *gini_entry_add_values(GIni *ini,
                                 GIniSection *sec,
                                 const gchar *key,
                                 const gchar *val)
{
    GIniEntry *e;

    g_return_val_if_fail(ini != NULL, FALSE);
    g_return_val_if_fail(sec != NULL, FALSE);
    g_return_val_if_fail(key != NULL, FALSE);

    if((e = gini_entry_new(key, val))) {
        if(gini_entry_add(ini, sec, e))
            return(e);

        gini_entry_free(e);
    }

    return(NULL);
}

/* add an entry to section */
gboolean gini_entry_add(GIni *ini, GIniSection *sec, GIniEntry *e)
{
    g_return_val_if_fail(ini != NULL, FALSE);
    g_return_val_if_fail(sec != NULL, FALSE);
    g_return_val_if_fail(e != NULL, FALSE);
    g_return_val_if_fail(e->key != NULL, FALSE);

    debug("gini_entry_add(): '%s'", e->key);

    if(!gini_entry_key_allowed(ini, e->key))
        return(FALSE);

    gini_mutex_lock();

    g_hash_table_replace(sec->entries, e->key, e);

    if(ini->entry_append)
        ini->entry_append(sec, e);

    e->sec = sec;

    gini_mutex_unlock();

    return(TRUE);
}

/* remove entry from GIniSection */
gboolean gini_entry_del(GIni *ini, GIniSection *sec, const gchar *key)
{
    gboolean ret;

    g_return_val_if_fail(ini != NULL, FALSE);
    g_return_val_if_fail(sec != NULL, FALSE);
    g_return_val_if_fail(key != NULL, FALSE);

    debug("gini_entry_del(): '%s/%s'", sec->name, key);

    gini_mutex_lock();

    ret = g_hash_table_remove(sec->entries, key);

    gini_mutex_unlock();

    return(ret);
}

/* remove entry from section_name */
gboolean gini_entry_remove(GIni *ini,
                           const gchar *section_name,
                           const gchar *key)
{
    GIniSection *sec;

    g_return_val_if_fail(ini != NULL, FALSE);
    g_return_val_if_fail(section_name != NULL, FALSE);
    g_return_val_if_fail(key != NULL, FALSE);

    if((sec = gini_section_lookup(ini, section_name)))
        return(gini_entry_del(ini, sec, key));

    return(FALSE);
}


/*
 * return the GIni of a selected entry. this will be NULL if the entry
 * and/or section has not yet been added to a GIni.
 */
G_CONST_RETURN GIni *gini_entry_get_ini(GIniEntry *e)
{
    g_return_val_if_fail(e != NULL, NULL);

    return((e->sec) ? e->sec->ini : NULL);
}

/*
 * return the GIniSection of a selected entry. this will be NULL if the
 * entry has not yet been added to a GIniSection.
 */
G_CONST_RETURN GIniSection *gini_entry_get_section(GIniEntry *e)
{
    g_return_val_if_fail(e != NULL, NULL);

    return(e->sec);
}

/* return the section_name of a selected entry */
G_CONST_RETURN gchar *gini_entry_get_section_name(GIniEntry *e)
{
    g_return_val_if_fail(e != NULL, NULL);

    return((e->sec) ? e->sec->name : NULL);
}

/* return the entry key */
G_CONST_RETURN gchar *gini_entry_get_key(GIniEntry *e)
{
    g_return_val_if_fail(e != NULL, NULL);

    return(e->key);
}

/* return the entry value */
G_CONST_RETURN GValue *gini_entry_get_val(GIniEntry *e)
{
    g_return_val_if_fail(e != NULL, NULL);

    return(e->val);
}

/* return the entry value type (string, int, long, etc) */
GType gini_entry_get_val_type(GIniEntry *e)
{
    g_return_val_if_fail(e != NULL, G_TYPE_INVALID);
    g_return_val_if_fail(e->val != NULL, G_TYPE_INVALID);

    return(G_VALUE_TYPE(e->val));
}

/*
 * get section entry by entry key
 *
 * one of section_name or sec must be set. if sec is NULL, a lookup will be
 * performed on section_name.
 *
 * if hundreds or thousands of operations will be performed, this would be the
 * preferred function. pass an gini_section_t * and a NULL section_name.
 */
GIniEntry *gini_entry_lookup(GIni *ini, GIniSection *sec, const gchar *key)
{
    g_return_val_if_fail(ini != NULL, NULL);
    g_return_val_if_fail(sec != NULL, NULL);
    g_return_val_if_fail(key != NULL, NULL);

    return(g_hash_table_lookup(sec->entries, key));
}

/* non-destructive value transform, returned values must be freed */
gboolean gini_entry_get_val_as_type(GIni *ini,
                                    const gchar *section_name,
                                    const gchar *key,
                                    GType type,
                                    ...)
{
    GValue value = { 0, };

    va_list args;

    GIniEntry *e;

    gchar *error;

    g_return_val_if_fail(ini != NULL, FALSE);
    g_return_val_if_fail(section_name != NULL, FALSE);
    g_return_val_if_fail(key != NULL, FALSE);

    {
        GIniSection *sec;

        if(!(sec = gini_section_lookup(ini, section_name)))
            return(FALSE);

        if(!(e = gini_entry_lookup(ini, sec, key)))
            return(FALSE);
    }

    g_return_val_if_fail(e->val != NULL, FALSE);

    if(!g_value_type_transformable(G_VALUE_TYPE(e->val), type)) {
        g_critical("Unable to transform '%s' from %lu to %lu.",
                   key, G_VALUE_TYPE(e->val), type);
        return(FALSE);
    }

    if(!g_value_init(&value, type))
        return(FALSE);

    if(!g_value_transform(e->val, &value))
        return(FALSE);

    va_start(args, type);
    G_VALUE_LCOPY(&value, args, 0, &error);
    va_end(args);

    if(error) {
        g_warning("%s", error);
        g_free(error);
        g_value_unset(&value);

        return(FALSE);
    }

    return(TRUE);
}

/* destructive transform, returned values should not be freed */
gboolean gini_entry_get_as_type(GIni *ini,
                                const gchar *section_name,
                                const gchar *key,
                                GType type,
                                ...)
{
    va_list args;

    GIniEntry *e;

    gchar *error;

    g_return_val_if_fail(ini != NULL, FALSE);
    g_return_val_if_fail(section_name != NULL, FALSE);
    g_return_val_if_fail(key != NULL, FALSE);

    {
        GIniSection *sec;

        if(!(sec = gini_section_lookup(ini, section_name)))
            return(FALSE);

        if(!(e = gini_entry_lookup(ini, sec, key)))
            return(FALSE);
    }

    g_return_val_if_fail(e->val != NULL, FALSE);

    /* convert type if necessary */
    if(G_VALUE_TYPE(e->val) != type) {
        GValue value = { 0, };

        if(!g_value_type_transformable(G_VALUE_TYPE(e->val), type)) {
            g_critical("Unable to transform '%s' from %lu to %lu.",
                       key, G_VALUE_TYPE(e->val), type);
            return(FALSE);
        }

        debug("gini_entry_get_as_type(): '%s' transformed from %lu to %lu",
              key, G_VALUE_TYPE(e->val), type);

        if(!g_value_init(&value, type))
            return(FALSE);

        if(!g_value_transform(e->val, &value))
            return(FALSE);

        gini_mutex_lock();

        g_value_unset(e->val);
        g_value_init(e->val, type);
        g_value_copy(&value, e->val);
        g_value_unset(&value);

        gini_mutex_unlock();
    }

    va_start(args, type);
    G_VALUE_LCOPY(e->val, args, G_VALUE_NOCOPY_CONTENTS, &error);
    va_end(args);

    if(error) {
        g_warning("%s", error);
        g_free(error);

        return(FALSE);
    }

    return(TRUE);
}

/* create and add an entry (convenience functions) */
static GIniEntry *gini_entry_create(GIni *ini,
                                    GIniSection *sec,
                                    const gchar *key,
                                    GType type)
{
    GIniEntry *e;

    g_return_val_if_fail(ini != NULL, NULL);
    g_return_val_if_fail(sec != NULL, NULL);
    g_return_val_if_fail(key != NULL, NULL);

    e = gini_entry_new_with_type(key, NULL, type);

    g_return_val_if_fail(e != NULL, NULL);

    if(!gini_entry_add(ini, sec, e)) {
        gini_entry_free(e);
        e = NULL;
    }

    return(e);
}

void gini_entry_set_as_type(GIni *ini,
                            const gchar *section_name,
                            const gchar *key,
                            GType type,
                            ...)
{
    va_list args;

    GIniEntry *e;

    gchar *error;

    g_return_if_fail(ini != NULL);
    g_return_if_fail(section_name != NULL);
    g_return_if_fail(key != NULL);

    gini_mutex_lock();

    {
        GIniSection *sec;

        if(!(sec = gini_section_lookup(ini, section_name)))
            if(!(sec = gini_section_add_value(ini, section_name)))
                return;  /* possible illegal section name */

        if(!(e = gini_entry_lookup(ini, sec, key)))
            if(!(e = gini_entry_create(ini, sec, key, type)))
                return;  /* possible illegal entry name */
    }

    if(G_VALUE_TYPE(e->val) != type) {
        debug("gini_entry_set_as_type(): '%s' transformed from %lu to %lu",
              key, G_VALUE_TYPE(e->val), type);

        g_value_unset(e->val);
        g_value_init(e->val, type);
    }

    va_start(args, type);
    G_VALUE_COLLECT(e->val, args, 0, &error);
    va_end(args);

    gini_mutex_unlock();

    if(error) {
        g_warning("%s", error);
        g_free(error);
    }
}

G_CONST_RETURN gchar *gini_get_version(void)
{
    static gchar buf[25];

    g_snprintf(buf, sizeof(buf), "libgini/%i.%i", GINI_MAJOR_VERSION,
                                                  GINI_MINOR_VERSION);

    return(buf);
}
