/*
 * libgini: a thread-safe, GLib-based .ini parser that uses hash
 *          tables for quick value lookup.
 *
 * Copyright (c) 2003-2005 Mike Hokenson <logan at gozer dot org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef __gini_h
#define __gini_h

#include <glib.h>
#include <glib-object.h>

G_BEGIN_DECLS

typedef enum {
    GINI_ORDER_ALPHA,                  /* sort alphabetically       */
    GINI_ORDER_ADDED,                  /* maintain added order      */
    GINI_ORDER_NONE                    /* disable sorting (fastest) */
} GIniOrderType;

typedef enum {
    GINI_FLAG_CASE_SENSITIVE = 1 << 0  /* make case-sensitive       */
} GIniFlags;

typedef struct _GIniEntry   GIniEntry;
typedef struct _GIniSection GIniSection;
typedef struct _GIni        GIni;

void gini_init(void);

GIni *gini_new_full(const gchar *path,
                    const gchar **allowed_sections,
                    const gchar **allowed_keys,
                    GIniOrderType order_type,
                    GIniFlags flags);

#define gini_new(path) \
            (gini_new_full(path, NULL, NULL, GINI_ORDER_ADDED, 0))

void gini_ref(GIni *ini);
void gini_unref(GIni *ini);

G_CONST_RETURN gchar *gini_get_file(GIni *ini);
void gini_get_file_set(GIni *ini, const gchar *file);

guint gini_get_section_count(GIni *ini);

GIni *gini_load_full(const gchar *path,
                     const gchar **allowed_sections,
                     const gchar **allowed_keys,
                     GIniOrderType order_type,
                     GIniFlags flags,
                     GError **err);

#define gini_load(path, err) \
            (gini_load_full(path, NULL, NULL, GINI_ORDER_ADDED, 0, err))

GIni *gini_parse_data_full(const gchar *path,
                           gchar *data,
                           const gchar **allowed_sections,
                           const gchar **allowed_keys,
                           GIniOrderType order_type,
                           GIniFlags flags);

#define gini_parse_data(path, data) \
            (gini_parse_data_full(path, data, NULL, NULL, GINI_ORDER_ADDED, 0))

void gini_write(GIni *ini, FILE *fp);

gboolean gini_save(GIni *ini, const gchar *path, GError **err);

gboolean gini_has_section(GIni *ini, const gchar *section_name);
gboolean gini_has_entry(GIni *ini, const gchar *section_name, const gchar *key);

GIniSection *gini_section_new(const gchar *section_name);
void gini_section_free(GIniSection *sec);

GIniSection *gini_section_new_from_string(const gchar *str);

G_CONST_RETURN gchar *gini_section_get_name(GIniSection *sec);

gboolean gini_section_add(GIni *ini, GIniSection *sec);
GIniSection *gini_section_add_value(GIni *ini, const gchar *section_name);

gboolean gini_section_del(GIni *ini, const gchar *section_name);
#define gini_section_remove gini_section_del

GIniSection *gini_section_lookup(GIni *ini, const gchar *section_name);

G_CONST_RETURN GIni *gini_section_get_ini(GIniSection *sec);

guint gini_section_get_entry_count(GIni *ini, const gchar *section_name);

GIniEntry *gini_entry_new(const gchar *key, const gchar *val);
GIniEntry *gini_entry_new_with_type(const gchar *key,
                                    const gpointer val,
                                    GType type);
void gini_entry_free(GIniEntry *e);

GIniEntry *gini_entry_new_from_string(const gchar *str);

GIniEntry *gini_entry_add_values(GIni *ini,
                                 GIniSection *sec,
                                 const gchar *key,
                                 const gchar *val);

gboolean gini_entry_add(GIni *ini, GIniSection *sec, GIniEntry *e);
gboolean gini_entry_del(GIni *ini, GIniSection *sec, const gchar *key);

gboolean gini_entry_remove(GIni *ini,
                           const gchar *section_name,
                           const gchar *key);

G_CONST_RETURN GIni *gini_entry_get_ini(GIniEntry *e);
G_CONST_RETURN GIniSection *gini_entry_get_section(GIniEntry *e);
G_CONST_RETURN gchar *gini_entry_get_section_name(GIniEntry *e);

G_CONST_RETURN gchar *gini_entry_get_key(GIniEntry *e);
G_CONST_RETURN GValue *gini_entry_get_val(GIniEntry *e);
GType gini_entry_get_val_type(GIniEntry *e);

GIniEntry *gini_entry_lookup(GIni *ini, GIniSection *sec, const gchar *key);

/* one-time, non-destructive transform (original value unchanged) */
gboolean gini_entry_get_val_as_type(GIni *ini,
                                    const gchar *section_name,
                                    const gchar *key,
                                    GType type,
                                    ...);

/* destructive transforms (permanent conversion) for cross-types calls */
gboolean gini_entry_get_as_type(GIni *ini,
                                const gchar *section_name,
                                const gchar *key,
                                GType type,
                                ...);

#define gini_entry_get_pointer(i, s, k, v) \
            (gini_entry_get_as_type(i, s, k, G_TYPE_POINTER, v))
#define gini_entry_get_string(i, s, k, v) \
            (gini_entry_get_as_type(i, s, k, G_TYPE_STRING, v))
#define gini_entry_get_boolean(i, s, k, v) \
            (gini_entry_get_as_type(i, s, k, G_TYPE_BOOLEAN, v))
#define gini_entry_get_int(i, s, k, v) \
            (gini_entry_get_as_type(i, s, k, G_TYPE_INT, v))
#define gini_entry_get_uint(i, s, k, v) \
            (gini_entry_get_as_type(i, s, k, G_TYPE_UINT, v))
#define gini_entry_get_long(i, s, k, v) \
            (gini_entry_get_as_type(i, s, k, G_TYPE_LONG, v))
#define gini_entry_get_ulong(i, s, k, v) \
            (gini_entry_get_as_type(i, s, k, G_TYPE_ULONG, v))
#define gini_entry_get_int64(i, s, k, v) \
            (gini_entry_get_as_type(i, s, k, G_TYPE_INT64, v))
#define gini_entry_get_uint64(i, s, k, v) \
            (gini_entry_get_as_type(i, s, k, G_TYPE_UINT64, v))
#define gini_entry_get_float(i, s, k, v) \
            (gini_entry_get_as_type(i, s, k, G_TYPE_FLOAT, v))
#define gini_entry_get_double(i, s, k, v) \
            (gini_entry_get_as_type(i, s, k, G_TYPE_DOUBLE, v))

/* destructive transforms (permanent conversion) for cross-types calls */
void gini_entry_set_as_type(GIni *ini,
                            const gchar *section_name,
                            const gchar *key,
                            GType type,
                            ...);

#define gini_entry_set_pointer(i, s, k, v) \
            (gini_entry_set_as_type(i, s, k, G_TYPE_POINTER, v))
#define gini_entry_set_string(i, s, k, v) \
            (gini_entry_set_as_type(i, s, k, G_TYPE_STRING, v))
#define gini_entry_set_boolean(i, s, k, v) \
            (gini_entry_set_as_type(i, s, k, G_TYPE_BOOLEAN, v))
#define gini_entry_set_int(i, s, k, v) \
            (gini_entry_set_as_type(i, s, k, G_TYPE_INT, v))
#define gini_entry_set_uint(i, s, k, v) \
            (gini_entry_set_as_type(i, s, k, G_TYPE_UINT, v))
#define gini_entry_set_long(i, s, k, v) \
            (gini_entry_set_as_type(i, s, k, G_TYPE_LONG, v))
#define gini_entry_set_ulong(i, s, k, v) \
            (gini_entry_set_as_type(i, s, k, G_TYPE_ULONG, v))
#define gini_entry_set_int64(i, s, k, v) \
            (gini_entry_set_as_type(i, s, k, G_TYPE_INT64, v))
#define gini_entry_set_uint64(i, s, k, v) \
            (gini_entry_set_as_type(i, s, k, G_TYPE_UINT64, v))
#define gini_entry_set_float(i, s, k, v) \
            (gini_entry_set_as_type(i, s, k, G_TYPE_FLOAT, v))
#define gini_entry_set_double(i, s, k, v) \
            (gini_entry_set_as_type(i, s, k, G_TYPE_DOUBLE, v))

G_CONST_RETURN gchar *gini_get_version(void);

#define GINI_MAJOR_VERSION 0
#define GINI_MINOR_VERSION 4
#define GINI_MICRO_VERSION 1

#define GINI_CHECK_VERSION(major, minor, micro) \
  (GINI_MAJOR_VERSION > (major) || \
   (GINI_MAJOR_VERSION == (major) && GINI_MINOR_VERSION > (minor)) || \
   (GINI_MAJOR_VERSION == (major) && GINI_MINOR_VERSION == (minor) && \
    GINI_MICRO_VERSION >= (micro)))

G_END_DECLS

#endif /*__gini_h */
