/*
 * mdh (MailDooHicky), a GTK+-based toolbar.
 *
 * Copyright (c) 2003-2005 by Mike Hokenson <mdh at gozer dot org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <unistd.h>

#include <gtk/gtk.h>

#ifdef HAVE_CURL_CURL_H
# include <curl/curl.h>
# include <curl/types.h>
# include <curl/easy.h>
#endif

#include "mdh.h"

#include "util.h"

#ifdef HAVE_CURL_CURL_H

#if DEBUG
# define _CURL_VERBOSE 1
#else
# define _CURL_VERBOSE 0
#endif

#define _CURL_MAX_SIZE 1048576

static gint curl_get_write(gpointer ptr,
                           size_t size,
                           size_t nmemb,
                           gpointer data)
{
    GString *string = data;
    
    g_return_val_if_fail(ptr != NULL, -1);
    g_return_val_if_fail(string != NULL, -1);

    if(string->len > _CURL_MAX_SIZE) {
        debug("curl_get_write(): max length hit");
        return(-1); /* string will be freed on ret != CURLE_OK */
    }

    g_string_append_len(string, ptr, nmemb);

    return(nmemb);
}

gboolean mdh_curl_get(const gchar *uri,
                      gint timeout,
                      gchar **result,
                      gint *result_len,
                      gint *status,
                      GError **err)
{
    gchar agent[128], error[CURL_ERROR_SIZE];

    CURL *curl;

    GString *string;

    g_return_val_if_fail(uri != NULL, FALSE);

    string = g_string_new(NULL);

    if(!(curl = curl_easy_init())) {
        g_set_error(err, 0, 0, "Failed to initialize libcurl.");
        return(FALSE);
    }

    error[0] = '\0';

    g_snprintf(agent, sizeof(agent), "MailDooHicky/%s (+%s)", PACKAGE_VERSION,
                                                              _MDH_URL);

    if(timeout < 1) {
#if (defined(G_THREADS_ENABLED) && defined(ENABLE_THREADS))
        curl_easy_setopt(curl, CURLOPT_NOSIGNAL, 1);
#endif
        curl_easy_setopt(curl, CURLOPT_TIMEOUT, 60);
    } else
        curl_easy_setopt(curl, CURLOPT_TIMEOUT, timeout);

    curl_easy_setopt(curl, CURLOPT_VERBOSE, _CURL_VERBOSE);
    curl_easy_setopt(curl, CURLOPT_USERAGENT, agent);
    curl_easy_setopt(curl, CURLOPT_FAILONERROR, 1);
    curl_easy_setopt(curl, CURLOPT_FOLLOWLOCATION, 0);
    curl_easy_setopt(curl, CURLOPT_MAXREDIRS, 3);
    curl_easy_setopt(curl, CURLOPT_NOPROGRESS, 1);
    curl_easy_setopt(curl, CURLOPT_HEADER, 0);
    curl_easy_setopt(curl, CURLOPT_NOBODY, 0);
    curl_easy_setopt(curl, CURLOPT_TRANSFERTEXT, 1);
    curl_easy_setopt(curl, CURLOPT_ERRORBUFFER, error);
    curl_easy_setopt(curl, CURLOPT_URL, uri);
    curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, curl_get_write);
    curl_easy_setopt(curl, CURLOPT_FILE, string);

    if(curl_easy_perform(curl) != CURLE_OK) {
        g_set_error(err, 0, 0, "Failed to load '%s': %s.", uri,
                               (error[0]) ? error : "empty error buffer");
        curl_easy_cleanup(curl);
        g_string_free(string, TRUE);
        return(FALSE);
    }

    if(status)
        curl_easy_getinfo(curl, CURLINFO_RESPONSE_CODE, status);

    curl_easy_cleanup(curl);

    if(result_len)
        *result_len = string->len;

    if(result)
        *result = g_string_free(string, FALSE);
    else
        g_string_free(string, TRUE);

    return(TRUE);
}

gchar *mdh_curl_escape(const gchar *str)
{
    g_return_val_if_fail(str != NULL, NULL);

    return(curl_escape(str, 0));
}

void mdh_curl_free(gpointer ptr)
{
    if(ptr)
        curl_free(ptr);
}

#endif /* HAVE_CURL_CURL_H */
