/*
 * mdh (MailDooHicky), a GTK+-based toolbar.
 *
 * Copyright (c) 2003-2005 by Mike Hokenson <mdh at gozer dot org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <unistd.h>

#include <gtk/gtk.h>

#include "toolbar.h"

#include "panel.h"

#include "error.h"
#include "threads.h"
#include "widget.h"
#include "window.h"
#include "util.h"

MdhPanel *mdh_panel_new(void)
{
    MdhPanel *obj;

    obj           = g_new0(MdhPanel, 1);

    obj->init     = FALSE;
    obj->value    = NULL;
    obj->display  = NULL;
    obj->command  = NULL;
    obj->tooltip  = NULL;
    obj->interval = 0;
    obj->handler  = 0;
    obj->enabled  = FALSE;
    obj->busy     = FALSE;
    obj->hidden   = TRUE;

    obj->data     = NULL;

    obj->flags    = 0;

    obj->funcs    = (MdhPanelFuncs) { NULL, NULL, NULL, NULL, NULL };

    obj->widgets.event_box = gtk_event_box_new();
    obj->widgets.frame     = gtk_frame_new(NULL);
    obj->widgets.label     = gtk_label_new(NULL);
    obj->widgets.icon      = gtk_image_new();

    return(obj);
}

void mdh_panel_free(MdhPanel *obj)
{
    g_return_if_fail(obj != NULL);

    g_free(obj->value);
    g_free(obj->display);
    g_free(obj->command);
    g_free(obj->tooltip);

    g_free(obj->data);

    g_free(obj);
}

void mdh_panel_start(MdhPanel *obj)
{
    g_return_if_fail(obj != NULL);
    g_return_if_fail(obj->interval >= 0);

    debug("mdh_panel_start(): %d", obj->handler);

    if(obj->handler > 0) /* existing timeout */
        return;

    obj->handler = g_timeout_add(obj->interval * 1000,
                                 (GSourceFunc) mdh_panel_execute_func,
                                 obj);

    mdh_panel_execute_func(obj);
}

/* stop the timer and close the resource(s) relating to the panel */
void mdh_panel_stop(MdhPanel *obj)
{
    g_return_if_fail(obj != NULL);

    debug("mdh_panel_stop(): %d", obj->handler);

    if(obj->handler > 0)
        g_source_remove(obj->handler);

    obj->handler = 0;

    if(obj->funcs.shutdown) {
        GError *err = NULL;

        if(!obj->funcs.shutdown(obj, &err) && err) {
            ADD_LOG("%s", err->message);
            g_error_free(err);
        }
    }
}

void mdh_panel_refresh(MdhPanel *obj)
{
    g_return_if_fail(obj != NULL);

    debug("mdh_panel_refresh(): %d", obj->handler);

    if(!obj->enabled)
        return;

    if(!obj->handler) {
        mdh_panel_stop(obj);
        mdh_panel_start(obj);
    } else if(obj->interval >= 5)
        mdh_panel_execute_func(obj);
}

void mdh_panel_toggle(MdhPanel *obj)
{
    g_return_if_fail(obj != NULL);

    if(!obj->enabled)
        return;

    if(obj->hidden)
        mdh_panel_show(obj);
    else
        mdh_panel_hide(obj);
}

void mdh_panel_show(MdhPanel *obj)
{
    g_return_if_fail(obj != NULL);

    gtk_widget_show_all(obj->widgets.event_box);

    if(!(obj->flags & MDH_PANEL_FLAG_ICON))
        gtk_widget_hide(obj->widgets.icon);

    obj->hidden = FALSE;
}

void mdh_panel_hide(MdhPanel *obj)
{
    g_return_if_fail(obj != NULL);

    gtk_widget_hide_all(obj->widgets.event_box);

    obj->hidden = TRUE;
}

void mdh_panel_reset(MdhPanel *obj)
{
    g_return_if_fail(obj != NULL);

    debug("mdh_panel_reset(): %d", obj->handler);

    if(obj->enabled) {
        mdh_panel_start(obj);
        mdh_panel_show(obj);
    } else {
        mdh_panel_stop(obj);
        mdh_panel_hide(obj);
    }
}

void mdh_panel_foreach(gpointer toolbar, GFunc func, gpointer data)
{
    g_return_if_fail(toolbar != NULL);
    g_return_if_fail(func != NULL);

    g_slist_foreach(MDH_TOOLBAR(toolbar)->panels, func, data);
}

gboolean mdh_panel_hidden(MdhPanel *obj)
{
    g_return_val_if_fail(obj != NULL, FALSE);

    return(obj->hidden);
}

static void panel_execute_func(MdhPanel *obj)
{
    gboolean ret = TRUE;

    GError *err = NULL;

    g_return_if_fail(obj != NULL);
    g_return_if_fail(obj->funcs.main != NULL);

    if(obj->busy == TRUE)
        return;

    obj->busy = TRUE;

    if(obj->funcs.startup)
        ret = obj->funcs.startup(obj, &err);

    if(ret)
        obj->funcs.main(obj, &err);

    obj->busy = FALSE;

    if(err) {
        ADD_LOG("%s", err->message);
        g_error_free(err);
    }
}

gboolean mdh_panel_execute_func(MdhPanel *obj)
{
    g_return_val_if_fail(obj != NULL, FALSE);

    if(!obj->busy)
        mdh_thread_create((GFunc) panel_execute_func, obj);
    else
        debug("panel_func(): busy (%d): %s", obj->handler, obj->value);

    return(TRUE);
}

gboolean mdh_panel_execute_callback(MdhPanel *obj, GError **err)
{
    gboolean ret;

    g_return_val_if_fail(obj != NULL, FALSE);
    g_return_val_if_fail(obj->command != NULL, FALSE);

    if(!*obj->command)
        return(TRUE);

    if(obj->funcs.callback)
        ret = obj->funcs.callback(obj, err);
    else
        ret = g_spawn_command_line_async(obj->command, err);

    return(ret);
}

MdhPanelFlags mdh_panel_flags(MdhPanel *obj)
{
    g_return_val_if_fail(obj != NULL, 0);

    return(obj->flags);
}

void mdh_panel_flags_set(MdhPanel *obj, MdhPanelFlags flags)
{
    g_return_if_fail(obj != NULL);

    obj->flags = flags;
}

void mdh_panel_set_shadow_type(MdhPanel *obj, GtkShadowType type)
{
    g_return_if_fail(obj != NULL);
    g_return_if_fail(obj->widgets.frame != NULL);

    gtk_frame_set_shadow_type(GTK_FRAME(obj->widgets.frame), type);
}

void mdh_panel_set_tip(MdhPanel *obj, const gchar *text)
{
    GtkTooltips *tips;

    g_return_if_fail(obj != NULL);
    g_return_if_fail(obj->toolbar != NULL);

    tips = MDH_TOOLBAR(obj->toolbar)->widgets.tips;

    g_return_if_fail(tips != NULL);

    mdh_widget_set_tip(tips, obj->widgets.event_box, text);
}

void mdh_panel_set_tip_r(MdhPanel *obj, const gchar *text)
{
    mdh_threads_enter();
    mdh_panel_set_tip(obj, text);
    mdh_threads_leave();
}

void mdh_panel_set_text_r(MdhPanel *obj, gboolean bold, const gchar *fmt, ...)
{
    gchar buf[100];  /* need some limit */

    va_list args;

    GtkLabel *label;

    g_return_if_fail(obj != NULL);
    g_return_if_fail(fmt != NULL);

    va_start(args, fmt);
    g_vsnprintf(buf, sizeof(buf), fmt, args);
    va_end(args);

    label = GTK_LABEL(obj->widgets.label);

    mdh_threads_enter();

    if(bold)
        mdh_label_set_markup(label, MDH_MARKUP_WEIGHT_BOLD, "%s", buf);
    else
        gtk_label_set_text(label, buf);
    
    mdh_threads_leave();
}

void mdh_panel_set_markup(MdhPanel *obj, const gchar *markup)
{
    g_return_if_fail(obj != NULL);
    g_return_if_fail(markup != NULL);
    
    gtk_label_set_markup(GTK_LABEL(obj->widgets.label), markup);
}

void mdh_panel_set_markup_r(MdhPanel *obj, const gchar *markup)
{
    mdh_threads_enter();
    mdh_panel_set_markup(obj, markup);
    mdh_threads_leave();
}

void mdh_panel_set_icon(MdhPanel *obj, const gchar *base)
{
    gchar path[PATH_MAX];
    
    g_return_if_fail(obj != NULL);
    g_return_if_fail(base != NULL);
    
    mdh_icon_path_r(base, path, sizeof(path));

    if(IS_FILE(path))
        gtk_image_set_from_file(GTK_IMAGE(obj->widgets.icon), path);
}

void mdh_panel_set_icon_r(MdhPanel *obj, const gchar *base)
{
    mdh_threads_enter();
    mdh_panel_set_icon(obj, base);
    mdh_threads_leave();
}

void mdh_panel_set_font_desc(MdhPanel *obj, PangoFontDescription *font_desc)
{
    g_return_if_fail(obj != NULL);
    g_return_if_fail(font_desc != NULL);

    mdh_widget_set_font_desc(obj->widgets.label, font_desc);
}

void mdh_panel_set_font_desc_r(MdhPanel *obj, PangoFontDescription *font_desc)
{
    mdh_threads_enter();
    mdh_panel_set_font_desc(obj, font_desc);
    mdh_threads_leave();
}

void mdh_panel_set_font(MdhPanel *obj, const gchar *font)
{
    g_return_if_fail(obj != NULL);
    g_return_if_fail(font != NULL);

    mdh_widget_set_font(obj->widgets.label, font);
}

void mdh_panel_set_font_r(MdhPanel *obj, const gchar *font)
{
    mdh_threads_enter();
    mdh_panel_set_font(obj, font);
    mdh_threads_leave();
}
