/*
 * mdh (MailDooHicky), a GTK+-based toolbar.
 *
 * Copyright (c) 2003-2005 by Mike Hokenson <mdh at gozer dot org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <unistd.h>

#include <gtk/gtk.h>

#include "mdh.h"

#include "panel.h"

#include "panel_cpu.h"

#include "util.h"

#include "sysdeps.h"

#if _MDH_HAS_CPU

#define SHOW_BOLD(p, v) \
    (mdh_panel_flags_bold(p) && mdh_panel_cpu_threshold(p) > 0 && \
     v >= mdh_panel_cpu_threshold(p))

typedef struct {
    gdouble loadavg[3];

    guint64 nr_running;
    guint64 nr_tasks;
} MdhDataCache;

typedef struct {
    gint threshold;
} MdhPanelData;

static gchar *panel_subst(MdhDataCache *cache, const gchar *fmt);

static gboolean panel_startup(gpointer data, GError **err)
{
    MdhPanel *obj = data;

    g_return_val_if_fail(obj != NULL, FALSE);
    g_return_val_if_fail(obj->display != NULL, FALSE);
    g_return_val_if_fail(obj->interval >= _CPU_MIN, FALSE);
    g_return_val_if_fail(obj->interval <= _CPU_MAX, FALSE);

    if(obj->init)
        return(TRUE);

    debug("panel_startup(): %d", obj->handler);

    if(!mdh_sys_init(err)) {
        mdh_panel_stop(obj);
        mdh_panel_set_text_r(obj, FALSE, "Error");
        return(FALSE);
    }

    obj->init = TRUE;

    return(TRUE);
}

static gboolean panel_shutdown(gpointer data, GError **err)
{
    MdhPanel *obj = data;

    g_return_val_if_fail(obj != NULL, FALSE);

    if(!obj->init)
        return(FALSE);

    debug("panel_shutdown(): %d", obj->handler);

    mdh_sys_close();

    obj->init = FALSE;

    return(FALSE);
}

static gboolean panel_main(gpointer data, GError **err)
{
    MdhPanel *obj = data;

    MdhDataCache cache;

    gdouble loadavg[3];
    guint64 nr_running, nr_tasks;

    gboolean bold;

    g_return_val_if_fail(obj != NULL, FALSE);

    debug("panel_main(): %d", obj->handler);

    if(!panel_startup(obj, err))
        return(FALSE);

    if(!mdh_sys_get_cpu(loadavg, &nr_running, &nr_tasks, err)) {
        mdh_panel_stop(obj);
        mdh_panel_set_text_r(obj, FALSE, "N/A");
        return(FALSE);
    }

    bold = SHOW_BOLD(obj, (loadavg[0] * 100) + .5);

    cache.loadavg[0] = loadavg[0];
    cache.loadavg[1] = loadavg[1];
    cache.loadavg[2] = loadavg[2];

    cache.nr_running = nr_running;
    cache.nr_tasks   = nr_tasks;

    {
        gchar *p = panel_subst(&cache, obj->display);
        mdh_panel_set_text_r(obj, bold, "%s", p);
        g_free(p);
    }

    return(TRUE);
}

/*
 * %p: load percentage (1 min): 43%
 * %1:  1 minute average: 0.43
 * %5:  5 minute average: 0.54
 * %F: 15 minute average: 0.44
 * %t: tasks running (gtop only): 1
 * %T: tasks total (gtop only): 182
 */

static gchar *panel_subst(MdhDataCache *cache, const gchar *fmt)
{
    const gchar *p;

    GString *string;

    gdouble load_p;

    g_return_val_if_fail(cache != NULL, NULL);
    g_return_val_if_fail(cache->loadavg != NULL, NULL);
    g_return_val_if_fail(fmt != NULL, NULL);

    string = g_string_new(NULL);

    if((load_p = (cache->loadavg[0] * 100)) > 100)
        load_p = 100;

    for(p = fmt; *p; p++) {
        if(*p == '%' && *(p + 1) && p++) {
            switch(*p) {
                case 'p': /* load percentage (1 min): 43% */
                    g_string_append_printf(string, "%3.0f%%", load_p);
                    break;

                case '1': /*  1 minute average: 0.43 */
                    g_string_append_printf(string, "%0.2f", cache->loadavg[0]);
                    break;
                case '5': /*  5 minute average: 0.54 */
                    g_string_append_printf(string, "%0.2f", cache->loadavg[1]);
                    break;
                case 'F': /* 15 minute average: 0.44 */
                    g_string_append_printf(string, "%0.2f", cache->loadavg[2]);
                    break;

                case 't': /* tasks running (gtop only): 1 */
                    g_string_append_printf(string, "%llu", cache->nr_running);
                    break;
                case 'T': /* tasks total (gtop only): 182 */
                    g_string_append_printf(string, "%llu", cache->nr_tasks);
                    break;

                default:
                    g_string_append_c(string, *p);
                    break;
            }
        } else
            g_string_append_c(string, *p);
    }

    return(g_string_free(string, FALSE));
}

gint mdh_panel_cpu_threshold(MdhPanel *obj)
{
    g_return_val_if_fail(obj != NULL, 0);
    g_return_val_if_fail(obj->data != NULL, 0);

    return(((MdhPanelData *)obj->data)->threshold);
}

void mdh_panel_cpu_threshold_set(MdhPanel *obj, gint threshold)
{
    g_return_if_fail(obj != NULL);
    g_return_if_fail(obj->data != NULL);

    ((MdhPanelData *)obj->data)->threshold = threshold;
}

static MdhPanelFuncs funcs = {
    panel_main,
    panel_startup,
    panel_shutdown,
    NULL,
    NULL
};

MdhPanel *mdh_panel_cpu_new(const gchar *display,
                            const gchar *command,
                            gboolean enabled)
{
    MdhPanel *obj = mdh_panel_new();

    g_return_val_if_fail(obj != NULL, NULL);

    obj->display  = (display) ? g_strdup(display) : g_strdup(_CPU_DIS);
    obj->command  = (command) ? g_strdup(command) :
                                g_strdup_printf("%s -e top", mdh_get_xterm());
    obj->tooltip  = g_strdup("Cpu");
    obj->interval = _CPU_DEF;
    obj->enabled  = enabled;
    obj->funcs    = funcs;

    obj->data     = g_new(MdhPanelData, 1);

    ((MdhPanelData *) obj->data)->threshold = 0;

    return(obj);
}

#endif /* _MDH_HAS_CPU */
