/*
 * mdh (MailDooHicky), a GTK+-based toolbar.
 *
 * Copyright (c) 2003-2005 by Mike Hokenson <mdh at gozer dot org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <unistd.h>

#include <gtk/gtk.h>

#include "mdh.h"

#include "panel.h"

#include "panel_custom.h"

#include "util.h"

typedef struct {
    gint count;
} MdhPanelData;

static gboolean panel_main(gpointer data, GError **err)
{
    MdhPanel *obj = data;

    gchar *output = NULL;

    g_return_val_if_fail(obj != NULL, FALSE);
    g_return_val_if_fail(obj->value != NULL, FALSE);
    g_return_val_if_fail(obj->interval >= _CUSTOM_MIN, FALSE);
    g_return_val_if_fail(obj->interval <= _CUSTOM_MAX, FALSE);

    debug("panel_main(): %d: %s", obj->handler, obj->value);

    if(!*obj->value) {
        g_set_error(err, 0, 0, "Failed to process 'custom': Empty value.");
        mdh_panel_stop(obj);
        mdh_panel_set_text_r(obj, FALSE, "Error");
        return(FALSE);
    }

    if(!g_spawn_command_line_sync(obj->value, &output, NULL, NULL, err)) {
        mdh_panel_set_text_r(obj, FALSE, "N/A");
        return(FALSE);
    }

    {
        const gchar *p = g_strstrip(output);

        if(pango_parse_markup(p, -1, 0, NULL, NULL, NULL, err))
            mdh_panel_set_markup_r(obj, p);
        else
            mdh_panel_set_text_r(obj, FALSE, "%s", p);
    }

    g_free(output);

    return(TRUE);
}

gint mdh_panel_custom_count(MdhPanel *obj)
{
    g_return_val_if_fail(obj != NULL, 0);
    g_return_val_if_fail(obj->data != NULL, 0);

    return(((MdhPanelData *)obj->data)->count);
}

void mdh_panel_custom_count_set(MdhPanel *obj, gint count)
{
    g_return_if_fail(obj != NULL);
    g_return_if_fail(obj->data != NULL);
    
    ((MdhPanelData *)obj->data)->count = count;
}

static void panel_config(const gchar *v_in, gchar **v_out,
                         const gchar *d_in, gchar **d_out,
                         const gchar *c_in, gchar **c_out,
                         const gint   i_in, gint   *i_out)
{
    g_return_if_fail(v_in != NULL);
    g_return_if_fail(v_out != NULL);
    g_return_if_fail(c_in != NULL);
    g_return_if_fail(c_out != NULL);
    g_return_if_fail(i_out != NULL);

    *v_out = mdh_expand_tilde(v_in);
    *c_out = mdh_expand_tilde(c_in);
    *i_out = i_in;
}

static MdhPanelFuncs funcs = {
    panel_main,
    NULL,
    NULL,
    NULL,
    panel_config
};

MdhPanel *mdh_panel_custom_new(const gchar *value,
                               const gchar *command,
                               gboolean enabled)
{
    MdhPanel *obj = mdh_panel_new();

    g_return_val_if_fail(obj != NULL, NULL);

    obj->value    = (value)   ? g_strdup(value)   : g_strdup("");
    obj->command  = (command) ? g_strdup(command) : g_strdup("");
    obj->tooltip  = g_strdup_printf("Custom: %s", obj->value);
    obj->interval = _CUSTOM_DEF;
    obj->enabled  = (enabled && value) ? TRUE : FALSE;
    obj->funcs    = funcs;

    obj->data     = g_new(MdhPanelData, 1);

    ((MdhPanelData *) obj->data)->count = 0;

    return(obj);
}
