/*
 * mdh (MailDooHicky), a GTK+-based toolbar.
 *
 * Copyright (c) 2003-2005 by Mike Hokenson <mdh at gozer dot org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <unistd.h>

#include <gtk/gtk.h>

#include "mdh.h"

#include "panel.h"

#include "panel_disk.h"

#include "util.h"

#include "sysdeps.h"

#if _MDH_HAS_DISK

#define SHOW_BOLD(p, v) \
    (mdh_panel_flags_bold(p) && mdh_panel_disk_threshold(p) > 0 && \
     v >= mdh_panel_disk_threshold(p))

typedef struct {
    const gchar *disk_name;

    guint64      disk_total;

    guint64      disk_free;
    gdouble      disk_free_p;
} MdhDataCache;

typedef struct {
    gint threshold;
    gint count;
} MdhPanelData;

static gchar *panel_subst(MdhDataCache *cache, const gchar *fmt);

static gboolean panel_startup(gpointer data, GError **err)
{
    MdhPanel *obj = data;

    g_return_val_if_fail(obj != NULL, FALSE);
    g_return_val_if_fail(obj->value != NULL, FALSE);
    g_return_val_if_fail(obj->display != NULL, FALSE);
    g_return_val_if_fail(obj->interval >= _DISK_MIN, FALSE);
    g_return_val_if_fail(obj->interval <= _DISK_MAX, FALSE);

    if(obj->init)
        return(TRUE);

    debug("panel_startup(): %d", obj->handler);

    if(!*obj->value) {
        g_set_error(err, 0, 0, "Failed to process 'disk': Empty value.");
        goto panel_startup_fail;
    }

    if(!IS_DIR(obj->value)) {
        g_set_error(err, 0, 0, "Failed to process 'disk': No such directory.");
        goto panel_startup_fail;
    }

    if(!mdh_sys_init(err))
        goto panel_startup_fail;

    obj->init = TRUE;

    return(TRUE);

panel_startup_fail:
    mdh_panel_stop(obj);

    mdh_panel_set_text_r(obj, FALSE, "Error");

    return(FALSE);
}

static gboolean panel_shutdown(gpointer data, GError **err)
{
    MdhPanel *obj = data;

    g_return_val_if_fail(obj != NULL, FALSE);

    if(!obj->init)
        return(FALSE);

    debug("panel_shutdown(): %d", obj->handler);

    mdh_sys_close();

    obj->init = FALSE;

    return(FALSE);
}

static gboolean panel_main(gpointer data, GError **err)
{
    MdhPanel *obj = data;

    MdhDataCache cache;

    guint64 disk_total, disk_free, disk_avail;
    gdouble disk_free_p;

    gboolean bold;

    g_return_val_if_fail(obj != NULL, FALSE);
    g_return_val_if_fail(obj->value != NULL, FALSE);

    debug("panel_main(): %d: %s", obj->handler, obj->value);

    if(!panel_startup(obj, err))
        return(FALSE);

    if(!mdh_sys_get_disk(obj->value,
                         &disk_total,
                         &disk_free,
                         &disk_avail,
                         err)) {
        disk_total = 0;
        disk_free  = 0;
        disk_avail = 0;
    }

    disk_free_p = 100 - ((disk_total - disk_free) * 100 /
                         (disk_total - disk_free + disk_avail) + .5);

    bold = SHOW_BOLD(obj, (100 - disk_free_p) + .5);

    cache.disk_name   = obj->value;
    cache.disk_total  = disk_total;
    cache.disk_free   = disk_free;
    cache.disk_free_p = disk_free_p;

    {
        gchar *p = panel_subst(&cache, obj->display);
        mdh_panel_set_text_r(obj, bold, "%s", p);
        g_free(p);
    }

    return(TRUE);
}

/*
 * %m: mount point: /var
 * %u: used: 75.6GB
 * %U: used percent: 44%
 * %f: free: 86.1GB
 * %F: free percent: 56%
 * %t: total disk: 182GB
 */

static gchar *panel_subst(MdhDataCache *cache, const gchar *fmt)
{
    const gchar *p;

    GString *string;

    gdouble disk_total, disk_free, disk_free_p;

    g_return_val_if_fail(cache != NULL, NULL);
    g_return_val_if_fail(cache->disk_name != NULL, NULL);
    g_return_val_if_fail(fmt != NULL, NULL);

    string = g_string_new(NULL);

    disk_total  = (gdouble) cache->disk_total / 1073741824;
    disk_free   = (gdouble) cache->disk_free  / 1073741824;
    disk_free_p = cache->disk_free_p;

    for(p = fmt; *p; p++) {
        if(*p == '%' && *(p + 1) && p++) {
            switch(*p) {
                case 'm': /* mount point: /var */
                    g_string_append(string, cache->disk_name);
                    break;

                case 'u': /* used: 75.6GB */
                    if(disk_total > 0)
                        g_string_append_printf(string, "%01.1fGB",
                                                       disk_total - disk_free);
                    else
                        g_string_append(string, "---");
                    break;
                case 'U': /* used percent: 44% */
                    if(disk_total > 0)
                        g_string_append_printf(string, "%1.0f%%",
                                                       100 - disk_free_p);
                    else
                        g_string_append(string, "---");
                    break;

                case 'f': /* free: 86.1GB */
                    if(disk_total > 0)
                        g_string_append_printf(string, "%01.1fGB", disk_free);
                    else
                        g_string_append(string, "---");
                    break;
                case 'F': /* free percent: 56% */
                    if(disk_total > 0)
                        g_string_append_printf(string, "%1.0f%%", disk_free_p);
                    else
                        g_string_append(string, "---");
                    break;

                case 't': /* total disk: 182GB */
                    if(disk_total > 0)
                        g_string_append_printf(string, "%01.1fGB", disk_total);
                    else
                        g_string_append(string, "---");
                    break;

                default:
                    g_string_append_c(string, *p);
                    break;
            }
        } else
            g_string_append_c(string, *p);
    }

    return(g_string_free(string, FALSE));
}

gint mdh_panel_disk_threshold(MdhPanel *obj)
{
    g_return_val_if_fail(obj != NULL, 0);
    g_return_val_if_fail(obj->data != NULL, 0);

    return(((MdhPanelData *)obj->data)->threshold);
}

void mdh_panel_disk_threshold_set(MdhPanel *obj, gint threshold)
{
    g_return_if_fail(obj != NULL);
    g_return_if_fail(obj->data != NULL);

    ((MdhPanelData *)obj->data)->threshold = threshold;
}

gint mdh_panel_disk_count(MdhPanel *obj)
{
    g_return_val_if_fail(obj != NULL, 0);
    g_return_val_if_fail(obj->data != NULL, 0);

    return(((MdhPanelData *)obj->data)->count);
}

void mdh_panel_disk_count_set(MdhPanel *obj, gint count)
{
    g_return_if_fail(obj != NULL);
    g_return_if_fail(obj->data != NULL);
    
    ((MdhPanelData *)obj->data)->count = count;
}

static gboolean panel_callback(gpointer data, GError **err)
{
    MdhPanel *obj = data;

    gchar *command;

    gboolean ret;

    g_return_val_if_fail(obj != NULL, FALSE);
    g_return_val_if_fail(obj->command != NULL, FALSE);

    if(strstr(obj->command, " %s"))
        command = g_strdup_printf(obj->command, obj->value);
    else
        command = obj->command;

    ret = g_spawn_command_line_async(command, err);

    if(command != obj->command)
        g_free(command);

    return(ret);
}

static MdhPanelFuncs funcs = {
    panel_main,
    panel_startup,
    panel_shutdown,
    panel_callback,
    NULL
};

MdhPanel *mdh_panel_disk_new(const gchar *value,
                             const gchar *display,
                             const gchar *command,
                             gboolean enabled)
{
    MdhPanel *obj = mdh_panel_new();

    g_return_val_if_fail(obj != NULL, NULL);

    obj->value    = (value)   ? g_strdup(value)   : g_strdup("");
    obj->display  = (display) ? g_strdup(display) : g_strdup(_DISK_DIS);
    obj->command  = (command) ? g_strdup(command) : g_strdup("gmc %s");
    obj->tooltip  = g_strdup_printf("Disk: %s", obj->value);
    obj->interval = _DISK_DEF;
    obj->enabled  = (enabled && value && IS_DIR(value)) ? TRUE : FALSE;
    obj->funcs    = funcs;

    obj->data     = g_new(MdhPanelData, 1);

    ((MdhPanelData *) obj->data)->threshold = 0;
    ((MdhPanelData *) obj->data)->count     = 0;

    return(obj);
}

#endif /* _MDH_HAS_DISK */
