/*
 * mdh (MailDooHicky), a GTK+-based toolbar.
 *
 * Copyright (c) 2003-2005 by Mike Hokenson <mdh at gozer dot org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include <gtk/gtk.h>

#include "mdh.h"

#include "panel.h"

#include "panel_mail.h"

#include "util.h"

#define SHOW_BOLD(p, v) \
    (mdh_panel_flags_bold(p) && mdh_panel_mail_threshold(p) > 0 && \
     v >= mdh_panel_mail_threshold(p))

typedef enum {
    MAIL_TYPE_NONE,
    MAIL_TYPE_BOX,
    MAIL_TYPE_DIR
} MdhMailType;

typedef struct {
    const gchar *mail_box;

    gint         mail_total;
    gint         mail_new;
} MdhDataCache;

typedef struct {
    gint          threshold;

    FILE         *fp;

    GDir         *dp_new;
    GDir         *dp_cur;

    gchar        *md_new;
    gchar        *md_cur;

    gulong        inode;

    MdhMailType   type;
    time_t        last;

    gint          count;

    MdhDataCache  cache;
} MdhPanelData;

static gboolean panel_get_mail(MdhPanel *obj,
                               gint *total,
                               gint *new,
                               GError **err);

static gchar *panel_subst(MdhDataCache *cache, const gchar *fmt);

static gboolean panel_startup(gpointer data, GError **err)
{
    MdhPanel *obj = data;

    MdhPanelData *pdata;

    g_return_val_if_fail(obj != NULL, FALSE);
    g_return_val_if_fail(obj->value != NULL, FALSE);
    g_return_val_if_fail(obj->display != NULL, FALSE);
    g_return_val_if_fail(obj->interval >= _MAIL_MIN, FALSE);
    g_return_val_if_fail(obj->interval <= _MAIL_MAX, FALSE);

    if(obj->init)
        return(TRUE);

    debug("panel_startup(): %d: %s", obj->handler, obj->value);

    g_return_val_if_fail(obj->value != NULL, FALSE);

    if(!*obj->value) {
        g_set_error(err, 0, 0, "Failed to process 'mail': Empty value.");
        goto panel_startup_fail;
    }

    pdata = obj->data;

    pdata->type  = MAIL_TYPE_NONE;
    pdata->last  = 0;
    pdata->inode = 0;

    {
      MdhDataCache *cache = &pdata->cache;

      cache->mail_box   = obj->value;
      cache->mail_total = 0;
      cache->mail_new   = 0;
    }

    if(IS_FILE(obj->value)) {
        struct stat st;

        stat(obj->value, &st);

        /* track inode for file replacement/changes */
        pdata->inode = st.st_ino;

        pdata->type  = MAIL_TYPE_BOX;

        if(!(pdata->fp = fopen(obj->value, "r"))) {
            g_set_error(err, 0, 0, "Failed to open: %s: %s.", obj->value,
                                   g_strerror(errno));
            goto panel_startup_fail;
        }
    } else if(IS_DIR(obj->value)) {
        pdata->type = MAIL_TYPE_DIR;

        pdata->md_cur = g_build_filename(obj->value, "cur", NULL);
        pdata->md_new = g_build_filename(obj->value, "new", NULL);

        if(!IS_DIR(pdata->md_cur) || !IS_DIR(pdata->md_new)) {
            g_set_error(err, 0, 0, "Failed to open '%s': invalid maildir.",
                                   obj->value);
            g_free(pdata->md_cur);
            g_free(pdata->md_new);
            goto panel_startup_fail;
        }

        if(!(pdata->dp_cur = g_dir_open(pdata->md_cur, 0, err)) ||
           !(pdata->dp_new = g_dir_open(pdata->md_new, 0, err))) {
            if(pdata->dp_cur)
                g_dir_close(pdata->dp_cur);
            if(pdata->dp_new)
                g_dir_close(pdata->dp_new);

            g_free(pdata->md_cur);
            g_free(pdata->md_new);

            goto panel_startup_fail;
        }
    } else {
        /* this may be a problem for mail clients which removes the mailbox
           after all messages have been deleted */
        /*return(FALSE);*/
    }

    obj->init = TRUE;

    return(TRUE);

panel_startup_fail:
    mdh_panel_stop(obj);

    mdh_panel_set_text_r(obj, FALSE, "Error");

    return(FALSE);
}

static gboolean panel_shutdown(gpointer data, GError **err)
{
    MdhPanel *obj = data;

    MdhPanelData *pdata;

    g_return_val_if_fail(obj != NULL, FALSE);

    if(!obj->init)
        return(FALSE);

    debug("panel_shutdown(): %d: %s", obj->handler, obj->value);

    pdata = obj->data;

    if(pdata->type == MAIL_TYPE_DIR) {
        g_dir_close(pdata->dp_cur);
        g_dir_close(pdata->dp_new);

        g_free(pdata->md_cur);
        g_free(pdata->md_new);
    } else if(pdata->type == MAIL_TYPE_BOX)
        fclose(pdata->fp);

    obj->init = FALSE;

    return(FALSE);
}

static gboolean panel_main(gpointer data, GError **err)
{
    MdhPanel *obj = data;

    MdhPanelData *pdata;

    gint mail_total = 0, mail_new = 0;

    gboolean bold;

    g_return_val_if_fail(obj != NULL, FALSE);

    debug("panel_main(): %d: %s", obj->handler, obj->value);

    pdata = obj->data;

    if(!obj->init || pdata->type == MAIL_TYPE_NONE) {
        obj->init = FALSE;

        if(!panel_startup(obj, err))
            return(FALSE);
    }

    if(!panel_get_mail(obj, &mail_total, &mail_new, err))
        if(pdata->type == MAIL_TYPE_DIR) {
            mail_total = -1;
            mail_new   = -1;
        }

    bold = SHOW_BOLD(obj, mail_new);

    {
      MdhDataCache *cache = &pdata->cache;

      cache->mail_total = mail_total;
      cache->mail_new   = mail_new;
    }

    {
        gchar *p = panel_subst(&pdata->cache, obj->display);
        mdh_panel_set_text_r(obj, bold, "%s", p);
        g_free(p);
    }

    return(TRUE);
}

/* bits taken from gkrellm-mailwatch */
static gboolean mdh_mail_valid_from(const gchar *str)
{
    gint day;

    g_return_val_if_fail(str != NULL, FALSE);

    if(sscanf(str, "From %*s %*s %*s %d", &day) != 1)
        if(sscanf(str, "From %*s %*s %d", &day) != 1)
            return(FALSE);

    if(day < 1 || day > 31)
        return(FALSE);

    return(TRUE);
}

static gboolean panel_mail_box_message_is_internal(const gchar *str)
{
    g_return_val_if_fail(str != NULL, FALSE);

    if(!strncmp(str, "From: Mail System Internal Data", 31))
        return(TRUE);

    return(FALSE);
}

static gboolean panel_mail_box_message_is_read(const gchar *str)
{
    g_return_val_if_fail(str != NULL, FALSE);

    if(!strncmp(str, "Status: ", 8) && strchr(str + 8, 'R'))
        return(TRUE);

    return(FALSE);
}

static gboolean panel_get_mail_box(MdhPanel *obj,
                                   gint *total,
                                   gint *new,
                                   GError **err)
{
    gint mail_total, mail_read, mail_new;

    MdhPanelData *pdata;

    struct stat st;

    time_t last;

    g_return_val_if_fail(obj != NULL, FALSE);
    g_return_val_if_fail(total != NULL, FALSE);
    g_return_val_if_fail(new != NULL, FALSE);

    debug("panel_get_mail_box(): %d: %s", obj->handler, obj->value);

    pdata = obj->data;

    stat(obj->value, &st);

    /* empty file */
    if(!st.st_size) {
        *total = 0;
        *new   = 0;

        return(TRUE);
    }

    /* inode has changed, close panel resources and re-run */
    if(pdata->inode != st.st_ino) {
        panel_shutdown(obj, err);
        mdh_panel_refresh(obj);
        return(TRUE);
    }

    pdata->inode = st.st_ino;

    last = st.st_mtime;

    if(st.st_ctime > last)
        last = st.st_ctime;

    /* see if modification time of file has changed */
    if(last <= pdata->last) {
        *total = ((MdhDataCache)pdata->cache).mail_total;
        *new   = ((MdhDataCache)pdata->cache).mail_new;

        return(TRUE);
    }

    mail_total = 0;
    mail_read  = 0;

    rewind(pdata->fp);
    fflush(pdata->fp);

    {
        gchar buf[LINE_MAX];

        gboolean is_header = FALSE;

        /* bits taken from gkrellm-mailwatch */
        while((fgets(buf, sizeof(buf), pdata->fp))) {
            if(*buf == '\n')
                is_header = FALSE;

            if(!is_header && *buf == 'F') {
                if(!strncmp(buf, "From ", 5)) {
                    if(mdh_mail_valid_from(buf)) {
                        mail_total++;
                        is_header = TRUE;
                    }
                }
            }

            if(!is_header)
                continue;

            if(panel_mail_box_message_is_internal(buf))
                mail_total--;
            else if(panel_mail_box_message_is_read(buf))
                mail_read++;
        }
    }

    if(mail_total < 0)
        mail_total = 0;

    if((mail_new = mail_total - mail_read) < 0)
        mail_read  = 0;

    if(mail_new < 0)
        mail_new = 0;

    *total = mail_total;
    *new   = mail_new;

    /* update cached data */
    pdata->last = last;

    return(TRUE);
}

static gboolean panel_get_mail_dir_count(MdhPanel *obj,
                                         GDir *dp,
                                         gint *count,
                                         GError **err)
{
    gint i = 0;

    g_return_val_if_fail(dp != NULL, FALSE);
    g_return_val_if_fail(count != NULL, FALSE);

    g_dir_rewind(dp);

    while((g_dir_read_name(dp)))
        i++;

    *count = i;

    return(TRUE);
}

static gboolean panel_get_mail_dir(MdhPanel *obj,
                                   gint *total,
                                   gint *new,
                                   GError **err)
{
    gint mail_total, mail_read, mail_new, last_cur, last_new;

    MdhPanelData *pdata;

    g_return_val_if_fail(obj != NULL, FALSE);
    g_return_val_if_fail(total != NULL, FALSE);
    g_return_val_if_fail(new != NULL, FALSE);

    debug("panel_get_mail_dir(): %d: %s", obj->handler, obj->value);

    pdata = obj->data;

    last_cur = last_new = 0;

    {
        struct stat st;

        /* get modification time */
        if(!stat(pdata->md_cur, &st))
            last_cur = st.st_mtime;

        if(!stat(pdata->md_new, &st))
            last_new = st.st_mtime;
    }

    /* see if modification time of dirs have changed */
    if((last_new <= pdata->last) && (last_cur <= pdata->last)) {
        *total = ((MdhDataCache)pdata->cache).mail_total;
        *new   = ((MdhDataCache)pdata->cache).mail_new;

        return(TRUE);
    }

    if(!panel_get_mail_dir_count(obj, pdata->dp_cur, &mail_read, err))
        return(FALSE);

    if(!panel_get_mail_dir_count(obj, pdata->dp_new, &mail_new, err))
        return(FALSE);

    mail_total = mail_read + mail_new;

    *total = mail_total;
    *new   = mail_new;

    /* update cached data */

    pdata->last = last_cur;

    if(last_new > last_cur)
        pdata->last = last_new;

    return(TRUE);
}

static gboolean panel_get_mail(MdhPanel *obj,
                               gint *total,
                               gint *new,
                               GError **err)
{
    MdhPanelData *pdata = obj->data;

    g_return_val_if_fail(obj != NULL, FALSE);
    g_return_val_if_fail(total != NULL, FALSE);
    g_return_val_if_fail(new != NULL, FALSE);

    /* mailbox or maildir? */
    if(pdata->type == MAIL_TYPE_BOX)
        return(panel_get_mail_box(obj, total, new, err));
    else if(pdata->type == MAIL_TYPE_DIR)
        return(panel_get_mail_dir(obj, total, new, err));

    if(err && !*err)
        g_set_error(err, 0, 0,
                    "Failed to process '%s': No such file or directory.",
                    obj->value);

    /* this may be a problem for mail clients which removes the mailbox
       after all messages have been deleted */
    /* mdh_panel_stop(obj); */

    return(FALSE);
}

/*
 * %m: mailbox/dir: /var/mail/logan
 * %n: new: 1
 * %r: read: 9
 * %t: total: 10
 */

static gchar *panel_subst(MdhDataCache *cache, const gchar *fmt)
{
    const gchar *p;

    GString *string;

    gint mail_total, mail_new;

    g_return_val_if_fail(cache != NULL, NULL);
    g_return_val_if_fail(cache->mail_box != NULL, NULL);
    g_return_val_if_fail(fmt != NULL, NULL);

    string = g_string_new(NULL);

    mail_total = cache->mail_total;
    mail_new   = cache->mail_new;

    for(p = fmt; *p; p++) {
        if(*p == '%' && *(p + 1) && p++) {
            switch(*p) {
                case 'm': /* mailbox/dir: /var/mail/logan */
                    g_string_append(string, cache->mail_box);
                    break;

                case 'n': /* new: 1 */
                    if(mail_total > -1 && mail_new > -1)
                        g_string_append_printf(string, "%3d", mail_new);
                    else
                        g_string_append(string, "---");
                    break;

                case 'r': /* read: 9 */
                    if(mail_total > -1 && mail_new > -1)
                        g_string_append_printf(string, "%3d",
                                               mail_total - mail_new);
                    else
                        g_string_append(string, "---");
                    break;

                case 't': /* total: 10 */
                    if(mail_total > -1 && mail_new > -1)
                        g_string_append_printf(string, "%3d", mail_total);
                    else
                        g_string_append(string, "---");
                    break;

                default:
                    g_string_append_c(string, *p);
                    break;
            }       
        } else
            g_string_append_c(string, *p);
    }

    return(g_string_free(string, FALSE));
}

gint mdh_panel_mail_threshold(MdhPanel *obj)
{
    g_return_val_if_fail(obj != NULL, 0);
    g_return_val_if_fail(obj->data != NULL, 0);

    return(((MdhPanelData *)obj->data)->threshold);
}

void mdh_panel_mail_threshold_set(MdhPanel *obj, gint threshold)
{
    g_return_if_fail(obj != NULL);
    g_return_if_fail(obj->data != NULL);

    ((MdhPanelData *)obj->data)->threshold = threshold;
}

gint mdh_panel_mail_count(MdhPanel *obj)
{
    g_return_val_if_fail(obj != NULL, 0);
    g_return_val_if_fail(obj->data != NULL, 0);

    return(((MdhPanelData *)obj->data)->count);
}

void mdh_panel_mail_count_set(MdhPanel *obj, gint count)
{
    g_return_if_fail(obj != NULL);
    g_return_if_fail(obj->data != NULL);
    
    ((MdhPanelData *)obj->data)->count = count;
}

static gboolean panel_callback(gpointer data, GError **err)
{
    MdhPanel *obj = data;

    gchar *command;

    gboolean ret;

    g_return_val_if_fail(obj != NULL, FALSE);
    g_return_val_if_fail(obj->command != NULL, FALSE);

    if(strstr(obj->command, " %s")) {
        command = g_strdup_printf(obj->command, obj->value);
    } else
        command = obj->command;

    ret = g_spawn_command_line_async(command, err);

    if(command != obj->command)
        g_free(command);

    return(ret);
}

static void panel_config(const gchar *v_in, gchar **v_out,
                         const gchar *d_in, gchar **d_out,
                         const gchar *c_in, gchar **c_out,
                         const gint   i_in, gint   *i_out)
{
    g_return_if_fail(v_in != NULL);
    g_return_if_fail(v_out != NULL);
    g_return_if_fail(d_in != NULL);
    g_return_if_fail(d_out != NULL);
    g_return_if_fail(c_in != NULL);
    g_return_if_fail(c_out != NULL);
    g_return_if_fail(i_out != NULL);

    *v_out = mdh_expand_tilde(v_in);
    *d_out = g_strdup(d_in);
    *c_out = mdh_expand_tilde(c_in);
    *i_out = i_in;
}

static MdhPanelFuncs funcs = {
    panel_main,
    panel_startup,
    panel_shutdown,
    panel_callback,
    panel_config
};

MdhPanel *mdh_panel_mail_new(const gchar *value,
                             const gchar *display,
                             const gchar *command,
                             gboolean enabled)
{
    MdhPanel *obj = mdh_panel_new();

    g_return_val_if_fail(obj != NULL, NULL);

    obj->value    = (value)   ? mdh_expand_tilde(value) : g_strdup("");
    obj->display  = (display) ? g_strdup(display)       : g_strdup(_MAIL_DIS);
    obj->command  = (command) ? g_strdup(command)       : g_strdup("balsa %s");
    obj->tooltip  = g_strdup_printf("Mail: %s", obj->value);
    obj->interval = _MAIL_DEF;
    obj->enabled  = (IS_EXIST(obj->value)) ? enabled : FALSE;
    obj->funcs    = funcs;

    obj->data     = g_new(MdhPanelData, 1);

    ((MdhPanelData *) obj->data)->threshold = 0;
    ((MdhPanelData *) obj->data)->count     = 0;

    return(obj);
}
