/*
 * mdh (MailDooHicky), a GTK+-based toolbar.
 *
 * Copyright (c) 2003-2005 by Mike Hokenson <mdh at gozer dot org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <unistd.h>

#include <gtk/gtk.h>

#include "mdh.h"

#include "panel.h"

#include "panel_mem.h"

#include "util.h"

#include "sysdeps.h"

#if _MDH_HAS_MEM

#define SHOW_BOLD(p, v) \
    (mdh_panel_flags_bold(p) && mdh_panel_mem_threshold(p) > 0 && \
     v >= mdh_panel_mem_threshold(p))

typedef struct {
    guint64 mem_total;

    guint64 mem_free;
    gdouble mem_free_p;
} MdhDataCache;

typedef struct {
    gint threshold;
} MdhPanelData;

static gchar *panel_subst(MdhDataCache *cache, const gchar *fmt);

static gboolean panel_startup(gpointer data, GError **err)
{
    MdhPanel *obj = data;

    g_return_val_if_fail(obj != NULL, FALSE);
    g_return_val_if_fail(obj->display != NULL, FALSE);
    g_return_val_if_fail(obj->interval >= _MEM_MIN, FALSE);
    g_return_val_if_fail(obj->interval <= _MEM_MAX, FALSE);

    if(obj->init)
        return(TRUE);

    debug("panel_startup(): %d", obj->handler);

    if(!mdh_sys_init(err)) {
        mdh_panel_stop(obj);
        mdh_panel_set_text_r(obj, FALSE, "Error");
        return(FALSE);
    }

    obj->init = TRUE;

    return(TRUE);
}

static gboolean panel_shutdown(gpointer data, GError **err)
{
    MdhPanel *obj = data;

    g_return_val_if_fail(obj != NULL, FALSE);

    if(!obj->init)
        return(FALSE);

    debug("panel_shutdown(): %d", obj->handler);

    mdh_sys_close();

    obj->init = FALSE;

    return(FALSE);
}

static gboolean panel_main(gpointer data, GError **err)
{
    MdhPanel *obj = data;

    MdhDataCache cache;

    guint64 mem_total, mem_free;
    gdouble mem_free_p;

    gboolean bold;

    g_return_val_if_fail(obj != NULL, FALSE);

    debug("panel_main(): %d", obj->handler);

    if(!panel_startup(obj, err))
        return(FALSE);

    if(!mdh_sys_get_mem(&mem_total, &mem_free, err)) {
        mdh_panel_stop(obj);
        mdh_panel_set_text_r(obj, FALSE, "N/A");
        return(FALSE);
    }

    mem_free_p = 100 - (((gdouble) (mem_total-mem_free) / mem_total*100) + .5);

    bold = SHOW_BOLD(obj, (100 - mem_free_p) + .5);

    cache.mem_total  = mem_total;
    cache.mem_free   = mem_free;
    cache.mem_free_p = mem_free_p;

    {
        gchar *p = panel_subst(&cache, obj->display);
        mdh_panel_set_text_r(obj, bold, "%s", p);
        g_free(p);
    }

    return(TRUE);
}

#define DVAL(v) ((gdouble) (v) / 1048576)

/*
 * %u: used: 1018MB
 * %U: used percent: 52%
 * %f: free: 973MB
 * %F: free percent: 48%
 * %t: total mem: 1991MB
 */

static gchar *panel_subst(MdhDataCache *cache, const gchar *fmt)
{
    const gchar *p;

    GString *string;

    guint64 mem_total, mem_free;
    gdouble mem_free_p;

    g_return_val_if_fail(cache != NULL, NULL);
    g_return_val_if_fail(fmt != NULL, NULL);

    string = g_string_new(NULL);

    mem_total  = cache->mem_total;
    mem_free   = cache->mem_free;
    mem_free_p = cache->mem_free_p;

    for(p = fmt; *p; p++) {
        if(*p == '%' && *(p + 1) && p++) {
            switch(*p) {
                case 'u': /* used: 1018MB */
                    g_string_append_printf(string, "%1.0fMB",
                                                   DVAL(mem_total - mem_free));
                    break;
                case 'U': /* used percent: 52% */
                    g_string_append_printf(string, "%01.0f%%",
                                                   100 - mem_free_p);
                    break;

                case 'f': /* free: 973MB */
                    g_string_append_printf(string, "%1.0fMB", DVAL(mem_free));
                    break;
                case 'F': /* free percent: 48% */
                    g_string_append_printf(string, "%01.0f%%", mem_free_p);
                    break;

                case 't': /* total mem: 1991MB */
                    g_string_append_printf(string, "%1.0fMB", DVAL(mem_total));
                    break;

                default:
                    g_string_append_c(string, *p);
                    break;
            }
        } else
            g_string_append_c(string, *p);
    }

    return(g_string_free(string, FALSE));
}

gint mdh_panel_mem_threshold(MdhPanel *obj)
{
    g_return_val_if_fail(obj != NULL, 0);
    g_return_val_if_fail(obj->data != NULL, 0);

    return(((MdhPanelData *)obj->data)->threshold);
}

void mdh_panel_mem_threshold_set(MdhPanel *obj, gint threshold)
{
    g_return_if_fail(obj != NULL);
    g_return_if_fail(obj->data != NULL);

    ((MdhPanelData *)obj->data)->threshold = threshold;
}

static MdhPanelFuncs funcs = {
    panel_main,
    panel_startup,
    panel_shutdown,
    NULL,
    NULL
};

MdhPanel *mdh_panel_mem_new(const gchar *display,
                            const gchar *command,
                            gboolean enabled)
{
    MdhPanel *obj = mdh_panel_new();

    g_return_val_if_fail(obj != NULL, NULL);

    obj->display  = (display) ? g_strdup(display) : g_strdup(_MEM_DIS);
    obj->command  = (command) ? g_strdup(command) : g_strdup("");
    obj->tooltip  = g_strdup("Mem");
    obj->interval = _MEM_DEF;
    obj->enabled  = enabled;
    obj->funcs    = funcs;

    obj->data     = g_new(MdhPanelData, 1);

    ((MdhPanelData *) obj->data)->threshold = 0;

    return(obj);
}

#endif /* _MDH_HAS_MEM */
