/*
 * mdh (MailDooHicky), a GTK+-based toolbar.
 *
 * Copyright (c) 2003-2005 by Mike Hokenson <mdh at gozer dot org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <unistd.h>

#include <gtk/gtk.h>

#include "mdh.h"

#include "panel.h"

#include "panel_uptime.h"

#include "util.h"

#include "sysdeps.h"

#if _MDH_HAS_UPTIME

typedef struct {
    gdouble seconds;
} MdhDataCache;

static gchar *panel_subst(MdhDataCache *cache, const gchar *fmt);

static gboolean panel_startup(gpointer data, GError **err)
{
    MdhPanel *obj = data;

    g_return_val_if_fail(obj != NULL, FALSE);
    g_return_val_if_fail(obj->display != NULL, FALSE);
    g_return_val_if_fail(obj->interval >= _UPTIME_MIN, FALSE);
    g_return_val_if_fail(obj->interval <= _UPTIME_MAX, FALSE);

    if(obj->init)
        return(TRUE);

    debug("panel_startup(): %d", obj->handler);

    if(!mdh_sys_init(err)) {
        mdh_panel_stop(obj);
        mdh_panel_set_text_r(obj, FALSE, "Error");
        return(FALSE);
    }

    obj->init = TRUE;

    return(TRUE);
}

static gboolean panel_shutdown(gpointer data, GError **err)
{
    MdhPanel *obj = data;

    g_return_val_if_fail(obj != NULL, FALSE);

    debug("panel_shutdown(): %d", obj->handler);

    if(!obj->init)
        return(FALSE);

    mdh_sys_close();

    obj->init = FALSE;

    return(FALSE);
}

static gboolean panel_main(gpointer data, GError **err)
{
    MdhPanel *obj = data;

    MdhDataCache cache;

    gulong seconds;

    g_return_val_if_fail(obj != NULL, FALSE);

    debug("panel_main(): %d", obj->handler);

    if(!panel_startup(obj, err))
        return(FALSE);

    if(!mdh_sys_get_uptime(&seconds, err)) {
        mdh_panel_stop(obj);
        mdh_panel_set_text_r(obj, FALSE, "N/A");
        return(FALSE);
    }

    cache.seconds = seconds;

    {
        gchar *p = panel_subst(&cache, obj->display);
        mdh_panel_set_text_r(obj, FALSE, "%s", p);
        g_free(p);
    }

    return(TRUE);
}

/*
 * %m: minutes: 126164:22
 * %h: hours: 2102:44
 * %d: days: 87
 * %U: uptime: 87d 14:44
 */

static gchar *panel_subst(MdhDataCache *cache, const gchar *fmt)
{
    const gchar *p;

    GString *string;

    gulong seconds;
    guint hours, minutes;

    g_return_val_if_fail(cache != NULL, NULL);
    g_return_val_if_fail(fmt != NULL, NULL);

    string = g_string_new(NULL);

    seconds = cache->seconds;

    minutes = seconds / 60;
    hours   = minutes / 60;

    for(p = fmt; *p; p++) {
        if(*p == '%' && *(p + 1) && p++) {
            switch(*p) {
                case 'm': /* minutes: 126164:22 */
                    g_string_append_printf(string, "%2d:%02d",
                                                    minutes,
                                                    (gint) seconds % 60);
                    break;

                case 'h': /* hours: 2102:44 */
                    g_string_append_printf(string, "%2d:%02d",
                                                   hours,
                                                   minutes % 60);
                    break;

                case 'd': /* days: 87 */
                    g_string_append_printf(string, "%d", hours / 24);
                    break;

                case 'U': /* uptime: 87d 14:44 */
                    g_string_append_printf(string, "%dd %2d:%02d",
                                                   hours / 24,
                                                   hours % 24,
                                                   minutes % 60);
                    break;

                default:
                    g_string_append_c(string, *p);
                    break;
            }   
        } else      
            g_string_append_c(string, *p);
    }           
                    
    return(g_string_free(string, FALSE));
}

static MdhPanelFuncs funcs = {
    panel_main,
    panel_startup,
    panel_shutdown,
    NULL,
    NULL
};

MdhPanel *mdh_panel_uptime_new(const gchar *display,
                               const gchar *command,
                               gboolean enabled)
{
    MdhPanel *obj = mdh_panel_new();

    g_return_val_if_fail(obj != NULL, NULL);

    obj->display  = (display) ? g_strdup(display) : g_strdup(_UPTIME_DIS);
    obj->command  = (command) ? g_strdup(command) : g_strdup("");
    obj->tooltip  = g_strdup("Uptime");
    obj->interval = _UPTIME_DEF;
    obj->enabled  = enabled;
    obj->funcs    = funcs;

    return(obj);
}

#endif /* _MDH_HAS_UPTIME */
