/*
 * mdh (MailDooHicky), a GTK+-based toolbar.
 *
 * Copyright (c) 2003-2005 by Mike Hokenson <mdh at gozer dot org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <stdarg.h>
#include <unistd.h>

#include <gtk/gtk.h>

#ifdef HAVE_GINI_H
#include <gini.h>
#endif

#include "config.h"
#include "scratch_pad.h"
#include "util.h"

#define _VALID_SCRATCH_PAD_TAB_RANGE(v) \
            (tab >= 0 && tab <= _MDH_MAX_PAD)

static GtkTextBuffer *pad_buffer[_MDH_MAX_PAD];

static GIni *ini = NULL;

gboolean mdh_scratch_pad_load(const gchar *file, GError **err)
{
    gboolean ret = TRUE;

    gint i;

    g_return_val_if_fail(file != NULL, FALSE);

    if(IS_FILE(file))
        if(!(ini = gini_load(file, err)))
            ret = FALSE;

    for(i = 0; i <= _MDH_MAX_PAD; i++)
        pad_buffer[i] = gtk_text_buffer_new(NULL);

    if(ini) {
        for(i = 0; i <= _MDH_MAX_PAD; i++) {
            gchar *p = g_strdup_printf("tab_%d", i), *s;

            if(gini_entry_get_string(ini, "scratch_pad", p, &s) && *s)
                gtk_text_buffer_set_text(pad_buffer[i], s, -1);

            g_free(p);
        }
    } else
        ini = gini_new(file);

    return(ret);
}

gboolean mdh_scratch_pad_save(GError **err)
{
    gint i;

    for(i = 0; i <= _MDH_MAX_PAD; i++)
      if(pad_buffer[i]) {
          GtkTextIter is, ie;

          gchar *p, *s;

          gtk_text_buffer_get_start_iter(pad_buffer[i], &is);
          gtk_text_buffer_get_end_iter(pad_buffer[i], &ie);

          p = g_strdup_printf("tab_%d", i);
          s = gtk_text_buffer_get_text(pad_buffer[i], &is, &ie, TRUE);

          gini_entry_set_string(ini, "scratch_pad", p, s);

          g_free(p);
          g_free(s);
      }

    return(gini_save(ini, NULL, err));
}

void mdh_scratch_pad_free(void)
{
    gini_unref(ini);
}

void mdh_scratch_pad_set_tab(gint tab)
{
    g_return_if_fail(_VALID_SCRATCH_PAD_TAB_RANGE(tab));

    mdh_config_set_int("scratch_pad", "last_tab", tab);
}

gboolean mdh_scratch_pad_get_tab(gint *tab)
{
    return(mdh_config_get_int("scratch_pad", "last_tab", tab));
}

void mdh_scratch_pad_set_geometry(gint x, gint y, gint w, gint h)
{
    mdh_config_set_int("scratch_pad", "geom_x", x);
    mdh_config_set_int("scratch_pad", "geom_y", y);
    mdh_config_set_int("scratch_pad", "geom_w", w);
    mdh_config_set_int("scratch_pad", "geom_h", h);
}

gboolean mdh_scratch_pad_get_geometry(gint *x, gint *y, gint *w, gint *h)
{
    gint _x, _y, _w, _h;

    if(!mdh_config_get_int("scratch_pad", "geom_x", &_x) ||
       !mdh_config_get_int("scratch_pad", "geom_y", &_y) ||
       !mdh_config_get_int("scratch_pad", "geom_w", &_w) ||
       !mdh_config_get_int("scratch_pad", "geom_h", &_h))
        return(FALSE);

    if(x) *x = _x;
    if(y) *y = _y;
    if(w) *w = _w;
    if(h) *h = _h;

    return(TRUE);
}

GtkTextBuffer *mdh_scratch_pad_get_buffer(gint tab)
{
    g_return_val_if_fail(_VALID_SCRATCH_PAD_TAB_RANGE(tab), NULL);

    return(pad_buffer[tab]);
}
