/*
 * mdh (MailDooHicky), a GTK+-based toolbar.
 *
 * Copyright (c) 2003-2005 by Mike Hokenson <mdh at gozer dot org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <glibtop.h>
#include <glibtop/open.h>
#include <glibtop/loadavg.h>
#include <glibtop/mem.h>
#include <glibtop/swap.h>
#include <glibtop/fsusage.h>
#include <glibtop/netload.h>
#include <glibtop/uptime.h>

static gboolean sys_init = FALSE;

gboolean mdh_sys_init(GError **err)
{
    if(sys_init)
        return(TRUE);

    glibtop_init();

    sys_init = TRUE;

    return(TRUE);
}

void mdh_sys_close(void)
{
    if(!sys_init)
        return;

    /*
     * this is probably just a useless close call as it's quite
     * likely another gtop-enabled panel is open and will just
     * call glibtop_init() a moment later.
     */

    /*glibtop_close();*/

    sys_init = FALSE;
}

gboolean mdh_sys_get_cpu(gdouble loadavg[3],
                         guint64 *nr_running,
                         guint64 *nr_tasks,
                         GError **err)
{
    glibtop_loadavg glt;

    g_return_val_if_fail(loadavg != NULL, FALSE);
    g_return_val_if_fail(nr_running != NULL, FALSE);
    g_return_val_if_fail(nr_tasks != NULL, FALSE);

    glibtop_get_loadavg(&glt);

    loadavg[0]  = glt.loadavg[0];
    loadavg[1]  = glt.loadavg[1];
    loadavg[2]  = glt.loadavg[2];

    *nr_running = glt.nr_running;
    *nr_tasks   = glt.nr_tasks;

    return(TRUE);
}

gboolean mdh_sys_get_mem(guint64 *total, guint64 *free, GError **err)
{
    glibtop_mem  glt_m;
    glibtop_swap glt_s;

    g_return_val_if_fail(total != NULL, FALSE);
    g_return_val_if_fail(free != NULL, FALSE);

    if(!mdh_sys_init(err))
        return(FALSE);

    glibtop_get_mem(&glt_m);
    glibtop_get_swap(&glt_s);

    *total = glt_m.total + glt_s.total;
    *free  = glt_m.free  + glt_s.free;

    return(TRUE);
}

gboolean mdh_sys_get_disk(const gchar *fs,
                          guint64 *total,
                          guint64 *free,
                          guint64 *avail,
                          GError **err)
{
    glibtop_fsusage glt;

    g_return_val_if_fail(fs != NULL, FALSE);
    g_return_val_if_fail(total != NULL, FALSE);
    g_return_val_if_fail(free != NULL, FALSE);
    g_return_val_if_fail(avail != NULL, FALSE);

    if(!mdh_sys_init(err))
        return(FALSE);

    glibtop_get_fsusage(&glt, fs);

    *total = glt.blocks * glt.block_size;
    *free  = glt.bfree  * glt.block_size;
    *avail = glt.bavail * glt.block_size;

    return(TRUE);
}

gboolean mdh_sys_get_net(const gchar *dev,
                         guint64 *rx,
                         guint64 *tx,
                         gboolean *up,
                         GError **err)
{
    guint64  net_rx = 0, net_tx = 0;
    gboolean net_up = FALSE;

    glibtop_netload glt;

    g_return_val_if_fail(dev != NULL, FALSE);
    g_return_val_if_fail(rx != NULL, FALSE);
    g_return_val_if_fail(tx != NULL, FALSE);
    g_return_val_if_fail(up != NULL, FALSE);

    if(!mdh_sys_init(err))
        return(FALSE);

    glibtop_get_netload(&glt, dev);

    if(glt.if_flags & (1 << GLIBTOP_IF_FLAGS_UP)) {
        net_rx = glt.bytes_in;
        net_tx = glt.bytes_out;
        net_up = TRUE;
    }

    *rx = net_rx;
    *tx = net_tx;
    *up = net_up;

    return(TRUE);
}

gboolean mdh_sys_get_uptime(gulong *seconds, GError **err)
{
    glibtop_uptime glt;

    g_return_val_if_fail(seconds != NULL, FALSE);

    if(!mdh_sys_init(err))
        return(FALSE);

    glibtop_get_uptime(&glt);

    *seconds = glt.uptime;

    return(TRUE);
}
