/*
 * mdh (MailDooHicky), a GTK+-based toolbar.
 *
 * Copyright (c) 2003-2005 by Mike Hokenson <mdh at gozer dot org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <sys/sysinfo.h>

#define _PROC_NET_DEV "/proc/net/dev"

static FILE *net_fp;

static gboolean sys_init = FALSE;

gboolean mdh_sys_init(GError **err)
{
    if(sys_init)
        return(TRUE);

    if(!(net_fp = fopen(_PROC_NET_DEV, "r"))) {
        g_set_error(err, 0, 0, "Failed to open '%s': %s.", _PROC_NET_DEV,
                               g_strerror(errno));
        return(FALSE);
    }

    sys_init = TRUE;

    return(TRUE);
}

void mdh_sys_close(void)
{
    if(!sys_init)
        return;

    fclose(net_fp);

    sys_init = FALSE;
}

gboolean mdh_sys_get_mem(guint64 *total, guint64 *free, GError **err)
{
    struct sysinfo sysin;

    g_return_val_if_fail(total != NULL, FALSE);
    g_return_val_if_fail(free != NULL, FALSE);

    if(sysinfo(&sysin) == -1) {
        g_set_error(err, 0, 0, "sysinfo: %s", g_strerror(errno));
        return(FALSE);
    }

    *total = (guint64) sysin.totalram + sysin.totalswap;
    *free  = (guint64) sysin.freeram  + sysin.freeswap;

    return(TRUE);
}

gboolean mdh_sys_get_net(const gchar *dev,
                         guint64 *rx,
                         guint64 *tx,
                         gboolean *up,
                         GError **err)
{
    guint64  net_rx = 0, net_tx = 0;
    gboolean net_up = FALSE;

    gchar buf[255];
    gint dev_len;

    g_return_val_if_fail(dev != NULL, FALSE);
    g_return_val_if_fail(rx != NULL, FALSE);
    g_return_val_if_fail(tx != NULL, FALSE);
    g_return_val_if_fail(up != NULL, FALSE);

    if(!mdh_sys_init(err))
        return(FALSE);

    rewind(net_fp);
    fflush(net_fp);

    /* first two lines are headers */
    fgets(buf, sizeof(buf), net_fp);
    fgets(buf, sizeof(buf), net_fp);

    dev_len = strlen(dev);

    while((fgets(buf, sizeof(buf), net_fp))) {
        gchar *p = g_strchug(buf);

        if(!strncmp(buf, dev, dev_len))
            if(sscanf(p + dev_len, ":%llu %*u %*u %*u %*u %*u %*u %*u %llu ",
                &net_rx, &net_tx) == 2) {
                    net_up = TRUE;
                    break;
            }
    }

    *rx = net_rx;
    *tx = net_tx;
    *up = net_up;

    return(TRUE);
}

gboolean mdh_sys_get_uptime(gulong *seconds, GError **err)
{
    struct sysinfo sysin;

    g_return_val_if_fail(seconds != NULL, FALSE);

    if(sysinfo(&sysin) == -1) {
        g_set_error(err, 0, 0, "sysinfo: %s", g_strerror(errno));
        return(FALSE);
    }

    *seconds = sysin.uptime;

    return(TRUE);
}
