/*
 * mdh (MailDooHicky), a GTK+-based toolbar.
 *
 * Copyright (c) 2003-2005 by Mike Hokenson <mdh at gozer dot org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <kstat.h>
#include <sys/swap.h>

static kstat_ctl_t *kc;

static gboolean sys_init = FALSE;

gboolean mdh_sys_init(GError **err)
{
    if(sys_init)
        return(TRUE);

    if(!(kc = kstat_open())) {
        g_set_error(err, 0, 0, "kstat_open: %s", g_strerror(errno));
        return(FALSE);
    }

    sys_init = TRUE;

    return(TRUE);
}

void mdh_sys_close(void)
{
    if(!sys_init)
        return;

    kstat_close(kc);

    sys_init = FALSE;
}

gboolean mdh_sys_get_mem(gdouble *total, gdouble *free, GError **err)
{
    struct anoninfo anon;
    gulong pagesize;

    g_return_val_if_fail(total != NULL, FALSE);
    g_return_val_if_fail(free != NULL, FALSE);

    if(swapctl(SC_AINFO, &anon) == -1) {
        g_set_error(err, 0, 0, "swapctl: %s", g_strerror(errno));
        return(FALSE);
    }

    pagesize = sysconf(_SC_PAGESIZE);

    *total = (sysconf(_SC_PHYS_PAGES) + anon.ani_max) * pagesize;
    *free  = (sysconf(_SC_AVPHYS_PAGES) + (anon.ani_max - anon.ani_resv)) *
                 pagesize;

    return(TRUE);
}

gboolean mdh_sys_get_net(const gchar *dev,
                         guint64 *rx,
                         guint64 *tx,
                         gboolean *up,
                         GError **err)
{
    guint64  net_rx = 0, net_tx = 0;
    gboolean net_up = FALSE;

    kstat_t *ksp;
    kstat_named_t *knp;

    g_return_val_if_fail(rx != NULL, FALSE);
    g_return_val_if_fail(tx != NULL, FALSE);
    g_return_val_if_fail(up != NULL, FALSE);

    if(!mdh_sys_init(err))
        return(FALSE);

    if(kstat_chain_update(kc) == -1) {
        g_set_error(err, 0, 0, "kstat_chain_update: %s", g_strerror(errno));
        return(FALSE);
    }

    for(ksp = kc->kc_chain; ksp; ksp = ksp->ks_next) {
        if(!strcmp(ksp->ks_class, "net")) {
            kstat_read(kc, ksp, NULL);

            if(!(knp = kstat_data_lookup(ksp, "rbytes")))
                continue;

            net_rx = (guint64) knp->value.ui32;

            if(!(knp = kstat_data_lookup(ksp, "obytes")))
                continue;

            net_tx = (guint64) knp->value.ui32;

            net_up = TRUE;

            break;
        }
    }

    *rx = net_rx;
    *tx = net_tx;
    *up = net_up;

    return(TRUE);
}

gboolean mdh_sys_get_uptime(gulong *seconds, GError **err)
{
    gulong sec = 0;

    kstat_t *ksp;
    kstat_named_t *knp;

    g_return_val_if_fail(seconds != NULL, FALSE);

    if(!mdh_sys_init(err))
        return(FALSE);

    if(kstat_chain_update(kc) == -1) {
        g_set_error(err, 0, 0, "kstat_chain_update: %s", g_strerror(errno));
        return(FALSE);
    }

    ksp = kstat_lookup(kc, "unix", -1, "system_misc");

    if(ksp && kstat_read(kc, ksp, NULL) >= 0)
        if((knp = kstat_data_lookup(ksp, "boot_time")))
            sec = time(NULL) - knp->value.ui32 + 30;

    *seconds = sec;

    return(TRUE);
}
