/*
 * mdh (MailDooHicky), a GTK+-based toolbar.
 *
 * Copyright (c) 2003-2005 by Mike Hokenson <mdh at gozer dot org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <unistd.h>

#include <gtk/gtk.h>

#ifdef HAVE_LIBXML_PARSER_H
# include <libxml/parser.h>
#endif

#include "mdh.h"

#include "toolbar.h"

#include "button.h"    
#include "custom.h"
#include "panel.h"

#include "panel_time.h"
#include "panel_mail.h"
#include "panel_cpu.h"
#include "panel_mem.h"
#include "panel_disk.h"
#include "panel_net.h"
#include "panel_weather.h"
#include "panel_uptime.h"
#include "panel_custom.h"

#include "ui_config.h"

#include "config.h"
#include "curl.h"
#include "error.h"
#include "widget.h"    
#include "window.h"    
#include "util.h"
#include "x11.h"

static gint geom_x, geom_y;

/* config widgets */

static GtkWidget *p_font_w, *p_tooltips_w, *p_sticky_w, *p_layer_w,
       *p_relief_w, *p_shadow_w;

static GtkWidget *b_menu_w, *b_hide_w, *b_run_w, *b_pad_w;

#if _MDH_HAS_MIXER
static GtkWidget *b_vol_w;
#endif

/*
 * could always just use GtkReliefStyle and GtkShadowType, but if any
 * new items were added, it'd screw everything up...
 */

typedef enum {
    CFG_RELIEF_NORMAL,
    CFG_RELIEF_HALF,
    CFG_RELIEF_NONE
} MdhConfigButtonReliefStyle;

typedef enum {
    CFG_SHADOW_IN,
    CFG_SHADOW_OUT,
    CFG_SHADOW_ETCHED_IN,
    CFG_SHADOW_ETCHED_OUT,
    CFG_SHADOW_NONE
} MdhConfigPanelShadowType;

typedef struct {
    const gchar *a;
    const gchar *b;
} MdhConfigHelp;

#if _MDH_HAS_TIME
typedef enum {
    CFG_TIME_ENABLED,
    CFG_TIME_DISPLAY,
    CFG_TIME_COMMAND,
    CFG_TIME_N
} MdhConfigPanelTimeWidgets;

static GtkWidget *p_time_c[CFG_TIME_N + 1];
#endif

typedef enum {
    CFG_MAIL_ENABLED,
    CFG_MAIL_VALUE,
    CFG_MAIL_DISPLAY,
    CFG_MAIL_COMMAND,
    CFG_MAIL_SPINNER,
    CFG_MAIL_N
} MdhConfigPanelMailWidgets;

static GtkWidget *p_mail_use_bold_w, *p_mail_thresh_w;
static GtkWidget *p_mail_c[_MDH_PANEL_COUNT_MAIL_MAX][CFG_MAIL_N + 1];

static MdhConfigHelp display_format_mail[] = {
        { "%m", "The name of the mailbox being processed." },
        { "%n", "The number of new messages." },
        { "%r", "The number of read messages." },
        { "%t", "The number of total messages." },
        { "%%", "A literal `%' character." },
        { NULL, NULL }
    };

#if _MDH_HAS_CPU
typedef enum {
    CFG_CPU_ENABLED,
    CFG_CPU_SPINNER,
    CFG_CPU_DISPLAY,
    CFG_CPU_COMMAND,
    CFG_CPU_BOLD,
    CFG_CPU_THRESH,
    CFG_CPU_N
} MdhConfigPanelCpuWidgets;

static GtkWidget *p_cpu_c[CFG_CPU_N + 1];

static MdhConfigHelp display_format_cpu[] = {
        { "%p", "The 1 minute load average as a percentage." },
        { "%1", "The 1 minute load average as a float." },
        { "%5", "The 5 minute load average as a float." },
        { "%F", "The 15 minute load average as a float." },
        { "%t", "The number of running tasks (gtop only)." },
        { "%T", "The number of total tasks (gtop only)." },
        { "%%", "A literal `%' character." },
        { NULL, NULL }
    };
#endif

#if _MDH_HAS_MEM
typedef enum {
    CFG_MEM_ENABLED,
    CFG_MEM_SPINNER,
    CFG_MEM_DISPLAY,
    CFG_MEM_COMMAND,
    CFG_MEM_BOLD,
    CFG_MEM_THRESH,
    CFG_MEM_N
} MdhConfigPanelMemWidgets;

static GtkWidget *p_mem_c[CFG_MEM_N + 1];

static MdhConfigHelp display_format_mem[] = {
        { "%u", "The amount of used memory in megabytes." },
        { "%U", "The amount of used memory as a percent." },
        { "%f", "The amount of free memory in megabytes." },
        { "%F", "The amount of free memory as a percent." },
        { "%t", "The amount of total memory in megabytes." },
        { "%%", "A literal `%' character." },
        { NULL, NULL }
    };
#endif

#if _MDH_HAS_DISK
typedef enum {
    CFG_DISK_ENABLED,
    CFG_DISK_VALUE,
    CFG_DISK_DISPLAY,
    CFG_DISK_COMMAND,
    CFG_DISK_SPINNER,
    CFG_DISK_N
} MdhConfigPanelDiskWidgets;

static GtkWidget *p_disk_use_bold_w, *p_disk_thresh_w;
static GtkWidget *p_disk_c[_MDH_PANEL_COUNT_DISK_MAX][CFG_DISK_N + 1];

static MdhConfigHelp display_format_disk[] = {
        { "%m", "The name of the mount point being processed." },
        { "%u", "The amount of used space in gigabytes." },
        { "%U", "The amount of used space as a percent." },
        { "%f", "The amount of free space in gigabytes." },
        { "%F", "The amount of free space as a percent." },
        { "%t", "The amount of total space in gigabytes." },
        { "%%", "A literal `%' character." },
        { NULL, NULL }
    };
#endif

#if _MDH_HAS_NET
typedef enum {
    CFG_NET_ENABLED,
#ifndef __sun__
    CFG_NET_VALUE,
#endif
    CFG_NET_DISPLAY,
    CFG_NET_COMMAND,
    CFG_NET_SPINNER,
    CFG_NET_N
} MdhConfigPanelNetWidgets;

static GtkWidget *p_net_c[_MDH_PANEL_COUNT_NET_MAX][CFG_NET_N + 1];

static MdhConfigHelp display_format_net[] = {
        { "%i", "The name of the interface being processed." },
        { "%r", "The receive rate average in kilobytes as a float." },
        { "%t", "The transfer rate average in kilobytes as a float." },
        { "%%", "A literal `%' character." },
        { NULL, NULL }
    };
#endif

#if _MDH_HAS_WEATHER
typedef enum {
    CFG_WEATHER_ENABLED,
    CFG_WEATHER_SPINNER,
    CFG_WEATHER_VALUE,
    CFG_WEATHER_COMMAND,
    CFG_WEATHER_DISPLAY,
    CFG_WEATHER_ICON,
    CFG_WEATHER_N
} MdhConfigPanelWeatherWidgets;

static GtkWidget *p_weather_c[CFG_WEATHER_N + 1];

static MdhConfigHelp display_format_weather[] = {
        { "%l", "The location id being processed (54311)." },
        { "%L", "The location name (Green Bay, WI)." },
        { "%i", "The time in AM/PM notation." },
        { "%s", "The time of sunrise in AM/PM notation." },
        { "%S", "The time of sunset in AM/PM notation." },
        { "%u", "The date and time of last update." },
        { "%t", "The current temperature." },
        { "%f", "The current temperature accounting for wind chill." },
        { "%c", "The description of current conditions." },
        { "%b", "The barometric pressure as a float." },
        { "%B", "The barometric pressure change direction." },
        { "%w", "The wind speed and direction." },
        { "%h", "The amount of humidity as a percent." },
        { "%v", "The visibility in miles as a float." },
        { "%I", "The UV index and rating description." },
        { "%d", "The dew point." },
        { "%%", "A literal `%' character." },
        { NULL, NULL }
    };

static const gchar *help_location_id =
    "Weather.com can accept two types of location id.\n"
    "The first is a 5 digit (US) zip code and the other\n"
    "an 8 character city code.\n\n"
    "Here's a few examples:\n"
    "  Green Bay, WI: 54311 or USWI0288\n"
    "  Beverly Hills, CA: 90210 or USCA0090\n"
    "  Zurich, Switzerland: SZXX0033\n"
    "  Mexico City, Mexico: MXDF0132\n\n"
    "If you are outside the US or don't know your zip\n"
    "code (heh?), search for your city name below.\n";

# define _WEATHER_URI_SEARCH "http://xoap.weather.com/search/search?where=%s"
#endif

#if _MDH_HAS_UPTIME
typedef enum {
    CFG_UPTIME_ENABLED,
    CFG_UPTIME_SPINNER,
    CFG_UPTIME_DISPLAY,
    CFG_UPTIME_COMMAND,
    CFG_UPTIME_N
} MdhConfigPanelUptimeWidgets;

static GtkWidget *p_uptime_c[CFG_UPTIME_N + 1];

static MdhConfigHelp display_format_uptime[] = {
        { "%m", "The uptime in minute notation (MM:SS)." },
        { "%h", "The uptime in hour notation (HH:MM)." },
        { "%d", "The uptime in days." },
        { "%U", "The uptime in day and hour notation (DDd HH:MM)." },
        { "%%", "A literal `%' character." },
        { NULL, NULL }
    };
#endif

typedef enum {
    CFG_CUSTOM_ENABLED,
    CFG_CUSTOM_VALUE,
    CFG_CUSTOM_COMMAND,
    CFG_CUSTOM_SPINNER,
    CFG_CUSTOM_N
} MdhConfigPanelCustomWidgets;

static GtkWidget *p_custom_c[_MDH_PANEL_COUNT_CUSTOM_MAX][CFG_CUSTOM_N + 1];

#define  TOGGLE_VAL(w) (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w)))
#define   ENTRY_VAL(w) (gtk_entry_get_text(GTK_ENTRY(w)))
#define SPINNER_VAL(w) (gtk_spin_button_get_value(GTK_SPIN_BUTTON(w)))

#if _MDH_HAS_WEATHER
static void ui_config_view_weather_home(void)
{
    GError *err = NULL;

    if(!mdh_browser_open_uri(_WEATHER_URI, &err)) {
        mdh_window_error("Failed to open URI.", err->message);
        g_error_free(err);
    }
}
#endif /* _MDH_HAS_WEATHER */

static void ui_config_toggle_panel(GtkWidget *widget, GtkWidget **w)
{
    gboolean enabled;

    gint i;

    g_return_if_fail(w != NULL);

    enabled = TOGGLE_VAL(w[0]);

    for(i = 1; w[i]; i++) {
        gtk_widget_set_sensitive(w[i], enabled);

        /*
         * XXX: hack for cpu & mem panels to keep anything following
         *      the threshold toggle button insensitive when the
         *      toggle button is active.
         */
#ifdef _MDH_HAS_CPU
        if(i > 1 && enabled && w[i] == p_cpu_c[CFG_CPU_BOLD])
            enabled = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w[i]));
#endif
#ifdef _MDH_HAS_MEM
        if(i > 1 && enabled && w[i] == p_mem_c[CFG_MEM_BOLD])
            enabled = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w[i]));
#endif
    }
}

static void ui_config_spinner_update_geom_x(GtkWidget *widget,
                                            GtkWidget *spinner)
{
    Toolbar->geom_x = gtk_spin_button_get_value(GTK_SPIN_BUTTON(spinner));

    mdh_toolbar_move(Toolbar, Toolbar->geom_x, Toolbar->geom_y);
}

static void ui_config_spinner_update_geom_y(GtkWidget *widget,
                                            GtkWidget *spinner)
{
    Toolbar->geom_y = gtk_spin_button_get_value(GTK_SPIN_BUTTON(spinner));

    mdh_toolbar_move(Toolbar, Toolbar->geom_x, Toolbar->geom_y);
}

static void ui_config_font_selection_ok(GtkFontSelectionDialog *fs)
{
    gchar *font = gtk_font_selection_dialog_get_font_name(fs);

    if(!font)
        return;

    gtk_entry_set_text(GTK_ENTRY(p_font_w), font);

    g_free(font);
}

static void ui_config_select_font(void)
{
    GtkFontSelectionDialog *fs;

    GtkWidget *window;

    const gchar *font;

    window = gtk_font_selection_dialog_new("Font Selection Dialog");

    fs = GTK_FONT_SELECTION_DIALOG(window);

    /* default is just [a-k][A-K] */
    gtk_font_selection_dialog_set_preview_text(fs,
                                        "abcdefghijk ABCDEFGHIJK 123456789");

    /* select current font */
    if((font = gtk_entry_get_text(GTK_ENTRY(p_font_w))) && *font)
        gtk_font_selection_dialog_set_font_name(fs, font);

    g_signal_connect_swapped(fs->ok_button, "clicked",
                             G_CALLBACK(ui_config_font_selection_ok), window);

    g_signal_connect_swapped(fs->ok_button, "clicked",
                             G_CALLBACK(gtk_widget_destroy), window);

    g_signal_connect_swapped(fs->cancel_button, "clicked",
                             G_CALLBACK(gtk_widget_destroy), window);

    gtk_widget_show(window);
}

static void ui_config_unselect_font(void)
{
    gtk_entry_set_text(GTK_ENTRY(p_font_w), "");
}

static void ui_config_panel_get_new(MdhPanel *obj,
                                    GtkWidget *_e,
                                    GtkWidget *_v,
                                    GtkWidget *_d,
                                    GtkWidget *_c,
                                    GtkWidget *_i)
{
    gboolean e;

    const gchar *v = NULL, *d = NULL, *c;
    gint i = 0;

    gchar *v_new = NULL, *d_new = NULL, *c_new;
    gint i_new = 0;

    gboolean changed = FALSE;

    g_return_if_fail(obj != NULL);
    g_return_if_fail(_e != NULL);
    g_return_if_fail(_c != NULL);

    e = TOGGLE_VAL(_e);

    if(_v)
        v = g_strstrip((gchar *) ENTRY_VAL(_v));

    if(_d)
        d = ENTRY_VAL(_d);

    c = g_strstrip((gchar *) ENTRY_VAL(_c));

    if(_i)
        i = SPINNER_VAL(_i);

    if(!obj->funcs.config) {
        v_new = g_strdup(v);
        d_new = g_strdup(d);
        c_new = mdh_expand_tilde(c);
        i_new = i;
    } else
        obj->funcs.config(v, &v_new, d, &d_new, c, &c_new, i, &i_new);

    if(obj->enabled != e) {
        changed = TRUE;
        obj->enabled = e;
    }

    if(obj->value && v_new) {
        if(strcmp(obj->value, v_new)) {
            changed = TRUE;
            g_free(obj->value);
            obj->value = v_new;
        } else
            g_free(v_new);
    }

    if(strcmp(obj->command, c_new)) {
        g_free(obj->command);
        obj->command = c_new;
    } else
        g_free(c_new);

    if(obj->display && d_new) {
        if(strcmp(obj->display, d_new)) {
            g_free(obj->display);
            obj->display = d_new;
        } else
            g_free(d_new);
    }

    if(i_new > 0)
        if(obj->interval != i_new) {
            changed = TRUE;
            obj->interval = i_new;
        }

    if(obj->value && changed) {
       g_free(obj->tooltip);
       obj->tooltip = NULL;
    }

    if(obj->enabled && changed) {
        mdh_panel_stop(obj);
        mdh_panel_start(obj);
    }
}

static void ui_config_toolbar_get_new(void)
{
    MdhPanelFlags flags;

    gint thresh;

    gint i;

    Toolbar->tooltips = TOGGLE_VAL(p_tooltips_w);

    if(mdh_x11_has_netwm() || mdh_x11_has_gnome())
        Toolbar->sticky = TOGGLE_VAL(p_sticky_w);

    if(mdh_x11_has_netwm_layers() || mdh_x11_has_gnome_layers())
        Toolbar->layer  = gtk_combo_box_get_active(GTK_COMBO_BOX(p_layer_w));

    switch(gtk_combo_box_get_active(GTK_COMBO_BOX(p_relief_w))) {
        case CFG_RELIEF_NORMAL:
            Toolbar->button_relief = GTK_RELIEF_NORMAL;
            break;
        case CFG_RELIEF_HALF:
            Toolbar->button_relief = GTK_RELIEF_HALF;
            break;
        case CFG_RELIEF_NONE:
            Toolbar->button_relief = GTK_RELIEF_NONE;
            break;
        default: 
            g_assert_not_reached();
    }

    switch(gtk_combo_box_get_active(GTK_COMBO_BOX(p_shadow_w))) {
        case CFG_SHADOW_IN:
            Toolbar->panel_shadow = GTK_SHADOW_IN;
            break;
        case CFG_SHADOW_OUT:
            Toolbar->panel_shadow = GTK_SHADOW_OUT;
            break;
        case CFG_SHADOW_ETCHED_IN:
            Toolbar->panel_shadow = GTK_SHADOW_ETCHED_IN;
            break;
        case CFG_SHADOW_ETCHED_OUT:
            Toolbar->panel_shadow = GTK_SHADOW_ETCHED_OUT;
            break;
        case CFG_SHADOW_NONE:
            Toolbar->panel_shadow = GTK_SHADOW_NONE;
            break;
        default:
            g_assert_not_reached();
    }

    B_Menu->enabled   = TOGGLE_VAL(b_menu_w);
    B_Toggle->enabled = TOGGLE_VAL(b_hide_w);
    B_Run->enabled    = TOGGLE_VAL(b_run_w);
    B_Pad->enabled    = TOGGLE_VAL(b_pad_w);

#if _MDH_HAS_MIXER
    B_Volume->enabled = TOGGLE_VAL(b_vol_w);
#endif

    g_free(Toolbar->font);
    Toolbar->font = g_strdup(ENTRY_VAL(p_font_w));

#if _MDH_HAS_TIME
    ui_config_panel_get_new(P_Time,
                            p_time_c[CFG_TIME_ENABLED],
                            NULL,
                            p_time_c[CFG_TIME_DISPLAY],
                            p_time_c[CFG_TIME_COMMAND],
                            NULL);
#endif

    flags = 0;

    if(TOGGLE_VAL(p_mail_use_bold_w))
        flags |= MDH_PANEL_FLAG_BOLD;

    thresh = SPINNER_VAL(p_mail_thresh_w);

    for(i = 0; i < mdh_panel_mail_count(P_Mail[0]); i++) {
        ui_config_panel_get_new(P_Mail[i],
                                p_mail_c[i][CFG_MAIL_ENABLED],
                                p_mail_c[i][CFG_MAIL_VALUE],
                                p_mail_c[i][CFG_MAIL_DISPLAY],
                                p_mail_c[i][CFG_MAIL_COMMAND],
                                p_mail_c[i][CFG_MAIL_SPINNER]);

        mdh_panel_flags_set(P_Mail[i], flags);
        mdh_panel_mail_threshold_set(P_Mail[i], thresh);

        if(!P_Mail[i]->tooltip)
            P_Mail[i]->tooltip = g_strdup_printf("Mail: %s", P_Mail[i]->value);
    }

#ifdef _MDH_HAS_CPU
    flags = 0;

    if(TOGGLE_VAL(p_cpu_c[CFG_CPU_BOLD]))
        flags |= MDH_PANEL_FLAG_BOLD;

    thresh = SPINNER_VAL(p_cpu_c[CFG_CPU_THRESH]);

    mdh_panel_flags_set(P_Cpu, flags);
    mdh_panel_cpu_threshold_set(P_Cpu, thresh);

    ui_config_panel_get_new(P_Cpu,
                            p_cpu_c[CFG_CPU_ENABLED],
                            NULL,
                            p_cpu_c[CFG_CPU_DISPLAY],
                            p_cpu_c[CFG_CPU_COMMAND],
                            p_cpu_c[CFG_CPU_SPINNER]);
#endif

#ifdef _MDH_HAS_MEM
    flags = 0;

    if(TOGGLE_VAL(p_mem_c[CFG_MEM_BOLD]))
        flags |= MDH_PANEL_FLAG_BOLD;

    thresh = SPINNER_VAL(p_mem_c[CFG_MEM_THRESH]);

    mdh_panel_flags_set(P_Mem, flags);
    mdh_panel_mem_threshold_set(P_Mem, thresh);

    ui_config_panel_get_new(P_Mem,
                            p_mem_c[CFG_MEM_ENABLED],
                            NULL,
                            p_mem_c[CFG_MEM_DISPLAY],
                            p_mem_c[CFG_MEM_COMMAND],
                            p_mem_c[CFG_MEM_SPINNER]);
#endif

#ifdef _MDH_HAS_DISK
    flags = 0;

    if(TOGGLE_VAL(p_disk_use_bold_w))
        flags |= MDH_PANEL_FLAG_BOLD;

    thresh = SPINNER_VAL(p_disk_thresh_w);

    for(i = 0; i < mdh_panel_disk_count(P_Disk[0]); i++) {
        ui_config_panel_get_new(P_Disk[i],
                                p_disk_c[i][CFG_DISK_ENABLED],
                                p_disk_c[i][CFG_DISK_VALUE],
                                p_disk_c[i][CFG_DISK_DISPLAY],
                                p_disk_c[i][CFG_DISK_COMMAND],
                                p_disk_c[i][CFG_DISK_SPINNER]);

        mdh_panel_flags_set(P_Disk[i], flags);
        mdh_panel_disk_threshold_set(P_Disk[i], thresh);

        if(!P_Disk[i]->tooltip)
            P_Disk[i]->tooltip = g_strdup_printf("Disk: %s", P_Disk[i]->value);
    }
#endif

#ifdef _MDH_HAS_NET
    for(i = 0; i < mdh_panel_net_count(P_Net[0]); i++) {
        ui_config_panel_get_new(P_Net[i],
                                p_net_c[i][CFG_NET_ENABLED],
#ifndef __sun__
                                p_net_c[i][CFG_NET_VALUE],
#else
                                NULL,
#endif
                                p_net_c[i][CFG_NET_DISPLAY],
                                p_net_c[i][CFG_NET_COMMAND],
                                p_net_c[i][CFG_NET_SPINNER]);

        if(!P_Net[i]->tooltip)
            P_Net[i]->tooltip = g_strdup_printf("Net: %s", P_Net[i]->value);
    }
#endif

#if _MDH_HAS_WEATHER
    flags = 0;

    if(TOGGLE_VAL(p_weather_c[CFG_WEATHER_ICON]))
        flags |= MDH_PANEL_FLAG_ICON;

    mdh_panel_flags_set(P_Weather, flags);

    ui_config_panel_get_new(P_Weather,
                            p_weather_c[CFG_WEATHER_ENABLED],
                            p_weather_c[CFG_WEATHER_VALUE],
                            p_weather_c[CFG_WEATHER_DISPLAY],
                            p_weather_c[CFG_WEATHER_COMMAND],
                            p_weather_c[CFG_WEATHER_SPINNER]);

    if(!P_Weather->tooltip)
        P_Weather->tooltip = g_strdup_printf("Weather: %s", P_Weather->value);
#endif /* _MDH_HAS_WEATHER */

#if _MDH_HAS_UPTIME
    ui_config_panel_get_new(P_Uptime,
                            p_uptime_c[CFG_UPTIME_ENABLED],
                            NULL,
                            p_uptime_c[CFG_UPTIME_DISPLAY],
                            p_uptime_c[CFG_UPTIME_COMMAND],
                            p_uptime_c[CFG_UPTIME_SPINNER]);
#endif /* _MDH_HAS_UPTIME */

    for(i = 0; i < mdh_panel_custom_count(P_Custom[0]); i++) {
        ui_config_panel_get_new(P_Custom[i],
                                p_custom_c[i][CFG_CUSTOM_ENABLED],
                                p_custom_c[i][CFG_CUSTOM_VALUE],
                                NULL,
                                p_custom_c[i][CFG_CUSTOM_COMMAND],
                                p_custom_c[i][CFG_CUSTOM_SPINNER]);

        if(!P_Custom[i]->tooltip)
            P_Custom[i]->tooltip = g_strdup_printf("Custom: %s",
                                                  P_Custom[i]->value);
    }
}

static gboolean ui_config_panel_has_value(GtkWidget **w,
                                          gint enabled,
                                          gint value)
{
    const gchar *v;

    g_return_val_if_fail(w != NULL, FALSE);

    if(TOGGLE_VAL(w[enabled]))
        if((v = ENTRY_VAL(w[value])) && !*v)
            return(FALSE);

    return(TRUE);
}

/* if panel value is not NULL and zero-length, return the name of the panel */
static gboolean ui_config_panels_have_empty_values(gchar **list)
{
    GString *string;

    gint i;

    g_return_val_if_fail(list != NULL, FALSE);

    string = g_string_new(NULL);

#if _MDH_HAS_TIME
    if(!ui_config_panel_has_value(p_time_c, CFG_TIME_ENABLED, CFG_TIME_DISPLAY))
        g_string_append(string, "Time, ");
#endif

    for(i = 0; i < mdh_panel_mail_count(P_Mail[0]); i++)
        if(!ui_config_panel_has_value(p_mail_c[i],
                                      CFG_MAIL_ENABLED, CFG_MAIL_VALUE) ||
           !ui_config_panel_has_value(p_mail_c[i],
                                      CFG_MAIL_ENABLED, CFG_MAIL_DISPLAY))
            g_string_append_printf(string, "Mail (%d), ", i);

#if _MDH_HAS_CPU
    if(!ui_config_panel_has_value(p_cpu_c, CFG_CPU_ENABLED, CFG_CPU_DISPLAY))
        g_string_append(string, "Cpu, ");
#endif

#if _MDH_HAS_MEM
    if(!ui_config_panel_has_value(p_mem_c, CFG_MEM_ENABLED, CFG_MEM_DISPLAY))
        g_string_append(string, "Mem, ");
#endif

#if _MDH_HAS_DISK
    for(i = 0; i < mdh_panel_disk_count(P_Disk[0]); i++)
        if(!ui_config_panel_has_value(p_disk_c[i],
                                      CFG_DISK_ENABLED, CFG_DISK_VALUE) ||
           !ui_config_panel_has_value(p_disk_c[i],
                                      CFG_DISK_ENABLED, CFG_DISK_DISPLAY))
            g_string_append_printf(string, "Disk (%d), ", i);
#endif

#if _MDH_HAS_NET
    for(i = 0; i < mdh_panel_net_count(P_Net[0]); i++)
#ifndef __sun__
        if(!ui_config_panel_has_value(p_net_c[i],
                                      CFG_NET_ENABLED, CFG_NET_VALUE) ||
           !ui_config_panel_has_value(p_net_c[i],
                                      CFG_NET_ENABLED, CFG_NET_DISPLAY))
#else  /* __sun __ */
        if(!ui_config_panel_has_value(p_net_c[i],
                                      CFG_NET_ENABLED, CFG_NET_DISPLAY))
#endif /* __sun __ */
            g_string_append_printf(string, "Net (%d), ", i);
#endif /* _MDH_HAS_NET */

#if _MDH_HAS_WEATHER
    if(!ui_config_panel_has_value(p_weather_c,
                                  CFG_WEATHER_ENABLED, CFG_WEATHER_VALUE) ||
       !ui_config_panel_has_value(p_weather_c,
                                  CFG_WEATHER_ENABLED, CFG_WEATHER_DISPLAY))
        g_string_append(string, "Weather, ");
#endif

#if _MDH_HAS_UPTIME
    if(!ui_config_panel_has_value(p_uptime_c,
                                  CFG_UPTIME_ENABLED, CFG_UPTIME_DISPLAY))
        g_string_append(string, "Uptime, ");
#endif

    for(i = 0; i < mdh_panel_custom_count(P_Custom[0]); i++)
        if(!ui_config_panel_has_value(p_custom_c[i],
                                      CFG_CUSTOM_ENABLED, CFG_CUSTOM_VALUE))
            g_string_append_printf(string, "Custom (%d), ", i);

    if(string->len) {
        g_string_truncate(string, string->len - 2); /* strip trailing comma */
        *list = g_string_free(string, FALSE);
        return(TRUE);
    }

    g_string_free(string, TRUE);

    return(FALSE);
}

static gboolean ui_config_apply(void)
{
    gchar *list = NULL;

    /* make sure the enabled panels have a value */
    if(ui_config_panels_have_empty_values(&list)) {
        gchar *text = g_strdup_printf("Missing value(s) for %s.", list);

        mdh_window_notice(text, "Every enabled panel that requires an argument to function must have a value and/or display format (eg. strftime format, mailbox, network interface). Disable the panel if you do not wish to use it.");

        g_free(text);
        g_free(list);

        return(FALSE);
    }

#if _MDH_HAS_WEATHER
    /* check LocID length */
    if(TOGGLE_VAL(p_weather_c[CFG_WEATHER_ENABLED])) {
        const gchar *v;

        v = ENTRY_VAL(p_weather_c[CFG_WEATHER_VALUE]);

        if(!_WEATHER_ID_VALID(v)) {
            gchar *text = g_strdup_printf("The weather location must be a zip code (%d digits) or a Weather.com city code (%d characters).", _WEATHER_ID_MIN, _WEATHER_ID_MAX);
            mdh_window_notice("Invalid Location ID.", text);
            g_free(text);
            return(FALSE);
        }
    }
#endif

    ui_config_toolbar_get_new();

    mdh_toolbar_reset(Toolbar);

    geom_x = Toolbar->geom_x;
    geom_y = Toolbar->geom_y;

    return(TRUE);
}

static void ui_config_close(void)
{
    mdh_window_item_open_set(&wi_config, FALSE);
}

static void ui_config_cancel(GtkWidget *widget, GtkWidget *window)
{
    Toolbar->geom_x = geom_x;
    Toolbar->geom_y = geom_y;

    mdh_toolbar_move(Toolbar, Toolbar->geom_x, Toolbar->geom_y);
}

#ifdef _MDH_HAS_WEATHER
static GtkWidget *weather_search_e;

static void ui_config_help_dialog_weather_update_location(GtkComboBox *combo)
{
    gchar *text, *p;

    if(!(text = gtk_combo_box_get_active_text(combo)))
        return;

    /* 'Green Bay, WI - USWI0288' */
    if((p = strstr(text, " - ")) && strlen(p) > 3) {
        p += 3;

        if(_WEATHER_ID_VALID(p))
            gtk_entry_set_text(GTK_ENTRY(p_weather_c[CFG_WEATHER_VALUE]), p);
    }

    g_free(text);
}

static void ui_config_help_dialog_weather_search(GtkComboBox *combo)
{
    const gchar *str = g_strstrip((gchar *) ENTRY_VAL(weather_search_e));

    xmlDocPtr doc;
    xmlNodePtr node, n;

    gchar *res;
    gint res_len;

    static gint res_count;
    gint i;

    if(!str || !*str)
        return;

    {
        gchar uri[PATH_MAX], *query;

        GError *err = NULL;

        query = mdh_curl_escape(str);
        g_snprintf(uri, sizeof(uri), _WEATHER_URI_SEARCH, query);
        mdh_curl_free(query);

        if(!mdh_curl_get(uri, 30, &res, &res_len, NULL, &err)) {
            mdh_window_error("Search failed.", err->message);
            return;
        }
    }

    if(!(doc = xmlParseMemory(res, res_len))) {
        mdh_window_error("Search failed.", "Invalid xml document received.");
        g_free(res);
        return;
    }

    g_free(res);

    if(!(node = xmlDocGetRootElement(doc))) {
        mdh_window_error("Search failed.", "Invalid xml document received.");
        goto config_help_dialog_weather_search_end;
    }

    if(!g_ascii_strcasecmp((const gchar *) node->name, "error")) {
        gchar *p = (gchar *) xmlNodeGetContent(node);
        mdh_window_error("Search failed.", p);
        g_free(p);

        goto config_help_dialog_weather_search_end;
    }

    if(g_ascii_strcasecmp((const gchar *) node->name, "search"))
        goto config_help_dialog_weather_search_end;

    if(!node->children) {
        mdh_window_error("Search failed.",
                         "No matching city names were found.");
        goto config_help_dialog_weather_search_end;
    }

    /* remove any previous results */
    for(i = res_count; i >= 0; i--)
        gtk_combo_box_remove_text(combo, i);

    gtk_combo_box_set_active(combo, -1);

    res_count = 0;

    n = node->children;

    while(n) {
        if(!g_ascii_strcasecmp((const gchar *) n->name, "loc")) {
            gchar *p = (gchar *) xmlNodeGetContent(n);
            gchar *s = (gchar *) xmlGetProp(n, (const xmlChar *) "id");

            if(p && s) {
                gchar *n = g_strdup_printf("%s - %s", p, s);
                gtk_combo_box_append_text(combo, n);
                g_free(n);

                res_count++;
            }

            g_free(p);
            g_free(s);
        }

        n = n->next;
    }

    /* select the first result */
    if(res_count)
        gtk_combo_box_set_active(combo, 0);

config_help_dialog_weather_search_end:
    xmlFreeDoc(doc);
}

static void ui_config_help_dialog_weather_close(void)
{
    weather_search_e = NULL;
}

static void ui_config_help_dialog_weather(GtkWidget *widget,
                                          GdkEvent *event,
                                          gpointer data)
{
    GtkWidget *window,
              *vbox,
              *hbox,
              *label,
              *combo,
              *entry,
              *button;

    if(weather_search_e)
        return;

    window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(window), "MailDooHicky: Config Help");
    gtk_window_set_resizable(GTK_WINDOW(window), FALSE);

    g_signal_connect(G_OBJECT(window), "event",
                     G_CALLBACK(mdh_window_event), NULL);

    g_signal_connect(G_OBJECT(window), "destroy",
                     G_CALLBACK(ui_config_help_dialog_weather_close), NULL);

    gtk_container_set_border_width(GTK_CONTAINER(window), 5);

    vbox = gtk_vbox_new(FALSE, 5);
    gtk_widget_show(vbox);

    gtk_container_add(GTK_CONTAINER(window), vbox);

    label = gtk_label_new(NULL);
    mdh_label_set_larger(GTK_LABEL(label), "Location ID:");
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
    gtk_widget_show(label);

    label = gtk_label_new(help_location_id);
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 0);
    gtk_widget_show(label);

    hbox = gtk_hbox_new(FALSE, 3);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);
    gtk_widget_show(hbox);

    weather_search_e = entry = gtk_entry_new();
    gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
    gtk_widget_show(entry);

    gtk_entry_set_text(GTK_ENTRY(entry), "<search string>");
    gtk_editable_select_region(GTK_EDITABLE(entry), 0, -1);

    mdh_widget_grab_default(entry);

    button = gtk_button_new_from_stock(GTK_STOCK_FIND);
    gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
    gtk_widget_show(button);

    combo = gtk_combo_box_new_text();
    gtk_box_pack_start(GTK_BOX(vbox), combo, FALSE, FALSE, 0);
    gtk_widget_show(combo);

    g_signal_connect_swapped(G_OBJECT(entry), "activate",
            G_CALLBACK(ui_config_help_dialog_weather_search), combo);

    g_signal_connect_swapped(G_OBJECT(button), "clicked",
            G_CALLBACK(ui_config_help_dialog_weather_search), combo);

    hbox = gtk_hbox_new(TRUE, 3);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);
    gtk_widget_show(hbox);

    button = gtk_button_new_from_stock(GTK_STOCK_CANCEL);
    gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
    gtk_widget_show(button);

    g_signal_connect_swapped(G_OBJECT(button), "clicked",
                             G_CALLBACK(gtk_widget_destroy), window);

    button = gtk_button_new_from_stock(GTK_STOCK_OK);
    gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
    gtk_widget_show(button);

    g_signal_connect_swapped(G_OBJECT(button), "clicked",
            G_CALLBACK(ui_config_help_dialog_weather_update_location), combo);

    g_signal_connect_swapped(G_OBJECT(button), "clicked",
                             G_CALLBACK(gtk_widget_destroy), window);

    gtk_widget_show(window);
}
#endif /* _MDH_HAS_WEATHER */

static void ui_config_help_format(GtkWidget *widget,
                                  GdkEvent *event,
                                  gpointer data)
{
    MdhConfigHelp *help = data;

    GtkWidget *window,
              *vbox,
              *label,
              *table,
              *button;

    gint i;

    window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(window), "MailDooHicky: Config Help");
    gtk_window_set_resizable(GTK_WINDOW(window), FALSE);

    g_signal_connect(G_OBJECT(window), "event",
                     G_CALLBACK(mdh_window_event), NULL);

    gtk_container_set_border_width(GTK_CONTAINER(window), 5);

    vbox = gtk_vbox_new(FALSE, 5);
    gtk_widget_show(vbox);

    gtk_container_add(GTK_CONTAINER(window), vbox);

    label = gtk_label_new(NULL);
    mdh_label_set_larger(GTK_LABEL(label), "Display format:");
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
    gtk_widget_show(label);

    for(i = 0; help[i].a; i++)
        ;

    table = gtk_table_new(i, 4, FALSE);
    gtk_box_pack_start(GTK_BOX(vbox), table, TRUE, TRUE, 0);
    gtk_widget_show(table);

    for(i = 0; help[i].a; i++) {
        label = gtk_label_new(" ");
        gtk_table_attach(GTK_TABLE(table), label, 0, 1, i + 1, i + 2,
                                                  GTK_FILL, GTK_FILL, 3, 0);
        gtk_widget_show(label);

        label = gtk_label_new(help[i].a);
        gtk_label_set_selectable(GTK_LABEL(label), TRUE);
        gtk_misc_set_alignment(GTK_MISC(label), 0.5, 0.0);
        gtk_table_attach(GTK_TABLE(table), label, 1, 2, i + 1, i + 2,
                                                  GTK_FILL, GTK_FILL, 3, 0);
        gtk_widget_show(label);

        label = gtk_label_new("-");
        gtk_table_attach(GTK_TABLE(table), label, 2, 3, i + 1, i + 2,
                                                  GTK_FILL, GTK_FILL, 3, 0);
        gtk_widget_show(label);

        label = gtk_label_new(help[i].b);
        gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
        gtk_table_attach(GTK_TABLE(table), label, 3, 4, i + 1, i + 2,
                                                  GTK_FILL, GTK_FILL, 3, 0);
        gtk_widget_show(label);
    }

    button = gtk_button_new_from_stock(GTK_STOCK_OK);
    gtk_box_pack_start(GTK_BOX(vbox), button, TRUE, TRUE, 0);
    gtk_widget_show(button);

    mdh_widget_grab_default(button);

    g_signal_connect_swapped(G_OBJECT(button), "clicked",
                             G_CALLBACK(gtk_widget_destroy), window);

    gtk_widget_show(window);
}

static void ui_config_toggle_sensitive(gpointer data, GtkWidget *widget)
{
    g_return_if_fail(widget != NULL);

    gtk_widget_set_sensitive(widget, (widget->state & GTK_STATE_INSENSITIVE));
}

void mdh_ui_config(void)
{
    GtkWidget *window,
              *vbox, *vbox3, *vbox4,
              *hbox, *hbox2,
              *notebook,
              *frame,
              *table,
              *label,
              *entry,
              *button, *p_button,
              *combo,
              *spinner,
              *hsep, *vsep;

    GtkObject *adj;

    gchar *text;

    gint thresh;

    gint i, max_geom_x, max_geom_y;

    if(mdh_window_item_open(&wi_config))
        return;

    /* expand panel */
    if(mdh_toolbar_hidden(Toolbar))
        mdh_toolbar_toggle(Toolbar);

    /* set after mdh_toolbar_toggle_all() otherwise the toolbar won't expand */
    mdh_window_item_open_set(&wi_config, TRUE);

    max_geom_x = gdk_screen_width()  - 5;
    max_geom_y = gdk_screen_height() - 5;

    /* allow toolbar geometry to be previewed, reset if canceled */
    geom_x = Toolbar->geom_x;
    geom_y = Toolbar->geom_y;

#if _MDH_HAS_TIME
    p_time_c[CFG_TIME_N] = NULL;
#endif

    for(i = 0; i < mdh_panel_mail_count(P_Mail[0]); i++)
        p_mail_c[i][CFG_MAIL_N] = NULL;

#if _MDH_HAS_CPU
    p_cpu_c[CFG_CPU_N] = NULL;
#endif

#if _MDH_HAS_MEM
    p_mem_c[CFG_MEM_N] = NULL;
#endif

#if _MDH_HAS_DISK
    for(i = 0; i < mdh_panel_disk_count(P_Disk[0]); i++)
        p_disk_c[i][CFG_DISK_N] = NULL;
#endif

#if _MDH_HAS_NET
    for(i = 0; i < mdh_panel_net_count(P_Net[0]); i++)
        p_net_c[i][CFG_NET_N] = NULL;
#endif

#if _MDH_HAS_WEATHER
    p_weather_c[CFG_WEATHER_N] = NULL;
#endif

#if _MDH_HAS_UPTIME
    p_uptime_c[CFG_UPTIME_N] = NULL;
#endif

    for(i = 0; i < mdh_panel_custom_count(P_Custom[0]); i++)
        p_custom_c[i][CFG_CUSTOM_N] = NULL;

    window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(window), "MailDooHicky: Preferences");
    gtk_window_set_resizable(GTK_WINDOW(window), FALSE);

    /* ignores the window managers's close request */
    g_signal_connect(GTK_OBJECT(window), "delete_event",
                     GTK_SIGNAL_FUNC(gtk_true), NULL);

    g_signal_connect(GTK_OBJECT(window), "destroy",
                     G_CALLBACK(ui_config_close), NULL);

    g_signal_connect(G_OBJECT(window), "expose_event",
                     G_CALLBACK(mdh_window_expose_event), NULL);

    gtk_container_set_border_width(GTK_CONTAINER(window), 5);

    vbox = gtk_vbox_new(FALSE, 5);
    gtk_widget_show(vbox);

    gtk_container_add(GTK_CONTAINER(window), vbox);

    notebook = gtk_notebook_new();
    gtk_notebook_set_tab_pos(GTK_NOTEBOOK(notebook), GTK_POS_TOP);
    gtk_notebook_set_scrollable(GTK_NOTEBOOK(notebook), TRUE);
    gtk_widget_show(notebook);

    gtk_box_pack_start(GTK_BOX(vbox), notebook, TRUE, TRUE, 0);

    /* General tab */

    vbox3 = gtk_vbox_new(FALSE, 3);
    gtk_widget_show(vbox3);

    gtk_container_set_border_width(GTK_CONTAINER(vbox3), 5);

    hbox2 = gtk_hbox_new(FALSE, 3);
    gtk_box_pack_start(GTK_BOX(vbox3), hbox2, TRUE, FALSE, 0);
    gtk_widget_show(hbox2);

    frame = mdh_frame_new_with_title("Location");
    gtk_box_pack_start(GTK_BOX(hbox2), frame, FALSE, FALSE, 0);
    gtk_widget_show(frame);

    hbox = gtk_hbox_new(TRUE, 3);
    gtk_widget_show(hbox);

    gtk_container_set_border_width(GTK_CONTAINER(hbox), 2);

    gtk_container_add(GTK_CONTAINER(frame), hbox);

    adj = gtk_adjustment_new(Toolbar->geom_x, 0.0, max_geom_x, 1.0, 50.0, 0.0);
    spinner = gtk_spin_button_new(GTK_ADJUSTMENT(adj), 0, 0);
    gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spinner), FALSE);
    gtk_box_pack_start(GTK_BOX(hbox), spinner, FALSE, TRUE, 2);
    gtk_widget_show(spinner);

    mdh_widget_set_tip(NULL, spinner,
                       "Adjust the toolbars horizontal location");

    g_signal_connect(G_OBJECT(adj), "value_changed",
                     G_CALLBACK(ui_config_spinner_update_geom_x), spinner);

    adj = gtk_adjustment_new(Toolbar->geom_y, 0.0, max_geom_y, 1.0, 50.0, 0.0);
    spinner = gtk_spin_button_new(GTK_ADJUSTMENT(adj), 0, 0);
    gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spinner), FALSE);
    gtk_box_pack_start(GTK_BOX(hbox), spinner, FALSE, TRUE, 2);
    gtk_widget_show(spinner);

    mdh_widget_set_tip(NULL, spinner,
                       "Adjust the toolbars vertical location");

    g_signal_connect(G_OBJECT(adj), "value_changed",
                     G_CALLBACK(ui_config_spinner_update_geom_y), spinner);

    vsep = gtk_vseparator_new();
    gtk_box_pack_start(GTK_BOX(hbox2), vsep, FALSE, FALSE, 5);
    gtk_widget_show(vsep);

    frame = mdh_frame_new_with_title("Font");
    gtk_box_pack_start(GTK_BOX(hbox2), frame, TRUE, TRUE, 0);
    gtk_widget_show(frame);

    hbox = gtk_hbox_new(FALSE, 3);
    gtk_widget_show(hbox);

    gtk_container_set_border_width(GTK_CONTAINER(hbox), 2);

    gtk_container_add(GTK_CONTAINER(frame), hbox);

    button = gtk_button_new_with_label("Select");
    gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
    gtk_widget_show(button);

    mdh_widget_set_tip(NULL, button, "Select a font for the main toolbar");

    g_signal_connect(G_OBJECT(button), "clicked",
                     G_CALLBACK(ui_config_select_font), NULL);

    entry = gtk_entry_new();
    gtk_editable_set_editable(GTK_EDITABLE(entry), FALSE);
    gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
    gtk_widget_show(entry);

    gtk_entry_set_text(GTK_ENTRY(entry), Toolbar->font);

    p_font_w = entry;

    button = gtk_button_new_with_label("Reset");
    gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
    gtk_widget_show(button);

    mdh_widget_set_tip(NULL, button, "Use the default (GTK) font");

    g_signal_connect(G_OBJECT(button), "clicked",
                     G_CALLBACK(ui_config_unselect_font), NULL);

    hsep = gtk_hseparator_new();
    gtk_box_pack_start(GTK_BOX(vbox3), hsep, TRUE, TRUE, 3);
    gtk_widget_show(hsep);

    hbox2 = gtk_hbox_new(FALSE, 3);
    gtk_box_pack_start(GTK_BOX(vbox3), hbox2, FALSE, TRUE, 0);
    gtk_widget_show(hbox2);

    frame = mdh_frame_new_with_title("Features");
    gtk_box_pack_start(GTK_BOX(hbox2), frame, TRUE, TRUE, 0);
    gtk_widget_show(frame);

    gtk_container_set_border_width(GTK_CONTAINER(hbox2), 2);

    hbox = gtk_hbox_new(FALSE, 3);
    gtk_widget_show(hbox);

    gtk_container_add(GTK_CONTAINER(frame), hbox);

    button = gtk_check_button_new_with_label("Tooltips");
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), Toolbar->tooltips);
    gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, FALSE, 3);
    gtk_widget_show(button);

    mdh_widget_set_tip(NULL, button, "Enable tooltips for the toolbar objects");

    p_tooltips_w = button;

    /* GTK_WINDOW_POPUP should already keep the window stuck */
    button = gtk_check_button_new_with_label("Sticky");
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), Toolbar->sticky);
    gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, FALSE, 3);
    gtk_widget_show(button);

    mdh_widget_set_tip(NULL, button, "Show the toolbar on all workspaces");

    p_sticky_w = button;

    if(!(mdh_x11_has_netwm() || mdh_x11_has_gnome()))
        gtk_widget_set_sensitive(GTK_WIDGET(button), FALSE);

    vsep = gtk_vseparator_new();
    gtk_box_pack_start(GTK_BOX(hbox2), vsep, FALSE, FALSE, 5);
    gtk_widget_show(vsep);

    frame = mdh_frame_new_with_title("Layer");
    gtk_box_pack_start(GTK_BOX(hbox2), frame, TRUE, TRUE, 0);
    gtk_widget_show(frame);

    gtk_container_set_border_width(GTK_CONTAINER(hbox2), 2);

    hbox = gtk_hbox_new(TRUE, 3);
    gtk_widget_show(hbox);

    gtk_container_add(GTK_CONTAINER(frame), hbox);

    p_layer_w = combo = gtk_combo_box_new_text();
    gtk_box_pack_start(GTK_BOX(hbox), combo, FALSE, TRUE, 0);
    gtk_widget_show(combo);

    gtk_combo_box_append_text(GTK_COMBO_BOX(combo), "Normal");
    gtk_combo_box_append_text(GTK_COMBO_BOX(combo), "Below");
    gtk_combo_box_append_text(GTK_COMBO_BOX(combo), "Above");

    gtk_combo_box_set_active(GTK_COMBO_BOX(combo), Toolbar->layer);

    if(!(mdh_x11_has_netwm_layers() || mdh_x11_has_gnome_layers()))
        gtk_widget_set_sensitive(GTK_WIDGET(combo), FALSE);

    vsep = gtk_vseparator_new();
    gtk_box_pack_start(GTK_BOX(hbox2), vsep, FALSE, FALSE, 5);
    gtk_widget_show(vsep);

    frame = mdh_frame_new_with_title("Button Relief");
    gtk_box_pack_start(GTK_BOX(hbox2), frame, TRUE, TRUE, 0);
    gtk_widget_show(frame);

    gtk_container_set_border_width(GTK_CONTAINER(hbox2), 2);

    hbox = gtk_hbox_new(TRUE, 3);
    gtk_widget_show(hbox);

    gtk_container_add(GTK_CONTAINER(frame), hbox);

    p_relief_w = combo = gtk_combo_box_new_text();
    gtk_box_pack_start(GTK_BOX(hbox), combo, FALSE, TRUE, 0);
    gtk_widget_show(combo);

    gtk_combo_box_append_text(GTK_COMBO_BOX(combo), "Normal");
    gtk_combo_box_append_text(GTK_COMBO_BOX(combo), "Half");
    gtk_combo_box_append_text(GTK_COMBO_BOX(combo), "None");

    {
        gint selection = -1;

        switch(Toolbar->button_relief) {
            case GTK_RELIEF_NORMAL:
                selection = CFG_RELIEF_NORMAL;
                break;
            case GTK_RELIEF_HALF:
                selection = CFG_RELIEF_HALF;
                break;
            case GTK_RELIEF_NONE:
                selection = CFG_RELIEF_NONE;
                break;
            default:
                g_assert_not_reached();
        }

        gtk_combo_box_set_active(GTK_COMBO_BOX(combo), selection);
    }

    frame = mdh_frame_new_with_title("Panel Shadow");
    gtk_box_pack_start(GTK_BOX(hbox2), frame, TRUE, TRUE, 0);
    gtk_widget_show(frame);

    gtk_container_set_border_width(GTK_CONTAINER(hbox2), 2);

    hbox = gtk_hbox_new(TRUE, 3);
    gtk_widget_show(hbox);

    gtk_container_add(GTK_CONTAINER(frame), hbox);

    p_shadow_w = combo = gtk_combo_box_new_text();
    gtk_box_pack_start(GTK_BOX(hbox), combo, FALSE, TRUE, 0);
    gtk_widget_show(combo);

    gtk_combo_box_append_text(GTK_COMBO_BOX(combo), "In");
    gtk_combo_box_append_text(GTK_COMBO_BOX(combo), "Out");
    gtk_combo_box_append_text(GTK_COMBO_BOX(combo), "Etched In");
    gtk_combo_box_append_text(GTK_COMBO_BOX(combo), "Etched Out");
    gtk_combo_box_append_text(GTK_COMBO_BOX(combo), "None");

    {
        gint selection = -1;

        switch(Toolbar->panel_shadow) {
            case GTK_SHADOW_IN:
                selection = CFG_SHADOW_IN;
                break;
            case GTK_SHADOW_OUT:
                selection = CFG_SHADOW_OUT;
                break;
            case GTK_SHADOW_ETCHED_IN:
                selection = CFG_SHADOW_ETCHED_IN;
                break;
            case GTK_SHADOW_ETCHED_OUT:
                selection = CFG_SHADOW_ETCHED_OUT;
                break;
            case GTK_SHADOW_NONE:
                selection = CFG_SHADOW_NONE;
                break;
            default:
                g_assert_not_reached();
        }

        gtk_combo_box_set_active(GTK_COMBO_BOX(combo), selection);
    }

    hsep = gtk_hseparator_new();
    gtk_box_pack_start(GTK_BOX(vbox3), hsep, TRUE, TRUE, 3);
    gtk_widget_show(hsep);

    hbox2 = gtk_hbox_new(FALSE, 3);
    gtk_box_pack_start(GTK_BOX(vbox3), hbox2, FALSE, FALSE, 0);
    gtk_widget_show(hbox2);

    frame = mdh_frame_new_with_title("Buttons");
    gtk_box_pack_start(GTK_BOX(hbox2), frame, TRUE, TRUE, 0);
    gtk_widget_show(frame);

    hbox = gtk_hbox_new(TRUE, 3);
    gtk_widget_show(hbox);

    gtk_container_set_border_width(GTK_CONTAINER(hbox), 2);

    gtk_container_add(GTK_CONTAINER(frame), hbox);

    button = gtk_check_button_new_with_label("Menu");
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), B_Menu->enabled);
    gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 3);
    gtk_widget_show(button);

    mdh_widget_set_tip(NULL, button, "Display the menu button");

    b_menu_w = button;

    button = gtk_check_button_new_with_label("Toggle");
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), B_Toggle->enabled);
    gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 3);
    gtk_widget_show(button);

    mdh_widget_set_tip(NULL, button, "Display the hide button");

    b_hide_w = button;

    button = gtk_check_button_new_with_label("Run");
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), B_Run->enabled);
    gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 3);
    gtk_widget_show(button);

    mdh_widget_set_tip(NULL, button, "Display the run button");

    b_run_w = button;

    button = gtk_check_button_new_with_label("Pad");
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), B_Pad->enabled);
    gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 3);
    gtk_widget_show(button);

    mdh_widget_set_tip(NULL, button, "Display the scratch-pad button");

    b_pad_w = button;

#if _MDH_HAS_MIXER
    button = gtk_check_button_new_with_label("Volume");
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), B_Volume->enabled);
    gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 3);
    gtk_widget_show(button);

    mdh_widget_set_tip(NULL, button, "Display the volume control button");

    b_vol_w = button;
#endif

    gtk_notebook_append_page(GTK_NOTEBOOK(notebook),
                             vbox3, gtk_label_new("General"));

#if _MDH_HAS_TIME
    /* Time tab */

    vbox3 = gtk_vbox_new(FALSE, 3);
    gtk_widget_show(vbox3);

    gtk_container_set_border_width(GTK_CONTAINER(vbox3), 5);

    frame = mdh_frame_new_with_title("Toolbar: Time");
    gtk_box_pack_start(GTK_BOX(vbox3), frame, TRUE, TRUE, 0);
    gtk_widget_show(frame);

    vbox4 = gtk_vbox_new(FALSE, 0);
    gtk_widget_show(vbox4);

    gtk_container_set_border_width(GTK_CONTAINER(vbox4), 5);

    gtk_container_add(GTK_CONTAINER(frame), vbox4);

    table = gtk_table_new(2, 3, FALSE);
    gtk_box_pack_start(GTK_BOX(vbox4), table, TRUE, TRUE, 0);
    gtk_widget_show(table);

    label = gtk_label_new("Display (see strftime(3)):");
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), label, 1, 2, 0, 1,
                                              GTK_FILL, GTK_FILL, 3, 0);
    gtk_widget_show(label);

    label = gtk_label_new("Click command:");
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), label, 2, 3, 0, 1,
                                              GTK_FILL, GTK_FILL, 3, 0);
    gtk_widget_show(label);

    p_button = gtk_check_button_new();
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(p_button), P_Time->enabled);
    gtk_table_attach(GTK_TABLE(table), p_button, 0, 1, 1, 2,
                                                 GTK_FILL, GTK_FILL, 3, 3);
    gtk_widget_show(p_button);

    p_time_c[CFG_TIME_ENABLED] = p_button;

    mdh_widget_set_tip(NULL, p_button, "Enable the panel");

    entry = gtk_entry_new();
    gtk_entry_set_text(GTK_ENTRY(entry), P_Time->display);
    gtk_table_attach(GTK_TABLE(table), entry, 1, 2, 1, 2,
                                              GTK_FILL, GTK_FILL, 3, 3);
    gtk_widget_show(entry);

    p_time_c[CFG_TIME_DISPLAY] = entry;

    gtk_widget_set_sensitive(GTK_WIDGET(entry), P_Time->enabled);

    mdh_widget_set_tip(NULL, entry,
                       "Display format (default: "_TIME_DIS")");

    entry = gtk_entry_new();
    gtk_entry_set_text(GTK_ENTRY(entry), P_Time->command);
    gtk_table_attach(GTK_TABLE(table), entry, 2, 3, 1, 2,
                                              GTK_FILL, GTK_FILL, 3, 3);
    gtk_widget_show(entry);

    p_time_c[CFG_TIME_COMMAND] = entry;

    gtk_widget_set_sensitive(GTK_WIDGET(entry), P_Time->enabled);

    mdh_widget_set_tip(NULL, entry,
            "Command to execute when panel is clicked (default: evolution)");

    g_signal_connect(G_OBJECT(p_button), "clicked",
                     G_CALLBACK(ui_config_toggle_panel), p_time_c);

    label = gtk_label_new("Displays system time.");
    gtk_widget_set_size_request(GTK_WIDGET(label), 450, -1);
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_box_pack_start(GTK_BOX(vbox3), label, FALSE, FALSE, 0);
    gtk_widget_show(label);

    gtk_notebook_append_page(GTK_NOTEBOOK(notebook),
                             vbox3, gtk_label_new("Time"));
#endif /* _MDH_HAS_TIME */

    /* Mail tab */

    vbox3 = gtk_vbox_new(FALSE, 3);
    gtk_widget_show(vbox3);

    gtk_container_set_border_width(GTK_CONTAINER(vbox3), 5);

    frame = mdh_frame_new_with_title("Toolbar: Mail");
    gtk_box_pack_start(GTK_BOX(vbox3), frame, TRUE, TRUE, 0);
    gtk_widget_show(frame);

    vbox4 = gtk_vbox_new(FALSE, 0);
    gtk_widget_show(vbox4);

    gtk_container_set_border_width(GTK_CONTAINER(vbox4), 5);

    gtk_container_add(GTK_CONTAINER(frame), vbox4);

    table = gtk_table_new(mdh_panel_mail_count(P_Mail[0]), 5, FALSE);
    gtk_box_pack_start(GTK_BOX(vbox4), table, TRUE, TRUE, 0);
    gtk_widget_show(table);

    label = gtk_label_new("Mailbox/dir:");
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), label, 1, 2, 0, 1,
                                              GTK_FILL, GTK_FILL, 3, 0);
    gtk_widget_show(label);

    hbox = gtk_hbox_new(FALSE, 0);
    gtk_table_attach(GTK_TABLE(table), hbox, 2, 3, 0, 1,
                                             GTK_FILL, GTK_FILL, 3, 0);
    gtk_widget_show(hbox);

    label = gtk_label_new("Display:");
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 0);
    gtk_widget_show(label);

    button = mdh_clickable_help("?",
                                ui_config_help_format,
                                display_format_mail);
    gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
    gtk_widget_show_all(button);

    mdh_widget_set_tip(NULL, button, "Display format help");

    label = gtk_label_new("Click command:");
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), label, 3, 4, 0, 1,
                                              GTK_FILL, GTK_FILL, 3, 0);
    gtk_widget_show(label);

    label = gtk_label_new("Sec(s):");
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), label, 4, 5, 0, 1,
                                              GTK_FILL, GTK_FILL, 3, 0);
    gtk_widget_show(label);

    for(i = 0; i < mdh_panel_mail_count(P_Mail[0]); i++) {
        p_button = gtk_check_button_new();
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(p_button),
            P_Mail[i]->enabled);
        gtk_table_attach(GTK_TABLE(table), p_button, 0, 1, i + 1, i + 2,
                                                     GTK_FILL, GTK_FILL, 3, 3);
        gtk_widget_show(p_button);

        p_mail_c[i][CFG_MAIL_ENABLED] = p_button;

        mdh_widget_set_tip(NULL, p_button, "Enable this panel");

        entry = gtk_entry_new();
        gtk_entry_set_text(GTK_ENTRY(entry), P_Mail[i]->value);
        gtk_table_attach(GTK_TABLE(table), entry, 1, 2, i + 1, i + 2,
                                                  GTK_FILL, GTK_FILL, 3, 3);
        gtk_widget_show(entry);

        p_mail_c[i][CFG_MAIL_VALUE] = entry;

        gtk_widget_set_sensitive(GTK_WIDGET(entry), P_Mail[i]->enabled);

        mdh_widget_set_tip(NULL, entry,
                           "Mailbox to process (default $MAIL, $MAILDIR)");

        entry = gtk_entry_new();
        gtk_entry_set_text(GTK_ENTRY(entry), P_Mail[i]->display);
        gtk_widget_set_size_request(GTK_WIDGET(entry), 75, -1);
        gtk_table_attach(GTK_TABLE(table), entry, 2, 3, i + 1, i + 2,
                                                  GTK_FILL, GTK_FILL, 3, 3);
        gtk_widget_show(entry);

        p_mail_c[i][CFG_MAIL_DISPLAY] = entry;

        gtk_widget_set_sensitive(GTK_WIDGET(entry), P_Mail[i]->enabled);

        mdh_widget_set_tip(NULL, entry,
                           "Display format (default: "_MAIL_DIS")");

        entry = gtk_entry_new();
        gtk_entry_set_text(GTK_ENTRY(entry), P_Mail[i]->command);
        gtk_table_attach(GTK_TABLE(table), entry, 3, 4, i + 1, i + 2,
                                                  GTK_FILL, GTK_FILL, 3, 3);
        gtk_widget_show(entry);

        p_mail_c[i][CFG_MAIL_COMMAND] = entry;

        gtk_widget_set_sensitive(GTK_WIDGET(entry), P_Mail[i]->enabled);

        mdh_widget_set_tip(NULL, entry,
            "Command to execute when panel is clicked (default: balsa %s)");

        adj = gtk_adjustment_new(P_Mail[i]->interval, _MAIL_MIN, _MAIL_MAX,
                                 5.0, 30.0, 0.0);
        spinner = gtk_spin_button_new(GTK_ADJUSTMENT(adj), 0, 0);
        gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spinner), FALSE);
        gtk_table_attach(GTK_TABLE(table), spinner, 4, 5, i + 1, i + 2,
                                                    GTK_FILL, GTK_FILL, 3, 3);
        gtk_widget_show(spinner);

        p_mail_c[i][CFG_MAIL_SPINNER] = spinner;

        gtk_widget_set_sensitive(GTK_WIDGET(spinner), P_Mail[i]->enabled);

        text = g_strdup_printf("Set the update interval (default: %d)",
                               _MAIL_DEF);
        mdh_widget_set_tip(NULL, spinner, text);
        g_free(text);

        g_signal_connect(G_OBJECT(p_button), "clicked",
                         G_CALLBACK(ui_config_toggle_panel), p_mail_c[i]);
    }

    hbox = gtk_hbox_new(FALSE, 3);
    gtk_box_pack_start(GTK_BOX(vbox4), hbox, FALSE, FALSE, 3);
    gtk_widget_show(hbox);

    button = gtk_check_button_new();
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button),
                                 mdh_panel_flags_bold(P_Mail[0]));
    gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 1);
    gtk_widget_show(button);

    p_mail_use_bold_w = button;

    label = gtk_label_new("Bold on");
    gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 1);
    gtk_widget_show(label);

    thresh = mdh_panel_mail_threshold(P_Mail[0]);

    adj = gtk_adjustment_new(thresh, _MAIL_THRESH_MIN, _MAIL_THRESH_MAX,
                             1.0, 5.0, 0.0);
    spinner = gtk_spin_button_new(GTK_ADJUSTMENT(adj), 0, 0);
    gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spinner), FALSE);
    gtk_box_pack_start(GTK_BOX(hbox), spinner, FALSE, FALSE, 1);
    gtk_widget_show(spinner);

    p_mail_thresh_w = spinner;

    gtk_widget_set_sensitive(GTK_WIDGET(spinner),
                             mdh_panel_flags_bold(P_Mail[0]));

    g_signal_connect(G_OBJECT(button), "clicked",
                     G_CALLBACK(ui_config_toggle_sensitive), p_mail_thresh_w);

    label = gtk_label_new("new message(s)");
    gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 1);
    gtk_widget_show(label);

    label = gtk_label_new("Displays message count of selected mailbox/dir(s).");
    gtk_widget_set_size_request(GTK_WIDGET(label), 450, -1);
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_box_pack_start(GTK_BOX(vbox3), label, FALSE, FALSE, 0);
    gtk_widget_show(label);

    gtk_notebook_append_page(GTK_NOTEBOOK(notebook),
                             vbox3, gtk_label_new("Mail"));

#if _MDH_HAS_CPU
    /* Cpu tab */

    vbox3 = gtk_vbox_new(FALSE, 3);
    gtk_widget_show(vbox3);

    gtk_container_set_border_width(GTK_CONTAINER(vbox3), 5);

    frame = mdh_frame_new_with_title("Toolbar: Cpu");
    gtk_box_pack_start(GTK_BOX(vbox3), frame, TRUE, TRUE, 0);
    gtk_widget_show(frame);

    vbox4 = gtk_vbox_new(FALSE, 0);
    gtk_widget_show(vbox4);

    gtk_container_set_border_width(GTK_CONTAINER(vbox4), 5);

    gtk_container_add(GTK_CONTAINER(frame), vbox4);

    table = gtk_table_new(2, 4, FALSE);
    gtk_box_pack_start(GTK_BOX(vbox4), table, TRUE, TRUE, 0);
    gtk_widget_show(table);

    hbox = gtk_hbox_new(FALSE, 0);
    gtk_table_attach(GTK_TABLE(table), hbox, 1, 2, 0, 1,
                                             GTK_FILL, GTK_FILL, 3, 0);
    gtk_widget_show(hbox);

    label = gtk_label_new("Display:");
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 0);
    gtk_widget_show(label);

    button = mdh_clickable_help("?",
                                ui_config_help_format,
                                display_format_cpu);
    gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
    gtk_widget_show_all(button);

    mdh_widget_set_tip(NULL, button, "Display format help");

    label = gtk_label_new("Click command:");
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), label, 2, 3, 0, 1,
                                              GTK_FILL, GTK_FILL, 3, 0);
    gtk_widget_show(label);

    label = gtk_label_new("Sec(s):");
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), label, 3, 4, 0, 1,
                                              GTK_FILL, GTK_FILL, 3, 0);
    gtk_widget_show(label);

    p_button = gtk_check_button_new();
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(p_button), P_Cpu->enabled);
    gtk_table_attach(GTK_TABLE(table), p_button, 0, 1, 1, 2,
                                                 GTK_FILL, GTK_FILL, 3, 3);
    gtk_widget_show(p_button);

    p_cpu_c[CFG_CPU_ENABLED] = p_button;

    mdh_widget_set_tip(NULL, p_button, "Enable the panel");

    entry = gtk_entry_new();
    gtk_entry_set_text(GTK_ENTRY(entry), P_Cpu->display);
    gtk_widget_set_size_request(GTK_WIDGET(entry), 75, -1);
    gtk_table_attach(GTK_TABLE(table), entry, 1, 2, 1, 2,
                                              GTK_FILL, GTK_FILL, 3, 3);
    gtk_widget_show(entry);

    p_cpu_c[CFG_CPU_DISPLAY] = entry;

    gtk_widget_set_sensitive(GTK_WIDGET(entry), P_Cpu->enabled);

    mdh_widget_set_tip(NULL, entry,
                       "Display format (default: "_CPU_DIS")");

    entry = gtk_entry_new();
    gtk_entry_set_text(GTK_ENTRY(entry), P_Cpu->command);
    gtk_table_attach(GTK_TABLE(table), entry, 2, 3, 1, 2,
                                              GTK_FILL, GTK_FILL, 3, 3);
    gtk_widget_show(entry);

    p_cpu_c[CFG_CPU_COMMAND] = entry;

    gtk_widget_set_sensitive(GTK_WIDGET(entry), P_Cpu->enabled);

    text = g_strdup_printf("Command to execute when panel is clicked (default: %s -e top)", mdh_get_xterm());
    mdh_widget_set_tip(NULL, entry, text);
    g_free(text);

    adj = gtk_adjustment_new(P_Cpu->interval, _CPU_MIN, _CPU_MAX,
                             1.0, 5.0, 0.0);
    spinner = gtk_spin_button_new(GTK_ADJUSTMENT(adj), 0, 0);
    gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spinner), FALSE);
    gtk_table_attach(GTK_TABLE(table), spinner, 3, 4, 1, 2,
                                                GTK_FILL, GTK_FILL, 3, 3);
    gtk_widget_show(spinner);

    p_cpu_c[CFG_CPU_SPINNER] = spinner;
    
    gtk_widget_set_sensitive(GTK_WIDGET(spinner), P_Cpu->enabled);
    
    text = g_strdup_printf("Set the update interval (default: %d)", _CPU_DEF);
    mdh_widget_set_tip(NULL, spinner, text);
    g_free(text);

    hbox = gtk_hbox_new(FALSE, 3);
    gtk_box_pack_start(GTK_BOX(vbox4), hbox, FALSE, FALSE, 3);
    gtk_widget_show(hbox);

    button = gtk_check_button_new();
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button),
                                 mdh_panel_flags_bold(P_Cpu));
    gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 1);
    gtk_widget_show(button);

    gtk_widget_set_sensitive(GTK_WIDGET(button), P_Cpu->enabled);

    p_cpu_c[CFG_CPU_BOLD] = button;

    label = gtk_label_new("Bold on");
    gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 1);
    gtk_widget_show(label);

    thresh = mdh_panel_cpu_threshold(P_Cpu);

    adj = gtk_adjustment_new(thresh, _CPU_THRESH_MIN, _CPU_THRESH_MAX,
                             1.0, 5.0, 0.0);
    spinner = gtk_spin_button_new(GTK_ADJUSTMENT(adj), 0, 0);
    gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spinner), FALSE);
    gtk_box_pack_start(GTK_BOX(hbox), spinner, FALSE, FALSE, 1);
    gtk_widget_show(spinner);

    p_cpu_c[CFG_CPU_THRESH] = spinner;

    gtk_widget_set_sensitive(GTK_WIDGET(spinner),
                             mdh_panel_flags_bold(P_Cpu));

    g_signal_connect(G_OBJECT(button), "clicked",
            G_CALLBACK(ui_config_toggle_sensitive), p_cpu_c[CFG_CPU_THRESH]);

    label = gtk_label_new("% usage");
    gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 1);
    gtk_widget_show(label);

    g_signal_connect(G_OBJECT(p_button), "clicked",
                     G_CALLBACK(ui_config_toggle_panel), p_cpu_c);

    label = gtk_label_new("Displays cpu usage.");
    gtk_widget_set_size_request(GTK_WIDGET(label), 450, -1);
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_box_pack_start(GTK_BOX(vbox3), label, FALSE, FALSE, 0);
    gtk_widget_show(label);

    gtk_notebook_append_page(GTK_NOTEBOOK(notebook),
                             vbox3, gtk_label_new("Cpu"));
#endif /* _MDH_AAS_CPU */

#if _MDH_HAS_MEM
    /* Mem tab */

    vbox3 = gtk_vbox_new(FALSE, 3);
    gtk_widget_show(vbox3);

    gtk_container_set_border_width(GTK_CONTAINER(vbox3), 5);

    frame = mdh_frame_new_with_title("Toolbar: Mem");
    gtk_box_pack_start(GTK_BOX(vbox3), frame, TRUE, TRUE, 0);
    gtk_widget_show(frame);

    vbox4 = gtk_vbox_new(FALSE, 0);
    gtk_widget_show(vbox4);

    gtk_container_set_border_width(GTK_CONTAINER(vbox4), 5);

    gtk_container_add(GTK_CONTAINER(frame), vbox4);

    table = gtk_table_new(2, 4, FALSE);
    gtk_box_pack_start(GTK_BOX(vbox4), table, TRUE, TRUE, 0);
    gtk_widget_show(table);

    hbox = gtk_hbox_new(FALSE, 0);
    gtk_table_attach(GTK_TABLE(table), hbox, 1, 2, 0, 1,
                                             GTK_FILL, GTK_FILL, 3, 0);
    gtk_widget_show(hbox);

    label = gtk_label_new("Display:");
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 0);
    gtk_widget_show(label);

    button = mdh_clickable_help("?",
                                ui_config_help_format,
                                display_format_mem);
    gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
    gtk_widget_show_all(button);

    mdh_widget_set_tip(NULL, button, "Display format help");

    label = gtk_label_new("Click command:");
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), label, 2, 3, 0, 1,
                                              GTK_FILL, GTK_FILL, 3, 0);
    gtk_widget_show(label);

    label = gtk_label_new("Sec(s):");
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), label, 3, 4, 0, 1,
                                              GTK_FILL, GTK_FILL, 3, 0);
    gtk_widget_show(label);

    p_button = gtk_check_button_new();
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(p_button), P_Mem->enabled);
    gtk_table_attach(GTK_TABLE(table), p_button, 0, 1, 1, 2,
                                                 GTK_FILL, GTK_FILL, 3, 3);
    gtk_widget_show(p_button);

    p_mem_c[CFG_MEM_ENABLED] = p_button;

    mdh_widget_set_tip(NULL, p_button, "Enable the panel");

    entry = gtk_entry_new();
    gtk_entry_set_text(GTK_ENTRY(entry), P_Mem->display);
    gtk_widget_set_size_request(GTK_WIDGET(entry), 75, -1);
    gtk_table_attach(GTK_TABLE(table), entry, 1, 2, 1, 2,
                                              GTK_FILL, GTK_FILL, 3, 3);
    gtk_widget_show(entry);

    p_mem_c[CFG_MEM_DISPLAY] = entry;

    gtk_widget_set_sensitive(GTK_WIDGET(entry), P_Mem->enabled);

    mdh_widget_set_tip(NULL, entry,
                       "Display format (default: "_MEM_DIS")");

    entry = gtk_entry_new();
    gtk_entry_set_text(GTK_ENTRY(entry), P_Mem->command);
    gtk_table_attach(GTK_TABLE(table), entry, 2, 3, 1, 2,
                                              GTK_FILL, GTK_FILL, 3, 3);
    gtk_widget_show(entry);

    p_mem_c[CFG_MEM_COMMAND] = entry;

    gtk_widget_set_sensitive(GTK_WIDGET(entry), P_Mem->enabled);

    mdh_widget_set_tip(NULL, entry,
            "Command to execute when panel is clicked (default: none)");

    adj = gtk_adjustment_new(P_Mem->interval, _MEM_MIN, _MEM_MAX,
                             1.0, 5.0, 0.0);
    spinner = gtk_spin_button_new(GTK_ADJUSTMENT(adj), 0, 0);
    gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spinner), FALSE);
    gtk_table_attach(GTK_TABLE(table), spinner, 3, 4, 1, 2,
                                                GTK_FILL, GTK_FILL, 3, 3);
    gtk_widget_show(spinner);

    p_mem_c[CFG_MEM_SPINNER] = spinner;

    gtk_widget_set_sensitive(GTK_WIDGET(spinner), P_Mem->enabled);

    text = g_strdup_printf("Set the update interval (default: %d)", _MEM_DEF);
    mdh_widget_set_tip(NULL, spinner, text);
    g_free(text);

    hbox = gtk_hbox_new(FALSE, 3);
    gtk_box_pack_start(GTK_BOX(vbox4), hbox, FALSE, FALSE, 3);
    gtk_widget_show(hbox);

    button = gtk_check_button_new();
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button),
                                 mdh_panel_flags_bold(P_Mem));
    gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 1);
    gtk_widget_show(button);

    p_mem_c[CFG_MEM_BOLD] = button;

    gtk_widget_set_sensitive(GTK_WIDGET(button), P_Mem->enabled);

    label = gtk_label_new("Bold on");
    gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 1);
    gtk_widget_show(label);

    thresh = mdh_panel_mem_threshold(P_Mem);

    adj = gtk_adjustment_new(thresh, _MEM_THRESH_MIN, _MEM_THRESH_MAX,
                             1.0, 5.0, 0.0);
    spinner = gtk_spin_button_new(GTK_ADJUSTMENT(adj), 0, 0);
    gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spinner), FALSE);
    gtk_box_pack_start(GTK_BOX(hbox), spinner, FALSE, FALSE, 1);
    gtk_widget_show(spinner);

    p_mem_c[CFG_MEM_THRESH] = spinner;

    gtk_widget_set_sensitive(GTK_WIDGET(spinner),
                             mdh_panel_flags_bold(P_Mem));

    g_signal_connect(G_OBJECT(button), "clicked",
            G_CALLBACK(ui_config_toggle_sensitive), p_mem_c[CFG_MEM_THRESH]);

    label = gtk_label_new("% usage");
    gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 1);
    gtk_widget_show(label);

    g_signal_connect(G_OBJECT(p_button), "clicked",
                     G_CALLBACK(ui_config_toggle_panel), p_mem_c);

    label = gtk_label_new("Displays memory usage.");
    gtk_widget_set_size_request(GTK_WIDGET(label), 450, -1);
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_box_pack_start(GTK_BOX(vbox3), label, FALSE, FALSE, 0);
    gtk_widget_show(label);

    gtk_notebook_append_page(GTK_NOTEBOOK(notebook),
                             vbox3, gtk_label_new("Mem"));
#endif /* _MDH_AAS_MEM */

#if _MDH_HAS_DISK
    /* Disk tab */

    vbox3 = gtk_vbox_new(FALSE, 3);
    gtk_widget_show(vbox3);

    gtk_container_set_border_width(GTK_CONTAINER(vbox3), 5);

    frame = mdh_frame_new_with_title("Toolbar: Disk");
    gtk_box_pack_start(GTK_BOX(vbox3), frame, TRUE, TRUE, 0);
    gtk_widget_show(frame);

    vbox4 = gtk_vbox_new(FALSE, 0);
    gtk_widget_show(vbox4);

    gtk_container_set_border_width(GTK_CONTAINER(vbox4), 5);

    gtk_container_add(GTK_CONTAINER(frame), vbox4);

    table = gtk_table_new(mdh_panel_disk_count(P_Disk[0]), 5, FALSE);
    gtk_box_pack_start(GTK_BOX(vbox4), table, TRUE, TRUE, 0);
    gtk_widget_show(table);

    label = gtk_label_new("Filesystem:");
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), label, 1, 2, 0, 1,
                                              GTK_FILL, GTK_FILL, 3, 0);
    gtk_widget_show(label);

    hbox = gtk_hbox_new(FALSE, 0);
    gtk_table_attach(GTK_TABLE(table), hbox, 2, 3, 0, 1,
                                             GTK_FILL, GTK_FILL, 3, 0);
    gtk_widget_show(hbox);

    label = gtk_label_new("Display:");
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 0);
    gtk_widget_show(label);

    button = mdh_clickable_help("?",
                                ui_config_help_format,
                                display_format_disk);
    gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
    gtk_widget_show_all(button);

    mdh_widget_set_tip(NULL, button, "Display format help");

    label = gtk_label_new("Click command:");
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), label, 3, 4, 0, 1,
                                              GTK_FILL, GTK_FILL, 3, 0);
    gtk_widget_show(label);

    label = gtk_label_new("Sec(s):");
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), label, 4, 5, 0, 1,
                                              GTK_FILL, GTK_FILL, 3, 0);
    gtk_widget_show(label);

    for(i = 0; i < mdh_panel_disk_count(P_Disk[0]); i++) {
        hbox = gtk_hbox_new(FALSE, 5);
        gtk_box_pack_start(GTK_BOX(vbox4), hbox, TRUE, TRUE, 2);
        gtk_widget_show(hbox);

        p_button = gtk_check_button_new();
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(p_button),
            P_Disk[i]->enabled);
        gtk_table_attach(GTK_TABLE(table), p_button, 0, 1, i + 1, i + 2,
                                                     GTK_FILL, GTK_FILL, 3, 3);
        gtk_widget_show(p_button);

        p_disk_c[i][CFG_DISK_ENABLED] = p_button;

        entry = gtk_entry_new();
        gtk_entry_set_text(GTK_ENTRY(entry), P_Disk[i]->value);
        gtk_table_attach(GTK_TABLE(table), entry, 1, 2, i + 1, i + 2,
                                                  GTK_FILL, GTK_FILL, 3, 3);
        gtk_widget_show(entry);

        p_disk_c[i][CFG_DISK_VALUE] = entry;

        gtk_widget_set_sensitive(GTK_WIDGET(entry), P_Disk[i]->enabled);

        mdh_widget_set_tip(NULL, entry, "Filesystem to monitor");

        entry = gtk_entry_new();
        gtk_entry_set_text(GTK_ENTRY(entry), P_Disk[i]->display);
        gtk_widget_set_size_request(GTK_WIDGET(entry), 75, -1);
        gtk_table_attach(GTK_TABLE(table), entry, 2, 3, i + 1, i + 2,
                                                  GTK_FILL, GTK_FILL, 3, 3);
        gtk_widget_show(entry);

        p_disk_c[i][CFG_DISK_DISPLAY] = entry;

        gtk_widget_set_sensitive(GTK_WIDGET(entry), P_Disk[i]->enabled);

        mdh_widget_set_tip(NULL, entry,
                           "Display format (default: "_DISK_DIS")");

        entry = gtk_entry_new();
        gtk_entry_set_text(GTK_ENTRY(entry), P_Disk[i]->command);
        gtk_table_attach(GTK_TABLE(table), entry, 3, 4, i + 1, i + 2,
                                                  GTK_FILL, GTK_FILL, 3, 3);
        gtk_widget_show(entry);

        p_disk_c[i][CFG_DISK_COMMAND] = entry;

        gtk_widget_set_sensitive(GTK_WIDGET(entry), P_Disk[i]->enabled);

        mdh_widget_set_tip(NULL, entry,
            "Command to execute when panel is clicked (default: gmc %s)");

        adj = gtk_adjustment_new(P_Disk[i]->interval, _DISK_MIN, _DISK_MAX,
                                 5.0, 30.0, 0.0);
        spinner = gtk_spin_button_new(GTK_ADJUSTMENT(adj), 0, 0);
        gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spinner), FALSE);
        gtk_table_attach(GTK_TABLE(table), spinner, 4, 5, i + 1, i + 2,
                                                    GTK_FILL, GTK_FILL, 3, 3);
        gtk_widget_show(spinner);

        p_disk_c[i][CFG_DISK_SPINNER] = spinner;

        gtk_widget_set_sensitive(GTK_WIDGET(spinner), P_Disk[i]->enabled);

        text = g_strdup_printf("Set the update interval (default: %d)",
                               _DISK_DEF);
        mdh_widget_set_tip(NULL, spinner, text);
        g_free(text);

        g_signal_connect(G_OBJECT(p_button), "clicked",
                         G_CALLBACK(ui_config_toggle_panel), p_disk_c[i]);
    }

    hbox = gtk_hbox_new(FALSE, 3);
    gtk_box_pack_start(GTK_BOX(vbox4), hbox, FALSE, FALSE, 3);
    gtk_widget_show(hbox);

    button = gtk_check_button_new();
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button),
                                 mdh_panel_flags_bold(P_Disk[0]));
    gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 1);
    gtk_widget_show(button);

    p_disk_use_bold_w = button;

    label = gtk_label_new("Bold on");
    gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 1);
    gtk_widget_show(label);

    thresh = mdh_panel_disk_threshold(P_Disk[0]);

    adj = gtk_adjustment_new(thresh, _DISK_THRESH_MIN, _DISK_THRESH_MAX,
                             1.0, 5.0, 0.0);
    spinner = gtk_spin_button_new(GTK_ADJUSTMENT(adj), 0, 0);
    gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spinner), FALSE);
    gtk_box_pack_start(GTK_BOX(hbox), spinner, FALSE, FALSE, 1);
    gtk_widget_show(spinner);

    p_disk_thresh_w = spinner;

    gtk_widget_set_sensitive(GTK_WIDGET(spinner),
                             mdh_panel_flags_bold(P_Disk[0]));

    g_signal_connect(G_OBJECT(button), "clicked",
                     G_CALLBACK(ui_config_toggle_sensitive), p_disk_thresh_w);

    label = gtk_label_new("% usage");
    gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 1);
    gtk_widget_show(label);

    label = gtk_label_new("Displays disk usage.");
    gtk_widget_set_size_request(GTK_WIDGET(label), 450, -1);
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_box_pack_start(GTK_BOX(vbox3), label, FALSE, FALSE, 0);
    gtk_widget_show(label);

    gtk_notebook_append_page(GTK_NOTEBOOK(notebook),
                             vbox3, gtk_label_new("Disk"));
#endif /* _MDH_HAS_DISK */

#if _MDH_HAS_NET
    /* Net tab */

    vbox3 = gtk_vbox_new(FALSE, 3);
    gtk_widget_show(vbox3);

    gtk_container_set_border_width(GTK_CONTAINER(vbox3), 5);

    frame = mdh_frame_new_with_title("Toolbar: Net");
    gtk_box_pack_start(GTK_BOX(vbox3), frame, TRUE, TRUE, 0);
    gtk_widget_show(frame);

    vbox4 = gtk_vbox_new(FALSE, 0);
    gtk_widget_show(vbox4);

    gtk_container_set_border_width(GTK_CONTAINER(vbox4), 5);

    gtk_container_add(GTK_CONTAINER(frame), vbox4);

    table = gtk_table_new(mdh_panel_net_count(P_Net[0]), 5, FALSE);
    gtk_box_pack_start(GTK_BOX(vbox4), table, TRUE, TRUE, 0);
    gtk_widget_show(table);

#ifndef __sun__
    label = gtk_label_new("Interface:");
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), label, 1, 2, 0, 1,
                                              GTK_FILL, GTK_FILL, 3, 0);
    gtk_widget_show(label);
#endif

    hbox = gtk_hbox_new(FALSE, 0);
    gtk_table_attach(GTK_TABLE(table), hbox, 2, 3, 0, 1,
                                             GTK_FILL, GTK_FILL, 3, 0);
    gtk_widget_show(hbox);

    label = gtk_label_new("Display:");
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 0);
    gtk_widget_show(label);

    button = mdh_clickable_help("?",
                                ui_config_help_format,
                                display_format_net);
    gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
    gtk_widget_show_all(button);

    mdh_widget_set_tip(NULL, button, "Display format help");

    label = gtk_label_new("Click command:");
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), label, 3, 4, 0, 1,
                                              GTK_FILL, GTK_FILL, 3, 0);
    gtk_widget_show(label);

    label = gtk_label_new("Sec(s):");
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), label, 4, 5, 0, 1,
                                              GTK_FILL, GTK_FILL, 3, 0);
    gtk_widget_show(label);

    for(i = 0; i < mdh_panel_net_count(P_Net[0]); i++) {
        p_button = gtk_check_button_new();
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(p_button),
            P_Net[i]->enabled);
        gtk_table_attach(GTK_TABLE(table), p_button, 0, 1, i + 1, i + 2,
                                                     GTK_FILL, GTK_FILL, 3, 3);
        gtk_widget_show(p_button);

        p_net_c[i][CFG_NET_ENABLED] = p_button;

        mdh_widget_set_tip(NULL, p_button, "Enable the panel");

#ifndef __sun__
        entry = gtk_entry_new();
        gtk_entry_set_text(GTK_ENTRY(entry), P_Net[i]->value);
        gtk_table_attach(GTK_TABLE(table), entry, 1, 2, i + 1, i + 2,
                                                  GTK_FILL, GTK_FILL, 3, 3);
        gtk_widget_show(entry);

        gtk_widget_set_sensitive(GTK_WIDGET(entry), P_Net[i]->enabled);

        p_net_c[i][CFG_NET_VALUE] = entry;

        mdh_widget_set_tip(NULL, entry, "Interface to monitor");
#endif

        entry = gtk_entry_new();
        gtk_entry_set_text(GTK_ENTRY(entry), P_Net[i]->display);
        gtk_widget_set_size_request(GTK_WIDGET(entry), 75, -1);
        gtk_table_attach(GTK_TABLE(table), entry, 2, 3, i + 1, i + 2,
                                                  GTK_FILL, GTK_FILL, 3, 3);
        gtk_widget_show(entry);

        p_net_c[i][CFG_NET_DISPLAY] = entry;

        gtk_widget_set_sensitive(GTK_WIDGET(entry), P_Net[i]->enabled);

        mdh_widget_set_tip(NULL, entry,
                           "Display format (default: "_NET_DIS")");

        entry = gtk_entry_new();
        gtk_entry_set_text(GTK_ENTRY(entry), P_Net[i]->command);
        gtk_table_attach(GTK_TABLE(table), entry, 3, 4, i + 1, i + 2,
                                                  GTK_FILL, GTK_FILL, 3, 3);
        gtk_widget_show(entry);

        p_net_c[i][CFG_NET_COMMAND] = entry;

        gtk_widget_set_sensitive(GTK_WIDGET(entry), P_Net[i]->enabled);

        mdh_widget_set_tip(NULL, entry,
            "Command to execute when panel is clicked (default: ethereal %s)");

        adj = gtk_adjustment_new(P_Net[i]->interval, 1.0, 30.0, 1.0, 5.0, 0.0);
        spinner = gtk_spin_button_new(GTK_ADJUSTMENT(adj), 0, 0);
        gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spinner), FALSE);
        gtk_table_attach(GTK_TABLE(table), spinner, 4, 5, i + 1, i + 2,
                                                    GTK_FILL, GTK_FILL, 3, 3);
        gtk_widget_show(spinner);

        p_net_c[i][CFG_NET_SPINNER] = spinner;

        gtk_widget_set_sensitive(GTK_WIDGET(spinner), P_Net[i]->enabled);

        text = g_strdup_printf("Set the update interval (default: %d)",
                               _NET_DEF);
        mdh_widget_set_tip(NULL, spinner, text);
        g_free(text);

        g_signal_connect(G_OBJECT(p_button), "clicked",
                         G_CALLBACK(ui_config_toggle_panel), p_net_c[i]);
    }

    label = gtk_label_new("Displays network interface utilization.");
    gtk_widget_set_size_request(GTK_WIDGET(label), 450, -1);
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_box_pack_start(GTK_BOX(vbox3), label, FALSE, FALSE, 0);
    gtk_widget_show(label);

    gtk_notebook_append_page(GTK_NOTEBOOK(notebook),
                             vbox3, gtk_label_new("Net"));
#endif /* _MDH_HAS_NET */

#if _MDH_HAS_WEATHER
    /* Weather tab */

    vbox3 = gtk_vbox_new(FALSE, 3);
    gtk_widget_show(vbox3);

    gtk_container_set_border_width(GTK_CONTAINER(vbox3), 5);

    frame = mdh_frame_new_with_title("Toolbar: Weather");
    gtk_box_pack_start(GTK_BOX(vbox3), frame, TRUE, TRUE, 0);
    gtk_widget_show(frame);

    vbox4 = gtk_vbox_new(FALSE, 0);
    gtk_widget_show(vbox4);

    gtk_container_set_border_width(GTK_CONTAINER(vbox4), 5);

    gtk_container_add(GTK_CONTAINER(frame), vbox4);

    table = gtk_table_new(2, 5, FALSE);
    gtk_box_pack_start(GTK_BOX(vbox4), table, TRUE, TRUE, 0);
    gtk_widget_show(table);

    hbox = gtk_hbox_new(FALSE, 0);
    gtk_table_attach(GTK_TABLE(table), hbox, 1, 2, 0, 1,
                                             GTK_FILL, GTK_FILL, 3, 0);
    gtk_widget_show(hbox);

    label = gtk_label_new("Location ID:");
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 0);
    gtk_widget_show(label);

    button = mdh_clickable_help("?", ui_config_help_dialog_weather, NULL);
    gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
    gtk_widget_show_all(button);

    mdh_widget_set_tip(NULL, button, "Search for a location id");

    hbox = gtk_hbox_new(FALSE, 0);
    gtk_table_attach(GTK_TABLE(table), hbox, 2, 3, 0, 1,
                                             GTK_FILL, GTK_FILL, 3, 0);
    gtk_widget_show(hbox);

    label = gtk_label_new("Display:");
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 0);
    gtk_widget_show(label);

    button = mdh_clickable_help("?",
                                ui_config_help_format,
                                display_format_weather);
    gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
    gtk_widget_show_all(button);

    mdh_widget_set_tip(NULL, button, "Display format help");

    label = gtk_label_new("Click command:");
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), label, 3, 4, 0, 1,
                                              GTK_FILL, GTK_FILL, 3, 0);
    gtk_widget_show(label);

    label = gtk_label_new("Hour(s):");
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), label, 4, 5, 0, 1,
                                              GTK_FILL, GTK_FILL, 3, 0);
    gtk_widget_show(label);

    p_button = gtk_check_button_new();
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(p_button),
                                 P_Weather->enabled);
    gtk_table_attach(GTK_TABLE(table), p_button, 0, 1, 1, 2,
                                                 GTK_FILL, GTK_FILL, 3, 3);
    gtk_widget_show(p_button);

    p_weather_c[CFG_WEATHER_ENABLED] = p_button;

    mdh_widget_set_tip(NULL, p_button, "Enable the panel");

    entry = gtk_entry_new();
    gtk_entry_set_text(GTK_ENTRY(entry), P_Weather->value);
    gtk_table_attach(GTK_TABLE(table), entry, 1, 2, 1, 2,
                                             GTK_FILL, GTK_FILL, 3, 3);
    gtk_widget_show(entry);

    p_weather_c[CFG_WEATHER_VALUE] = entry;

    gtk_widget_set_sensitive(GTK_WIDGET(entry), P_Weather->enabled);

    mdh_widget_set_tip(NULL, entry,
                       "Location to use (default: "_WEATHER_LOC")");

    entry = gtk_entry_new();
    gtk_entry_set_text(GTK_ENTRY(entry), P_Weather->display);
    gtk_widget_set_size_request(GTK_WIDGET(entry), 75, -1);
    gtk_table_attach(GTK_TABLE(table), entry, 2, 3, 1, 2,
                                              GTK_FILL, GTK_FILL, 3, 3);
    gtk_widget_show(entry);

    p_weather_c[CFG_WEATHER_DISPLAY] = entry;

    gtk_widget_set_sensitive(GTK_WIDGET(entry), P_Weather->enabled);

    mdh_widget_set_tip(NULL, entry,
                       "Display format (default: "_WEATHER_DIS")");

    entry = gtk_entry_new();
    gtk_entry_set_text(GTK_ENTRY(entry), P_Weather->command);
    gtk_table_attach(GTK_TABLE(table), entry, 3, 4, 1, 2,
                                              GTK_FILL, GTK_FILL, 3, 3);
    gtk_widget_show(entry);

    p_weather_c[CFG_WEATHER_COMMAND] = entry;

    gtk_widget_set_sensitive(GTK_WIDGET(entry), P_Weather->enabled);

    {
        gchar *text = g_strdup_printf("Command to execute when panel is clicked (default: %s %%s)", mdh_get_browser());
        mdh_widget_set_tip(NULL, entry, text);
        g_free(text);
    }

    adj = gtk_adjustment_new(P_Weather->interval / 3600, _WEATHER_MIN / 3600,
                             _WEATHER_MAX / 3600, 1.0, 3.0, 0.0);
    spinner = gtk_spin_button_new(GTK_ADJUSTMENT(adj), 0, 0);
    gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spinner), FALSE);
    gtk_table_attach(GTK_TABLE(table), spinner, 4, 5, 1, 2,
                                                GTK_FILL, GTK_FILL, 3, 3);
    gtk_widget_show(spinner);

    p_weather_c[CFG_WEATHER_SPINNER] = spinner;

    gtk_widget_set_sensitive(GTK_WIDGET(spinner), P_Weather->enabled);

    button = gtk_check_button_new_with_label("Display condition icon");
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button),
                                 mdh_panel_flags_icon(P_Weather));
    gtk_box_pack_start(GTK_BOX(vbox4), button, FALSE, FALSE, 3);
    gtk_widget_show(button);

    mdh_widget_set_tip(NULL, button,
                       "Display the current weather condition icon");

    p_weather_c[CFG_WEATHER_ICON] = button;

    gtk_widget_set_sensitive(GTK_WIDGET(button), P_Weather->enabled);

    g_signal_connect(G_OBJECT(p_button), "clicked",
                     G_CALLBACK(ui_config_toggle_panel), p_weather_c);

    hbox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(vbox3), hbox, FALSE, FALSE, 0);
    gtk_widget_set_size_request(GTK_WIDGET(hbox), 450, -1);
    gtk_widget_show(hbox);

    label = gtk_label_new("Displays current weather conditions, provided by ");
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
    gtk_widget_show(label);

    button = mdh_clickable_uri("Weather.com",
                               ui_config_view_weather_home,
                               NULL);
    gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
    gtk_widget_show_all(button);

    mdh_widget_set_tip(NULL, button, "View Weather.com");

    {
        static gchar REGISTERED[3] = { 0xc2, 0xae, 0 };
        label = gtk_label_new(REGISTERED);
    }

    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
    gtk_widget_show(label);

    gtk_notebook_append_page(GTK_NOTEBOOK(notebook),
                             vbox3, gtk_label_new("Weather"));
#endif /* _MDH_HAS_WEATHER */

#if _MDH_HAS_UPTIME
    /* Uptime tab */

    vbox3 = gtk_vbox_new(FALSE, 3);
    gtk_widget_show(vbox3);

    gtk_container_set_border_width(GTK_CONTAINER(vbox3), 5);

    frame = mdh_frame_new_with_title("Toolbar: Uptime");
    gtk_box_pack_start(GTK_BOX(vbox3), frame, TRUE, TRUE, 0);
    gtk_widget_show(frame);

    vbox4 = gtk_vbox_new(FALSE, 0);
    gtk_widget_show(vbox4);

    gtk_container_set_border_width(GTK_CONTAINER(vbox4), 5);

    gtk_container_add(GTK_CONTAINER(frame), vbox4);

    table = gtk_table_new(2, 4, FALSE);
    gtk_box_pack_start(GTK_BOX(vbox4), table, TRUE, TRUE, 0);
    gtk_widget_show(table);

    hbox = gtk_hbox_new(FALSE, 0);
    gtk_table_attach(GTK_TABLE(table), hbox, 1, 2, 0, 1,
                                             GTK_FILL, GTK_FILL, 3, 0);
    gtk_widget_show(hbox);

    label = gtk_label_new("Display:");
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 0);
    gtk_widget_show(label);

    button = mdh_clickable_help("?",
                                ui_config_help_format,
                                display_format_uptime);
    gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
    gtk_widget_show_all(button);

    mdh_widget_set_tip(NULL, button, "Display format help");

    label = gtk_label_new("Click command:");
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), label, 2, 3, 0, 1,
                                              GTK_FILL, GTK_FILL, 3, 0);
    gtk_widget_show(label);

    label = gtk_label_new("Sec(s):");
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), label, 3, 4, 0, 1,
                                              GTK_FILL, GTK_FILL, 3, 0);
    gtk_widget_show(label);

    p_button = gtk_check_button_new();
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(p_button), P_Uptime->enabled);
    gtk_table_attach(GTK_TABLE(table), p_button , 0, 1, 1, 2,
                                                  GTK_FILL, GTK_FILL, 3, 3);
    gtk_widget_show(p_button);

    p_uptime_c[CFG_UPTIME_ENABLED] = p_button;

    mdh_widget_set_tip(NULL, p_button, "Enable the panel");

    entry = gtk_entry_new();
    gtk_entry_set_text(GTK_ENTRY(entry), P_Uptime->display);
    gtk_widget_set_size_request(GTK_WIDGET(entry), 75, -1);
    gtk_table_attach(GTK_TABLE(table), entry, 1, 2, 1, 2,
                                              GTK_FILL, GTK_FILL, 3, 3);
    gtk_widget_show(entry);

    p_uptime_c[CFG_UPTIME_DISPLAY] = entry;

    gtk_widget_set_sensitive(GTK_WIDGET(entry), P_Uptime->enabled);

    mdh_widget_set_tip(NULL, entry,
                       "Display format (default: "_UPTIME_DIS")");

    entry = gtk_entry_new();
    gtk_entry_set_text(GTK_ENTRY(entry), P_Uptime->command);
    gtk_table_attach(GTK_TABLE(table), entry, 2, 3, 1, 2,
                                              GTK_FILL, GTK_FILL, 3, 3);
    gtk_widget_show(entry);

    p_uptime_c[CFG_UPTIME_COMMAND] = entry;

    gtk_widget_set_sensitive(GTK_WIDGET(entry), P_Uptime->enabled);

    mdh_widget_set_tip(NULL, entry,
        "Command to execute when panel is clicked (default: none)");

    adj = gtk_adjustment_new(P_Uptime->interval, _UPTIME_MIN, _UPTIME_MAX,
                             1.0, 5.0, 0.0);
    spinner = gtk_spin_button_new(GTK_ADJUSTMENT(adj), 0, 0);
    gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spinner), FALSE);
    gtk_table_attach(GTK_TABLE(table), spinner, 3, 4, 1, 2,
                                                GTK_FILL, GTK_FILL, 3, 3);
    gtk_widget_show(spinner);

    p_uptime_c[CFG_UPTIME_SPINNER] = spinner;

    gtk_widget_set_sensitive(GTK_WIDGET(spinner), P_Uptime->enabled);

    text = g_strdup_printf("Set the update interval (default: %d)",
                           _UPTIME_DEF);
    mdh_widget_set_tip(NULL, spinner, text);
    g_free(text);

    g_signal_connect(G_OBJECT(p_button), "clicked",
                     G_CALLBACK(ui_config_toggle_panel), p_uptime_c);

    label = gtk_label_new("Displays system uptime.");
    gtk_widget_set_size_request(GTK_WIDGET(label), 450, -1);
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_box_pack_start(GTK_BOX(vbox3), label, FALSE, FALSE, 0);
    gtk_widget_show(label);

    gtk_notebook_append_page(GTK_NOTEBOOK(notebook),
                             vbox3, gtk_label_new("Uptime"));
#endif /* _MDH_HAS_UPTIME */

    /* Custom tab */

    vbox3 = gtk_vbox_new(FALSE, 3);
    gtk_widget_show(vbox3);

    gtk_container_set_border_width(GTK_CONTAINER(vbox3), 5);

    frame = mdh_frame_new_with_title("Toolbar: Custom");
    gtk_box_pack_start(GTK_BOX(vbox3), frame, TRUE, TRUE, 0);
    gtk_widget_show(frame);

    vbox4 = gtk_vbox_new(FALSE, 0);
    gtk_widget_show(vbox4);

    gtk_container_set_border_width(GTK_CONTAINER(vbox4), 5);

    gtk_container_add(GTK_CONTAINER(frame), vbox4);

    table = gtk_table_new(mdh_panel_custom_count(P_Custom[0]), 4, FALSE);
    gtk_box_pack_start(GTK_BOX(vbox4), table, TRUE, TRUE, 0);
    gtk_widget_show(table);

    label = gtk_label_new("Script/command:");
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), label, 1, 2, 0, 1,
                                              GTK_FILL, GTK_FILL, 3, 0);
    gtk_widget_show(label);

    label = gtk_label_new("Click command:");
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), label, 2, 3, 0, 1,
                                              GTK_FILL, GTK_FILL, 3, 0);
    gtk_widget_show(label);

    label = gtk_label_new("Sec(s):");
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), label, 3, 4, 0, 1,
                                              GTK_FILL, GTK_FILL, 3, 0);
    gtk_widget_show(label);

    for(i = 0; i < mdh_panel_custom_count(P_Custom[0]); i++) {
        p_button = gtk_check_button_new();
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(p_button),
            P_Custom[i]->enabled);
        gtk_table_attach(GTK_TABLE(table), p_button, 0, 1, i + 1, i + 2,
                                                     GTK_FILL, GTK_FILL, 3, 3);
        gtk_widget_show(p_button);

        p_custom_c[i][CFG_CUSTOM_ENABLED] = p_button;

        mdh_widget_set_tip(NULL, p_button, "Enable the panel");

        entry = gtk_entry_new();
        gtk_entry_set_text(GTK_ENTRY(entry), P_Custom[i]->value);
        gtk_table_attach(GTK_TABLE(table), entry, 1, 2, i + 1, i + 2,
                                                  GTK_FILL, GTK_FILL, 3, 3);
        gtk_widget_show(entry);

        p_custom_c[i][CFG_CUSTOM_VALUE] = entry;

        gtk_widget_set_sensitive(GTK_WIDGET(entry), P_Custom[i]->enabled);

        mdh_widget_set_tip(NULL, entry, "Script or command to execute");

        entry = gtk_entry_new();
        gtk_entry_set_text(GTK_ENTRY(entry), P_Custom[i]->command);
        gtk_table_attach(GTK_TABLE(table), entry, 2, 3, i + 1, i + 2,
                                                  GTK_FILL, GTK_FILL, 3, 3);
        gtk_widget_show(entry);

        p_custom_c[i][CFG_CUSTOM_COMMAND] = entry;

        gtk_widget_set_sensitive(GTK_WIDGET(entry), P_Custom[i]->enabled);

        mdh_widget_set_tip(NULL, entry,
                           "Command to execute when panel is clicked");

        adj = gtk_adjustment_new(P_Custom[i]->interval,
                                 _CUSTOM_MIN, _CUSTOM_MAX,
                                 1.0, 15.0, 0.0);
        spinner = gtk_spin_button_new(GTK_ADJUSTMENT(adj), 0, 0);
        gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spinner), FALSE);
        gtk_table_attach(GTK_TABLE(table), spinner, 3, 4, i + 1, i + 2,
                                                    GTK_FILL, GTK_FILL, 3, 3);
        gtk_widget_show(spinner);

        p_custom_c[i][CFG_CUSTOM_SPINNER] = spinner;

        gtk_widget_set_sensitive(GTK_WIDGET(spinner), P_Custom[i]->enabled);

        text = g_strdup_printf("Set the update interval (default: %d)",
                               _CUSTOM_DEF);
        mdh_widget_set_tip(NULL, spinner, text);
        g_free(text);

        g_signal_connect(G_OBJECT(p_button), "clicked",
                         G_CALLBACK(ui_config_toggle_panel), p_custom_c[i]);
    }

    label = gtk_label_new("Displays output of selected command(s).");
    gtk_widget_set_size_request(GTK_WIDGET(label), 450, -1);
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_box_pack_start(GTK_BOX(vbox3), label, FALSE, FALSE, 0);
    gtk_widget_show(label);

    gtk_notebook_append_page(GTK_NOTEBOOK(notebook),
                             vbox3, gtk_label_new("Custom"));

    /* Custom tab end */

    hbox = gtk_hbox_new(TRUE, 3);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);
    gtk_widget_show(hbox);

    button = gtk_button_new_from_stock(GTK_STOCK_CANCEL);
    gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
    gtk_widget_show(button);

    g_signal_connect(G_OBJECT(button), "clicked",
                     G_CALLBACK(ui_config_cancel), window);

    g_signal_connect_swapped(G_OBJECT(button), "clicked",
                             G_CALLBACK(gtk_widget_destroy), window);

    mdh_widget_set_tip(NULL, button,
                       "Cancel new (or unapplied) panel settings");

    button = gtk_button_new_from_stock(GTK_STOCK_APPLY);
    gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
    gtk_widget_show(button);

    mdh_widget_set_tip(NULL, button, "Apply new panel settings");

    g_signal_connect(G_OBJECT(button), "clicked",
                     G_CALLBACK(ui_config_apply), NULL);

    button = gtk_button_new_from_stock(GTK_STOCK_OK);
    gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
    gtk_widget_show(button);

    g_signal_connect(G_OBJECT(button), "clicked",
                     G_CALLBACK(ui_config_apply), window);

    g_signal_connect_swapped(G_OBJECT(button), "clicked",
                             G_CALLBACK(gtk_widget_destroy), window);

    mdh_widget_set_tip(NULL, button, "Close and Apply new panel settings");

    gtk_widget_show(window);
}
