/*
 * mdh (MailDooHicky), a GTK+-based toolbar.
 *
 * Copyright (c) 2003-2005 by Mike Hokenson <mdh at gozer dot org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <unistd.h>

#include <gtk/gtk.h>

#include "mdh.h"

#include "toolbar.h"

#include "ui_run.h"

#include "config.h"
#include "run_history.h"
#include "widget.h"
#include "window.h"

static gboolean run_close,
                run_save;

static const gchar *run_xterm;

static GtkWidget *run_window;
static GtkWidget *run_terminal_w;

static GtkUIManager *run_ui;

static const gchar *run_ui_info =
    "<ui>"
    "  <menubar name='MenuBar'>"
    "    <menu action='OptionMenu'>"
    "      <menuitem action='SaveHist'/>"
    "      <menuitem action='CloseWin'/>"
    "    </menu>"
    "  </menubar>"
    "</ui>";

#define _RUN_MENU_SAVE_HIST "/MenuBar/OptionMenu/SaveHist"
#define _RUN_MENU_CLOSE_WIN "/MenuBar/OptionMenu/CloseWin"

static void ui_run_save_toggle(GtkAction *action)
{
    GtkWidget *w = gtk_ui_manager_get_widget(run_ui, _RUN_MENU_SAVE_HIST);

    run_save = gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(w));

    mdh_config_set_boolean("run", "save", run_save);
}

static void ui_run_close_toggle(GtkAction *action)
{
    GtkWidget *w = gtk_ui_manager_get_widget(run_ui, _RUN_MENU_CLOSE_WIN);

    run_close = gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(w));

    mdh_config_set_boolean("run", "close", run_close);
}

static GtkActionEntry entries[] = {
        { "OptionMenu", NULL, "_Options" }
    };

static guint n_entries = G_N_ELEMENTS(entries);

static GtkToggleActionEntry toggle_entries[] = {
        { "SaveHist", NULL, "Save history",     NULL, NULL,
          G_CALLBACK(ui_run_save_toggle),  FALSE },
        { "CloseWin", NULL, "Close on execute", NULL, NULL,
          G_CALLBACK(ui_run_close_toggle), FALSE }
    };

static guint n_toggle_entries = G_N_ELEMENTS(toggle_entries);

static GtkWidget *ui_run_menu_create(void)
{
    GtkActionGroup *actions;
    GtkWidget *w;
    GError *err;

    run_ui = gtk_ui_manager_new();

    actions = gtk_action_group_new("Actions");

    gtk_action_group_add_actions(actions, entries, n_entries, NULL);

    gtk_action_group_add_toggle_actions(actions,
                                        toggle_entries,
                                        n_toggle_entries,
                                        NULL);

    gtk_ui_manager_insert_action_group(run_ui, actions, 0);

    if(!gtk_ui_manager_add_ui_from_string(run_ui, run_ui_info, -1, &err)) {
        g_critical("%s", err->message);
        g_error_free(err);
        return(NULL);
    }

    w = gtk_ui_manager_get_widget(run_ui, _RUN_MENU_SAVE_HIST);

    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(w), run_save);

    w = gtk_ui_manager_get_widget(run_ui, _RUN_MENU_CLOSE_WIN);

    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(w), run_close);

    return(gtk_ui_manager_get_widget(run_ui, "/MenuBar"));
}

static void ui_run_close(GtkWidget *widget)
{
    mdh_window_item_open_set(&wi_run, FALSE);
}

static void ui_run_execute(GtkComboBox *combo)
{
    gchar *command, *text, *cmd;

    GError *err = NULL;

    text = g_strdup(gtk_entry_get_text(GTK_ENTRY(GTK_BIN(combo)->child)));

    cmd = g_strstrip(text);

    if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(run_terminal_w))) {
        gchar buf[LINE_MAX];

        gint len;

        len = g_snprintf(buf, sizeof(buf), "%s -e", run_xterm);

        /*
         * quick test to see if what's already in the entry contains
         * '<Toolbar->run_item> -e'. probably shouldn't do much more testing
         * in this area, don't want to get annoying.
         */

        if(!strncmp(cmd, buf, len))
            command = g_strdup(cmd);
        else
            command = g_strdup_printf("%s -e %s", run_xterm, cmd);
    } else
        command = g_strdup(cmd);

    g_free(text);

    /* add in reverse order for history dropdown */
    mdh_run_history_prepend(command);

    gtk_editable_select_region(GTK_EDITABLE(GTK_BIN(combo)->child), 0, -1);

    /* keep run window open if enabled, or execution failed */
    if(!g_spawn_command_line_async(command, &err)) {
        GList *p;

        for(p = mdh_run_history_list(); p; p = g_list_next(p))
            gtk_combo_box_append_text(GTK_COMBO_BOX(combo), p->data);

        mdh_window_error("Execution failed.", err->message);

        g_error_free(err);
    } else if(run_close)
        gtk_widget_destroy(run_window);

    /* don't free @command, stored in run history */
}

void mdh_ui_run(void)
{
    GtkWidget *window;
    GtkWidget *vbox;
    GtkWidget *hbox;
    GtkWidget *menubar;
    GtkWidget *button;

    GtkComboBox *combo;

    if(mdh_window_item_open(&wi_run))
        return;

    mdh_window_item_open_set(&wi_run, TRUE);

    if(!run_xterm) {   /* first run */
        if(!mdh_config_get_boolean("run", "save", &run_save))
            run_save = FALSE;

        if(!mdh_config_get_boolean("run", "close", &run_close))
            run_close = TRUE;

        if(!mdh_config_get_string("run", "term", &run_xterm))
            run_xterm = mdh_get_xterm();
    }

    run_window = window = gtk_window_new(GTK_WINDOW_TOPLEVEL);

    gtk_window_set_title(GTK_WINDOW(window), "Run...");
    gtk_window_set_resizable(GTK_WINDOW(window), FALSE);
    gtk_widget_set_size_request(GTK_WIDGET(window), 300, -1);

    g_signal_connect(G_OBJECT(window), "destroy",
                     G_CALLBACK(ui_run_close), NULL);

    g_signal_connect(G_OBJECT(window), "event",
                     G_CALLBACK(mdh_window_event), NULL);

    g_signal_connect(G_OBJECT(window), "expose_event",
                     G_CALLBACK(mdh_window_expose_event), NULL);

    gtk_container_set_border_width(GTK_CONTAINER(window), 5);

    vbox = gtk_vbox_new(FALSE, 5);
    gtk_widget_show(vbox);

    gtk_container_add(GTK_CONTAINER(window), vbox);

    menubar = ui_run_menu_create();
    gtk_box_pack_start(GTK_BOX(vbox), GTK_WIDGET(menubar), FALSE, FALSE, 0);
    gtk_widget_show(menubar);

    combo = (GtkComboBox *) gtk_combo_box_entry_new_text();
    gtk_box_pack_start(GTK_BOX(vbox), GTK_WIDGET(combo), FALSE, FALSE, 5);
    gtk_widget_show(GTK_WIDGET(combo));

    mdh_widget_grab_default(GTK_WIDGET(combo));

    if(mdh_run_history_list()) {
        GList *p;
        GtkEntry *entry = GTK_ENTRY(GTK_BIN(combo)->child);

        /* insert and select the previous command */
        gtk_entry_set_text(entry, mdh_run_history_list()->data);
        gtk_editable_select_region(GTK_EDITABLE(entry), 0, -1);

        for(p = mdh_run_history_list(); p; p = g_list_next(p))
            gtk_combo_box_append_text(GTK_COMBO_BOX(combo), p->data);
    }

    gtk_entry_set_activates_default(GTK_ENTRY(GTK_BIN(combo)->child), TRUE);

    button = gtk_check_button_new_with_label("Run in terminal");
    gtk_box_pack_start(GTK_BOX(vbox), button, FALSE, FALSE, 0);
    gtk_widget_show(button);

    if(!*run_xterm)
        gtk_widget_set_sensitive(GTK_WIDGET(button), FALSE);

    run_terminal_w = button;

    hbox = gtk_hbox_new(TRUE, 3);
    gtk_widget_show(hbox);

    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

    button = mdh_button_new_from_stock(GTK_STOCK_EXECUTE, "Run");
    gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
    gtk_widget_show(button);

    GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);

    gtk_window_set_default(GTK_WINDOW(window), button);

    g_signal_connect_swapped(G_OBJECT(button), "clicked",
                             G_CALLBACK(ui_run_execute), combo);

    button = mdh_button_new_from_stock(GTK_STOCK_CLOSE, "Close");
    gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
    gtk_widget_show(button);

    g_signal_connect_swapped(G_OBJECT(button), "clicked",
                             G_CALLBACK(gtk_widget_destroy), window);

    gtk_widget_show(window);
}
