/*
 MeCab -- Yet Another Part-of-Speech and Morphological Analyzer

 $Id: viterbi.cpp,v 1.15 2003/01/04 05:20:21 taku-ku Exp $;

 Copyright (C) 2001-2002  Taku Kudo <taku-ku@is.aist-nara.ac.jp>
 All rights reserved.

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Library General Public
 License as published by the Free Software Foundation; either
 version 2 of the License, or (at your option) any later verjsion.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Library General Public License for more details.

 You should have received a copy of the GNU Library General Public
 License along with this library; if not, write to the
 Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 Boston, MA 02111-1307, USA.
*/

#include "viterbi.h"
#include "common.h"
#include <cstring>
#include <algorithm>
#include <stdexcept>

namespace MeCab {

  Viterbi::Viterbi (Param &param, Tokenizer *t)
     : alloc_size (MIN_INPUT_BUFFER_SIZE)
  {
    try {
      tokenizer = t;
     
      if (param.getProfileInt ("build-all-lattice")) {
	pathFreeList = new FreeList<Path> (PATH_FREELIST_SIZE);
	_connect = &Viterbi::connectWithAllPath;
      } else {
	pathFreeList = 0;
	_connect = &Viterbi::connectNormal;
      }

      endNodeList = new Node * [alloc_size];
      lookupCache = new Node * [alloc_size];
    }
     
    catch (std::exception &e) {
      _what = std::string ("Viterbi::Viterbi(): ") +  e.what ();
      throw std::runtime_error (_what);
    }
  }

  Viterbi::~Viterbi () 
  { 
     delete [] endNodeList;
     delete [] lookupCache;
     delete pathFreeList; 
  }

  void Viterbi::clear () 
  {
     tokenizer->clear ();
     if (pathFreeList) pathFreeList->free();
  }
  
  Node *Viterbi::lookup (unsigned int pos)
  {
    Node *resultNode = 0;

    if (lookupCache[pos]) { 
      for (Node *node = lookupCache[pos]; node; node = node->rnext) { 
	Node *newNode = tokenizer->getNewNode ();
	unsigned int id = newNode->id; 
	memcpy (newNode, node, sizeof (Node));
	newNode->rnext = resultNode;
	newNode->id = id;
	resultNode = newNode; 
      }
    } else {
      resultNode = tokenizer->lookup (begin + pos, end);
      lookupCache[pos] = resultNode;
    }

    return resultNode;
  }
  
  Node* Viterbi::analyze (const char *sentence, unsigned int len)
  {
    try {
      if (len >= alloc_size) {
	do { alloc_size *= 2; } while (len > alloc_size);
	delete [] endNodeList;
	delete [] lookupCache;
	endNodeList = new Node * [alloc_size];
	lookupCache = new Node * [alloc_size];
      }

      memset (endNodeList, 0, sizeof (Node *) * (len+1));
      memset (lookupCache, 0, sizeof (Node *) * (len+1));
     
      clear ();     
      begin = const_cast<char *>(sentence);
      end   = begin + len;
      bosNode = tokenizer->getBOSNode ();
      bosNode->surface = begin;
      endNodeList[0] = bosNode;

      for (unsigned int pos = 0; pos < len;  pos++){
	if (endNodeList[pos]) {
	  Node *rNode = lookup (pos);
	  connect (pos, rNode);
	}
      }

      eosNode = tokenizer->getEOSNode ();
      eosNode->surface = end; 
      lookupCache[len] = eosNode;
      for (unsigned int pos = len; (int)pos >= 0;  pos--) {
	if (endNodeList[pos]) {
	  connect (pos, eosNode);
	  break;
	}
      }
       
      Node *node = eosNode;
      for (Node *prevNode; node->prev; ){
	prevNode = node->prev;
	prevNode->next = node;
	node = prevNode;
      }

      return bosNode;
    }
     
    catch (std::exception &e) {
       _what = std::string ("Viterbi::analyze()") + e.what ();
       return 0;
    }
  }
}

#undef _VITERBI_WITH_ALL_PATH
#include "viterbisub.h"
#define _VITERBI_WITH_ALL_PATH
#include "viterbisub.h"
