#include "konst.socket.h"

#include <netdb.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <limits.h>
#include <arpa/inet.h>
#include <netinet/in.h>

#include <sys/time.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/wait.h>

int atoport(char *service, char *proto) {
    int port;
    long int lport;
    struct servent *serv;
    char *errpos;

    serv = getservbyname(service,proto);
    if(serv) port = serv->s_port; else {
        lport = strtol(service, &errpos, 0);
        if(errpos[0] || lport < 1 || lport > 65535) return -1;
        port = htons(lport);
    }
    return port;
}

struct in_addr *atoaddr(const char *address) {
    struct hostent *host;
    static struct in_addr saddr;

    saddr.s_addr = inet_addr(address);
    if(saddr.s_addr != -1) return &saddr;
    host = gethostbyname(address);
    if(host) return (struct in_addr *) *host->h_addr_list;
    return 0;
}

void sig_chld(int signal_type) {
    int pid, status;
    while ((pid = wait3(&status, WNOHANG, NULL)) > 0);
}

void ignore_pipe() {
    struct sigaction sig;

    sig.sa_handler = SIG_IGN;
    sig.sa_flags = 0;
    sigemptyset(&sig.sa_mask);
    sigaction(SIGPIPE, &sig, 0);
}

int sockWaitForData(int sock, int seconds) {
    struct timeval tv;
    fd_set ds;
    int res;

    FD_ZERO(&ds);
    FD_SET(sock, &ds);

    tv.tv_sec = seconds != -1 ? seconds : 0;
    tv.tv_usec = 0;
    res = select(sock+1, &ds, 0, 0, seconds ? &tv : 0);

    if(res < 0) return res; else res = 0;
    if(FD_ISSET(sock, &ds)) res = 1;
    return res;
}

int sockWaitForWrite(int sock, int seconds) {
    int res;
    struct timeval tv;
    fd_set ds;

    FD_ZERO(&ds);
    FD_SET(sock, &ds);

    tv.tv_sec = seconds != -1 ? seconds : 0;
    tv.tv_usec = 0;
    res = select(sock+1, 0, &ds, 0, &tv);

    if(res < 0) return res;
    res = 0;
    if(FD_ISSET(sock, &ds)) res |= 1;
    return res;
}

// ----------------------------------------------------------------------------

streamsocketabstract::streamsocketabstract() {
    sockfd = -1;
    fconnected = false;
}

streamsocketabstract::~streamsocketabstract() {
    disconnect();
}

int streamsocketabstract::getfd() {
    return sockfd;
}

bool streamsocketabstract::connected() {
    return fconnected;
}

bool streamsocketabstract::readready() {
    return true;
}

bool streamsocketabstract::writeready() {
    return true;
}

void streamsocketabstract::disconnect() {
}

// ----------------------------------------------------------------------------

streamsocketclient::streamsocketclient() {
}

streamsocketclient::streamsocketclient(string dest, int port) {
    connect(dest, port);
}

bool streamsocketclient::readready() {
    fd_set fds;
    struct timeval tm;
    
    tm.tv_sec = tm.tv_usec = 0;
    FD_ZERO(&fds);
    FD_SET(sockfd, &fds);
    
    select(sockfd+1, &fds, 0, 0, &tm);
    return FD_ISSET(sockfd, &fds);
}

bool streamsocketclient::writeready() {
    fd_set fds;
    struct timeval tm;
    
    tm.tv_sec = tm.tv_usec = 0;
    FD_ZERO(&fds);
    FD_SET(sockfd, &fds);
    
    select(sockfd+1, 0, &fds, 0, &tm);
    return FD_ISSET(sockfd, &fds);
}

void streamsocketclient::connect(string dest, int port) {
    struct in_addr *addr;
    struct sockaddr_in address;

    if(addr = atoaddr(dest.c_str())) {
        memset(&address, 0, sizeof(address));
        address.sin_family = AF_INET;
        address.sin_port = htons(port);
        address.sin_addr.s_addr = addr->s_addr;
        sockfd = socket(AF_INET, SOCK_STREAM, 0);

        int res = ::connect(sockfd, (struct sockaddr *) &address, sizeof(address));
        fconnected = (res >= 0);
    }
}

void streamsocketclient::disconnect() {
    if(sockfd > 0) {
        shutdown(sockfd, 2);
        close(sockfd);
        sockfd = -1;
        fconnected = false;
    }
}

string streamsocketclient::readline() {
    string text;
    char last_read = 0;
    int res;

    if(fconnected) {
        while(last_read != 10) {
            res = read(sockfd, &last_read, 1);
            if(res <= 0) {
                fconnected = false;
                break;
            } else {
                if((last_read != 10) && (last_read != 13)) text += last_read;
            }
        }
    }

    return text;
}

void streamsocketclient::write(string text) {
    int res, bytes_sent = 0;
    const char *buf = text.c_str();

    while(bytes_sent < text.size()) {
        do res = ::write(sockfd, buf, text.size()-bytes_sent);
        while((res < 0) && (errno == EINTR));

        if(res < 0) fconnected = false; else {
            bytes_sent += res;
            buf += res;
        }
    }
}

void streamsocketclient::writeline(string text) {
    write(text + "\n");
}

int streamsocketclient::readblock(void *buf, int size) {
    int toread, res;
    fd_set fds;
    struct timeval tm;

    tm.tv_sec = 10;
    tm.tv_usec = 0;

    FD_ZERO(&fds);
    FD_SET(sockfd, &fds);

    res = select(sockfd+1, &fds, 0, 0, &tm);

    if((res != -1) && !ioctl(sockfd, FIONREAD, &toread)) {
        if(FD_ISSET(sockfd, &fds)) {
            if(toread) {
                if(toread > size) toread = size;
                return recv(sockfd, buf, toread, 0);
            } else {
                fconnected = false;
            }
        }
    }

    return 0;
}

streamsocketclient& streamsocketclient::operator<<(string s) {
    write(s);
    return *this;
}

streamsocketclient& streamsocketclient::operator<<(int n) {
    write(i2str(n));
    return *this;
}
