#!/usr/bin/perl

########################################################################
# metaf.pl -- CGI wrapper script to display METAR/TAF conversion results
#
# copyright (c) metaf2xml 2006
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License as
# published by the Free Software Foundation; either version 2 of the
# License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA
########################################################################

########################################################################
# some things strictly perl
########################################################################
use strict;
use warnings;

=head1 NAME

metaf.pl -- CGI wrapper script to display METAR/TAF conversion results

=head1 SYNOPSIS

as a standalone script:

  metaf.pl 'type=...&lang=...&format=...&source=...&msg=...'

or with HTML:

  <form method="post" action="metaf.pl">
   <input name="type" ... >
   <input name="lang" ... >
   <input name="format" ... >
   <input name="source" ... >
   <input name="msg" ... >
  </form>

=head1 ARGUMENTS

The script uses the following 5 CGI parameters:

=over

=item B<type>

type of message to be parsed: metar (default), taf

=item B<lang>

language for output of script and display of parsed message: de (default), en,
es, ru

=item B<format>

output format: html (default), text, or xml (with style sheet reference)

=item B<source>

source for current weather: noaa (default), adds, addsds, file

=item B<msg>

message to be parsed or the keyword C<FETCH> and a list of airport ids

=back

=head1 DESCRIPTION

This CGI script provides a web interface to display METAR/TAF conversion
results. It can be used with GET or POST methods. It uses the perl module
C<CGI>.

=cut

use CGI qw/:standard -no_xhtml/;

#
# restrict use of CGI:
#

# uploads are not allowed
$CGI::DISABLE_UPLOADS = 1;

=pod

The length of the body of the POST request (not the length of query string of
the URL) is restricted to 350 bytes.

=cut

$CGI::POST_MAX = 350;

=pod

The following directories and files are used by the script and can be adapted to
local needs by modifying the script (e.g. during install):

=over

=item B<$METAF2XML_BIN>

directory where the C<metaf2xml.pl> is expected (default: C</opt/metaf2xml/bin>)

=item B<$METAF2XML_XSL>

directory where the XSL scripts are located (default: C</opt/metaf2xml/share>)

=item B<$CGI_DATA_DIR>

directory for the local copies of messages (default: C</tmp>)

=item B<$CGI_TMP_DIR>

directory for the temporary XML file (default: C</tmp>)

=item B<$CURL>

path and options for C<curl> (default: C<curl -sSf --stderr ->)

=item B<$CURL_PROXY_OPTS>

options for C<curl> if a HTTP proxy should be used (default: empty)

=back

=cut

########################################################################
# config section, possibly changed by install
########################################################################
my $METAF2XML_BIN = '/opt/metaf2xml/bin';
my $METAF2XML_XSL = '/opt/metaf2xml/share';
my $CGI_DATA_DIR = '/tmp';
my $CGI_LOG_FILE = '/tmp/metaf2xml.log';
my $CGI_TMP_DIR = '/tmp';
my $CURL = 'curl -sSf --stderr -';
my $CURL_PROXY_OPTS = '-x localhost:3128';

#
# filter line(s) for $icao from $filename
#
sub process_file {
    my ($icao, $filename) = @_;
    my ($found, $response, @arr);

    $icao =~ s/ /|/g;

    if (!open(FP, $filename)) {
        return ("$icao - ERROR - " . uc "$filename - $!");
    }
    $found = 0;
    while (<FP>) {
        if (!$found) {
            next unless /^(?:METAR |TAF |SPECI )?(?:$icao) /;
            $found = 1;
            chomp();
            ($response = $_) =~ s/^(?:METAR |TAF |SPECI )//;
        } else {
            if (/^$/) {
                # remove existing entries with same $id from @arr
                my $id;
                $id = ($response =~ /^(....)/)[0];
                @arr = grep { !/^$id / } @arr;

                $response =~ s/[\n\r]+//g;
                $response =~ s/ +/ /g;
                push @arr, $response;
                $found = 0;
            } else {
                chomp();
                $response .= ' ' . $_;
            }
        }
    }
    close FP;
    push @arr, $response if $found;
    return @arr;
}

#
# fetch line(s) for $icao from NOAA
#
sub process_noaa {
    my ($icao, $type) = @_;
    my ($response, $url);

    $icao =~ s/ /+/g;

    $url = 'http://weather.noaa.gov/cgi-bin/mget' . $type . '.pl?cccc=';

    if (!open(FP, "$CURL $CURL_PROXY_OPTS $url'$icao' || echo ERROR |")) {
        return ("$icao - ERROR - " . uc $!);
    }
    {
        local $/;
        $response = <FP>;
    }
    close FP;
    if ($response =~ /ERROR/) {
        $response =~ s/[\n\r]+/ /g;
        return ("$icao - ERROR - " . uc "$url - $response");
    }
    $response =~ s/[\n\r]+/ /g;
    $response =~ s/ +/ /g;
    return $response =~ /<font face=.couri[^>]*?> *(?:<pre>)? *(?:METAR|SPECI|TAF)? *([^<]+)/go;
}

#
# fetch line(s) for $icao from ADDS
#
sub process_adds {
    my ($icao) = @_;
    my ($response, $url, $data);

    $icao =~ s/ /+/g;

    $url  = 'http://adds.aviationweather.gov/metars/';
    $data = 'std_trans=standard&chk_metars=on&hoursStr=most+recent+only&chk_tafs=on&submitmet=Submit&station_ids=';

    if (!open(FP, "$CURL $CURL_PROXY_OPTS -d '$data$icao' $url || echo ERROR |"))
    {
        return ("$icao - ERROR - " . uc $!);
    }
    {
        local $/;
        $response = <FP>;
    }
    close FP;
    if ($response =~ /ERROR/) {
        $response =~ s/[\n\r]+/ /g;
        return ("$icao - ERROR - " . uc "$url - $response");
    }
    $response =~ s/[\n\r]+//g;
    $response =~ s/ +/ /g;
    $response =~ s/(<font face=.mono[^>]*?>)/$1METAR /igo;
    $response =~ s/(<pre>)/$1TAF /igo;
    return $response =~ /(?:<font face=.mono[^>]*?>|<pre>)([^<]+)/igo;
}

#
# fetch line(s) for $icao from ADDS dataserver
#
sub process_addsds {
    my ($icao, $type) = @_;
    my ($response, $url);
    my (%issue_time, %raw_text, %metar, %taf);

    $icao =~ s/ /,/g;

    $url = 'http://weather.aero/dataserver1_0/httpparam?requestType=retrieve&format=xml&dataSource=';
    if ($type eq 'metar') {
        $url .= 'metars&fields=raw_text,station_id,observation_time,metar_type&hoursBeforeNow=4';
    } else {
        $url .= 'tafs&hoursBeforeNow=0';
    }
    $url .= '&stationString=';

    if (!open(FP, "$CURL $CURL_PROXY_OPTS '$url$icao' || echo ERROR |")) {
        return ("$icao - ERROR - " . uc $!);
    }
    {
        local $/;
        $response = <FP>;
    }
    close FP;
    if ($response =~ /ERROR/) {
        $response =~ s/[\n\r]+/ /g;
        return ("$icao - ERROR - " . uc "$url - $response");
    }
    $response =~ s/[\n\r]+/ /g;
    $response =~ s/ +/ /g;
    if ($type eq 'metar') {
        # copy to hash, key is observation_time
        for ($response =~ /<((?:raw_text|station_id|observation_time|metar_type)>[^<]+)/igo)
        {
            /^([^>]+).(.*)/;
            if ($1 eq 'raw_text') {
                if (scalar keys %metar eq 4) {
                    my $key = "$metar{station_id},$metar{metar_type}";
                    if (   !exists $issue_time{$key}
                        || $issue_time{$key} < $metar{observation_time})
                    {
                        $issue_time{$key} = $metar{observation_time};

                        # add SPECI only if there is no METAR yet
                        if ($metar{metar_type} eq 'METAR') {
                            $raw_text{$key} = $metar{raw_text};
                        } elsif (!exists $issue_time{"$metar{station_id},METAR"})
                        {
                            ($raw_text{$key} = $metar{raw_text})
                                =~ s/^(?:METAR )?/SPECI /;
                        }
                    }
                }
                %metar = ();
            }
            $metar{$1} = $2;
        }
        return values %raw_text;
    }

    # copy to hash, key is issue_time
    for ($response =~ /<((?:raw_text|station_id|issue_time|valid_time_from|valid_time_to)>[^<]+)/igo)
    {
        /^([^>]+).(.*)/;
        if ($1 eq 'raw_text') {
            if (scalar keys %taf eq 5) {
                my $hours =
                        ($taf{valid_time_to} - $taf{valid_time_from}) / 60 / 60;
                my $key = "$taf{station_id},$hours";
                if (   !exists $issue_time{$key}
                    || $issue_time{$key} < $taf{issue_time})
                {
                    $issue_time{$key} = $taf{issue_time};
                    $raw_text{$key} = "$hours,$taf{raw_text}";
                }
            }
            %taf = ();
        }
        $taf{$1} = $2;
    }

    return values %raw_text;
}

=pod

If there are any CGI errors the type of error is reported and the script
terminates.

=cut

if (cgi_error()) {
    print header({ status => cgi_error() }) .
        start_html({ title => cgi_error() }) . "\n" .
        b(cgi_error()) . "\n" .
        end_html();
    exit 0;
}

=pod

The parameter C<msg> is truncated to 300 characters. Letters are converted to
upper case. Any characters that are not allowed in a message are removed.

=cut

my $msg = param('msg') ? substr(param('msg'), 0, 300) : '';
$msg = uc $msg;
$msg =~ s/[^ -~]//gos; # avoid invalid XML and HTML

=pod

The other parameters are checked to have one of the allowed values.

=cut

#
# determine type from request (default metar)
#
my $type = param('type');
$type = 'metar' unless defined $type && $type eq 'taf';

=pod

If the parameter C<lang> is not set the first allowed value of the HTTP header
C<Accept-Language> is used.

=cut

#
# determine language from request, browser header (default de)
#
my $lang = param('lang');
if (!defined $lang) {
    my $accept_lang = http('Accept-Language');
    if (defined $accept_lang) {
        for (split(/,/, $accept_lang)) {
            last if $_ =~ /^de/i;
            if ($_ =~ /^en/i) {
                $lang = 'en';
                last;
            }
            if ($_ =~ /^es/i) {
                $lang = 'es';
                last;
            }
            if ($_ =~ /^ru/i) {
                $lang = 'ru';
                last;
            }
        }
    }
}
$lang = 'de'
    unless defined $lang && ($lang eq 'en' || $lang eq 'es' || $lang eq 'ru');

#
# determine format from request (default html)
#
my $format = param('format');
$format = 'html'
    unless defined $format && ($format eq 'text' || $format eq 'xml');

#
# determine source from request (default noaa)
#
my $source = param('source');
if (-f "$CGI_DATA_DIR/metar.txt") {
    $source = 'file';
} else {
    $source = 'noaa'
        unless defined $source && ($source eq 'adds' || $source eq 'addsds');
}
#
# determine if XML file name provided
#
my $xml_out_provided = param('xml_out_provided');
my $xml_out;
if (defined $xml_out_provided && ! -e "$CGI_TMP_DIR/metaf2xml-provided.xml") {
    $xml_out = "$CGI_TMP_DIR/metaf2xml-provided.xml";
} else {
    $xml_out_provided = undef;
    $xml_out = "$CGI_TMP_DIR/metaf2xml-$$.xml";
}

#
# log request
#
if (request_method() && open(LOG, '>>', $CGI_LOG_FILE)) {
    # anonymize IP address, we only want to group the requests
    my $addr = remote_addr();
    $addr =~ s/\.[0-9]+/.x/;
    $addr =~ s/[0-9]+$/x/;
    print LOG gmtime() . ' ' . $addr . ' ' . request_method()
              . " $type $lang $format $source $msg\n";
    close LOG;
}

=pod

If C<msg> starts with C<METAR> or C<TAF> this message type is used instead of
the parameter C<type>.

=cut

#
# overwrite type if specified in message
#
$type = 'metar' if $msg =~ /^(?:METAR|SPECI) /;
$type = 'taf' if $msg =~ /^TAF /;
my $is_taf = $type eq 'taf' ? '-t ' : '';

#
# translated strings
#
my %trans = (
    title_en => 'METAR/TAF decoder',
    title_de => 'METAR/TAF-Übersetzer',
    title_es => 'METAR/TAF traductor',
    title_ru => 'МЕТАР/ТАФ декодер',
    header_en => 'METAR/TAF decoder',
    header_de => 'METAR/TAF-Übersetzer',
    header_es => 'METAR/TAF traductor',
    header_ru => 'МЕТАР/ТАФ декодер',
    infosrc_en => 'A good source for up-to-date METAR and TAF messages is',
    infosrc_de => 'Eine gute Quelle für aktuelle METAR- und TAF-Meldungen ist',
    infosrc_es => 'Una buena fuente por mensajes actuales de METAR y TAF es',
    infosrc_ru => 'Хороший источник для современных сообщений МЕТАР и ТАФ',
    outdated_en => 'Warning: forecast data for many stations has not been updated at NOAA since August, 3rd, 2006.',
    outdated_de => 'Warnung: Die Wetter-Vorhersagen bei NOAA werden seit 3. August 2006 für viele Stationen nicht mehr aktualisiert.',
    outdated_es => 'Warning: forecast data for many stations has not been updated at NOAA since August, 3rd, 2006.',
    outdated_ru => 'предупреждение: Данные прогноза для многих станций не были обновлены в НОАА с тех пор August, 3rd, 2006.',
    infofetch_en => 'If the keyword "FETCH" and one or more Airport-Ids are used, messages are fetched from the selected source (or if this option is disabled from a file on the server where this web page comes from).',
    infofetch_de => 'Wird das Schlüsselwort "FETCH" und ein oder mehrere Flugplatz-Kürzel verwendet, werden Meldungen von der ausgewählten Quelle (oder, falls diese Option gesperrt ist, von einer Datei auf dem Server, von dem diese Web-Seite kommt) geholt.',
    infofetch_es => 'Si se usa la palabra clave "FETCH" y uno o mas ids de la estación, los mensajes serán traído de la fuente elegida (o si esta opción esta minusválida de un fichero del servidor de donde viene este página web).',
    infofetch_ru => 'Если ключевое слово "FETCH" и одни или более ид аэропорта используется, сообщения принесены из отобранного источника (или если этот выбор инвалиды от файла на сервере, откуда эта веб-страница прибывает).',
    example_en => 'examples for messages',
    example_de => 'Beispiele für Meldungen',
    example_es => 'ejemplos para mensajes',
    example_ru => 'примеры для сообщений',
    message_en => 'message',
    message_de => 'Meldung',
    message_es => 'mensaje',
    message_ru => 'сообщение',
    type_en => 'is of type',
    type_de => 'ist vom Typ',
    type_es => 'es del typo',
    type_ru => 'имеет тип',
    source_en   => 'fetch from',
    source_de   => 'hole von',
    source_es   => 'trae de',
    source_ru   => 'усилия от',
    language_en => 'language',
    language_de => 'Sprache',
    language_es => 'lengua',
    language_ru => 'язык',
    format_en   => 'output as',
    format_de   => 'gib aus als',
    format_es   => 'presenta como',
    format_ru   => 'показа в',
    decode_en   => 'decode',
    decode_de   => 'Übersetzen',
    decode_es   => 'traduce',
    decode_ru   => 'переводить',
    disclaimer_en => 'Warning: The information on this page may be out-dated, inaccurate, or both. It is not suited for use in aviation.',
    disclaimer_de => 'Warnung: Die Informationen auf dieser Seite könnten veraltet, falsch oder beides sein. Sie sind nicht für die Verwendung in der Luftfahrt geeignet.',
    disclaimer_es => 'Aviso: Las informaciones de esta página web puedan ser antiguas o falsas o los dos. No son apropiadas para el uso en la aviación.',
    disclaimer_ru => 'Предупреждение: Информация относительно этой страницы может быть устарелой, неточной, или оба. Этому не удовлетворяют для использования в авиации.',
);

#
# save message parameter
#
my $msg_orig = $msg;

=pod

If C<msg> starts with C<FETCH> and is followed by one or more airport ICAO codes
messages for the specified airports are fetched from the local files
C<$CGI_DATA_DIR/{metar,staf,taf}.txt> (if they exist) or the NOAA web site.

=cut

#
# process special contents of message
#
if ($msg =~ /^FETCH((?: [A-Z][A-Z0-9]{3})+)$/) {
    my ($icao, $response, @metar, @staf, @taf);

    $is_taf = '';
    $icao = $1;
    $icao =~ s/ //;

    if ($source ne 'file') {
        if ($source eq 'noaa') {
            @metar = process_noaa $icao, 'metar';
            @staf  = process_noaa $icao, 'staf';
            @taf   = process_noaa $icao, 'taf';
        } elsif ($source eq 'adds') {
            for (process_adds $icao) {
                if (s/^METAR //) {
                    push @metar, $_;
                } elsif (s/^TAF //) {
                    push @taf, $_;
                } else {
                    push @metar, $_;
                }
            }
        } else {
            @metar = process_addsds $icao, 'metar';
            for (process_addsds $icao, 'taf') {
                if (s/^[1-9],//) {
                    push @staf, $_;
                } else {
                    s/^[0-9]+,//;
                    push @taf, $_;
                }
            }
        }
    } else {
        my ($metarfile, $staffile, $taffile);

        $metarfile = "$CGI_DATA_DIR/metar.txt";
        $staffile  = "$CGI_DATA_DIR/staf.txt";
        $taffile   = "$CGI_DATA_DIR/taf.txt";

        #$icao='[A-Z][A-Z0-9]{3}';
        @metar = process_file $icao, $metarfile;
        #$icao='qqqq';
        @staf  = process_file $icao, $staffile;
        @taf   = process_file $icao, $taffile;
    }

    $msg = '';
    for my $id (split / /, $icao) {
        $msg .= "\nMETAR\n" . join("\n", grep { /^(?:METAR |SPECI )?$id/ } @metar);
        $msg .= "\nTAF\n"   . join("\n", grep { /^$id/ } @staf);
        $msg .= "\nTAF\n"   . join("\n", grep { /^$id/ } @taf);
    }
    $msg =~ s/ +/ /g;
    $msg =~ s/ $//g;
    $msg =~ s/[^\n -~]/?/gos; # avoid invalid XML and HTML
    $msg =~ s/[\n]+/\n/g;
    $msg =~ s/^\n//;
}

if ($format eq 'xml') {
    print "Content-Type: text/xml; charset=UTF-8\n\n";
    open(METAF2XML, "|$METAF2XML_BIN/metaf2xml.pl $is_taf -Xx - -S/metaf-fullhtml.xsl -O'$type $lang $format $source $msg_orig' 2>&1");
    print METAF2XML $msg;
    close METAF2XML;
    exit 0;
}

print header({charset => 'UTF-8'}) .
    start_html({ title => $trans{'title_'.$lang},
                 lang  => $lang,
                 dtd   => [ '-//W3C//DTD HTML 4.01 Transitional//EN',
                            'http://www.w3.org/TR/html4/loose.dtd' ],
                 meta  => { robots   => 'noindex,nofollow',
                            keywords =>
                     'metaf2xml,METAR,TAF,XML,aviation,weather,report,forecast',
                          },
                 head  => meta({ http_equiv => 'Content-Type',
                                 content    => 'text/html; charset=UTF-8'})
               }
              ) . "\n" .
    h2($trans{'header_'.$lang}) . "\n" .
    $trans{'infosrc_'.$lang} . ":\n" .
    kbd('http://weather.noaa.gov/weather/coded.html') . p . "\n" .
    #b($trans{'outdated_'.$lang}) . p . "\n" .
    $trans{'infofetch_'.$lang} . p . "\n" .
    start_form({ action => 'metaf.pl' }) .
    table({ border => 0, cellspacing => 0 }, "\n" .
        Tr(td({ colspan => 5 }, $trans{'example_'.$lang} . ':')) . "\n" .
        Tr("\n" .
           td('') . "\n" .
           td({ colspan => 4, style => 'font-family: monospace;' }, "\n" .
              'METAR YUDO 090600Z 00000KT CAVOK 22/15 Q1021' . "\n") . "\n"
        ) . "\n" .
        Tr("\n" .
           td('') . "\n" .
           td({ colspan => 4, style => 'font-family: monospace;' }, "\n" .
              'TAF YUDO 090600Z 090716 VRB03KT 8000 SKC PROB40 TEMPO 0708 0200 +TSRA FM0800 CAVOK' . "\n") . "\n"
        ) . "\n" .
        Tr("\n" .
           td('') . "\n" .
           td({ colspan => 4, style => 'font-family: monospace;' }, "\n" .
              'FETCH SBGL KJFK RJTT' . "\n") . "\n"
        ) . "\n" .
        Tr("\n" .
           td({ nowrap => undef }, $trans{'message_'.$lang} . ':') . "\n" .
           td({ colspan => 4 }, "\n" .
              textfield({ name      => 'msg',
                          size      => 90,
                          maxlength => 300,
                          style     => 'font-family: monospace;',
                          value     => $msg_orig }) . "\n") . "\n"
        ) . "\n" .
        Tr("\n" .
           td({ nowrap => undef }, $trans{'type_'.$lang} . ':') . "\n" .
           td(input({ type => 'radio', name => 'type', value => 'metar',
                      ($type eq 'metar' ? 'checked' : '') => undef }
                   ) .
              'METAR') . "\n" .
           td({ colspan => 3 },
              input({ type => 'radio', name => 'type', value => 'taf',
                      ($type eq 'taf' ? 'checked' : '') => undef }
                   ) .
              'TAF') . "\n"
        ) . "\n" .
        Tr("\n" .
           td({ nowrap => undef }, $trans{'source_'.$lang} . ':') . "\n" .
           td(input({ type => 'radio', name => 'source', value => 'noaa',
                      ($source eq 'noaa' ? 'checked' : '') => undef,
                      ($source eq 'file' ? 'readonly' : '') => undef,
                      ($source eq 'file' ? 'disabled' : '') => undef,
                    }
                   ) .
              'NOAA/IWS') . "\n" .
           td(input({ type => 'radio', name => 'source', value => 'adds',
                      ($source eq 'adds' ? 'checked' : '') => undef,
                      ($source eq 'file' ? 'readonly' : '') => undef,
                      ($source eq 'file' ? 'disabled' : '') => undef,
                    }
                   ) .
              'NOAA/ADDS') . "\n" .
           td({ colspan => 2 },
              input({ type => 'radio', name => 'source', value => 'addsds',
                      ($source eq 'addsds' ? 'checked' : '') => undef,
                      ($source eq 'file' ? 'readonly' : '') => undef,
                      ($source eq 'file' ? 'disabled' : '') => undef,
                    }
                   ) .
              'NOAA/ADDS dataserver') . "\n"
        ) . "\n" .
        Tr("\n" .
           td({ nowrap => undef }, $trans{'language_'.$lang} . ':') . "\n" .
           td({ width => '16%' },
              input({ type => 'radio', name => 'lang', value => 'de',
                      ($lang eq 'de' ? 'checked' : '') => undef }
                   ) .
              'Deutsch') . "\n" .
           td({ width => '19%' },
              input({ type => 'radio', name => 'lang', value => 'en',
                      ($lang eq 'en' ? 'checked' : '') => undef }
                   ) .
              'English') . "\n" .
           td({ width => '25%' },
              input({ type => 'radio', name => 'lang', value => 'es',
                      ($lang eq 'es' ? 'checked' : '') => undef }
                   ) .
              'Español') . "\n" .
           td(input({ type => 'radio', name => 'lang', value => 'ru',
                      ($lang eq 'ru' ? 'checked' : '') => undef }
                   ) .
              'Русский') . "\n"
        ) . "\n" .
        Tr("\n" .
           td({ nowrap => undef }, $trans{'format_'.$lang} . ':') . "\n" .
           td(input({ type => 'radio', name => 'format', value => 'html',
                      ($format eq 'html' ? 'checked' : '') => undef }
                   ) .
              'HTML') . "\n" .
           td(input({ type => 'radio', name => 'format', value => 'text',
                      ($format eq 'text' ? 'checked' : '') => undef }
                   ) .
              'text') . "\n" .
           td({ colspan => 2 },
              input({ type => 'radio', name => 'format', value => 'xml',
                      ($format eq 'xml' ? 'checked' : '') => undef }
                   ) .
              'XML (+XSL -> HTML)') . "\n"
        ) . "\n"
    ) . "\n" .
    submit( value => $trans{'decode_'.$lang }) .
    end_form . p . "\n";

if ($msg) {
    my $response;

    print b($trans{'disclaimer_'.$lang}), p, "\n";

    open(METAF2XML, "|$METAF2XML_BIN/metaf2xml.pl $is_taf -Xx $xml_out 2>&1");
    print METAF2XML $msg;
    close METAF2XML;
    open(XSLTPROC, "xsltproc --nonet --nowrite --nomkdir --stringparam lang $lang $METAF2XML_XSL/metaf-$format.xsl $xml_out 2>&1 |");
    {
        local $/;
        $response = <XSLTPROC>;
    }
    close XSLTPROC;
    unlink $xml_out unless defined $xml_out_provided;
    if ($format eq 'html') {
        print $response;
    } else {
        print table({ border => 1 }, "\n" .
              Tr("\n" .
                 td("\n" .
                    pre({ style => 'margin: 0;' }, $response)) . "\n") . "\n");
    }
}

print comment('metaf.pl: $Id: metaf.pl,v 1.39 2006/11/09 19:37:38 metaf2xml Exp $'), "\n", p,
      "copyright (c) 2006 metaf2xml @\n",
      a({ href => 'http://metaf2xml.sourceforge.net/' }, "\n" .
      img({ src    =>
              'http://sflogo.sourceforge.net/sflogo.php?group_id=168043&type=1',
            border => 0,
            alt    => 'SourceForge.net Logo' }) . "\n") . "\n" .
      end_html();
