<?xml version="1.0" encoding="UTF-8"?>
<!--
########################################################################
# metaf.xsl
#
# copyright (c) metaf2xml 2006
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License as
# published by the Free Software Foundation; either version 2 of the
# License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA
########################################################################
-->

<xsl:stylesheet
    xmlns:xsl = "http://www.w3.org/1999/XSL/Transform"
    version = "1.0">

 <!-- those 2 must be passed with "- -stringparam name value" from xsltproc -->
 <xsl:param name="strictfgfs"/>
 <xsl:param name="lang"/>

 <xsl:variable name="NM2KM" select="1.852"/>
 <xsl:variable name="FT2M" select="0.3048"/>
 <xsl:variable name="SM2M" select="1609.3412196"/> <!-- statute mile! -->
 <xsl:variable name="METER_PER_SEC2KT" select="3.6 div $NM2KM"/>
 <xsl:variable name="INHG2HPA" select="33.86388640341"/>

 <xsl:template match="data">
  <xsl:if test="$method = 'html'">
   <xsl:comment> start of data </xsl:comment>
   <xsl:text>&#10;</xsl:text>
  </xsl:if>
  <xsl:choose>
   <xsl:when test="$lang = 'en'">
    <xsl:apply-templates select="metafList/taf|metafList/metar">
     <xsl:with-param name="trans" select="document('metaf-lang.xsl')//trans/en"/>
    </xsl:apply-templates>
   </xsl:when>
   <xsl:when test="$lang = 'es'">
    <xsl:apply-templates select="metafList/taf|metafList/metar">
     <xsl:with-param name="trans" select="document('metaf-lang.xsl')//trans/es"/>
    </xsl:apply-templates>
   </xsl:when>
   <xsl:when test="$lang = 'ru'">
    <xsl:apply-templates select="metafList/taf|metafList/metar">
     <xsl:with-param name="trans" select="document('metaf-lang.xsl')//trans/ru"/>
    </xsl:apply-templates>
   </xsl:when>
   <xsl:otherwise>
    <xsl:apply-templates select="metafList/taf|metafList/metar">
     <xsl:with-param name="trans" select="document('metaf-lang.xsl')//trans/de"/>
    </xsl:apply-templates>
   </xsl:otherwise>
  </xsl:choose>
  <xsl:if test="$method = 'html'">
   <xsl:comment> metaf: $Id: metaf.xsl,v 1.37 2006/11/09 19:38:18 metaf2xml Exp $ </xsl:comment>
   <xsl:text>&#10;</xsl:text>
   <xsl:comment>
   <xsl:value-of select="concat(' metaf-lang: ', document('metaf-lang.xsl')//trans/version, ' ')"/>
   </xsl:comment>
   <xsl:text>&#10;</xsl:text>
   <xsl:apply-templates select="versionList/version"/>
   <xsl:comment> end of data </xsl:comment>
  </xsl:if>
 </xsl:template>

 <xsl:template match="version">
  <xsl:comment><xsl:value-of select="concat(' ', ., ' ')"/></xsl:comment>
  <xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="taf">
  <xsl:param name="trans"/>
  <table border="1" cellpadding="2">
  <tr>
  <td colspan="4" style="font-family: monospace; font-size: 90%;">
  <xsl:value-of select="concat('msg: ', @s)"/>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
  <xsl:apply-templates select="WARNING">
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <tr>
  <xsl:if test="$method = 'html'">
   <td><b>TAF</b></td>
  </xsl:if>
  <th nowrap="1" colspan="3">
  <xsl:choose>
   <xsl:when test="not(boolean(fcstPeriod/hourFrom))">
    <xsl:value-of select="$trans/hdr[@o='taf']"/>
   </xsl:when>
   <xsl:otherwise>
    <xsl:variable name="hours">
     <xsl:choose>
      <xsl:when test="fcstPeriod/hourFrom/@v &lt; fcstPeriod/hourTill/@v">
       <xsl:value-of select="fcstPeriod/hourTill/@v - fcstPeriod/hourFrom/@v"/>
      </xsl:when>
      <xsl:otherwise>
       <xsl:value-of select="fcstPeriod/hourTill/@v - fcstPeriod/hourFrom/@v + 24"/>
      </xsl:otherwise>
     </xsl:choose>
    </xsl:variable>
    <xsl:value-of select="concat(substring-before($trans/hdr[@o='taf_hour'], '|'), $hours, substring-after($trans/hdr[@o='taf_hour'], '|'))"/>
   </xsl:otherwise>
  </xsl:choose>
  <xsl:apply-templates select="reportModifier">
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:if test="$method != 'html'">
   <xsl:value-of select="concat('&#10;', $trans/hdr[@o='taf_ul'])"/>
  </xsl:if>
  </th>
  </tr><xsl:text>&#10;</xsl:text>
  <xsl:apply-templates select="obsStation">
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:apply-templates select="issueTime">
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:apply-templates select="fcstPeriod">
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:apply-templates select="fcstNotAvbl">
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:apply-templates select="sfcWind">
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:apply-templates select="CAVOK">
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:apply-templates select="visPrev">
   <xsl:with-param name="unitThreshold" select="1000"/>
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:apply-templates select="weatherList">
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:apply-templates select="cloudList">
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:apply-templates select="visVert">
   <xsl:with-param name="unitThreshold" select="1000"/>
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:apply-templates select="tempMax|tempMin">
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:apply-templates select="windShearLvl|turbulence|icing|tempAt">
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:apply-templates select="amendment">
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:apply-templates select="trendList">
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:apply-templates select="tafRemarkList">
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:if test="boolean(ERROR/descr)">
   <xsl:choose>
    <xsl:when test="$method = 'html'">
     <xsl:apply-templates select="ERROR">
      <xsl:with-param name="trans" select="$trans"/>
     </xsl:apply-templates>
    </xsl:when>
    <xsl:otherwise>
     <xsl:value-of select="concat('ERROR: invalid format: ', ERROR/pos/@v, '&#10;')"/>
    </xsl:otherwise>
   </xsl:choose>
  </xsl:if>
  </table>
  <xsl:text>&#10;</xsl:text>
  <xsl:if test="position()!=last()"><br /></xsl:if>
 </xsl:template>

 <xsl:template match="metar">
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <table border="1" cellpadding="2">
  <tr>
  <td colspan="4" style="font-family: monospace; font-size: 90%;">
  <xsl:value-of select="concat('msg: ', @s)"/>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
  <xsl:if test="$strictfgfs != 1">
   <xsl:apply-templates select="WARNING" />
  </xsl:if>
  <tr>
   <xsl:choose>
    <xsl:when test="boolean(SPECI)">
     <xsl:if test="$method = 'html'">
      <td><b>SPECI</b></td>
     </xsl:if>
     <th nowrap="1" colspan="3">
      <xsl:value-of select="$trans/hdr[@o='speci']"/>
      <xsl:apply-templates select="reportModifier">
       <xsl:with-param name="trans" select="$trans"/>
      </xsl:apply-templates>
      <xsl:if test="$method != 'html'">
       <xsl:value-of select="concat('&#10;', $trans/hdr[@o='speci_ul'])"/>
      </xsl:if>
     </th>
    </xsl:when>
    <xsl:otherwise>
     <xsl:if test="$method = 'html'">
      <td><b>METAR</b></td>
     </xsl:if>
     <th nowrap="1" colspan="3">
      <xsl:value-of select="$trans/hdr[@o='metar']"/>
      <xsl:apply-templates select="reportModifier">
       <xsl:with-param name="trans" select="$trans"/>
      </xsl:apply-templates>
      <xsl:if test="$method != 'html'">
       <xsl:value-of select="concat('&#10;', $trans/hdr[@o='metar_ul'])"/>
      </xsl:if>
     </th>
    </xsl:otherwise>
   </xsl:choose>
  </tr><xsl:text>&#10;</xsl:text>
  <xsl:apply-templates select="obsStation">
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:apply-templates select="obsTime">
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:apply-templates select="sfcWind">
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:if test="$strictfgfs != 1">
   <xsl:apply-templates select="CAVOK">
    <xsl:with-param name="trans" select="$trans"/>
   </xsl:apply-templates>
  </xsl:if>
  <xsl:apply-templates select="visPrev">
   <xsl:with-param name="unitThreshold" select="1000"/>
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:apply-templates select="visRwyList/visRwy">
   <xsl:with-param name="unitThreshold" select="1000"/>
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:apply-templates select="RVRNO">
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:apply-templates select="weatherList">
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:apply-templates select="cloudList">
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:apply-templates select="visVert">
   <xsl:with-param name="unitThreshold" select="1000"/>
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:apply-templates select="temperature">
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:apply-templates select="QNH">
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:apply-templates select="windShearList/windShear">
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:apply-templates select="rwyStateList/rwyState">
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:if test="$strictfgfs = 1">
   <xsl:apply-templates select="remarkRwyState">
    <xsl:with-param name="trans" select="$trans"/>
   </xsl:apply-templates>
  </xsl:if>
  <xsl:if test="$strictfgfs != 1">
   <xsl:if test="boolean(somePressure)">
    <xsl:apply-templates select="somePressure">
     <xsl:with-param name="trans" select="$trans"/>
    </xsl:apply-templates>
   </xsl:if>
   <xsl:apply-templates select="recentWeatherList">
    <xsl:with-param name="trans" select="$trans"/>
   </xsl:apply-templates>
   <xsl:apply-templates select="colourCode">
    <xsl:with-param name="trans" select="$trans"/>
   </xsl:apply-templates>
   <xsl:apply-templates select="cloudMaxCover">
    <xsl:with-param name="trans" select="$trans"/>
   </xsl:apply-templates>
   <xsl:apply-templates select="NEFO_PLAYA">
    <xsl:with-param name="trans" select="$trans"/>
   </xsl:apply-templates>
   <xsl:apply-templates select="RH">
    <xsl:with-param name="trans" select="$trans"/>
   </xsl:apply-templates>
   <xsl:apply-templates select="rwyWindList/rwyWind">
    <xsl:with-param name="trans" select="$trans"/>
   </xsl:apply-templates>
   <xsl:apply-templates select="trendList">
    <xsl:with-param name="trans" select="$trans"/>
   </xsl:apply-templates>
   <xsl:apply-templates select="remarkList">
    <xsl:with-param name="trans" select="$trans"/>
   </xsl:apply-templates>
  </xsl:if>
  <xsl:if test="boolean(ERROR/descr)">
   <xsl:choose>
    <xsl:when test="$method = 'html'">
     <xsl:apply-templates select="ERROR">
      <xsl:with-param name="trans" select="$trans"/>
     </xsl:apply-templates>
    </xsl:when>
    <xsl:otherwise>
     <xsl:value-of select="concat('ERROR: ', $trans/error[@o=$this/ERROR/descr/@v], ': ', ERROR/pos/@v, '&#10;')"/>
    </xsl:otherwise>
   </xsl:choose>
  </xsl:if>
  </table>
  <xsl:text>&#10;</xsl:text>
  <xsl:if test="position()!=last()"><br /></xsl:if>
 </xsl:template>

 <xsl:template match="reportModifier">
  <xsl:param name="trans"/>
  <xsl:text>&#9;&#9;(</xsl:text>
  <xsl:variable name="this" select="."/>
  <xsl:value-of select="$trans/reportModifierType[@o=$this/modifierType/@v]"/>
  <xsl:if test="type/@v = 'P'">
    <xsl:if test="boolean(isLastSegment)">
     <xsl:value-of select="concat($trans/t[@o='last'],' ')"/>
    </xsl:if>
   <xsl:value-of select="concat($trans/t[@o='segment'],' ')"/>
   <xsl:value-of select="segment/@v"/>
  </xsl:if>
  <xsl:if test="boolean(over24hLate)">
   <xsl:text>, bulletin over 24 hours after observation</xsl:text>
  </xsl:if>
  <xsl:if test="boolean(sequenceLost)">
   <xsl:text>, bulletin sequence lost</xsl:text>
  </xsl:if>
  <xsl:if test="boolean(bulletinSeq)">
   <xsl:text>, bulletin sequence </xsl:text>
   <xsl:value-of select="bulletinSeq/@v"/>
  </xsl:if>
  <xsl:text>)</xsl:text>
 </xsl:template>

 <xsl:template match="obsStation">
  <xsl:param name="trans"/>
  <tr>
   <xsl:if test="$method = 'html'">
    <td nowrap="1"><b><xsl:value-of select="@s"/></b></td>
   </xsl:if>
   <td nowrap="1">
   <xsl:value-of select="$trans/hdr[@o='obsStation']"/>
   </td>
   <td nowrap="1" colspan="2"><xsl:value-of select="id/@v"/></td>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="fcstPeriod">
  <xsl:param name="trans"/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1"><b><xsl:value-of select="@s"/></b></td>
  </xsl:if>
  <td nowrap="1">
  <xsl:value-of select="$trans/hdr[@o='fcstPeriod']"/>
  </td>
  <td nowrap="1" colspan="2">
  <xsl:choose>
   <xsl:when test="boolean(invalidFormat)">
    <xsl:value-of select="concat('(', $trans/t[@o='invalidFormat'])"/>
    <xsl:text>: '</xsl:text>
    <xsl:value-of select="invalidFormat/@v"/>
    <xsl:text>')</xsl:text>
   </xsl:when>
   <xsl:otherwise>
    <xsl:value-of select="concat(format-number(day/@v, '0'), '., ', hourFrom/@v, ':00 - ', hourTill/@v, ':00 UTC')"/>
    <xsl:if test="hourFrom/@v &gt;= hourTill/@v">
     <xsl:value-of select="$trans/t[@o='nextDay']"/>
    </xsl:if>
   </xsl:otherwise>
  </xsl:choose>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="obsTime|issueTime">
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <tr>
   <xsl:if test="$method = 'html'">
    <td nowrap="1"><b><xsl:value-of select="@s"/></b></td>
   </xsl:if>
   <td nowrap="1">
   <xsl:value-of select="$trans/hdr[@o=name($this)]"/>
   </td>
   <td nowrap="1" colspan="2">
    <xsl:choose>
     <xsl:when test="boolean(invalidFormat)">
      <xsl:value-of select="concat('(', $trans/t[@o='invalidFormat'])"/>
      <xsl:text>: '</xsl:text>
      <xsl:value-of select="invalidFormat/@v"/>
      <xsl:text>')</xsl:text>
     </xsl:when>
     <xsl:otherwise>
      <xsl:value-of select="substring-before($trans/t[@o='onThe'], '|')"/>
      <xsl:value-of select="format-number(day/@v, '0')"/>
      <xsl:value-of select="substring-after($trans/t[@o='onThe'], '|')"/>
      <xsl:value-of select="concat(hour/@v, ':', minute/@v, ' UTC')"/>
     </xsl:otherwise>
    </xsl:choose>
   </td>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="visPrev">
  <xsl:param name="unitThreshold"/>
  <xsl:param name="trans"/>
  <xsl:variable name="rows">
   <xsl:choose>
    <xsl:when test="boolean(NDV) and boolean(../visMin)">3</xsl:when>
    <xsl:when test="boolean(NDV) or  boolean(../visMin)">2</xsl:when>
    <xsl:otherwise>1</xsl:otherwise>
   </xsl:choose>
  </xsl:variable>
  <xsl:variable name="str">
   <xsl:choose>
    <xsl:when test="boolean(../visMin)">
     <xsl:value-of select="concat(@s, ' ', ../visMin/@s)"/>
    </xsl:when>
    <xsl:otherwise><xsl:value-of select="@s"/></xsl:otherwise>
   </xsl:choose>
  </xsl:variable>
  <tr>
  <xsl:call-template name="visibility_str">
   <xsl:with-param name="str" select="$str"/>
   <xsl:with-param name="rows" select="$rows"/>
  </xsl:call-template>
  <xsl:call-template name="visibility_header">
   <xsl:with-param name="header" select="$trans/hdr[@o='visPrev']"/>
   <xsl:with-param name="rows" select="$rows"/>
  </xsl:call-template>
  <xsl:call-template name="visibility">
   <xsl:with-param name="notavail" select="$trans/t[@o='notAvailable']"/>
   <xsl:with-param name="rnd_m" select="50"/>
   <xsl:with-param name="unitThreshold" select="$unitThreshold"/>
   <xsl:with-param name="visVal" select="."/>
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:call-template>
  </tr><xsl:text>&#10;</xsl:text>
  <xsl:if test="boolean(NDV)">
   <tr><td nowrap="1" colspan="2">
   <xsl:text>&#9;&#9;&#9;</xsl:text>
   <xsl:text>(No Directional Variation reporting capabilty)</xsl:text>
   </td></tr><xsl:text>&#10;</xsl:text>
  </xsl:if>
  <xsl:if test="boolean(../visMin)">
   <tr>
   <xsl:text>&#9;&#9;&#9;</xsl:text>
   <xsl:call-template name="visibility">
    <xsl:with-param name="notavail" select="$trans/t[@o='notAvailable']"/>
    <xsl:with-param name="rnd_m" select="50"/>
    <xsl:with-param name="unitThreshold" select="$unitThreshold"/>
    <xsl:with-param name="visVal" select="../visMin"/>
    <xsl:with-param name="trans" select="$trans"/>
   </xsl:call-template>
   </tr><xsl:text>&#10;</xsl:text>
  </xsl:if>
 </xsl:template>

 <xsl:template match="visVert">
  <xsl:param name="unitThreshold"/>
  <xsl:param name="trans"/>
  <tr>
  <xsl:call-template name="visibility_str">
   <xsl:with-param name="str" select="@s"/>
   <xsl:with-param name="rows" select="1"/>
  </xsl:call-template>
  <xsl:call-template name="visibility_header">
   <xsl:with-param name="header" select="$trans/hdr[@o='visVert']"/>
   <xsl:with-param name="rows" select="1"/>
  </xsl:call-template>
  <xsl:call-template name="visibility">
   <xsl:with-param name="notavail" select="$trans/t[@o='visVertNA']"/>
   <xsl:with-param name="rnd_m" select="30"/>
   <xsl:with-param name="unitThreshold" select="$unitThreshold"/>
   <xsl:with-param name="visVal" select="."/>
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:call-template>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="visRwy">
  <xsl:param name="unitThreshold"/>
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <xsl:variable name="hdr" select="$trans/hdr[@o='visRwy']"/>
  <xsl:variable name="rows">
   <xsl:choose>
    <xsl:when test="boolean(RVRVariations) and boolean(visTrend)">3</xsl:when>
    <xsl:when test="boolean(RVRVariations) or  boolean(visTrend)">2</xsl:when>
    <xsl:otherwise>1</xsl:otherwise>
   </xsl:choose>
  </xsl:variable>
  <tr>
  <xsl:call-template name="visibility_str">
   <xsl:with-param name="str" select="@s"/>
   <xsl:with-param name="rows" select="$rows"/>
  </xsl:call-template>
  <xsl:call-template name="visibility_header">
   <xsl:with-param name="header" select="concat(substring-before($hdr, '|'), rwyDesig/@v, substring-after($hdr, '|'))"/>
   <xsl:with-param name="rows" select="$rows"/>
  </xsl:call-template>
  <xsl:variable name="from">
   <xsl:choose>
    <xsl:when test="boolean(RVRVariations)">
     <xsl:value-of select="concat($trans/t[@o='variable-from'], ' ')"/>
    </xsl:when>
    <xsl:otherwise><xsl:text></xsl:text></xsl:otherwise>
   </xsl:choose>
  </xsl:variable>
  <xsl:call-template name="visibility">
   <xsl:with-param name="notavail" select="$trans/t[@o='rwyVisNotAvailable']"/>
   <xsl:with-param name="fromto" select="$from"/>
   <xsl:with-param name="rnd_m" select="25"/>
   <xsl:with-param name="unitThreshold" select="$unitThreshold"/>
   <xsl:with-param name="visVal" select="RVR"/>
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:call-template>
  </tr><xsl:text>&#10;</xsl:text>
  <xsl:if test="boolean(RVRVariations)">
   <xsl:text>&#9;&#9;&#9;</xsl:text>
   <tr>
   <xsl:call-template name="visibility">
    <xsl:with-param name="notavail" select="$trans/t[@o='rwyVisNotAvailable']"/>
    <xsl:with-param name="fromto" select="concat($trans/t[@o='variable-to'], ' ')"/>
    <xsl:with-param name="rnd_m" select="25"/>
    <xsl:with-param name="unitThreshold" select="$unitThreshold"/>
    <xsl:with-param name="visVal" select="RVRVariations"/>
    <xsl:with-param name="trans" select="$trans"/>
   </xsl:call-template>
   </tr><xsl:text>&#10;</xsl:text>
  </xsl:if>
  <xsl:if test="boolean(visTrend) and $strictfgfs != 1">
   <tr>
   <td nowrap="1" colspan="2">
    <xsl:value-of select="concat($trans/hdr[@o='visTrend'], $trans/visTrend[@o=$this/visTrend/@v])"/>
   </td></tr><xsl:text>&#10;</xsl:text>
  </xsl:if>
  <xsl:if test="(boolean(RVRVariations) or (boolean(visTrend) and $strictfgfs != 1))"><xsl:text>&#10;</xsl:text></xsl:if>
 </xsl:template>

 <xsl:template name="visibility_str">
  <xsl:param name="rows"/>
  <xsl:param name="str"/>
  <xsl:if test="$method = 'html'">
   <xsl:element name="td">
    <xsl:attribute name="nowrap">1</xsl:attribute>
    <xsl:attribute name="rowspan"><xsl:value-of select="$rows"/></xsl:attribute>
    <b><xsl:value-of select="$str"/></b>
   </xsl:element>
  </xsl:if>
 </xsl:template>

 <xsl:template name="visibility_header">
  <xsl:param name="header"/>
  <xsl:param name="rows"/>
  <xsl:element name="td">
   <xsl:attribute name="nowrap">1</xsl:attribute>
   <xsl:attribute name="rowspan"><xsl:value-of select="$rows"/></xsl:attribute>
   <xsl:value-of select="$header"/>
  </xsl:element>
 </xsl:template>

 <xsl:template name="visibility">
  <xsl:param name="notavail"/>
  <xsl:param name="fromto"/>
  <xsl:param name="rnd_m"/>
  <xsl:param name="unitThreshold"/>
  <xsl:param name="visVal"/>
  <xsl:param name="trans"/>
  <xsl:choose>
   <xsl:when test="boolean($visVal/invalidFormat)">
    <td nowrap="1" colspan="2">
    <xsl:value-of select="concat('(', $trans/t[@o='invalidFormat'])"/>
    <xsl:text>: '</xsl:text>
    <xsl:value-of select="$visVal/invalidFormat/@v"/>
    <xsl:text>')</xsl:text>
    </td>
   </xsl:when>
   <xsl:when test="boolean($visVal/notAvailable)">
    <td nowrap="1" colspan="2">
    <xsl:value-of select="$notavail"/>
    </td>
   </xsl:when>
   <xsl:otherwise>
    <xsl:variable name="lt_gt">
     <xsl:choose>
      <xsl:when test="boolean($visVal/isLess)">
       <xsl:text>&lt;</xsl:text>
      </xsl:when>
      <xsl:when test="boolean($visVal/isGreater)">
       <xsl:text>&gt;</xsl:text>
      </xsl:when>
      <xsl:when test="boolean($visVal/isEqualGreater)">
       <xsl:text>&gt;=</xsl:text>
      </xsl:when>
      <xsl:otherwise></xsl:otherwise>
     </xsl:choose>
    </xsl:variable>
    <xsl:variable name="vis">
     <xsl:choose>
      <xsl:when test="$visVal/unitLength/@v = 'M'">
       <xsl:choose>
        <xsl:when test="$visVal/distance/@v = 0">&lt;|50</xsl:when>
        <xsl:when test="$visVal/distance/@v = 9999">&gt;=|10000</xsl:when>
        <xsl:otherwise>
        <xsl:value-of select="concat($lt_gt, '|', $visVal/distance/@v)"/>
        </xsl:otherwise>
       </xsl:choose>
      </xsl:when>
      <xsl:when test="$visVal/unitLength/@v = 'KM'">
       <xsl:text>|</xsl:text><xsl:value-of select="$visVal/distance/@v * 1000"/>
      </xsl:when>
      <xsl:when test="$visVal/unitLength/@v = 'SM'">
       <xsl:value-of select="concat($lt_gt, '|', $visVal/distance/@v * $SM2M)"/>
      </xsl:when>
      <xsl:when test="$visVal/unitLength/@v = 'FT' or $visVal/unitHeight/@v = 'FT'">
       <xsl:value-of select="concat($lt_gt, '|', $visVal/distance/@v * $FT2M)"/>
      </xsl:when>
      <xsl:when test="$visVal/unitHeight/@v = 'FL'">
       <xsl:value-of select="concat($lt_gt, '|', $visVal/distance/@v * 100 * $FT2M)"/>
      </xsl:when>
     </xsl:choose>
    </xsl:variable>
    <xsl:call-template name="printVis">
     <xsl:with-param name="vis" select="$vis"/>
     <xsl:with-param name="compassDir" select="$visVal/compassDir/@v"/>
     <xsl:with-param name="fromto" select="$fromto"/>
     <xsl:with-param name="rnd_m" select="$rnd_m"/>
     <xsl:with-param name="unitThreshold" select="$unitThreshold"/>
     <xsl:with-param name="trans" select="$trans"/>
    </xsl:call-template>
   </xsl:otherwise>
  </xsl:choose>
 </xsl:template>

 <xsl:template name="printVis">
  <xsl:param name="vis"/>
  <xsl:param name="compassDir"/>
  <xsl:param name="fromto"/>
  <xsl:param name="rnd_m"/>
  <xsl:param name="unitThreshold"/>
  <xsl:param name="trans"/>
  <xsl:variable name="amountMeter" select="substring-after($vis, '|')"/>
  <xsl:variable name="lt_gt" select="substring-before($vis, '|')"/>
  <td nowrap="1">
  <xsl:value-of select="$fromto"/>
  <xsl:value-of select="$lt_gt"/>
  <xsl:choose>
   <xsl:when test="$amountMeter &gt;= $unitThreshold">
    <xsl:value-of select="concat(format-number($amountMeter div 1000,'0.#'), $trans/units[@o='KM'])"/>
   </xsl:when>
   <xsl:otherwise>
    <xsl:value-of select="concat(format-number($amountMeter div $rnd_m ,'#0') * $rnd_m, $trans/units[@o='M'])"/>
   </xsl:otherwise>
  </xsl:choose>
  <xsl:if test="$compassDir != ''">
    <xsl:value-of select="concat(' ', $compassDir)"/>
  </xsl:if>
  </td>
  <xsl:text>&#9;&#9;&#9;&#9;&#9;</xsl:text>
  <td nowrap="1">
  <xsl:value-of select="$lt_gt"/>
  <xsl:value-of select="concat(format-number($amountMeter div $SM2M,'0.#'), $trans/units[@o='SM'])"/>
  <xsl:if test="$compassDir != ''">
   <xsl:value-of select="concat(' ', $compassDir)"/>
  </xsl:if>
  </td>
 </xsl:template>

 <xsl:template name="printTime">
  <xsl:param name="time"/>
  <xsl:param name="trans"/>
  <xsl:value-of select="substring-before($trans/t[@o='time-at'], '|')"/>
  <xsl:choose>
   <xsl:when test="boolean($time/hour)">
    <xsl:value-of select="concat(format-number($time/hour/@v, '00'), ':', format-number($time/minute/@v, '00 UTC'))"/>
   </xsl:when>
   <xsl:otherwise>
    <xsl:value-of select="concat(substring-before($trans/t[@o='afterHour'], '|'), format-number($time/minute/@v, '0'), substring-after($trans/t[@o='afterHour'], '|'))"/>
   </xsl:otherwise>
  </xsl:choose>
  <xsl:value-of select="substring-after($trans/t[@o='time-at'], '|')"/>
 </xsl:template>

 <xsl:template match="peakWind">
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <xsl:variable name="hdr">
   <xsl:value-of select="substring-before($trans/hdr[@o='peakWind'], '|')"/>
   <xsl:if test="boolean(minute)">
    <xsl:call-template name="printTime">
     <xsl:with-param name="time"  select="."/>
     <xsl:with-param name="trans" select="$trans"/>
    </xsl:call-template>
   </xsl:if>
   <xsl:value-of select="substring-after($trans/hdr[@o='peakWind'], '|')"/>
  </xsl:variable>
  <xsl:apply-templates select="wind">
   <xsl:with-param name="hdr"   select="$hdr"/>
   <xsl:with-param name="s"     select="@s"/>
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
 </xsl:template>

 <xsl:template match="rwyWind|thrWind">
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <xsl:apply-templates select="wind">
   <xsl:with-param name="hdr"   select="concat(substring-before($trans/hdr[@o=name($this)], '|'), rwyDesig/@v, substring-after($trans/hdr[@o=name($this)], '|'))"/>
   <xsl:with-param name="s"     select="@s"/>
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
 </xsl:template>

 <xsl:template match="sfcWind|gridWind">
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <xsl:apply-templates select="wind">
   <xsl:with-param name="hdr"   select="$trans/hdr[@o=name($this)]"/>
   <xsl:with-param name="s"     select="@s"/>
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
 </xsl:template>

 <xsl:template match="wind">
  <xsl:param name="hdr"/>
  <xsl:param name="s"/>
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <xsl:variable name="rows">
   <xsl:choose>
    <xsl:when test="boolean(gustSpeed) and boolean(windVarLeft)">3</xsl:when>
    <xsl:when test="boolean(gustSpeed) or  boolean(windVarLeft)">2</xsl:when>
    <xsl:otherwise>1</xsl:otherwise>
   </xsl:choose>
  </xsl:variable>
  <tr>
   <xsl:if test="$method = 'html' and $s != ''">
    <xsl:element name="td">
     <xsl:attribute name="nowrap">1</xsl:attribute>
     <xsl:attribute name="rowspan"><xsl:value-of select="$rows"/></xsl:attribute>
     <b><xsl:value-of select="$s"/></b>
    </xsl:element>
   </xsl:if>
   <xsl:element name="td">
    <xsl:attribute name="nowrap">1</xsl:attribute>
    <xsl:attribute name="rowspan"><xsl:value-of select="$rows"/></xsl:attribute>
    <xsl:value-of select="$hdr"/>
   </xsl:element>
   <xsl:choose>
    <xsl:when test="boolean(invalidFormat)">
     <td nowrap="1" colspan="2">
     <xsl:value-of select="concat('(', $trans/t[@o='invalidFormat'])"/>
     <xsl:text>: '</xsl:text>
     <xsl:value-of select="invalidFormat/@v"/>
     <xsl:text>')</xsl:text>
     </td>
    </xsl:when>
    <xsl:when test="boolean(notAvailable)">
     <td nowrap="1" colspan="2">
     <xsl:value-of select="$trans/t[@o='notAvailable']"/>
     </td>
    </xsl:when>
    <xsl:when test="dir/@v = '0' and speed/@v = '0' and not(boolean(gustSpeed))">
     <td nowrap="1" colspan="2">
     <xsl:value-of select="$trans/t[@o='calm']"/>
     </td>
    </xsl:when>
    <xsl:otherwise>
     <xsl:variable name="dir">
      <xsl:choose>
       <xsl:when test="boolean(dirNotAvailable)">
        <xsl:value-of select="$trans/t[@o='dirNotAvailable']"/>
       </xsl:when>
       <xsl:when test="boolean(dirVariable)">
        <xsl:value-of select="$trans/t[@o='dirVariable']"/>
       </xsl:when>
       <xsl:otherwise>
        <xsl:value-of select="concat($trans/t[@o='wind-from'],' ')"/>
        <xsl:value-of select="$trans/compassDir16/arr[position()=1+format-number($this/dir/@v div 45 * 2, '0')]"/>
        <xsl:value-of select="concat(' (', dir/@v, '°)')"/>
       </xsl:otherwise>
      </xsl:choose>
     </xsl:variable>
     <xsl:choose>
      <xsl:when test="dir/@v = '0' and speed/@v = '0'">
       <td nowrap="1" colspan="2">
       <xsl:value-of select="$trans/t[@o='calm']"/>
       </td>
      </xsl:when>
      <xsl:when test="boolean(speedNotAvailable)">
       <td nowrap="1" colspan="2">
       <xsl:value-of select="$dir"/>
       <xsl:if test="not(boolean(dirNotAvailable))">
        <xsl:text>,</xsl:text>
       </xsl:if>
       <xsl:value-of select="$trans/t[@o='speedNotAvailable']"/>
       </td>
      </xsl:when>
      <xsl:otherwise>
       <xsl:variable name="speed">
        <xsl:call-template name="speed2kt">
         <xsl:with-param name="speed" select="speed/@v"/>
         <xsl:with-param name="unit" select="unitSpeed/@v"/>
        </xsl:call-template>
       </xsl:variable>
       <xsl:call-template name="printSpeed">
        <xsl:with-param name="dir" select="$dir"/>
        <xsl:with-param name="speed" select="$speed"/>
        <xsl:with-param name="delim" select="'&#9;&#9;'"/>
        <xsl:with-param name="greater" select="speedGreater/@v"/>
        <xsl:with-param name="trans" select="$trans"/>
       </xsl:call-template>
      </xsl:otherwise>
     </xsl:choose>
    </xsl:otherwise>
   </xsl:choose>
  </tr><xsl:text>&#10;</xsl:text>
  <xsl:if test="boolean(gustSpeed)">
   <tr>
   <xsl:variable name="gustSpeed">
    <xsl:call-template name="speed2kt">
     <xsl:with-param name="speed" select="gustSpeed/@v"/>
     <xsl:with-param name="unit" select="unitSpeed/@v"/>
    </xsl:call-template>
   </xsl:variable>
   <xsl:call-template name="printSpeed">
    <xsl:with-param name="dir" select="$trans/t[@o='wind-gusts']"/>
    <xsl:with-param name="speed" select="$gustSpeed"/>
    <xsl:with-param name="delim" select="'&#9;&#9;&#9;'"/>
    <xsl:with-param name="greater" select="gustSpeedGreater/@v"/>
    <xsl:with-param name="trans" select="$trans"/>
   </xsl:call-template>
   </tr><xsl:text>&#10;</xsl:text>
  </xsl:if>
  <xsl:if test="boolean(windVarLeft)">
   <tr>
   <td nowrap="1" colspan="2">
   <xsl:value-of select="concat($trans/t[@o='variable'], ' ')"/>
   <xsl:value-of select="concat($trans/t[@o='variable-from'],' ')"/>
   <xsl:value-of select="concat($trans/compassDir16/arr[position()=1+format-number($this/windVarLeft/@v div 45 * 2, '0')], ' ')"/>
   <xsl:value-of select="concat($trans/t[@o='variable-to'], ' ')"/>
   <xsl:value-of select="concat($trans/compassDir16/arr[position()=1+format-number($this/windVarRight/@v div 45 * 2, '0')], ' ')"/>
   <xsl:value-of select="concat('(', windVarLeft/@v, '°--', windVarRight/@v, '°)')"/>
   </td></tr><xsl:text>&#10;</xsl:text>
  </xsl:if>
 </xsl:template>

 <xsl:template name="printSpeed">
  <xsl:param name="dir"/>
  <xsl:param name="speed"/>
  <xsl:param name="delim"/>
  <xsl:param name="greater"/>
  <xsl:param name="trans"/>
  <xsl:variable name="gt">
   <xsl:if test="$greater != ''">&gt;</xsl:if>
  </xsl:variable>
  <td nowrap="1">
  <xsl:value-of select="concat($dir, ' ', $trans/t[@o='wind-at'], ' ', $gt, format-number($speed * $NM2KM, '0.#'), $trans/units[@o='KMH'])"/>
  </td>
  <xsl:value-of select="$delim"/>
  <td nowrap="1">
  <xsl:value-of select="concat($gt, format-number($speed, '0.#'), $trans/units[@o='KT'])"/>
  <xsl:value-of select="concat(' = ', $gt, format-number($speed * $NM2KM div $SM2M * 1000,'0.#'), $trans/units[@o='MPH'])"/>
  <xsl:value-of select="concat(' = ', $gt, format-number($speed * $NM2KM div 3.6, '0.#'), $trans/units[@o='MPS'])"/>
  </td>
 </xsl:template>

 <xsl:template name="speed2kt">
  <xsl:param name="speed"/>
  <xsl:param name="unit"/>
  <xsl:choose>
   <xsl:when test="$unit = 'KT'">
    <xsl:value-of select="$speed"/>
   </xsl:when>
   <xsl:when test="$unit = 'KMH'">
    <xsl:value-of select="$speed div $NM2KM"/>
   </xsl:when>
   <xsl:otherwise>
    <xsl:value-of select="$speed * $METER_PER_SEC2KT"/>
   </xsl:otherwise>
  </xsl:choose>
 </xsl:template>

 <xsl:template match="weatherList">
  <xsl:param name="trans"/>
  <xsl:variable name="s_list">
   <xsl:apply-templates select="weather" mode="list"/>
  </xsl:variable>
  <xsl:apply-templates select="weather">
   <xsl:with-param name="s_list" select="$s_list"/>
   <xsl:with-param name="hdr1" select="$trans/hdr[@o='weather-1']"/>
   <xsl:with-param name="is_recent" select="0"/>
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="weather" mode="list">
  <xsl:value-of select="concat(' ', @s)"/>
 </xsl:template>

 <xsl:template match="weather">
  <xsl:param name="s_list"/>
  <xsl:param name="hdr1"/>
  <xsl:param name="is_recent"/>
  <xsl:param name="trans"/>
  <xsl:choose>
   <xsl:when test="position() = 1">
    <tr>
    <xsl:if test="$method = 'html'">
     <xsl:element name="td">
      <xsl:attribute name="nowrap">1</xsl:attribute>
      <xsl:attribute name="rowspan"><xsl:value-of select="last()"/></xsl:attribute>
      <b><xsl:value-of select="$s_list"/></b>
     </xsl:element>
    </xsl:if>
    <xsl:element name="td">
     <xsl:attribute name="nowrap">1</xsl:attribute>
     <xsl:attribute name="rowspan"><xsl:value-of select="last()"/></xsl:attribute>
     <xsl:value-of select="$hdr1"/>
    </xsl:element>
    <td nowrap="1" colspan="2">
    <xsl:call-template name="printWeather">
     <xsl:with-param name="is_recent" select="$is_recent"/>
     <xsl:with-param name="this" select="."/>
     <xsl:with-param name="trans" select="$trans"/>
    </xsl:call-template>
    </td></tr>
   </xsl:when>
   <xsl:otherwise>
    <tr><td nowrap="1" colspan="2">
    <xsl:if test="$method = 'text'">, </xsl:if>
    <xsl:call-template name="printWeather">
     <xsl:with-param name="is_recent" select="$is_recent"/>
     <xsl:with-param name="this" select="."/>
     <xsl:with-param name="trans" select="$trans"/>
    </xsl:call-template>
    </td></tr>
   </xsl:otherwise>
  </xsl:choose>
 </xsl:template>

 <xsl:template name="printWeather">
  <xsl:param name="is_recent"/>
  <xsl:param name="this"/>
  <xsl:param name="trans"/>
  <xsl:choose>
   <xsl:when test="boolean($this/notAvailable)">
     <xsl:value-of select="$trans/t[@o='notAvailable']"/>
   </xsl:when>
   <xsl:when test="boolean($this/invalidFormat)">
    <xsl:value-of select="concat('(', $trans/t[@o='invalidFormat'])"/>
    <xsl:text>: '</xsl:text>
    <xsl:value-of select="$this/invalidFormat/@v"/>
    <xsl:text>')</xsl:text>
   </xsl:when>
   <xsl:when test="boolean($this/NSW)">
     <xsl:value-of select="$trans/weather[@o='NSW']"/>
   </xsl:when>
   <xsl:when test="boolean($this/tornado)">
     <xsl:value-of select="$trans/weather[@o='tornado']"/>
   </xsl:when>
   <xsl:otherwise>
    <xsl:variable name="w_all">
     <xsl:apply-templates select="$this/descriptor"/>
     <xsl:apply-templates select="$this/phenomenon"/>
    </xsl:variable>
    <xsl:variable name="intensity">
     <xsl:choose>
      <xsl:when test="$this/intensity/@v = 'LIGHT'">
       <xsl:value-of select="$trans/weather[@o='phen-light']"/>
      </xsl:when>
      <xsl:when test="$this/intensity/@v = 'HEAVY'">
       <xsl:value-of select="$trans/weather[@o='phen-heavy']"/>
      </xsl:when>
      <xsl:otherwise>
       <xsl:choose>
        <xsl:when test="$is_recent = 1"></xsl:when>
        <xsl:when test="boolean($this/inVicinity)"></xsl:when>
        <xsl:when test="contains($w_all, 'DZ') or contains($w_all, 'RA') or contains($w_all, 'SN') or contains($w_all, 'SG') or contains($w_all, 'PL') or contains($w_all, 'GR') or contains($w_all, 'GS') or contains($w_all, 'DS') or contains($w_all, 'SS')">
         <xsl:value-of select="$trans/weather[@o='phen-moderate']"/>
        </xsl:when>
        <xsl:otherwise></xsl:otherwise>
       </xsl:choose>
      </xsl:otherwise>
     </xsl:choose>
    </xsl:variable>
    <xsl:variable name="w_all_trans">
     <xsl:choose>
      <xsl:when test="$trans/weather[@o=$w_all] != ''">
       <xsl:variable name="w_trans" select="$trans/weather[@o=$w_all]"/>
       <xsl:value-of select="concat(substring-before($w_trans, '|'), $intensity, substring-after($w_trans, '|'))"/>
      </xsl:when>
      <xsl:otherwise>
       <xsl:value-of select="$this/@s"/>
      </xsl:otherwise>
     </xsl:choose>
    </xsl:variable>
    <xsl:value-of select="$w_all_trans"/>
    <xsl:if test="boolean($this/inVicinity)">
     <xsl:value-of select="$trans/weather[@o='inVicinity']"/>
    </xsl:if>
   </xsl:otherwise>
  </xsl:choose>
 </xsl:template>

 <xsl:template match="descriptor">
  <xsl:value-of select="@v"/>
 </xsl:template>

 <xsl:template match="phenomenon">
  <xsl:value-of select="@v"/>
 </xsl:template>

 <xsl:template match="cloudList">
  <xsl:param name="trans"/>
  <xsl:variable name="s_list">
   <xsl:apply-templates select="cloud" mode="list"/>
  </xsl:variable>
  <xsl:variable name="ceiling">
   <xsl:apply-templates select="cloud" mode="ceiling"/>
  </xsl:variable>
  <xsl:apply-templates select="cloud">
   <xsl:with-param name="ceiling" select="$ceiling"/>
   <xsl:with-param name="s_list" select="$s_list"/>
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
 </xsl:template>

 <xsl:template match="cloud" mode="list">
  <xsl:value-of select="concat(' ', @s)"/>
 </xsl:template>

 <xsl:template match="cloud" mode="ceiling">
  <xsl:param name="trans"/>
  <xsl:if test="boolean(isCeiling) and $strictfgfs != 1">
   <xsl:value-of select="cloudBase/@v"/>
  </xsl:if>
 </xsl:template>

 <xsl:template match="cloud">
  <xsl:param name="ceiling"/>
  <xsl:param name="s_list"/>
  <xsl:param name="trans"/>
  <xsl:choose>
   <xsl:when test="position() = 1">
    <xsl:variable name="rows">
     <xsl:choose>
      <xsl:when test="$ceiling != ''"><xsl:value-of select="last() + 1"/></xsl:when>
      <xsl:otherwise><xsl:value-of select="last()"/></xsl:otherwise>
     </xsl:choose>
    </xsl:variable>
    <tr>
    <xsl:if test="$method = 'html'">
     <xsl:element name="td">
      <xsl:attribute name="nowrap">1</xsl:attribute>
      <xsl:attribute name="rowspan"><xsl:value-of select="$rows"/></xsl:attribute>
      <b><xsl:value-of select="$s_list"/></b>
     </xsl:element>
    </xsl:if>
    <xsl:choose>
     <xsl:when test="$ceiling != ''">
      <td nowrap="1"><xsl:value-of select="$trans/hdr[@o='ceiling']"/></td>
      <td nowrap="1">
      <xsl:value-of select="concat($trans/t[@o='cloud-at'], ' ', $ceiling * 100, $trans/units[@o='FT'])"/>
      </td>
      <xsl:text>&#9;&#9;&#9;&#9;</xsl:text>
      <td nowrap="1">
      <xsl:value-of select="concat(format-number($ceiling * 10 * $FT2M, '0') * 10, $trans/units[@o='M'])"/>
      </td>
     </xsl:when>
     <xsl:otherwise>
      <xsl:element name="td">
       <xsl:attribute name="nowrap">1</xsl:attribute>
       <xsl:attribute name="rowspan"><xsl:value-of select="last()"/></xsl:attribute>
       <xsl:value-of select="$trans/hdr[@o='cloud-1']"/>
      </xsl:element>
      <xsl:call-template name="printCloud">
       <xsl:with-param name="at_abv" select="'cloud-at'"/>
       <xsl:with-param name="trans" select="$trans"/>
      </xsl:call-template>
     </xsl:otherwise>
    </xsl:choose>
    </tr><xsl:text>&#10;</xsl:text>
    <xsl:if test="$ceiling != ''">
     <tr>
     <xsl:element name="td">
      <xsl:attribute name="nowrap">1</xsl:attribute>
      <xsl:attribute name="rowspan"><xsl:value-of select="last()"/></xsl:attribute>
      <xsl:value-of select="$trans/hdr[@o='cloud-1']"/>
     </xsl:element>
     <xsl:call-template name="printCloud">
      <xsl:with-param name="at_abv" select="'cloud-at'"/>
      <xsl:with-param name="trans" select="$trans"/>
     </xsl:call-template>
     </tr><xsl:text>&#10;</xsl:text>
    </xsl:if>
   </xsl:when>
   <xsl:otherwise>
    <tr>
    <xsl:text>&#9;&#9;&#9;</xsl:text>
    <xsl:call-template name="printCloud">
     <xsl:with-param name="at_abv" select="'cloud-at'"/>
     <xsl:with-param name="trans" select="$trans"/>
    </xsl:call-template>
    </tr><xsl:text>&#10;</xsl:text>
   </xsl:otherwise>
  </xsl:choose>
 </xsl:template>

 <xsl:template name="printCloud">
  <xsl:param name="trans"/>
  <xsl:param name="at_abv"/>
  <xsl:variable name="this" select="."/>
  <xsl:choose>
   <xsl:when test="boolean(notAvailable)">
    <td nowrap="1" colspan="2">
     <xsl:value-of select="$trans/t[@o='notAvailable']"/>
    </td>
   </xsl:when>
   <xsl:when test="boolean(invalidFormat)">
    <td nowrap="1" colspan="2">
    <xsl:value-of select="concat('(', $trans/t[@o='invalidFormat'])"/>
    <xsl:text>: '</xsl:text>
    <xsl:value-of select="invalidFormat/@v"/>
    <xsl:text>')</xsl:text>
    </td>
   </xsl:when>
   <xsl:when test="boolean(noClouds)">
    <td nowrap="1" colspan="2">
    <xsl:choose>
     <xsl:when test="$strictfgfs != 1">
      <xsl:value-of select="$trans/cloudCover[@o=$this/noClouds/@v]"/>
     </xsl:when>
     <xsl:otherwise>
      <xsl:value-of select="$trans/cloudCover[@o=concat($this/noClouds/@v,'-s')]"/>
     </xsl:otherwise>
    </xsl:choose>
    </td>
   </xsl:when>
   <xsl:otherwise>
    <xsl:element name="td">
     <xsl:attribute name="nowrap">1</xsl:attribute>
     <xsl:if test="not(boolean(cloudBase))">
      <xsl:attribute name="colspan">2</xsl:attribute>
     </xsl:if>
     <xsl:value-of select="$trans/cloudCover[@o=$this/cloudCover/@v]"/>
     <xsl:choose>
      <xsl:when test="boolean(cloudBase)">
       <xsl:value-of select="concat(' ', $trans/t[@o=$at_abv], ' ', cloudBase/@v * 100, $trans/units[@o='FT'])"/>
      </xsl:when>
      <xsl:otherwise>
       <xsl:value-of select="concat(', ', $trans/t[@o='baseBelowStation'])"/>
      </xsl:otherwise>
     </xsl:choose>
     <xsl:if test="boolean(cloudType)">
      <br />
      <xsl:value-of select="concat(' (', $trans/cloudType[@o=$this/cloudType/@v], ')')"/>
     </xsl:if>
     <xsl:if test="boolean(cloudTypeNotAvailable)">
      <xsl:value-of select="concat(' ', $trans/t[@o='cloudTypeNotAvailable'])"/>
     </xsl:if>
    </xsl:element>
    <xsl:if test="boolean(cloudBase)">
     <xsl:text>&#9;&#9;&#9;</xsl:text>
     <td nowrap="1">
     <xsl:value-of select="concat(format-number(cloudBase/@v * 10 * $FT2M, '0') * 10, $trans/units[@o='M'])"/>
     </td>
    </xsl:if>
   </xsl:otherwise>
  </xsl:choose>
 </xsl:template>

 <xsl:template match="temperature">
  <xsl:param name="trans"/>
  <xsl:variable name="rows">
   <xsl:choose>
    <xsl:when test="boolean(air/temp) and boolean(dewpoint/temp)">3</xsl:when>
    <xsl:when test="boolean(air) and boolean(dewpoint)">2</xsl:when>
    <xsl:otherwise>1</xsl:otherwise>
   </xsl:choose>
  </xsl:variable>
  <xsl:if test="boolean(air/temp)">
   <tr>
   <xsl:call-template name="printTemp">
    <xsl:with-param name="hdr" select="$trans/hdr[@o='temperature-temp']"/>
    <xsl:with-param name="rows" select="$rows"/>
    <xsl:with-param name="s" select="@s"/>
    <xsl:with-param name="temp" select="air"/>
    <xsl:with-param name="trans" select="$trans"/>
   </xsl:call-template>
   </tr><xsl:text>&#10;</xsl:text>
  </xsl:if>
  <xsl:if test="boolean(dewpoint/temp) or (boolean(dewpoint) and $method = 'html')">
   <tr>
   <xsl:call-template name="printTemp">
    <xsl:with-param name="hdr" select="$trans/hdr[@o='temperature-dew']"/>
    <xsl:with-param name="temp" select="dewpoint"/>
    <xsl:with-param name="trans" select="$trans"/>
   </xsl:call-template>
   </tr><xsl:text>&#10;</xsl:text>
  </xsl:if>
  <xsl:if test="boolean(relHumid1)">
   <tr>
    <td nowrap="1"><xsl:value-of select="$trans/hdr[@o='relHumid']"/></td>
    <td nowrap="1" colspan="2">
     <xsl:value-of select="format-number(relHumid1/@v, '0')"/>
     <xsl:text>%</xsl:text>
    </td>
   </tr><xsl:text>&#10;</xsl:text>
  </xsl:if>
 </xsl:template>

 <xsl:template match="tempHourly">
  <xsl:param name="trans"/>
  <xsl:variable name="rows">
   <xsl:choose>
    <xsl:when test="boolean(air/temp) and boolean(dewpoint/temp)">2</xsl:when>
    <xsl:otherwise>1</xsl:otherwise>
   </xsl:choose>
  </xsl:variable>
  <xsl:if test="boolean(air/temp)">
   <tr>
   <xsl:call-template name="printTemp">
    <xsl:with-param name="hdr" select="$trans/hdr[@o='tempHourly-temp']"/>
    <xsl:with-param name="rows" select="$rows"/>
    <xsl:with-param name="s" select="@s"/>
    <xsl:with-param name="temp" select="air"/>
    <xsl:with-param name="trans" select="$trans"/>
   </xsl:call-template>
   </tr><xsl:text>&#10;</xsl:text>
  </xsl:if>
  <xsl:if test="boolean(dewpoint/temp)">
   <tr>
   <xsl:call-template name="printTemp">
    <xsl:with-param name="hdr" select="$trans/hdr[@o='tempHourly-dew']"/>
    <xsl:with-param name="temp" select="dewpoint"/>
    <xsl:with-param name="trans" select="$trans"/>
   </xsl:call-template>
   </tr><xsl:text>&#10;</xsl:text>
  </xsl:if>
 </xsl:template>

 <xsl:template match="tempMax|tempMin|tempAt|temp6hMax|temp6hMin|SST|OAT|tempMaxFQ">
  <xsl:param name="trans"/>
  <tr>
  <xsl:variable name="this" select="."/>
  <xsl:call-template name="printTemp">
   <xsl:with-param name="hdr" select="$trans/hdr[@o=name($this)]"/>
   <xsl:with-param name="rows" select="1"/>
   <xsl:with-param name="s" select="@s"/>
   <xsl:with-param name="temp" select="."/>
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:call-template>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="temp24h">
  <xsl:param name="trans"/>
  <xsl:variable name="rows">
   <xsl:choose>
    <xsl:when test="boolean(temp24hMin/temp)">2</xsl:when>
    <xsl:otherwise>1</xsl:otherwise>
   </xsl:choose>
  </xsl:variable>
  <tr>
  <xsl:call-template name="printTemp">
   <xsl:with-param name="hdr" select="$trans/hdr[@o='temp24hMax']"/>
   <xsl:with-param name="rows" select="$rows"/>
   <xsl:with-param name="s" select="@s"/>
   <xsl:with-param name="temp" select="temp24hMax"/>
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:call-template>
  </tr><xsl:text>&#10;</xsl:text>
  <xsl:if test="boolean(temp24hMin/temp)">
   <tr>
   <xsl:call-template name="printTemp">
    <xsl:with-param name="hdr" select="$trans/hdr[@o='temp24hMin']"/>
    <xsl:with-param name="temp" select="temp24hMin"/>
    <xsl:with-param name="trans" select="$trans"/>
   </xsl:call-template>
   </tr><xsl:text>&#10;</xsl:text>
  </xsl:if>
 </xsl:template>

 <xsl:template name="printTemp">
  <xsl:param name="hdr"/>
  <xsl:param name="notavail"/>
  <xsl:param name="rows"/>
  <xsl:param name="s"/>
  <xsl:param name="temp"/>
  <xsl:param name="trans"/>
  <xsl:if test="$method = 'html' and $s != ''">
   <xsl:element name="td">
    <xsl:attribute name="nowrap">1</xsl:attribute>
    <xsl:attribute name="rowspan"><xsl:value-of select="$rows"/></xsl:attribute>
    <b><xsl:value-of select="$s"/></b>
   </xsl:element>
  </xsl:if>
  <td nowrap="1">
  <xsl:value-of select="$hdr"/>
  </td>
  <xsl:choose>
   <xsl:when test="boolean($temp/notAvailable)">
    <td nowrap="1" colspan="2">
    <xsl:value-of select="$trans/t[@o='notAvailable']"/>
    </td>
   </xsl:when>
   <xsl:otherwise>
    <td nowrap="1">
    <xsl:choose>
     <xsl:when test="$temp/unitTemp/@v = 'C'">
      <xsl:value-of select="$temp/temp/@v"/>
     </xsl:when>
     <xsl:otherwise>
      <xsl:value-of select="format-number(($temp/temp/@v - 32) div 1.8, '0.#')"/>
     </xsl:otherwise>
    </xsl:choose>
    <xsl:text> °C</xsl:text>
    <xsl:if test="boolean($temp/hour)">
     <xsl:value-of select="concat(substring-before($trans/t[@o='time-at'], '|'), $temp/hour/@v, ':00 UTC', substring-after($trans/t[@o='time-at'], '|'))"/>
    </xsl:if>
    </td>
    <xsl:choose>
     <xsl:when test="boolean($temp/hour)">
      <xsl:text>&#9;&#9;&#9;</xsl:text>
     </xsl:when>
     <xsl:otherwise>
      <xsl:text>&#9;&#9;&#9;&#9;&#9;</xsl:text>
     </xsl:otherwise>
    </xsl:choose>
    <td nowrap="1">
    <xsl:choose>
     <xsl:when test="$temp/unitTemp/@v = 'C'">
      <!--xsl:value-of select="format-number($temp/temp/@v * 1.8 + 32, '0.#')"/-->
      <xsl:value-of select="floor(10 * ($temp/temp/@v * 1.8 + 32 + 0.05)) div 10"/>
     </xsl:when>
     <xsl:otherwise>
      <xsl:value-of select="$temp/temp/@v"/>
     </xsl:otherwise>
    </xsl:choose>
    <xsl:text> °F</xsl:text>
    </td>
   </xsl:otherwise>
  </xsl:choose>
 </xsl:template>

 <xsl:template match="QNH|somePressure|SLPdPa|SLPinHg|regQNH|QFE|QFF">
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1"><b><xsl:value-of select="@s"/></b></td>
  </xsl:if>
  <td nowrap="1">
  <xsl:value-of select="$trans/hdr[@o=name($this)]"/>
  </td>
  <xsl:if test="boolean(inHg) or boolean(hPa) or boolean(mmHg)">
   <td nowrap="1">
   <xsl:choose>
    <xsl:when test="boolean(hPa)">
     <xsl:value-of select="hPa/@v"/>
    </xsl:when>
    <xsl:when test="boolean(mmHg)">
     <xsl:value-of select="format-number(mmHg/@v div ($FT2M div 12 * 1000) * $INHG2HPA, '0')"/>
    </xsl:when>
    <xsl:otherwise>
     <xsl:value-of select="format-number(inHg/@v * $INHG2HPA, '0')"/>
    </xsl:otherwise>
   </xsl:choose>
   <xsl:value-of select="$trans/units[@o='hPa']"/>
   </td>
   <xsl:text>&#9;&#9;&#9;&#9;</xsl:text>
   <td nowrap="1">
   <xsl:choose>
    <xsl:when test="boolean(inHg)">
     <xsl:value-of select="inHg/@v"/>
    </xsl:when>
    <xsl:when test="boolean(hPa)">
     <xsl:value-of select="format-number(hPa/@v div $INHG2HPA, '0.##')"/>
    </xsl:when>
    <xsl:otherwise>
     <xsl:value-of select="format-number(mmHg/@v div ($FT2M div 12 * 1000), '0.##')"/>
    </xsl:otherwise>
   </xsl:choose>
   <xsl:value-of select="$trans/units[@o='inHg']"/>
   </td>
  </xsl:if>
  <xsl:if test="boolean(notAvailable)">
   <td nowrap="1" colspan="2">
    <xsl:value-of select="$trans/t[@o='notAvailable']"/>
   </td>
  </xsl:if>
  <xsl:if test="boolean(invalidFormat)">
   <td nowrap="1" colspan="2">
   <xsl:value-of select="concat('(', $trans/t[@o='invalidFormat'])"/>
   <xsl:text>: '</xsl:text>
   <xsl:value-of select="invalidFormat/@v"/>
   <xsl:text>')</xsl:text>
   </td>
  </xsl:if>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="cloudMaxCover">
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1"><b><xsl:value-of select="@s"/></b></td>
  </xsl:if>
  <td nowrap="1">
  <xsl:value-of select="$trans/hdr[@o='cloudMaxCover']"/>
  </td>
  <td nowrap="1" colspan="2">
  <xsl:choose>
   <xsl:when test="boolean(SKC)">
    <xsl:value-of select="$trans/cloudCover[@o='SKC']"/>
   </xsl:when>
   <xsl:otherwise>
    <xsl:value-of select="$trans/cloudCover[@o=$this/cloudCover/@v]"/>
   </xsl:otherwise>
  </xsl:choose>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="cloudAbove">
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1"><b><xsl:value-of select="@s"/></b></td>
  </xsl:if>
  <td nowrap="1">
  <xsl:choose>
   <xsl:when test="boolean(isThin)">
    <xsl:value-of select="$trans/hdr[@o='cloudLayerThin']"/>
   </xsl:when>
   <xsl:otherwise>
    <xsl:value-of select="$trans/hdr[@o='cloudLayer']"/>
   </xsl:otherwise>
  </xsl:choose>
  </td>
  <xsl:call-template name="printCloud">
   <xsl:with-param name="at_abv" select="'cloud-above'"/>
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:call-template>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="cloudCoverVar">
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1" rowspan="2"><b><xsl:value-of select="@s"/></b></td>
  </xsl:if>
  <td nowrap="1"><xsl:value-of select="$trans/hdr[@o='cloudCoverVar']"/></td>
  <xsl:choose>
   <xsl:when test="boolean(cloudBase)">
    <xsl:call-template name="printCloud">
     <xsl:with-param name="at_abv" select="'cloud-at'"/>
     <xsl:with-param name="trans" select="$trans"/>
    </xsl:call-template>
   </xsl:when>
   <xsl:otherwise>
   <td nowrap="1" colspan="2">
   <xsl:value-of select="$trans/cloudCover[@o=$this/cloudCover/@v]"/>
   </td>
   </xsl:otherwise>
  </xsl:choose>
  </tr><xsl:text>&#10;</xsl:text>
  <tr>
  <td nowrap="1"><xsl:value-of select="$trans/hdr[@o='cloudCoverVar2']"/></td>
  <td nowrap="1" colspan="2">
  <xsl:value-of select="$trans/cloudCover[@o=$this/cloudCover2/@v]"/>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="reportConcerns">
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1"><b><xsl:value-of select="@s"/></b></td>
  </xsl:if>
  <td nowrap="1">
  <xsl:value-of select="$trans/hdr[@o='reportConcerns']"/>
  </td>
  <td nowrap="1" colspan="2">
  <xsl:variable name="chg">
   <xsl:value-of select="$trans/concernsChange[@o=$this/change/@v]"/>
  </xsl:variable>
  <xsl:value-of select="concat(substring-before($chg, '|'), $trans/concernsSubject[@o=$this/subject/@v], substring-after($chg, '|'))"/>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="cloudOpacityLvl">
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1"><b><xsl:value-of select="@s"/></b></td>
  </xsl:if>
  <td nowrap="1">
  <xsl:value-of select="$trans/hdr[@o='cloudOpacityLvl']"/>
  </td>
  <td nowrap="1">
  <xsl:value-of select="concat(eights/@v, '/8')"/>
  <xsl:if test="boolean(weather)">
   <xsl:text> </xsl:text>
   <xsl:call-template name="printWeather">
    <xsl:with-param name="is_recent" select="1"/>
    <xsl:with-param name="this" select="weather"/>
    <xsl:with-param name="trans" select="$trans"/>
   </xsl:call-template>
  </xsl:if>
  <xsl:if test="boolean(cloudType)">
   <xsl:value-of select="concat(' ', $trans/cloudType[@o=$this/cloudType/@v])"/>
  </xsl:if>
  <xsl:value-of select="concat(' ', $trans/t[@o='cloud-at'], ' ', cloudBase/@v * 100, $trans/units[@o='FT'])"/>
  <xsl:apply-templates select="locationAndList">
   <xsl:with-param name="locOrDir" select="'compassDirLocation'"/>
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  </td>
  <xsl:text>&#9;&#9;</xsl:text>
  <td nowrap="1">
  <xsl:value-of select="concat(format-number(cloudBase/@v * 10 * $FT2M, '0') * 10, $trans/units[@o='M'])"/>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="RH">
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1"><b><xsl:value-of select="@s"/></b></td>
  </xsl:if>
  <td nowrap="1"><xsl:value-of select="$trans/hdr[@o='relHumid']"/></td>
  <td nowrap="1" colspan="2">
  <xsl:value-of select="relHumid/@v"/>
  <xsl:text>%</xsl:text>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="AI">
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1"><b><xsl:value-of select="@s"/></b></td>
  </xsl:if>
  <td nowrap="1"><xsl:value-of select="$trans/hdr[@o='AI']"/></td>
  <td nowrap="1" colspan="2">
   <xsl:value-of select="AIVal/@v"/>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="rwyState">
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1"><b><xsl:value-of select="@s"/></b></td>
  </xsl:if>
  <xsl:choose>
   <xsl:when test="boolean(SNOCLO)">
    <td nowrap="1" colspan="3">
    <xsl:value-of select="$trans/t[@o='SNOCLO']"/>
    </td>
   </xsl:when>
   <xsl:otherwise>
    <td nowrap="1">
    <xsl:value-of select="substring-before($trans/hdr[@o='stateOfRwy'], '|')"/>
    <xsl:choose>
     <xsl:when test="boolean(rwyDesigAll)">
      <xsl:value-of select="$trans/t[@o='rwyDesigAll']"/>
     </xsl:when>
     <xsl:when test="boolean(rwyDesigRep)">
      <xsl:value-of select="$trans/t[@o='rwyDesigRep']"/>
     </xsl:when>
     <xsl:otherwise>
      <xsl:value-of select="concat(substring-before($trans/t[@o='rwyDesig'], '|'), rwyDesig/@v, substring-after($trans/t[@o='rwyDesig'], '|'))"/>
     </xsl:otherwise>
    </xsl:choose>
    <xsl:value-of select="substring-after($trans/hdr[@o='stateOfRwy'], '|')"/>
    </td>
    <td colspan="2">
    <xsl:choose>
     <xsl:when test="boolean(depositType/notAvailable) and boolean(depositExtent/notAvailable) and boolean(depositDepth/notAvailable) and boolean(friction/notAvailable)">
      <xsl:value-of select="$trans/t[@o='notAvailable']"/>
     </xsl:when>

     <xsl:otherwise>

      <xsl:if test="boolean(cleared)">
       <xsl:value-of select="$trans/t[@o='cleared']"/>
      </xsl:if>

      <xsl:variable name="delim1">
       <xsl:if test="boolean(cleared)">, </xsl:if>
      </xsl:variable>
      <xsl:if test="boolean(depositType/depositTypeVal)">
       <xsl:value-of select="$delim1"/>
       <xsl:value-of select="$trans/depositType[@o=$this/depositType/depositTypeVal/@v]"/>
      </xsl:if>

      <xsl:variable name="delim2">
       <xsl:if test="$delim1 != '' or boolean(depositType/depositTypeVal)">, </xsl:if>
      </xsl:variable>
      <xsl:if test="boolean(depositExtent/invalidFormat)">
       <xsl:value-of select="$delim2"/>
       <xsl:value-of select="concat('(', $trans/t[@o='invalidFormat'])"/>
       <xsl:text>: '</xsl:text>
       <xsl:value-of select="depositExtent/invalidFormat/@v"/>
       <xsl:text>')</xsl:text>
      </xsl:if>
      <xsl:if test="boolean(depositExtent/depositExtentVal)">
       <xsl:value-of select="$delim2"/>
       <xsl:value-of select="substring-before($trans/depositExtent[@o=$this/depositExtent/depositExtentVal/@v], '|')"/>
       <xsl:if test="$strictfgfs != 1">
        <xsl:value-of select="$trans/t[@o='depositExtent']"/>
       </xsl:if>
       <xsl:value-of select="substring-after($trans/depositExtent[@o=$this/depositExtent/depositExtentVal/@v], '|')"/>
      </xsl:if>

      <xsl:variable name="delim3">
       <xsl:if test="$delim2 != '' or boolean(depositExtent/invalidFormat) or boolean(depositExtent/depositExtentVal)">, </xsl:if>
      </xsl:variable>
      <xsl:if test="boolean(depositDepth/invalidFormat)">
       <xsl:value-of select="$delim3"/>
       <xsl:value-of select="concat('(', $trans/t[@o='invalidFormat'])"/>
       <xsl:text>: '</xsl:text>
       <xsl:value-of select="depositDepth/invalidFormat/@v"/>
       <xsl:text>')</xsl:text>
      </xsl:if>
      <xsl:if test="boolean(depositDepth/depositDepthVal)">
       <xsl:value-of select="$delim3"/>
       <xsl:if test="$strictfgfs != 1 and depositDepth/depositDepthVal/@v &lt; 99">
        <xsl:value-of select="substring-before($trans/t[@o='depositDepth'], '|')"/>
       </xsl:if>
       <xsl:choose>
        <xsl:when test="depositDepth/depositDepthVal/@v &lt; 1">
         <xsl:value-of select="concat('&lt;1', $trans/units[@o='MM'])"/>
        </xsl:when>
        <xsl:when test="depositDepth/depositDepthVal/@v &lt; 91">
         <xsl:value-of select="concat(depositDepth/depositDepthVal/@v, $trans/units[@o='MM'])"/>
        </xsl:when>
        <xsl:when test="depositDepth/depositDepthVal/@v &lt; 99">
         <xsl:value-of select="concat(50 * (depositDepth/depositDepthVal/@v - 90), $trans/units[@o='MM'])"/>
        </xsl:when>
        <xsl:otherwise>
         <xsl:value-of select="$trans/t[@o='rwyNotInUse']"/>
        </xsl:otherwise>
       </xsl:choose>
       <xsl:if test="$strictfgfs != 1">
        <xsl:value-of select="substring-after($trans/t[@o='depositDepth'], '|')"/>
       </xsl:if>
      </xsl:if>
      <xsl:variable name="delim4">
       <xsl:if test="$delim3 != '' or boolean(depositDepth/invalidFormat) or boolean(depositDepth/depositDepthVal)">, </xsl:if>
      </xsl:variable>
      <xsl:if test="boolean(friction/invalidFormat)">
       <xsl:value-of select="$delim4"/>
       <xsl:value-of select="concat('(', $trans/t[@o='invalidFormat'])"/>
       <xsl:text>: '</xsl:text>
       <xsl:value-of select="friction/invalidFormat/@v"/>
       <xsl:text>')</xsl:text>
      </xsl:if>
      <xsl:if test="boolean(friction/coefficient)">
       <xsl:value-of select="$delim4"/>
       <xsl:value-of select="concat($trans/t[@o='friction'], format-number(friction/coefficient/@v div 100, '0.00'))"/>
      </xsl:if>
      <xsl:if test="boolean(friction/brakingAction)">
       <xsl:value-of select="$delim4"/>
       <xsl:value-of select="$trans/friction[@o=$this/friction/brakingAction/@v]"/>
      </xsl:if>
      <xsl:if test="boolean(friction/unreliable)">
       <xsl:value-of select="$delim4"/>
       <xsl:value-of select="$trans/friction[@o='99']"/>
      </xsl:if>
     </xsl:otherwise>
    </xsl:choose>
    </td>
    </xsl:otherwise>
   </xsl:choose>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="windShear">
  <xsl:param name="trans"/>
  <tr>
   <xsl:if test="$method = 'html'">
    <td nowrap="1"><b><xsl:value-of select="@s"/></b></td>
   </xsl:if>
  <td nowrap="1">
  <xsl:value-of select="substring-before($trans/hdr[@o='windShearRwy'], '|')"/>
  <xsl:choose>
   <xsl:when test="boolean(rwyDesigAll)">
    <xsl:value-of select="$trans/t[@o='rwyDesigAll']"/>
   </xsl:when>
   <xsl:otherwise>
    <xsl:value-of select="concat(substring-before($trans/t[@o='rwyDesig'], '|'), rwyDesig/@v, substring-after($trans/t[@o='rwyDesig'], '|'))"/>
   </xsl:otherwise>
  </xsl:choose>
  <xsl:value-of select="substring-after($trans/hdr[@o='windShearRwy'], '|')"/>
  </td>
  <td nowrap="1" colspan="2">
    <xsl:value-of select="$trans/t[@o='windShear']"/>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="recentWeatherList">
  <xsl:param name="trans"/>
  <xsl:variable name="s_list">
   <xsl:apply-templates select="weather" mode="list"/>
  </xsl:variable>
  <xsl:apply-templates select="weather">
   <xsl:with-param name="s_list" select="$s_list"/>
   <xsl:with-param name="hdr1" select="$trans/hdr[@o='recentWeather-1']"/>
   <xsl:with-param name="is_recent" select="1"/>
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="colourCode">
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <xsl:variable name="rows">
   <xsl:choose>
    <xsl:when test="boolean(BLACK) and boolean(currentColour) and boolean(predictedColour)">3</xsl:when>
    <xsl:when test="boolean(BLACK) and boolean(currentColour)">2</xsl:when>
    <xsl:when test="boolean(currentColour) and boolean(predictedColour)">2</xsl:when>
    <xsl:otherwise>1</xsl:otherwise>
   </xsl:choose>
  </xsl:variable>
  <tr>
  <xsl:if test="$method = 'html'">
   <xsl:element name="td">
    <xsl:attribute name="nowrap">1</xsl:attribute>
    <xsl:attribute name="rowspan"><xsl:value-of select="$rows"/></xsl:attribute>
    <b><xsl:value-of select="@s"/></b>
   </xsl:element>
  </xsl:if>
  <xsl:element name="td">
   <xsl:attribute name="nowrap">1</xsl:attribute>
   <xsl:attribute name="rowspan"><xsl:value-of select="$rows"/></xsl:attribute>
   <xsl:value-of select="$trans/hdr[@o='colourCode']"/>
  </xsl:element>
  <td nowrap="1" colspan="2">
  <xsl:choose>
   <xsl:when test="boolean(BLACK)">
    <xsl:value-of select="$trans/colour[@o='BLACK']"/>
   </xsl:when>
   <xsl:otherwise>
    <xsl:value-of select="$trans/colour[@o=$this/currentColour/@v]"/>
   </xsl:otherwise>
  </xsl:choose>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
  <xsl:if test="boolean(currentColour) and boolean(BLACK)">
   <xsl:text>&#9;&#9;&#9;</xsl:text>
   <tr>
   <td nowrap="1" colspan="2">
   <xsl:value-of select="$trans/colour[@o=$this/currentColour/@v]"/>
   </td>
   </tr><xsl:text>&#10;</xsl:text>
  </xsl:if>
  <xsl:if test="boolean(predictedColour)">
   <xsl:text>&#9;&#9;&#9;</xsl:text>
   <tr>
   <td nowrap="1" colspan="2">
   <xsl:value-of select="$trans/t[@o='predictedColour']"/>
   <br /><xsl:text>&#10;&#9;&#9;&#9;</xsl:text>
   <xsl:choose>
    <xsl:when test="currentColour/@v != predictedColour/@v">
     <xsl:value-of select="$trans/colour[@o=$this/predictedColour/@v]"/>
    </xsl:when>
    <xsl:otherwise>
     <xsl:value-of select="$trans/colour[@o='noChange']"/>
    </xsl:otherwise>
   </xsl:choose>
   </td>
   </tr><xsl:text>&#10;</xsl:text>
  </xsl:if>
 </xsl:template>

 <xsl:template match="windShearLvl">
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1" rowspan="2"><b><xsl:value-of select="@s"/></b></td>
  </xsl:if>
  <td nowrap="1" rowspan="2">
  <xsl:value-of select="$trans/hdr[@o='windShearLvl']"/>
  </td>
  <td nowrap="1">
  <xsl:value-of select="concat($trans/t[@o='topOfLayer'], ' ', level/@v * 100, $trans/units[@o='FT'])"/>
  </td>
  <xsl:text>&#9;&#9;&#9;</xsl:text>
  <td nowrap="1">
  <xsl:value-of select="concat(format-number(level/@v * 10 * $FT2M, '0') * 10, $trans/units[@o='M'])"/>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
  <tr>
  <xsl:text>&#9;&#9;&#9;</xsl:text>
  <xsl:variable name="dir">
   <xsl:value-of select="concat($trans/t[@o='wind-from'], ' ')"/>
   <xsl:value-of select="$trans/compassDir16/arr[position()=1+format-number($this/wind/dir/@v div 45 * 2, '0')]"/>
   <xsl:value-of select="concat(' (', wind/dir/@v, '°)')"/>
  </xsl:variable>
  <xsl:variable name="speed">
   <xsl:call-template name="speed2kt">
    <xsl:with-param name="speed" select="wind/speed/@v"/>
    <xsl:with-param name="unit" select="wind/unitSpeed/@v"/>
   </xsl:call-template>
  </xsl:variable>
  <xsl:call-template name="printSpeed">
   <xsl:with-param name="dir" select="$dir"/>
   <xsl:with-param name="speed" select="$speed"/>
   <xsl:with-param name="delim" select="'&#9;&#9;'"/>
   <xsl:with-param name="greater" select="speedGreater/@v"/>
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:call-template>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template name="printLayerHeight">
  <xsl:param name="height"/>
  <xsl:param name="trans"/>
  <xsl:value-of select="concat($height, $trans/units[@o='M'], ' (', format-number($height div $FT2M div 100, '#0'), '00', $trans/units[@o='FT'], ') AGL')"/>
 </xsl:template>

 <xsl:template match="turbulence|icing">
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1"><b><xsl:value-of select="@s"/></b></td>
  </xsl:if>
  <td nowrap="1">
  <xsl:value-of select="$trans/hdr[@o=name($this)]"/>
  </td>
  <td nowrap="1" colspan="2">
  <xsl:value-of select="$trans/turbulenceDescr[@o=$this/turbulenceDescr/@v]"/>
  <xsl:value-of select="$trans/icingDescr[@o=$this/icingDescr/@v]"/>
  <xsl:value-of select="concat(', ', $trans/t[@o='level-from'], ' ')"/>
  <xsl:choose>
   <!-- WMO 306 Vol I.1 Part A, coding table 1690 -->
   <xsl:when test="layerBase/@v = '0'">
    <xsl:value-of select="concat('&lt;30', $trans/units[@o='M'], ' (&lt;100', $trans/units[@o='FT'], ') AGL')"/>
   </xsl:when>
   <xsl:otherwise>
    <xsl:call-template name="printLayerHeight">
     <xsl:with-param name="height" select="layerBase/@v * 30"/>
     <xsl:with-param name="trans" select="$trans"/>
    </xsl:call-template>
   </xsl:otherwise>
  </xsl:choose>
  <xsl:choose>
   <xsl:when test="boolean(layerTop)">
    <xsl:value-of select="concat(' ', $trans/t[@o='level-to'], ' ')"/>
    <xsl:call-template name="printLayerHeight">
     <xsl:with-param name="height" select="layerTop/@v * 30"/>
     <xsl:with-param name="trans" select="$trans"/>
    </xsl:call-template>
   </xsl:when>
   <xsl:when test="layerThickness/@v = '0'">
    <xsl:value-of select="$trans/t[@o='toTopOfCloud']"/>
   </xsl:when>
   <xsl:otherwise>
    <xsl:value-of select="concat(' ', $trans/t[@o='level-to'], ' ')"/>
    <!-- layerThickness: WMO 306 Vol I.1 Part A, coding table 0300 -->
    <xsl:call-template name="printLayerHeight">
     <xsl:with-param name="height" select="layerBase/@v * 30 + layerThickness/@v * 300"/>
     <xsl:with-param name="trans" select="$trans"/>
    </xsl:call-template>
   </xsl:otherwise>
  </xsl:choose>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="amendment">
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1"><b><xsl:value-of select="@s"/></b></td>
  </xsl:if>
  <td nowrap="1" colspan="3">
  <xsl:value-of select="concat($trans/t[@o='amendment'], ' ')"/>
  <xsl:choose>
   <xsl:when test="boolean(isNotScheduled)">
    <xsl:value-of select="$trans/t[@o='isNotScheduled']"/>
   </xsl:when>
   <xsl:otherwise>
    <xsl:value-of select="$trans/t[@o='isLtdToCldVisWind']"/>
    <xsl:if test="boolean(hourFrom)">
     <xsl:value-of select="concat(' ', substring-before($trans/t[@o='time-from'], '|'), hourFrom/@v, ':00 UTC', substring-after($trans/t[@o='time-from'], '|'))"/>
    </xsl:if>
    <xsl:if test="boolean(hourTill)">
     <xsl:value-of select="concat(' ', substring-before($trans/t[@o='time-till'], '|'), hourTill/@v, ':00 UTC', substring-after($trans/t[@o='time-till'], '|'))"/>
    </xsl:if>
   </xsl:otherwise>
  </xsl:choose>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="trendList">
  <xsl:param name="trans"/>
  <xsl:text>&#10;</xsl:text>
  <tr>
  <td></td>
  <td nowrap="1" colspan="3"><b>
   <xsl:if test="name(..) = 'metar'">
    <xsl:value-of select="$trans/hdr[@o='trendList']"/>
   </xsl:if>
   <xsl:if test="name(..) != 'metar'">
    <xsl:value-of select="$trans/hdr[@o='trendListTAF']"/>
   </xsl:if>
   <xsl:if test="$method != 'html'">
    <xsl:if test="name(..) = 'metar'">
     <xsl:value-of select="concat('&#10;', $trans/hdr[@o='trendList_ul'])"/>
    </xsl:if>
    <xsl:if test="name(..) != 'metar'">
     <xsl:value-of select="concat('&#10;', $trans/hdr[@o='trendListTAF_ul'])"/>
    </xsl:if>
   </xsl:if>
  </b></td>
  </tr><xsl:text>&#10;</xsl:text>
  <xsl:apply-templates select="trend">
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
 </xsl:template>

 <xsl:template match="trend">
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <tr>
  <xsl:choose>
   <xsl:when test="$method = 'html'">
    <td nowrap="1"><b><xsl:value-of select="@s"/></b></td>
   </xsl:when>
   <xsl:when test="position() != 1">
   <xsl:text>&#10;</xsl:text>
   </xsl:when>
  </xsl:choose>
  <td nowrap="1" colspan="3">
  <xsl:choose>
   <xsl:when test="trendType/@v = 'NOSIG'">
    <xsl:value-of select="$trans/t[@o='NOSIG']"/>
   </xsl:when>
   <xsl:otherwise>
    <b>
    <xsl:if test="boolean(probability)">
     <xsl:value-of select="concat(substring-before($trans/t[@o='prob'], '|'), probability/@v, substring-after($trans/t[@o='prob'], '|'))"/>
    </xsl:if>
    <xsl:if test="boolean(trendTime1)">
     <xsl:if test="boolean(probability)"><xsl:text> </xsl:text></xsl:if>
     <xsl:variable name="timeSpec">
      <xsl:choose>
       <xsl:when test="trendTime1/timeSpec/@v = 'AT'">
        <xsl:value-of select="$trans/t[@o='time-at']"/>
       </xsl:when>
       <xsl:when test="trendTime1/timeSpec/@v = 'FM'">
        <xsl:value-of select="$trans/t[@o='time-from']"/>
       </xsl:when>
       <xsl:otherwise>
        <xsl:value-of select="$trans/t[@o='time-till']"/>
       </xsl:otherwise>
      </xsl:choose>
     </xsl:variable>
     <xsl:value-of select="concat(substring-before($timeSpec, '|'), trendTime1/hour/@v, ':', trendTime1/minute/@v, ' UTC', substring-after($timeSpec, '|'))"/>
     <xsl:if test="boolean(trendTime2)">
      <xsl:value-of select="concat(' ', substring-before($trans/t[@o='time-till'], '|'), trendTime2/hour/@v, ':', trendTime2/minute/@v, ' UTC', substring-after($trans/t[@o='time-till'], '|'))"/>
     </xsl:if>
    </xsl:if>
    <xsl:if test="$trans/t[@o=$this/trendType/@v] != ':' and (boolean(probability) or boolean(trendTime1))">
     <xsl:text> </xsl:text>
    </xsl:if>
    <xsl:value-of select="$trans/t[@o=$this/trendType/@v]"/>
    </b>
   </xsl:otherwise>
  </xsl:choose>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
  <xsl:apply-templates select="sfcWind">
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:apply-templates select="CAVOK">
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:apply-templates select="visPrev">
   <xsl:with-param name="unitThreshold" select="1000"/>
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:apply-templates select="visVert">
   <xsl:with-param name="unitThreshold" select="1000"/>
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:apply-templates select="weatherList">
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:apply-templates select="cloudList">
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:apply-templates select="rwyStateList/rwyState">
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:apply-templates select="colourCode">
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:apply-templates select="windShearLvl|turbulence|icing">
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
 </xsl:template>

 <xsl:template match="tafRemarkList">
  <xsl:param name="trans"/>
  <xsl:text>&#10;</xsl:text>
  <tr>
  <xsl:if test="$method = 'html'">
   <td><b>RMK</b></td>
  </xsl:if>
  <td nowrap="1" colspan="3"><b>
   <xsl:value-of select="$trans/hdr[@o='remarkList']"/>
   <xsl:if test="$method != 'html'">
    <xsl:value-of select="concat('&#10;', $trans/hdr[@o='remarkList_ul'])"/>
   </xsl:if>
  </b></td>
  </tr><xsl:text>&#10;</xsl:text>
  <xsl:apply-templates select="tafRemark">
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
 </xsl:template>

 <xsl:template match="tafRemark">
  <xsl:param name="trans"/>
  <xsl:apply-templates select="child::*[last()]">
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
 </xsl:template>

 <xsl:template match="remarkList">
  <xsl:param name="trans"/>
  <xsl:text>&#10;</xsl:text>
  <tr>
  <xsl:if test="$method = 'html'">
   <td><b>RMK</b></td>
  </xsl:if>
  <td nowrap="1" colspan="3"><b>
   <xsl:value-of select="$trans/hdr[@o='remarkList']"/>
   <xsl:if test="$method != 'html'">
    <xsl:value-of select="concat('&#10;', $trans/hdr[@o='remarkList_ul'])"/>
   </xsl:if>
  </b></td>
  </tr><xsl:text>&#10;</xsl:text>
  <xsl:apply-templates select="remark">
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
 </xsl:template>

 <xsl:template match="remark">
  <xsl:param name="trans"/>
  <xsl:variable name="rmk">
   <xsl:value-of select="name(child::*[last()])"/>
  </xsl:variable>
  <xsl:choose>
   <xsl:when test="$rmk = 'cloud'">
    <xsl:apply-templates select="cloud">
     <xsl:with-param name="trans" select="$trans"/>
     <xsl:with-param name="s_list" select="cloud/@s"/>
    </xsl:apply-templates>
   </xsl:when>
   <xsl:when test="$rmk = 'visRwy'">
    <xsl:apply-templates select="visRwy">
     <xsl:with-param name="trans" select="$trans"/>
     <xsl:with-param name="unitThreshold" select="1000"/>
    </xsl:apply-templates>
   </xsl:when>
   <xsl:otherwise>
    <xsl:apply-templates select="child::*[last()]">
     <xsl:with-param name="trans" select="$trans"/>
    </xsl:apply-templates>
   </xsl:otherwise>
  </xsl:choose>
 </xsl:template>

 <xsl:template match="WARNING">
  <tr>
  <td nowrap="1" colspan="4">
  <pre style="margin: 0; font-size: 90%;">
  <xsl:value-of select="."/>
  </pre>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="CAVOK|RVRNO|needMaint|fcstAutoMETAR">
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1"><b>
   <xsl:choose>
    <xsl:when test="boolean(@s)">
     <xsl:value-of select="@s"/>
    </xsl:when>
    <xsl:otherwise>
     <xsl:value-of select="name(.)"/>
    </xsl:otherwise>
   </xsl:choose>
   </b></td>
  </xsl:if>
  <td colspan="3">
   <xsl:value-of select="$trans/t[@o=name($this)]"/>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="keyword">
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1"><b><xsl:value-of select="@s"/></b></td>
  </xsl:if>
  <td colspan="3">
  <xsl:choose>
   <xsl:when test="$trans/t[@o=$this/@v] != ''">
    <xsl:value-of select="$trans/t[@o=$this/@v]"/>
    <xsl:if test="@v = 'EPO' or @v = 'EPC' or @v = 'EPM'">
     <!-- Sheep Mountain Airport -->
     <xsl:if test="../../../obsStation/id/@v = 'PASP'"> (Tahetna pass)</xsl:if>
     <!-- Cantwell -->
     <xsl:if test="../../../obsStation/id/@v = 'PATW'"></xsl:if>
     <!-- Merrill Pass West -->
     <xsl:if test="../../../obsStation/id/@v = 'PAER'"></xsl:if>
     <!-- Farewell Lake Seaplane Base -->
     <xsl:if test="../../../obsStation/id/@v = 'PAFK'"></xsl:if>
     <!-- Healy River Airport -->
     <xsl:if test="../../../obsStation/id/@v = 'PAHV'"></xsl:if>
     <!-- Port Alsworth -->
     <xsl:if test="../../../obsStation/id/@v = 'PALJ'"></xsl:if>
     <!-- Big River Lake -->
     <xsl:if test="../../../obsStation/id/@v = 'PALV'"></xsl:if>
    </xsl:if>
   </xsl:when>
   <xsl:otherwise>
    <xsl:value-of select="@s"/>
   </xsl:otherwise>
  </xsl:choose>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="obsStationType">
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1"><b><xsl:value-of select="@s"/></b></td>
  </xsl:if>
  <td nowrap="1" colspan="3">
  <xsl:value-of select="$trans/stationType[@o=$this/stationType/@v]"/>
  </td></tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="visListLoc">
  <xsl:param name="trans"/>
  <xsl:apply-templates select="visLocData">
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
 </xsl:template>

 <xsl:template match="visLocData">
  <xsl:param name="trans"/>
  <tr>
  <xsl:if test="position() = 1">
   <xsl:if test="$method = 'html'">
    <xsl:element name="td">
     <xsl:attribute name="rowspan"><xsl:value-of select="last()"/></xsl:attribute>
     <b><xsl:value-of select="../@s"/></b>
    </xsl:element>
   </xsl:if>
  </xsl:if>
  <td nowrap="1">
  <xsl:value-of select="substring-before($trans/hdr[@o='visLoc'], '|')"/>
  <xsl:apply-templates select="locationAndList">
   <xsl:with-param name="locOrDir" select="'compassDirTo'"/>
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:value-of select="substring-after($trans/hdr[@o='visLoc'], '|')"/>
  </td>
  <xsl:call-template name="visibility">
   <xsl:with-param name="notavail" select="$trans/t[@o='notAvailable']"/>
   <xsl:with-param name="rnd_m" select="10"/>
   <xsl:with-param name="unitThreshold" select="1000"/>
   <xsl:with-param name="visVal" select="visibility"/>
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:call-template>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="MOV|MOVD">
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <xsl:value-of select="$trans/t[@o=name($this)]"/>
  <xsl:apply-templates select="locationAndList">
   <xsl:with-param name="locOrDir" select="'compassDirTo'"/>
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
 </xsl:template>

 <xsl:template match="locationAndList">
  <xsl:param name="locOrDir"/>
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <xsl:value-of select="$trans/t[@o=name($this)]"/>
  <xsl:apply-templates select="locationThruList">
   <xsl:with-param name="locOrDir" select="$locOrDir"/>
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:if test="boolean(isStationary)">
   <xsl:if test="not(name(..) = 'MOV' or name(..) = 'MOVD')">
    <xsl:text>,</xsl:text>
   </xsl:if>
   <xsl:value-of select="$trans/t[@o='isStationary']"/>
  </xsl:if>
 </xsl:template>

 <xsl:template match="locationThruList">
  <xsl:param name="locOrDir"/>
  <xsl:param name="trans"/>
  <xsl:choose>
   <xsl:when test="position() != 1">
     <xsl:text>,</xsl:text>
     <xsl:apply-templates select="location">
      <xsl:with-param name="locOrDirBefore"/>
      <xsl:with-param name="locOrDirAfter"/>
      <xsl:with-param name="trans" select="$trans"/>
     </xsl:apply-templates>
   </xsl:when>
   <xsl:otherwise>
    <xsl:apply-templates select="location">
     <xsl:with-param name="locOrDirBefore" select="substring-before($trans/t[@o=$locOrDir], '|')"/>
     <xsl:with-param name="locOrDirAfter" select="substring-after($trans/t[@o=$locOrDir], '|')"/>
     <xsl:with-param name="trans" select="$trans"/>
    </xsl:apply-templates>
   </xsl:otherwise>
  </xsl:choose>
 </xsl:template>

 <xsl:template match="location">
  <xsl:param name="locOrDirBefore"/>
  <xsl:param name="locOrDirAfter"/>
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <xsl:if test="position() != 1">
   <xsl:value-of select="substring-before($trans/t[@o='compassDirThrough'], '|')"/>
  </xsl:if>
  <xsl:if test="boolean(isDistant)">
   <xsl:value-of select="$trans/t[@o='distant']"/>
  </xsl:if>
  <xsl:if test="boolean(inVicinity)">
   <xsl:value-of select="$trans/weather[@o='inVicinity']"/>
  </xsl:if>
  <xsl:if test="boolean(distance)">
   <xsl:value-of select="concat(' ', distance/@v, $trans/units[@o=$this/unitLength/@v])"/>
  </xsl:if>
  <xsl:if test="position() = 1 and not(boolean(locationSpec))">
   <xsl:value-of select="$locOrDirBefore"/>
  </xsl:if>
  <xsl:if test="boolean(locationSpec)">
   <xsl:value-of select="concat(' ', $trans/locationSpec[@o=$this/locationSpec/@v])"/>
  </xsl:if>
  <xsl:if test="boolean(obscgMtns)">
   <xsl:value-of select="$trans/t[@o='obscgMtns']"/>
  </xsl:if>
  <xsl:if test="position() = 1 and boolean(isGrid)">
   <xsl:value-of select="$trans/t[@o='isGrid']"/>
  </xsl:if>
  <xsl:if test="boolean(quadrantList)">
   <xsl:text> </xsl:text>
   <xsl:for-each select="quadrantList/quadrant">
    <xsl:variable name="this2" select="."/>
    <xsl:if test="position() != 1">
     <xsl:value-of select="concat(' ', $trans/t[@o='loc-and'], ' ')"/>
    </xsl:if>
    <xsl:value-of select="$trans/quadrant[@o=$this2/@v]"/>
   </xsl:for-each>
   <xsl:value-of select="concat(' ', $trans/t[@o='quadrant'])"/>
  </xsl:if>
  <xsl:if test="boolean(compassDir)">
   <xsl:variable name="quad">
    <xsl:choose>
     <xsl:when test="boolean(isQuadrant)">Quad</xsl:when>
     <xsl:otherwise/>
    </xsl:choose>
   </xsl:variable>
   <xsl:variable name="ccd">
    <xsl:value-of select="$trans/t[@o=concat('compassDirDescr', $quad)]"/>
   </xsl:variable>
   <xsl:value-of select="substring-before($ccd, '|')"/>
   <xsl:value-of select="$trans/compassDir[@o=$this/compassDir/@v]"/>
   <xsl:value-of select="substring-after($ccd, '|')"/>
  </xsl:if>
  <xsl:if test="position() = 1 and not(boolean(locationSpec))">
   <xsl:value-of select="$locOrDirAfter"/>
  </xsl:if>
  <xsl:if test="position() != 1">
   <xsl:value-of select="substring-after($trans/t[@o='compassDirThrough'], '|')"/>
  </xsl:if>
 </xsl:template>

 <xsl:template match="visibilityAtLoc">
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1"><b><xsl:value-of select="@s"/></b></td>
  </xsl:if>
  <td nowrap="1">
  <xsl:if test="boolean(locationAt)">
   <xsl:value-of select="$trans/visibilityAtLoc[@o=$this/locationAt/@v]"/>
  </xsl:if>
  <xsl:if test="boolean(rwyDesig)">
   <xsl:value-of select="$trans/t[@o='visibilityAt']"/>
   <xsl:if test="boolean(isApproach)">
    <xsl:value-of select="concat(' ', $trans/t[@o='approach'])"/>
   </xsl:if>
   <xsl:value-of select="concat(' ', substring-before($trans/t[@o='rwyDesig'], '|'), rwyDesig/@v, substring-after($trans/t[@o='rwyDesig'], '|'))"/>
   <xsl:text>:&#9;</xsl:text>
  </xsl:if>
  </td>
  <xsl:call-template name="visibility">
   <xsl:with-param name="notavail" select="$trans/t[@o='notAvailable']"/>
   <xsl:with-param name="rnd_m" select="10"/>
   <xsl:with-param name="unitThreshold" select="1000"/>
   <xsl:with-param name="visVal" select="visibility"/>
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:call-template>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="visMin">
  <xsl:param name="trans"/>
  <tr>
  <xsl:call-template name="visibility_str">
   <xsl:with-param name="str" select="@s"/>
   <xsl:with-param name="rows" select="1"/>
  </xsl:call-template>
  <xsl:call-template name="visibility_header">
   <xsl:with-param name="header" select="$trans/hdr[@o='rmkVisMin']"/>
   <xsl:with-param name="rows" select="1"/>
  </xsl:call-template>
  <xsl:call-template name="visibility">
   <xsl:with-param name="notavail" select="$trans/t[@o='notAvailable']"/>
   <xsl:with-param name="rnd_m" select="10"/>
   <xsl:with-param name="unitThreshold" select="1000"/>
   <xsl:with-param name="visVal" select="."/>
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:call-template>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="visVar2">
  <xsl:param name="trans"/>
  <tr>
  <xsl:call-template name="visibility_str">
   <xsl:with-param name="str" select="../visVar1/@s"/>
   <xsl:with-param name="rows" select="2"/>
  </xsl:call-template>
  <xsl:call-template name="visibility_header">
   <xsl:with-param name="header" select="$trans/hdr[@o='rmkVisVar']"/>
   <xsl:with-param name="rows" select="1"/>
  </xsl:call-template>
  <xsl:call-template name="visibility">
   <xsl:with-param name="notavail" select="$trans/t[@o='notAvailable']"/>
   <xsl:with-param name="rnd_m" select="10"/>
   <xsl:with-param name="unitThreshold" select="1000"/>
   <xsl:with-param name="visVal" select="../visVar1"/>
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:call-template>
  </tr><xsl:text>&#10;</xsl:text>
  <tr>
  <xsl:call-template name="visibility_header">
   <xsl:with-param name="header" select="$trans/t[@o='varying-to']"/>
   <xsl:with-param name="rows" select="1"/>
  </xsl:call-template>
  <xsl:call-template name="visibility">
   <xsl:with-param name="notavail" select="$trans/t[@o='notAvailable']"/>
   <xsl:with-param name="rnd_m" select="10"/>
   <xsl:with-param name="unitThreshold" select="1000"/>
   <xsl:with-param name="visVal" select="."/>
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:call-template>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="correctedAt">
  <xsl:param name="trans"/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1"><b><xsl:value-of select="@s"/></b></td>
  </xsl:if>
  <td nowrap="1">
  <xsl:value-of select="$trans/hdr[@o='correctedAt']"/>
  </td>
  <td nowrap="1" colspan="3">
  <xsl:value-of select="concat(format-number(hour/@v, '00'), ':', format-number(minute/@v, '00'), ' UTC')"/>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="precipHourly">
  <xsl:param name="trans"/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1"><b><xsl:value-of select="@s"/></b></td>
  </xsl:if>
  <td nowrap="1">
  <xsl:choose>
   <xsl:when test="precipHours/@v = '1'">
    <xsl:value-of select="$trans/hdr[@o='precip1Hour']"/>
   </xsl:when>
   <xsl:when test="precipHours/@v = '3'">
    <xsl:value-of select="$trans/hdr[@o='precip3Hour']"/>
   </xsl:when>
   <xsl:when test="precipHours/@v = '6'">
    <xsl:value-of select="$trans/hdr[@o='precip6Hour']"/>
   </xsl:when>
   <xsl:when test="precipHours/@v = '24'">
    <xsl:value-of select="$trans/hdr[@o='precip24Hour']"/>
   </xsl:when>
   <xsl:otherwise>
    <xsl:value-of select="$trans/hdr[@o='precip36Hour']"/>
   </xsl:otherwise>
  </xsl:choose>
  </td>
  <xsl:choose>
   <xsl:when test="boolean(notAvailable)">
    <td nowrap="1" colspan="2">
    <xsl:value-of select="$trans/t[@o='notAvailable']"/>
    </td>
   </xsl:when>
   <xsl:otherwise>
    <td nowrap="1">
    <xsl:value-of select="concat(format-number(precipAmountInch/@v * ($FT2M div 12 * 1000), '0'), $trans/units[@o='MM'])"/>
    </td>
    <xsl:text>&#9;&#9;&#9;&#9;&#9;</xsl:text>
    <td nowrap="1">
    <xsl:value-of select="concat(precipAmountInch/@v, $trans/units[@o='IN'])"/>
    </td>
   </xsl:otherwise>
  </xsl:choose>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="snowIncr">
  <xsl:param name="trans"/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1"><b><xsl:value-of select="@s"/></b></td>
  </xsl:if>
  <td nowrap="1">
  <xsl:value-of select="$trans/hdr[@o='snowIncr']"/>
  </td>
  <td nowrap="1" colspan="2">
  <xsl:value-of select="substring-before($trans/t[@o='snowIncrPastHour'],'|')"/>
  <xsl:value-of select="pastHour/@v"/>
  <xsl:value-of select="substring-after($trans/t[@o='snowIncrPastHour'], '|')"/>
  <xsl:text>, </xsl:text>
  <xsl:value-of select="substring-before($trans/t[@o='snowIncrOnGround'],'|')"/>
  <xsl:value-of select="onGround/@v"/>
  <xsl:value-of select="substring-after($trans/t[@o='snowIncrOnGround'], '|')"/>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="cloudTypeFamily">
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1" rowspan="3"><b><xsl:value-of select="@s"/></b></td>
  </xsl:if>
  <td rowspan="3">
  <xsl:value-of select="$trans/hdr[@o='cloudTypeFamily']"/>
  </td>
  <td colspan="2">
  <xsl:text>&#9;</xsl:text><xsl:value-of select="$trans/t[@o='low']"/>
  <xsl:choose>
   <xsl:when test="boolean(cloudTypeLow)">
    <xsl:value-of select="$trans/cloudTypeLow[@o=$this/cloudTypeLow/@v]"/>
   </xsl:when>
   <xsl:otherwise>
    <xsl:value-of select="$trans/t[@o='cloudLayerAboveOvercast']"/>
   </xsl:otherwise>
  </xsl:choose>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
  <tr>
  <td colspan="2">
  <xsl:text>&#9;</xsl:text><xsl:value-of select="$trans/t[@o='middle']"/>
  <xsl:choose>
   <xsl:when test="boolean(cloudTypeMiddle)">
    <xsl:value-of select="$trans/cloudTypeMiddle[@o=$this/cloudTypeMiddle/@v]"/>
   </xsl:when>
   <xsl:otherwise>
    <xsl:value-of select="$trans/t[@o='cloudLayerAboveOvercast']"/>
   </xsl:otherwise>
  </xsl:choose>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
  <tr>
  <td colspan="2">
  <xsl:text>&#9;</xsl:text><xsl:value-of select="$trans/t[@o='high']"/>
  <xsl:choose>
   <xsl:when test="boolean(cloudTypeHigh)">
    <xsl:value-of select="$trans/cloudTypeHigh[@o=$this/cloudTypeHigh/@v]"/>
   </xsl:when>
   <xsl:otherwise>
    <xsl:value-of select="$trans/t[@o='cloudLayerAboveOvercast']"/>
   </xsl:otherwise>
  </xsl:choose>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="windShift">
  <xsl:param name="trans"/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1"><b><xsl:value-of select="@s"/></b></td>
  </xsl:if>
  <td nowrap="1" colspan="3">
  <xsl:value-of select="$trans/hdr[@o='windShift']"/>
  <xsl:call-template name="printTime">
   <xsl:with-param name="time"  select="."/>
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:call-template>
  <xsl:if test="boolean(FROPA)">
   <xsl:value-of select="concat(' ', $trans/t[@o='FROPA'])"/>
  </xsl:if>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="beginEndPrecip">
  <xsl:param name="trans"/>
  <xsl:variable name="rows">
   <xsl:apply-templates select="precip" mode="count"/>
  </xsl:variable>
  <tr>
  <xsl:if test="$method = 'html'">
   <xsl:element name="td">
    <xsl:attribute name="nowrap">1</xsl:attribute>
    <xsl:attribute name="rowspan"><xsl:value-of select="$rows"/></xsl:attribute>
    <b><xsl:value-of select="@s"/></b>
   </xsl:element>
  </xsl:if>
  <xsl:element name="td">
   <xsl:attribute name="nowrap">1</xsl:attribute>
   <xsl:attribute name="rowspan"><xsl:value-of select="$rows"/></xsl:attribute>
   <xsl:value-of select="$trans/hdr[@o='beginEndPrecip']"/>
  </xsl:element>
  <xsl:apply-templates select="precip">
   <xsl:with-param name="is_first" select="1"/>
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  </tr>
  <xsl:apply-templates select="precip">
   <xsl:with-param name="is_first" select="0"/>
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:if test="boolean(MOV) or boolean(MOVD) or boolean(locationAndList)">
   <tr>
   <td nowrap="1" colspan="2">
   <xsl:text>&#9;&#9;&#9;</xsl:text>
   <xsl:if test="boolean(locationAndList)">
    <xsl:apply-templates select="locationAndList">
     <xsl:with-param name="locOrDir" select="'compassDirLocation'"/>
     <xsl:with-param name="trans" select="$trans"/>
    </xsl:apply-templates>
    <xsl:if test="boolean(MOV) or boolean(MOVD)">
     <xsl:text>,</xsl:text>
    </xsl:if>
   </xsl:if>
   <xsl:apply-templates select="MOV|MOVD">
    <xsl:with-param name="trans" select="$trans"/>
   </xsl:apply-templates>
   </td>
   </tr><xsl:text>&#10;</xsl:text>
  </xsl:if>
 </xsl:template>

 <xsl:template match="precip" mode="count">
  <xsl:if test="position() = last()">
   <xsl:choose>
    <xsl:when test="boolean(../MOV) or boolean(../MOVD) or boolean(../locationAndList)">
     <xsl:value-of select="last() + 1"/>
    </xsl:when>
    <xsl:otherwise>
     <xsl:value-of select="last()"/>
    </xsl:otherwise>
   </xsl:choose>
  </xsl:if>
 </xsl:template>

 <xsl:template match="precip">
  <xsl:param name="is_first"/>
  <xsl:param name="trans"/>
  <xsl:if test="$is_first = '1' and position() = 1">
   <td nowrap="1" colspan="2">
   <dl style="margin: 0em;"><dt>
   <xsl:call-template name="printWeather">
    <xsl:with-param name="is_recent" select="1"/>
    <xsl:with-param name="this" select="."/>
    <xsl:with-param name="trans" select="$trans"/>
   </xsl:call-template>
   <xsl:text>:</xsl:text></dt>
   <xsl:apply-templates select="startTime|endTime">
    <xsl:with-param name="trans" select="$trans"/>
   </xsl:apply-templates>
   <xsl:text>&#10;</xsl:text>
   </dl></td>
  </xsl:if>
  <xsl:if test="$is_first = '0' and position() != 1">
   <xsl:text>&#9;&#9;&#9;</xsl:text>
   <tr>
   <td nowrap="1" colspan="2">
   <dl style="margin: 0em;"><dt>
   <xsl:call-template name="printWeather">
    <xsl:with-param name="is_recent" select="1"/>
    <xsl:with-param name="this" select="."/>
    <xsl:with-param name="trans" select="$trans"/>
   </xsl:call-template>
   <xsl:text>:</xsl:text></dt>
   <xsl:apply-templates select="startTime|endTime">
    <xsl:with-param name="trans" select="$trans"/>
   </xsl:apply-templates>
   <xsl:text>&#10;</xsl:text>
   </dl></td>
   </tr>
  </xsl:if>
 </xsl:template>

 <xsl:template match="startTime|endTime">
  <xsl:param name="trans"/>
  <xsl:text>&#10;&#9;&#9;&#9;&#9;</xsl:text><dd>
  <xsl:if test="name(.) = 'startTime'">
   <xsl:value-of select="$trans/t[@o='began']"/>
  </xsl:if>
  <xsl:if test="name(.) = 'endTime'">
   <xsl:value-of select="$trans/t[@o='ended']"/>
  </xsl:if>
  <xsl:call-template name="printTime">
   <xsl:with-param name="time"  select="."/>
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:call-template>
  </dd>
 </xsl:template>

 <xsl:template match="cloudTypeList">
  <xsl:param name="trans"/>
  <xsl:for-each select="cloudType">
   <xsl:variable name="this" select="."/>
   <xsl:if test="position() != 1">
    <xsl:value-of select="concat(' ', $trans/t[@o='loc-and'], ' ')"/>
   </xsl:if>
   <xsl:value-of select="$trans/cloudType[@o=$this/@v]"/>
  </xsl:for-each>
 </xsl:template>

 <xsl:template match="lightningTypeList">
  <xsl:param name="trans"/>
  <xsl:for-each select="lightningType">
   <xsl:variable name="this" select="."/>
   <xsl:if test="position() != 1">
    <xsl:value-of select="concat(' ', $trans/t[@o='loc-and'], ' ')"/>
   </xsl:if>
   <xsl:value-of select="$trans/lightningType[@o=$this/@v]"/>
  </xsl:for-each>
 </xsl:template>

 <xsl:template match="phenomenonAtLoc|phenomenonOnly">
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1"><b><xsl:value-of select="@s"/></b></td>
  </xsl:if>
  <td colspan="3">
  <xsl:apply-templates select="phenomDescrPre">
   <xsl:with-param name="post" select="' '"/>
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:choose>
   <xsl:when test="boolean(weather)">
    <xsl:call-template name="printWeather">
     <xsl:with-param name="is_recent" select="1"/>
     <xsl:with-param name="this" select="weather"/>
     <xsl:with-param name="trans" select="$trans"/>
    </xsl:call-template>
   </xsl:when>
   <xsl:when test="boolean(cloudTypeList)">
    <xsl:apply-templates select="cloudTypeList">
     <xsl:with-param name="trans" select="$trans"/>
    </xsl:apply-templates>
   </xsl:when>
   <xsl:when test="boolean(cloudCover)">
    <xsl:value-of select="$trans/cloudCover[@o=$this/cloudCover/@v]"/>
   </xsl:when>
   <xsl:when test="boolean(lightningTypeList)">
    <xsl:value-of select="concat($trans/phenomenon[@o='LTG'], ' ')"/>
    <xsl:apply-templates select="lightningTypeList">
     <xsl:with-param name="trans" select="$trans"/>
    </xsl:apply-templates>
   </xsl:when>
   <xsl:otherwise>
    <xsl:value-of select="$trans/phenomenon[@o=$this/otherPhenom/@v]"/>
   </xsl:otherwise>
  </xsl:choose>
  <xsl:apply-templates select="phenomDescrPost">
   <xsl:with-param name="pre" select="' '"/>
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:if test="boolean(locationAndList)">
   <xsl:if test="boolean(phenomDescrPost)">
    <xsl:text>,</xsl:text>
   </xsl:if>
   <xsl:apply-templates select="locationAndList">
    <xsl:with-param name="locOrDir" select="'compassDirLocation'"/>
    <xsl:with-param name="trans" select="$trans"/>
   </xsl:apply-templates>
   <xsl:if test="boolean(MOV) or boolean(MOVD)">
    <xsl:text>,</xsl:text>
   </xsl:if>
  </xsl:if>
  <xsl:apply-templates select="MOV|MOVD">
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:if test="boolean(baseBelowStation)">
   <xsl:value-of select="$trans/t[@o='baseBelowStation']"/>
  </xsl:if>
  <xsl:if test="boolean(isAloft)">
   <xsl:value-of select="$trans/t[@o='isAloft']"/>
  </xsl:if>
  <xsl:if test="boolean(isAround)">
   <xsl:value-of select="$trans/t[@o='isAround']"/>
  </xsl:if>
  <xsl:if test="boolean(cloudTypeAsoctd) or boolean(cloudTypeEmbd)">
   <xsl:text>, </xsl:text>
   <xsl:apply-templates select="cloudTypeAsoctd|cloudTypeEmbd">
    <xsl:with-param name="trans" select="$trans"/>
   </xsl:apply-templates>
  </xsl:if>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="phenomDescrPre|phenomDescrPost">
  <xsl:param name="pre"/>
  <xsl:param name="post"/>
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <xsl:if test="position() = 1"><xsl:value-of select="$pre"/></xsl:if>
  <xsl:if test="position() != 1">
   <xsl:text>, </xsl:text>
  </xsl:if>
  <xsl:value-of select="$trans/t[@o=$this/@v]"/>
  <xsl:if test="position() = last()"><xsl:value-of select="$post"/></xsl:if>
 </xsl:template>

 <xsl:template match="conditionMountainList">
  <xsl:param name="trans"/>
  <xsl:apply-templates select="conditionMountain">
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
 </xsl:template>

 <xsl:template match="conditionMountain">
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <xsl:variable name="rows" select="last()"/>
  <tr>
  <xsl:if test="position() = 1">
   <xsl:if test="$method = 'html'">
    <xsl:element name="td">
     <xsl:attribute name="nowrap">1</xsl:attribute>
     <xsl:attribute name="rowspan"><xsl:value-of select="$rows"/></xsl:attribute>
     <b><xsl:value-of select="../@s"/></b>
    </xsl:element>
   </xsl:if>
   <xsl:element name="td">
    <xsl:attribute name="nowrap">1</xsl:attribute>
    <xsl:attribute name="rowspan"><xsl:value-of select="$rows"/></xsl:attribute>
    <xsl:value-of select="$trans/hdr[@o='conditionMountain']"/>
   </xsl:element>
  </xsl:if>
  <xsl:if test="position() != 1">
   <xsl:text>&#9;&#9;&#9;</xsl:text>
  </xsl:if>
  <td nowrap="1" colspan="2">
  <xsl:apply-templates select="locationAndList">
   <xsl:with-param name="locOrDir" select="'compassDirLocation'"/>
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:if test="boolean(locationAndList)">
   <xsl:text>: </xsl:text>
  </xsl:if>
  <xsl:value-of select="$trans/condMounType[@o=$this/condMounType/@v]"/>
  <xsl:if test="boolean(condMounChange)">
   <xsl:value-of select="concat(', ', $trans/condMounChange[@o=$this/condMounChange/@v])"/>
  </xsl:if>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="conditionValleyList">
  <xsl:param name="trans"/>
  <xsl:apply-templates select="conditionValley">
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
 </xsl:template>

 <xsl:template match="conditionValley">
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <xsl:variable name="rows" select="last()"/>
  <tr>
  <xsl:if test="position() = 1">
   <xsl:if test="$method = 'html'">
    <xsl:element name="td">
     <xsl:attribute name="nowrap">1</xsl:attribute>
     <xsl:attribute name="rowspan"><xsl:value-of select="$rows"/></xsl:attribute>
     <b><xsl:value-of select="../@s"/></b>
    </xsl:element>
   </xsl:if>
   <xsl:element name="td">
    <xsl:attribute name="nowrap">1</xsl:attribute>
    <xsl:attribute name="rowspan"><xsl:value-of select="$rows"/></xsl:attribute>
    <xsl:value-of select="$trans/hdr[@o='conditionValley']"/>
   </xsl:element>
  </xsl:if>
  <xsl:if test="position() != 1">
   <xsl:text>&#9;&#9;&#9;</xsl:text>
  </xsl:if>
  <td nowrap="1" colspan="2">
  <xsl:apply-templates select="locationAndList">
   <xsl:with-param name="locOrDir" select="'compassDirLocation'"/>
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:if test="boolean(locationAndList)">
   <xsl:text>: </xsl:text>
  </xsl:if>
  <xsl:value-of select="$trans/condVallType[@o=$this/condVallType/@v]"/>
  <xsl:if test="boolean(condVallChange)">
   <xsl:value-of select="concat(', ', $trans/condVallChange[@o=$this/condVallChange/@v])"/>
  </xsl:if>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="ceilingAtLoc">
  <xsl:param name="trans"/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1"><b><xsl:value-of select="@s"/></b></td>
  </xsl:if>
  <td nowrap="1">
  <xsl:value-of select="$trans/t[@o='ceiling']"/>
  <xsl:if test="boolean(isApproach)">
   <xsl:value-of select="concat(' ', $trans/t[@o='approach'])"/>
  </xsl:if>
  <xsl:if test="boolean(rwyDesig)">
   <xsl:value-of select="concat(' ', substring-before($trans/t[@o='rwyDesig'], '|'), rwyDesig/@v, substring-after($trans/t[@o='rwyDesig'], '|'))"/>
  </xsl:if>
  <xsl:apply-templates select="locationAndList">
   <xsl:with-param name="locOrDir" select="'compassDirLocation'"/>
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:text>:</xsl:text>
  </td>
  <xsl:text>&#9;</xsl:text>
  <td nowrap="1">
  <xsl:value-of select="concat($trans/t[@o='cloud-at'], ' ', cloudBase/@v * 100, $trans/units[@o='FT'])"/>
  </td>
  <xsl:text>&#9;&#9;&#9;&#9;</xsl:text>
  <td nowrap="1">
  <xsl:value-of select="concat(format-number(cloudBase/@v * 10 * $FT2M, '0') * 10, $trans/units[@o='M'])"/>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="phenomOpacityList">
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <xsl:apply-templates select="phenomOpacity">
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  <xsl:if test="boolean(cloudTypeAsoctd) or boolean(cloudTypeEmbd)">
   <tr>
   <xsl:text>&#9;&#9;&#9;</xsl:text>
   <td nowrap="1" colspan="2">
   <xsl:apply-templates select="cloudTypeAsoctd|cloudTypeEmbd">
    <xsl:with-param name="trans" select="$trans"/>
   </xsl:apply-templates>
   </td>
   </tr><xsl:text>&#10;</xsl:text>
  </xsl:if>
 </xsl:template>

 <xsl:template match="cloudTypeAsoctd|cloudTypeEmbd">
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <xsl:value-of select="concat(substring-before($trans/t[@o=name($this)], '|'), $trans/cloudType[@o=$this/@v], substring-after($trans/t[@o=name($this)], '|'))"/>
 </xsl:template>

 <xsl:template match="phenomOpacity">
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <xsl:variable name="rows">
   <xsl:choose>
    <xsl:when test="boolean(../cloudTypeAsoctd) or boolean(../cloudTypeEmbd)">
     <xsl:value-of select="last() + 1"/>
    </xsl:when>
    <xsl:otherwise>
     <xsl:value-of select="last()"/>
    </xsl:otherwise>
   </xsl:choose>
  </xsl:variable>
  <tr>
  <xsl:choose>
   <xsl:when test="position() = 1">
    <xsl:if test="$method = 'html'">
     <xsl:element name="td">
      <xsl:attribute name="nowrap">1</xsl:attribute>
      <xsl:attribute name="rowspan"><xsl:value-of select="$rows"/></xsl:attribute>
      <b><xsl:value-of select="../@s"/></b>
     </xsl:element>
    </xsl:if>
    <xsl:element name="td">
     <xsl:attribute name="nowrap">1</xsl:attribute>
     <xsl:attribute name="rowspan"><xsl:value-of select="$rows"/></xsl:attribute>
     <xsl:value-of select="$trans/hdr[@o='cloudOpacityLvl']"/>
    </xsl:element>
   </xsl:when>
   <xsl:otherwise>
    <xsl:text>&#9;&#9;&#9;</xsl:text>
   </xsl:otherwise>
  </xsl:choose>
  <td nowrap="1" colspan="2">
   <xsl:value-of select="concat(eights/@v, '/8')"/>
   <xsl:if test="boolean(weather)">
    <xsl:text> </xsl:text>
    <xsl:call-template name="printWeather">
     <xsl:with-param name="is_recent" select="1"/>
     <xsl:with-param name="this" select="weather"/>
     <xsl:with-param name="trans" select="$trans"/>
    </xsl:call-template>
   </xsl:if>
   <xsl:if test="boolean(cloudType)">
    <xsl:value-of select="concat(' ', $trans/cloudType[@o=$this/cloudType/@v])"/>
   </xsl:if>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="cloudTypeLvl">
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1"><b><xsl:value-of select="@s"/></b></td>
  </xsl:if>
  <td nowrap="1">
  <xsl:value-of select="$trans/hdr[@o='cloud-1']"/>
  </td>
  <td nowrap="1">
  <xsl:value-of select="$trans/cloudType[@o=$this/cloudType/@v]"/>
  <xsl:value-of select="concat(' ', $trans/t[@o='cloud-at'], ' ', cloudBase/@v * 100, $trans/units[@o='FT'])"/>
  </td>
  <xsl:text>&#9;&#9;&#9;</xsl:text>
  <td nowrap="1">
  <xsl:value-of select="concat(format-number(cloudBase/@v * 10 * $FT2M, '0') * 10, $trans/units[@o='M'])"/>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="cloudTrace">
  <xsl:param name="trans"/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1"><b><xsl:value-of select="@s"/></b></td>
  </xsl:if>
  <td nowrap="1">
  <xsl:value-of select="$trans/hdr[@o='tracesOf']"/>
  </td>
  <td nowrap="1" colspan="2">
  <xsl:if test="boolean(isLower)">
   <xsl:value-of select="concat($trans/t[@o='isLower'], ' ')"/>
  </xsl:if>
  <xsl:if test="boolean(cloudTypeNotAvailable)">
   <xsl:value-of select="$trans/t[@o='cloudTypeNotAvailable']"/>
  </xsl:if>
  <xsl:for-each select="cloudType">
   <xsl:variable name="this" select="."/>
   <xsl:if test="position() != 1"><xsl:text>, </xsl:text></xsl:if>
   <xsl:value-of select="$trans/cloudType[@o=$this/@v]"/>
  </xsl:for-each>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="seaCondition|swellCondition">
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1"><b><xsl:value-of select="@s"/></b></td>
  </xsl:if>
  <td nowrap="1">
  <xsl:value-of select="$trans/hdr[@o=name($this)]"/>
  </td>
  <td nowrap="1" colspan="2">
  <xsl:if test="boolean(notAvailable)">
   <xsl:value-of select="$trans/t[@o='notAvailable']"/>
  </xsl:if>
  <xsl:if test="boolean(seaCondVal)">
   <xsl:value-of select="$trans/seaCondition[@o=$this/seaCondVal/@v]"/>
  </xsl:if>
  <xsl:if test="boolean(swellCondVal)">
   <xsl:value-of select="$trans/swellCondition[@o=$this/swellCondVal/@v]"/>
  </xsl:if>
  <xsl:if test="boolean(locationAndList)">
   <xsl:text>,</xsl:text>
   <xsl:apply-templates select="locationAndList">
    <xsl:with-param name="locOrDir" select="'compassDirFrom'"/>
    <xsl:with-param name="trans" select="$trans"/>
   </xsl:apply-templates>
  </xsl:if>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="obscuration">
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1"><b><xsl:value-of select="@s"/></b></td>
  </xsl:if>
  <td nowrap="1">
  <xsl:value-of select="substring-before($trans/hdr[@o='obscuration'], '|')"/>
  <xsl:if test="boolean(weather)">
   <xsl:call-template name="printWeather">
    <xsl:with-param name="is_recent" select="1"/>
    <xsl:with-param name="this" select="weather"/>
    <xsl:with-param name="trans" select="$trans"/>
   </xsl:call-template>
  </xsl:if>
  <xsl:if test="boolean(cloudPhenom)">
   <xsl:value-of select="$trans/cloudPhenom[@o=$this/cloudPhenom/@v]"/>
  </xsl:if>
  <xsl:value-of select="substring-after($trans/hdr[@o='obscuration'], '|')"/>
  </td>
  <xsl:for-each select="cloud">
   <xsl:call-template name="printCloud">
    <xsl:with-param name="at_abv" select="'cloud-at'"/>
    <xsl:with-param name="trans" select="$trans"/>
   </xsl:call-template>
  </xsl:for-each>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="variableCeiling">
  <xsl:param name="trans"/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1" rowspan="2"><b><xsl:value-of select="@s"/></b></td>
  </xsl:if>
  <td nowrap="1" rowspan="2">
  <xsl:value-of select="$trans/hdr[@o='variableCeiling']"/>
  </td>
  <td nowrap="1">
  <xsl:value-of select="concat($trans/t[@o='variable-from'], ' ', cloudBaseFrom/@v * 100, $trans/units[@o='FT'])"/>
  </td>
  <xsl:text>&#9;&#9;&#9;&#9;</xsl:text>
  <td nowrap="1">
  <xsl:value-of select="concat(format-number(cloudBaseFrom/@v * 10 * $FT2M, '0') * 10, $trans/units[@o='M'])"/>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
  <xsl:text>&#9;&#9;&#9;</xsl:text>
  <tr>
  <td nowrap="1">
  <xsl:value-of select="concat($trans/t[@o='variable-to'], ' ', cloudBaseTo/@v * 100, $trans/units[@o='FT'])"/>
  </td>
  <xsl:text>&#9;&#9;&#9;&#9;</xsl:text>
  <td nowrap="1">
  <xsl:value-of select="concat(format-number(cloudBaseTo/@v * 10 * $FT2M, '0') * 10, $trans/units[@o='M'])"/>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="pressureTendency3h">
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1"><b><xsl:value-of select="@s"/></b></td>
  </xsl:if>
  <td nowrap="1">
  <xsl:value-of select="$trans/hdr[@o='pressureTendency3h']"/>
  </td>
  <td nowrap="1" colspan="2">
  <xsl:choose>
   <xsl:when test="boolean(notAvailable)">
    <xsl:value-of select="$trans/t[@o='notAvailable']"/>
   </xsl:when>
   <xsl:otherwise>
    <xsl:value-of select="$trans/pressureTend[@o=$this/pressureTendency/@v]"/>
    <xsl:if test="pressureChange/@v != 0">
     <xsl:value-of select="concat($trans/t[@o='change-by'], ' ', pressureChange/@v, $trans/units[@o='hPa'])"/>
    </xsl:if>
   </xsl:otherwise>
  </xsl:choose>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="ceilVisVariable">
  <xsl:param name="trans"/>
  <tr>
  <xsl:call-template name="visibility_str">
   <xsl:with-param name="str" select="@s"/>
   <xsl:with-param name="rows" select="2"/>
  </xsl:call-template>
  <xsl:call-template name="visibility_header">
   <xsl:with-param name="header" select="$trans/hdr[@o='ceilVisVariable']"/>
   <xsl:with-param name="rows" select="1"/>
  </xsl:call-template>
  <xsl:call-template name="visibility">
   <xsl:with-param name="notavail" select="$trans/t[@o='notAvailable']"/>
   <xsl:with-param name="rnd_m" select="10"/>
   <xsl:with-param name="unitThreshold" select="1000"/>
   <xsl:with-param name="visVal" select="visibilityFrom"/>
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:call-template>
  </tr><xsl:text>&#10;</xsl:text>
  <tr>
  <xsl:call-template name="visibility_header">
   <xsl:with-param name="header" select="$trans/t[@o='varying-to']"/>
   <xsl:with-param name="rows" select="1"/>
  </xsl:call-template>
  <xsl:call-template name="visibility">
   <xsl:with-param name="notavail" select="$trans/t[@o='notAvailable']"/>
   <xsl:with-param name="rnd_m" select="10"/>
   <xsl:with-param name="unitThreshold" select="1000"/>
   <xsl:with-param name="visVal" select="visibilityTo"/>
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:call-template>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="rwySfcCondition">
  <xsl:param name="trans"/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1"><b><xsl:value-of select="@s"/></b></td>
  </xsl:if>
  <td nowrap="1">
  <xsl:value-of select="$trans/hdr[@o='rwySfcCondition']"/>
  </td>
  <td nowrap="1" colspan="2">
  <xsl:for-each select="child::*">
   <xsl:variable name="this" select="."/>
   <xsl:if test="name(.) = 'notAvailable'">
    <xsl:value-of select="$trans/t[@o='decelerometerReadingNA']"/>
   </xsl:if>
   <xsl:if test="name(.) = 'decelerometer'">
    <xsl:value-of select="concat($trans/t[@o='decelerometerReading'], ' ',@v)"/>
   </xsl:if>
   <xsl:if test="name(.) = 'rwySfc'">
    <xsl:value-of select="$trans/rwySfc[@o=$this/@v]"/>
   </xsl:if>
   <xsl:if test="position() != last()">
    <xsl:text>, </xsl:text>
   </xsl:if>
  </xsl:for-each>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="rainfall">
  <xsl:param name="trans"/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1"><b><xsl:value-of select="@s"/></b></td>
  </xsl:if>
  <td nowrap="1">
  <xsl:value-of select="$trans/hdr[@o='rainfall']"/>
  </td>
  <td nowrap="1" colspan="2">
  <xsl:value-of select="concat(rainfall10min/@v, $trans/t[@o='rainfall10min'], rainfall0900/@v, $trans/t[@o='rainfall0900'])"/>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="NEFO_PLAYA">
  <xsl:param name="trans"/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1"><b><xsl:value-of select="@s"/></b></td>
  </xsl:if>
  <td nowrap="1">
  <xsl:value-of select="$trans/hdr[@o='NEFO_PLAYA']"/>
  </td>
  <xsl:choose>
   <xsl:when test="boolean(SKC)">
    <td nowrap="1" colspan="2">
    <xsl:value-of select="$trans/cloudCover[@o='SKC']"/>
    </td>
   </xsl:when>
   <xsl:otherwise>
    <td nowrap="1">
    <xsl:value-of select="concat($trans/t[@o='cloud-at'], ' ', cloudBaseFT/@v , $trans/units[@o='FT'])"/>
    </td>
    <xsl:text>&#9;&#9;&#9;</xsl:text>
    <td nowrap="1">
    <xsl:value-of select="concat(format-number(cloudBaseFT/@v div 10 * $FT2M, '0') * 10, $trans/units[@o='M'])"/>
    </td>
   </xsl:otherwise>
  </xsl:choose>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="precipPastHour">
  <xsl:param name="trans"/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1"><b><xsl:value-of select="@s"/></b></td>
  </xsl:if>
  <td nowrap="1">
  <xsl:value-of select="$trans/hdr[@o='precip1Hour']"/>
  </td>
  <td nowrap="1">
  <xsl:value-of select="concat(precipAmountMM/@v, $trans/units[@o='MM'])"/>
  </td>
  <xsl:text>&#9;&#9;&#9;&#9;&#9;</xsl:text>
  <td nowrap="1">
  <xsl:value-of select="concat(format-number(precipAmountMM/@v div ($FT2M div 12 * 1000), '0.00'), $trans/units[@o='IN'])"/>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="densityAlt|pressureAlt">
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1"><b><xsl:value-of select="@s"/></b></td>
  </xsl:if>
  <td nowrap="1">
  <xsl:value-of select="$trans/hdr[@o=name($this)]"/>
  </td>
  <td nowrap="1">
  <xsl:value-of select="concat(altitude/@v, $trans/units[@o='FT'])"/>
  </td>
  <xsl:text>&#9;&#9;&#9;&#9;&#9;</xsl:text>
  <td nowrap="1">
  <xsl:value-of select="concat(format-number(altitude/@v div 10 * $FT2M, '0.0') * 10, $trans/units[@o='M'])"/>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="VISNO|CHINO">
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1"><b><xsl:value-of select="@s"/></b></td>
  </xsl:if>
  <td nowrap="1" colspan="3">
  <xsl:variable name="s" select="$trans/t[@o=name($this)]"/>
  <xsl:variable name="p" select="$trans/t[@o='rwyDesig']"/>
  <xsl:value-of select="substring-before($s, '|')"/>
  <xsl:value-of select="substring-before($p, '|')"/>
  <xsl:value-of select="rwyDesig/@v"/>
  <xsl:value-of select="substring-after($p, '|')"/>
  <xsl:value-of select="substring-after($s, '|')"/>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="obsTimeOffset">
  <xsl:param name="trans"/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1"><b><xsl:value-of select="@s"/></b></td>
  </xsl:if>
  <td nowrap="1">
  <xsl:value-of select="$trans/hdr[@o='obsTimeOffset']"/>
  </td>
  <td nowrap="1" colspan="2">
  <xsl:value-of select="concat(minutes/@v, ' ', $trans/t[@o='minutes'])"/>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="nextFcstBy|nextFcstAt|amdAt">
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1"><b><xsl:value-of select="@s"/></b></td>
  </xsl:if>
  <td nowrap="1">
  <xsl:value-of select="$trans/hdr[@o=name($this)]"/>
  </td>
  <td nowrap="1" colspan="2">
  <xsl:if test="boolean(day/@v)">
   <xsl:value-of select="substring-before($trans/t[@o='onThe'], '|')"/>
   <xsl:value-of select="format-number(day/@v, '0')"/>
   <xsl:value-of select="substring-after($trans/t[@o='onThe'], '|')"/>
  </xsl:if>
  <xsl:value-of select="concat(hour/@v, ':', minute/@v, ' UTC')"/>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="fcstAutoObs">
  <xsl:param name="trans"/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1"><b><xsl:value-of select="@s"/></b></td>
  </xsl:if>
  <td nowrap="1" colspan="3">
  <xsl:value-of select="$trans/hdr[@o='fcstAutoObs']"/>
  <xsl:if test="boolean(hourFrom/@v)">
   <xsl:value-of select="concat(' ', hourFrom/@v, ':00 - ', hourTill/@v, ':00 UTC')"/>
   <xsl:if test="hourFrom/@v &gt;= hourTill/@v">
    <xsl:value-of select="$trans/t[@o='nextDay']"/>
   </xsl:if>
  </xsl:if>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="waterEquivOfSnow|snowOnGround">
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1"><b><xsl:value-of select="@s"/></b></td>
  </xsl:if>
  <td nowrap="1">
  <xsl:value-of select="$trans/hdr[@o=name($this)]"/>
  </td>
  <td nowrap="1">
  <xsl:value-of select="concat(format-number(precipAmountInch/@v * ($FT2M div 12 * 1000), '0'), $trans/units[@o='MM'])"/>
  </td>
  <xsl:text>&#9;&#9;&#9;&#9;&#9;</xsl:text>
  <td nowrap="1">
  <xsl:value-of select="concat(precipAmountInch/@v, $trans/units[@o='IN'])"/>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="snowCover">
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1"><b><xsl:value-of select="@s"/></b></td>
  </xsl:if>
  <td nowrap="1">
  <xsl:value-of select="$trans/hdr[@o='snowCover']"/>
  </td>
  <td nowrap="1" colspan="2">
  <xsl:value-of select="$trans/snowCoverType[@o=$this/snowCoverType/@v]"/>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="climate">
  <xsl:param name="trans"/>
  <xsl:variable name="rows">
   <xsl:choose>
    <xsl:when test="boolean(precipAmount2Inch) or boolean(precipAmount2MM)">4</xsl:when>
    <xsl:when test="boolean(precipAmount1Inch) or boolean(precipAmount1MM) or boolean(precip1Traces)">3</xsl:when>
    <xsl:otherwise>2</xsl:otherwise>
   </xsl:choose>
  </xsl:variable>
  <tr>
  <xsl:call-template name="printTemp">
   <xsl:with-param name="hdr" select="$trans/hdr[@o='temp1']"/>
   <xsl:with-param name="rows" select="$rows"/>
   <xsl:with-param name="s" select="@s"/>
   <xsl:with-param name="temp" select="temp1"/>
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:call-template>
  </tr><xsl:text>&#10;</xsl:text>
  <tr>
  <xsl:call-template name="printTemp">
   <xsl:with-param name="hdr" select="$trans/hdr[@o='temp2']"/>
   <xsl:with-param name="temp" select="temp2"/>
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:call-template>
  </tr><xsl:text>&#10;</xsl:text>
  <xsl:if test="boolean(precipAmount1Inch) or boolean(precipAmount1MM) or boolean(precip1Traces)">
   <tr>
   <td nowrap="1">
   <xsl:value-of select="$trans/hdr[@o='precipAmount1']"/>
   </td>
   <xsl:choose>
    <xsl:when test="boolean(precipAmount1Inch)">
     <td nowrap="1">
     <xsl:value-of select="concat(format-number(precipAmount1Inch/@v * ($FT2M div 12 * 1000), '0'), $trans/units[@o='MM'])"/>
     </td>
     <xsl:text>&#9;&#9;&#9;&#9;&#9;</xsl:text>
     <td nowrap="1">
     <xsl:value-of select="concat(precipAmount1Inch/@v, $trans/units[@o='IN'])"/>
     </td>
    </xsl:when>
    <xsl:when test="boolean(precipAmount1MM)">
     <td nowrap="1">
     <xsl:value-of select="concat(precipAmount1MM/@v, $trans/units[@o='MM'])"/>
     </td>
     <xsl:text>&#9;&#9;&#9;&#9;&#9;</xsl:text>
     <td nowrap="1">
     <xsl:value-of select="concat(format-number(precipAmount1MM/@v div ($FT2M div 12 * 1000), '0.00'), $trans/units[@o='IN'])"/>
     </td>
    </xsl:when>
    <xsl:otherwise>
     <td nowrap="1" colspan="2">
      <xsl:value-of select="$trans/t[@o='traces']"/>
     </td>
    </xsl:otherwise>
   </xsl:choose>
   </tr><xsl:text>&#10;</xsl:text>
  </xsl:if>
  <xsl:if test="boolean(precipAmount2Inch) or boolean(precipAmount2MM)">
   <tr>
   <td nowrap="1">
   <xsl:value-of select="$trans/hdr[@o='precipAmount2']"/>
   </td>
   <xsl:choose>
    <xsl:when test="boolean(precipAmount2Inch)">
     <td nowrap="1">
     <xsl:value-of select="concat(format-number(precipAmount2Inch/@v * ($FT2M div 12 * 1000), '0'), $trans/units[@o='MM'])"/>
     </td>
     <xsl:text>&#9;&#9;&#9;&#9;&#9;</xsl:text>
     <td nowrap="1">
     <xsl:value-of select="concat(precipAmount2Inch/@v, $trans/units[@o='IN'])"/>
     </td>
    </xsl:when>
    <xsl:otherwise>
     <td nowrap="1">
     <xsl:value-of select="concat(precipAmount2MM/@v, $trans/units[@o='MM'])"/>
     </td>
     <xsl:text>&#9;&#9;&#9;&#9;&#9;</xsl:text>
     <td nowrap="1">
     <xsl:value-of select="concat(format-number(precipAmount2MM/@v div ($FT2M div 12 * 1000), '0.00'), $trans/units[@o='IN'])"/>
     </td>
    </xsl:otherwise>
   </xsl:choose>
   </tr><xsl:text>&#10;</xsl:text>
  </xsl:if>
 </xsl:template>

 <xsl:template match="tornadicActivity">
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1"><b><xsl:value-of select="@s"/></b></td>
  </xsl:if>
  <td nowrap="1" colspan="3">
  <dl style="margin: 0em;"><dt>
  <xsl:value-of select="$trans/tornadicActivity[@o=$this/tornadicActivityType/@v]"/>
   <xsl:if test="boolean(locationAndList)">
    <xsl:apply-templates select="locationAndList">
     <xsl:with-param name="locOrDir" select="'compassDirLocation'"/>
     <xsl:with-param name="trans" select="$trans"/>
    </xsl:apply-templates>
    <xsl:if test="boolean(MOV) or boolean(MOVD)">
     <xsl:text>,</xsl:text>
    </xsl:if>
   </xsl:if>
   <xsl:apply-templates select="MOV|MOVD">
    <xsl:with-param name="trans" select="$trans"/>
   </xsl:apply-templates>
  <xsl:text>:</xsl:text></dt>
  <xsl:apply-templates select="startTime|endTime">
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
  </dl></td>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="balloon">
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1"><b><xsl:value-of select="@s"/></b></td>
  </xsl:if>
  <td nowrap="1" colspan="3">
  <xsl:if test="boolean(disappearedAt)">
   <xsl:value-of select="substring-before($trans/t[@o='balloonDisapp'], '|')"/>
   <xsl:value-of select="concat(disappearedAt/distance/@v, $trans/units[@o=$this/disappearedAt/unitHeight/@v])"/>
   <xsl:value-of select="substring-after($trans/t[@o='balloonDisapp'], '|')"/>
  </xsl:if>
  <xsl:if test="boolean(visibleTo)">
   <xsl:value-of select="substring-before($trans/t[@o='balloonVisib'], '|')"/>
   <xsl:value-of select="concat(visibleTo/distance/@v, $trans/units[@o=$this/visibleTo/unitHeight/@v])"/>
   <xsl:value-of select="substring-after($trans/t[@o='balloonVisib'], '|')"/>
  </xsl:if>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="firstObs|nextObs|lastObs">
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1"><b><xsl:value-of select="@s"/></b></td>
  </xsl:if>
  <td nowrap="1" colspan="3">
  <xsl:value-of select="substring-before($trans/t[@o=name($this)], '|')"/>
  <xsl:if test="boolean(isManned)">
   <xsl:value-of select="$trans/t[@o='isManned']"/>
  </xsl:if>
  <xsl:if test="boolean(isStaffed)">
   <xsl:value-of select="$trans/t[@o='isStaffed']"/>
  </xsl:if>
  <xsl:value-of select="substring-after($trans/t[@o=name($this)], '|')"/>
  <xsl:if test="boolean(obsAt)">
   <xsl:value-of select="concat(' ', substring-before($trans/t[@o='onThe'], '|'))"/>
   <xsl:value-of select="format-number(obsAt/day/@v, '0')"/>
   <xsl:value-of select="substring-after($trans/t[@o='onThe'], '|')"/>
   <xsl:value-of select="concat(obsAt/hour/@v, ':', obsAt/minute/@v, ' UTC')"/>
  </xsl:if>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="estimated">
  <xsl:param name="trans"/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1"><b><xsl:value-of select="@s"/></b></td>
  </xsl:if>
  <td nowrap="1" colspan="3">
  <xsl:for-each select="estimatedItem">
   <xsl:variable name="this2" select="."/>
   <xsl:if test="position() != 1"><xsl:text>, </xsl:text></xsl:if>
   <xsl:value-of select="$trans/estimatedItem[@o=$this2/@v]"/>
  </xsl:for-each>
  <xsl:value-of select="concat(' ', $trans/t[@o='estimated'])"/>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="RSNK|LAG_PK">
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <xsl:variable name="rows">
   <xsl:choose>
    <xsl:when test="boolean(wind/gustSpeed) and boolean(dewpoint)">5</xsl:when>
    <xsl:when test="boolean(wind/gustSpeed) or boolean(dewpoint)">4</xsl:when>
    <xsl:otherwise>3</xsl:otherwise>
   </xsl:choose>
  </xsl:variable>
  <tr>
  <xsl:if test="$method = 'html'">
   <xsl:element name="td">
    <xsl:attribute name="nowrap">1</xsl:attribute>
    <xsl:attribute name="rowspan"><xsl:value-of select="$rows"/></xsl:attribute>
    <b><xsl:value-of select="@s"/></b>
   </xsl:element>
  </xsl:if>
  <td nowrap="1" colspan="3">
  <b><xsl:value-of select="$trans/hdr[@o=name($this)]"/></b>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
  <tr>
  <xsl:call-template name="printTemp">
   <xsl:with-param name="hdr" select="$trans/hdr[@o='temperature-temp']"/>
   <xsl:with-param name="rows" select="1"/>
   <xsl:with-param name="temp" select="air"/>
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:call-template>
  </tr><xsl:text>&#10;</xsl:text>
  <xsl:if test="boolean(dewpoint)">
   <tr>
   <xsl:call-template name="printTemp">
    <xsl:with-param name="hdr" select="$trans/hdr[@o='temperature-dew']"/>
    <xsl:with-param name="rows" select="1"/>
    <xsl:with-param name="temp" select="dewpoint"/>
    <xsl:with-param name="trans" select="$trans"/>
   </xsl:call-template>
   </tr><xsl:text>&#10;</xsl:text>
  </xsl:if>
  <xsl:apply-templates select="wind">
   <xsl:with-param name="hdr"   select="$trans/hdr[@o='sfcWind']"/>
   <xsl:with-param name="trans" select="$trans"/>
  </xsl:apply-templates>
 </xsl:template>

 <xsl:template match="RADAT">
  <xsl:param name="trans"/>
  <xsl:variable name="rows">
   <xsl:choose>
    <xsl:when test="boolean(missing)">2</xsl:when>
    <xsl:otherwise>3</xsl:otherwise>
   </xsl:choose>
  </xsl:variable>
  <tr>
  <xsl:if test="$method = 'html'">
   <xsl:element name="td">
    <xsl:attribute name="nowrap">1</xsl:attribute>
    <xsl:attribute name="rowspan"><xsl:value-of select="$rows"/></xsl:attribute>
    <b><xsl:value-of select="@s"/></b>
   </xsl:element>
  </xsl:if>
  <td nowrap="1" colspan="3">
  <b><xsl:value-of select="$trans/hdr[@o='RADAT']"/></b>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
  <xsl:choose>
   <xsl:when test="boolean(missing)">
    <tr>
    <td nowrap="1" colspan="3">
    <xsl:value-of select="$trans/t[@o='missing']"/>
    </td>
    </tr><xsl:text>&#10;</xsl:text>
   </xsl:when>
   <xsl:otherwise>
    <tr>
    <td nowrap="1"><xsl:value-of select="$trans/hdr[@o='relHumid']"/></td>
    <td nowrap="1" colspan="2">
    <xsl:value-of select="relHumid/@v"/>
    <xsl:text>%</xsl:text>
    </td>
    </tr><xsl:text>&#10;</xsl:text>
    <tr>
    <td nowrap="1"><xsl:value-of select="$trans/hdr[@o='freezingLvl']"/></td>
    <td nowrap="1">
    <xsl:value-of select="concat($trans/t[@o='cloud-at'], ' ', distance/@v * 100, $trans/units[@o='FT'])"/>
    </td>
    <xsl:text>&#9;&#9;&#9;&#9;</xsl:text>
    <td nowrap="1">
    <xsl:value-of select="concat(format-number(distance/@v * 10 * $FT2M, '0') * 10, $trans/units[@o='M'])"/>
    </td>
    </tr><xsl:text>&#10;</xsl:text>
   </xsl:otherwise>
  </xsl:choose>
 </xsl:template>

 <xsl:template match="durationOfSunshine">
  <xsl:param name="trans"/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1"><b><xsl:value-of select="@s"/></b></td>
  </xsl:if>
  <td nowrap="1"><xsl:value-of select="$trans/hdr[@o='durationOfSunshine']"/></td>
  <td nowrap="1" colspan="2">
  <xsl:choose>
   <xsl:when test="boolean(durationMinutes)">
    <xsl:choose>
     <xsl:when test="durationMinutes/@v &gt; 59">
      <xsl:variable name="min">
       <xsl:value-of select="durationMinutes/@v mod 60"/>
      </xsl:variable>
      <xsl:value-of select="concat((durationMinutes/@v - $min) div 60, $trans/units[@o='H'])"/>
      <xsl:if test="$min &gt; 0">
       <xsl:value-of select="concat(' ', $min, $trans/units[@o='MIN'])"/>
      </xsl:if>
     </xsl:when>
     <xsl:otherwise>
      <xsl:value-of select="concat(durationMinutes/@v, $trans/units[@o='MIN'])"/>
     </xsl:otherwise>
    </xsl:choose>
   </xsl:when>
   <xsl:otherwise>
    <xsl:value-of select="$trans/t[@o='notAvailable']"/>
   </xsl:otherwise>
  </xsl:choose>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="hailStones">
  <xsl:param name="trans"/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1"><b><xsl:value-of select="@s"/></b></td>
  </xsl:if>
  <td nowrap="1">
  <xsl:value-of select="$trans/hdr[@o='hailStones']"/>
  </td>
  <td colspan="2">
  <xsl:value-of select="$trans/t[@o='hailStoneSize']"/>
  <xsl:if test="boolean(isLess)">
   <xsl:text>&lt;</xsl:text>
  </xsl:if>
  <xsl:value-of select="concat(hailStoneSize/@v, $trans/units[@o='IN'])"/>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="notRecognised">
  <xsl:param name="trans"/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1"><b><xsl:value-of select="@s"/></b></td>
  </xsl:if>
  <td nowrap="1">
  <xsl:value-of select="$trans/hdr[@o='notRecognised']"/>
  </td>
  <td colspan="2">
  <xsl:value-of select="@s"/>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="fcstNotAvbl">
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <tr>
  <xsl:if test="$method = 'html'">
   <td nowrap="1"><b><xsl:value-of select="@s"/></b></td>
  </xsl:if>
  <td nowrap="1">
  <xsl:value-of select="$trans/hdr[@o='fcstNotAvbl']"/>
  </td>
  <td nowrap="1" colspan="2">
  <xsl:value-of select="$trans/fcstNotAvblReason[@o=$this/fcstNotAvblReason/@v]"/>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

 <xsl:template match="ERROR">
  <xsl:param name="trans"/>
  <xsl:variable name="this" select="."/>
  <tr>
  <td nowrap="1">
  <xsl:value-of select="$trans/hdr[@o='ERROR']"/>
  </td>
  <td colspan="3">
  <xsl:value-of select="$trans/error[@o=$this/descr/@v]"/>
  <br /><xsl:text>&#10;&#9;&#9;&#9;</xsl:text>
  <xsl:value-of select="pos/@v"/>
  </td>
  </tr><xsl:text>&#10;</xsl:text>
 </xsl:template>

</xsl:stylesheet>
