########################################################################
# metaf2xml/XML.pm -- write data from a METAR or TAF message to XML file
#
# copyright (c) metaf2xml 2006
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License as
# published by the Free Software Foundation; either version 2 of the
# License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA
########################################################################

package metaf2xml::XML;

########################################################################
# some things strictly perl
########################################################################
use strict;
use warnings;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

=head1 NAME

metaf2xml::XML - write data from a METAR or TAF message to XML file

=head1 SYNOPSIS

  use metaf2xml::XML;

  xml_start(\%opts, @versions);
  printReport_XML(\%metar, $is_taf);
  xml_end();

=head1 DESCRIPTION

This module writes the data of a METAR or TAF message to an XML file.

=head1 FUNCTIONS

=over

=cut

########################################################################
# export the functions provided by this module
########################################################################
require Exporter;
@ISA = qw(Exporter);
@EXPORT = qw(
    xml_start
    printReport_XML
    xml_end
);

########################################################################
# the one and only global variable, the IO handle for XML::Writer
########################################################################
my $writer;

########################################################################
# helper functions
########################################################################

sub _mkXMLTag {
    my ($root, $attr_name) = @_;
    my @attr;

    for (sort keys %{$root->{$attr_name}}) {
        push @attr, ($_,  $root->{$attr_name}{$_});
    }
    return @attr;
}

sub _printXMLEmptyTag {
    my ($r, $tag) = @_;

    return unless exists $r->{$tag};

    if (defined $r->{$tag}) {
        $writer->emptyTag($tag, 'v', $r->{$tag});
    } else {
        $writer->emptyTag($tag);
    }
}

sub _printXMLLocations {
    my $r = shift;

    $writer->startTag('locationAndList');
    for (@$r) {
        if (   scalar @$_ == 1
            && exists $_->[0]{locationSpec}
            && $_->[0]{locationSpec} eq 'STNRY')
        {
            $writer->emptyTag('isStationary');
            next;
        }
        $writer->startTag('locationThruList');
        for (@$_) {
            $writer->startTag('location');
            _printXMLEmptyTag $_, 'isDistant';
            _printXMLEmptyTag $_, 'inVicinity';
            if (exists $_->{distance}) {
                _printXMLEmptyTag $_, 'distance';
                _printXMLEmptyTag $_, 'unitLength';
            }
            _printXMLEmptyTag $_, 'isGrid';
            _printXMLEmptyTag $_, 'isQuadrant';
            _printXMLEmptyTag $_, 'obscgMtns';
            if (exists $_->{locationSpec}) {
                _printXMLEmptyTag $_, 'locationSpec';
            } elsif (exists $_->{compassDir}) {
                _printXMLEmptyTag $_, 'compassDir';
            } else {
                $writer->startTag('quadrantList');
                for (@{$_->{quadrant}}) {
                    $writer->emptyTag('quadrant', 'v', $_);
                }
                $writer->endTag('quadrantList');
            }
            $writer->endTag('location');
        }
        $writer->endTag('locationThruList');
    }
    $writer->endTag('locationAndList');
}

sub _printXMLLocSpecs {
    my $r = shift;
    _printXMLLocations $r->{locationAndList}
        if exists $r->{locationAndList};
    if (exists $r->{MOV}) {
        $writer->startTag('MOV');
        _printXMLLocations $r->{MOV};
        $writer->endTag('MOV');
    } elsif (exists $r->{MOVD}) {
        $writer->startTag('MOVD');
        _printXMLLocations $r->{MOVD};
        $writer->endTag('MOVD');
    }
}

sub _printXMLTag {
    my ($r, $tag, $skip_s) = @_;

    return unless exists $r->{$tag};

    $writer->startTag($tag, !$skip_s && exists $r->{$tag}{s}
                            ? ('s', $r->{$tag}{s}) : ());
    for (sort keys %{$r->{$tag}}) {
        next if $_ eq 's';
        if ($tag eq 'rwyState' && ($_ eq 'friction' || $_ =~ /^deposit/)) {
            _printXMLTag($r->{$tag}, $_);
        } elsif ($_ =~ /^(?:locationAndList|MOVD?)$/) {
            _printXMLLocSpecs $r->{$tag};
        } elsif ($_ =~ /^(?:cloud|visibility)$/) {
            _printXMLTag($r->{$tag}, $_);
        } elsif ($_ =~ /^weather$/) {
            _printXMLWeather([$r->{$tag}{$_}]);
        } else {
            _printXMLEmptyTag $r->{$tag}, $_;
        }
    }
    $writer->endTag($tag);
}

sub _printXMLTagList {
    my ($r, $tag) = @_;

    return unless exists $r->{$tag};

    $writer->startTag($tag . 'List');
    for (@{$r->{$tag}}) {
        _printXMLTag { $tag => $_ }, $tag;
    }
    $writer->endTag($tag . 'List');
}

sub _printXMLWeather {
    my ($weather, $tag) = @_;

    $writer->startTag($tag) if defined $tag;
    for (@$weather) {
        $writer->startTag('weather', 's', $_->{s});
        _printXMLEmptyTag $_, 'notAvailable';
        _printXMLEmptyTag $_, 'invalidFormat';
        _printXMLEmptyTag $_, 'NSW';
        _printXMLEmptyTag $_, 'tornado';
        _printXMLEmptyTag $_, 'inVicinity';
        _printXMLEmptyTag $_, 'intensity';
        for (exists $_->{descriptor} ? @{$_->{descriptor}} : ()) {
            $writer->emptyTag('descriptor', 'v', $_);
        }
        for (exists $_->{phenomenon} ? @{$_->{phenomenon}} : ()) {
            $writer->emptyTag('phenomenon', 'v', $_);
        }
        $writer->endTag('weather');
    }
    $writer->endTag($tag) if defined $tag;
}

sub _printXMLWind {
    my ($wind, $tag) = @_;

    $writer->startTag($tag, exists $wind->{s} ? ('s', $wind->{s}) : ());
    for (sort keys %$wind) {
        next if $_ eq 's';
        if ($_ eq 'wind') {
            _printXMLTag $wind, $_;
        } else {
            $writer->emptyTag($_, 'v', $wind->{$_});
        }
    }
    $writer->endTag($tag);
}

########################################################################
# printReport_XML
########################################################################

=item printReport_XML(B<metar>, B<is_taf>)

This function writes the data contained the argument B<metar> to the XML file.
The data will be enclosed in a tag <metar> or <taf>. The function can be invoked
repeatedly. xml_start() must have been invoked already.

The following arguments are expected:

=over

=item B<metar>

reference to hash of data to be written to the file

=item B<is_taf>

boolean value, indicating whether it is a TAF message or not

=back

=cut

sub printReport_XML {
    my ($metar, $is_taf) = @_;

    return unless $writer;

    $writer->startTag($is_taf ? 'taf' : 'metar', 's', $metar->{msg});
    _printXMLTag $metar, 'ERROR';
    $writer->dataElement('WARNING', ($metar->{WARNING} =~ /(.*?)\n?$/s))
        if $metar->{WARNING} ne '';
    _printXMLEmptyTag $metar, 'SPECI';
    _printXMLTag $metar, 'obsStation';
    _printXMLTag $metar, 'obsTime';
    _printXMLTag $metar, 'issueTime';
    _printXMLTag $metar, 'fcstPeriod';
    _printXMLTag $metar, 'reportModifier';
    _printXMLTag $metar, 'fcstNotAvbl';
    _printXMLWind $metar->{sfcWind}, 'sfcWind'
        if exists $metar->{sfcWind};
    _printXMLEmptyTag $metar, 'CAVOK';
    _printXMLTag $metar, 'visPrev';
    _printXMLTag $metar, 'visMin';
    if (exists $metar->{visRwy}) {
        $writer->startTag('visRwyList');
        for my $v (@{$metar->{visRwy}}) {
            $writer->startTag('visRwy', 's', $v->{s});
            for (sort keys %$v) {
                next if $_ eq 's';
                if ($_ eq 'RVR' || $_ eq 'RVRVariations') {
                    _printXMLTag $v, $_;
                } else {
                    $writer->emptyTag($_, 'v', $v->{$_});
                }
            }
            $writer->endTag('visRwy');
        }
        $writer->endTag('visRwyList');
    }
    _printXMLEmptyTag $metar, 'RVRNO';
    _printXMLWeather $metar->{weather}, 'weatherList'
        if exists $metar->{weather};
    _printXMLTagList $metar, 'cloud';
    _printXMLTag $metar, 'visVert';
    if (exists $metar->{temperature}) {
        $writer->startTag('temperature', 's', $metar->{temperature}{s});
        _printXMLTag $metar->{temperature}, 'air';
        _printXMLTag $metar->{temperature}, 'dewpoint';
        if (exists $metar->{temperature}{relHumid1}) {
            for (1, 2, 3, 4) {
              $writer->emptyTag('relHumid' . $_,
                                'v', $metar->{temperature}{'relHumid' . $_});
            }
        }
        $writer->endTag('temperature');
    }
    for (@{$metar->{tempMaxMin}}) {
        _printXMLTag $_, keys %$_;
    }
    _printXMLTagList $metar, 'tempAt';
    _printXMLTag $metar, 'QNH';
    _printXMLTag $metar, 'somePressure';
    _printXMLTag $metar, 'cloudMaxCover';
    _printXMLWeather $metar->{recWeather}, 'recentWeatherList'
        if exists $metar->{recWeather};
    _printXMLTagList $metar, 'windShear';
    _printXMLTagList $metar, 'rwyState';
    _printXMLTag $metar, 'colourCode';
    _printXMLTag $metar, 'NEFO_PLAYA';
    _printXMLTag $metar, 'RH';
    if (exists $metar->{rwyWind}) {
        $writer->startTag('rwyWindList');
        for (@{$metar->{rwyWind}}) {
            _printXMLWind $_, 'rwyWind';
        }
        $writer->endTag('rwyWindList');
    }
    for my $suppl (@{$metar->{TAFsuppl}}) {
        my $r_type = (keys %{$suppl})[0];
        if ($r_type eq 'windShearLvl') {
            $writer->startTag('windShearLvl', 's', $suppl->{windShearLvl}{s});
            $writer->emptyTag('level', 'v', $suppl->{windShearLvl}{level});
            _printXMLWind $suppl->{windShearLvl}{wind}, 'wind';
            $writer->endTag('windShearLvl');
        } elsif ($r_type =~ /^(?:turbulence|icing|tempAt)$/) {
            _printXMLTag $suppl, $r_type;
        }
    }

    if (exists $metar->{trend}) {
        $writer->startTag('trendList');
        for my $td (@{$metar->{trend}}) {
            $writer->startTag('trend', 's', $td->{s});
            $writer->emptyTag('trendType', 'v', $td->{trendType});
            _printXMLTag $td, 'trendTime1';
            _printXMLTag $td, 'trendTime2';
            $writer->emptyTag('probability', 'v', $td->{probability})
                if exists $td->{probability};
            _printXMLWind $td->{sfcWind}, 'sfcWind'
                if exists $td->{sfcWind};
            $writer->emptyTag('CAVOK') if exists $td->{CAVOK};
            _printXMLTag $td, 'visPrev';
            _printXMLTag $td, 'visVert';
            _printXMLWeather $td->{weather}, 'weatherList'
                if exists $td->{weather};
            _printXMLTagList $td, 'cloud';
            _printXMLTagList $td, 'rwyState';
            _printXMLTag $td, 'colourCode';
            for my $suppl (@{$td->{TAFsuppl}}) {
                my $r_type = (keys %{$suppl})[0];
                if ($r_type eq 'windShearLvl') {
                    $writer->startTag('windShearLvl', 's',
                                      $suppl->{windShearLvl}{s});
                    $writer->emptyTag('level', 'v',
                                      $suppl->{windShearLvl}{level});
                    _printXMLWind $suppl->{windShearLvl}{wind}, 'wind';
                    $writer->endTag('windShearLvl');
                } elsif ($r_type =~ /^(?:turbulence|icing)$/) {
                    _printXMLTag $suppl, $r_type;
                }
            }

            $writer->endTag('trend');
        }
        $writer->endTag('trendList');
    }

    if (exists $metar->{remark}) {
        my $rmk_label = $is_taf ? 'tafRemark' : 'remark';

        $writer->startTag($rmk_label . 'List');
        for my $r (@{$metar->{remark}}) {
            $writer->startTag($rmk_label);
            for my $r_type (sort keys %$r) {
                my $e = $r->{$r_type};

                if ($r_type =~ '^(?:obsStationType|visVar.|cloudMaxCover|cloudCoverVar|cloud|colourCode|QNH|SLP.*|regQNH|QFE|QFF|windShift|snowIncr|correctedAt|rwyState|reportConcerns|temp6hMax|temp6hMin|precipHourly|AI|RH|OAT|SST|cloudTypeFamily|cloudTypeLvl|seaCondition|swellCondition|ceilingAtLoc|visAtLoc|obscuration|cloudAbove|variableCeiling|pressureTendency3h|rainfall|precipPastHour|densityAlt|pressureAlt|VISNO|CHINO|obsTimeOffset|nextFcstBy|nextFcstAt|amdAt|fcstAutoObs|fcstAutoMETAR|waterEquivOfSnow|snowOnGround|snowCover|RADAT|tempMaxFQ|durationOfSunshine|hailStones)$') {
                    _printXMLTag $r, $r_type;
                } elsif ($r_type eq 'notRecognised') {
                    $writer->emptyTag($r_type, 's', $r->{$r_type}{s});
                } elsif ($r_type =~ '^(?:recWeather)$') {
                    _printXMLWeather $e, 'recentWeatherList';
                } elsif ($r_type =~ '^(?:visibilityAtLoc|visMin)$') {
                    $writer->startTag($r_type, 's', $e->{s});
                    for (sort keys %$e) {
                        next if $_ eq 's';
                        if ($_ eq 'visibility') {
                            _printXMLTag $e, $_;
                        } else {
                            _printXMLEmptyTag $e, $_;
                        }
                    }
                    $writer->endTag($r_type);
                } elsif ($r_type =~ '^(?:thrWind|peakWind|rwyWind|gridWind)$') {
                    _printXMLWind $e, $r_type;
                } elsif ($r_type =~ '^(?:visListLoc)$') {
                    $writer->startTag($r_type, 's', $e->{s});
                    for (@{$e->{arr}}) {
                        $writer->startTag('visLocData');
                        _printXMLTag $_, 'visibility';
                        _printXMLLocSpecs $_;
                        $writer->endTag('visLocData');
                    }
                    $writer->endTag($r_type);
                } elsif ($r_type =~ '^(?:cloudOpacityLvl)$') {
                    $writer->startTag($r_type, 's', $e->{s});
                    _printXMLLocSpecs $e;
                    $writer->emptyTag('eights', 'v', $e->{eights});
                    _printXMLWeather [$e->{weather}]
                        if exists $e->{weather};
                    $writer->emptyTag('cloudType', 'v',$e->{cloudType})
                        if exists $e->{cloudType};
                    $writer->emptyTag('cloudBase', 'v', $e->{cloudBase});
                    $writer->endTag($r_type);
                } elsif ($r_type =~ '^(?:visRwy)$') {
                    $writer->startTag($r_type, 's', $e->{s});
                    for (sort keys %$e) {
                        next if $_ eq 's';
                        if ($_ eq 'RVR') {
                            _printXMLTag $e, $_;
                        } else {
                            $writer->emptyTag($_, 'v', $e->{$_});
                        }
                    }
                    $writer->endTag($r_type);
                } elsif ($r_type =~ '^(?:tempHourly)$') {
                    $writer->startTag($r_type, 's', $e->{s});
                    _printXMLTag $e, 'air';
                    _printXMLTag $e, 'dewpoint';
                    $writer->endTag($r_type);
                } elsif ($r_type =~ '^(?:temp24h)$') {
                    $writer->startTag($r_type, 's', $e->{s});
                    _printXMLTag $e, 'temp24hMax';
                    _printXMLTag $e, 'temp24hMin';
                    $writer->endTag($r_type);
                } elsif ($r_type eq 'keyword') {
                    $writer->emptyTag($r_type, 's', $e->{s}, 'v',$e->{keyword});
                } elsif ($r_type eq 'needMaint') {
                    $writer->emptyTag($r_type, 's', '$');
                } elsif ($r_type eq 'beginEndPrecip') {
                    $writer->startTag($r_type, 's', $e->{s});
                    for (@{$e->{precip}}) {
                        $writer->startTag('precip');
                        for (exists $_->{weather}{descriptor}
                                    ? @{$_->{weather}{descriptor}} : ()) {
                            $writer->emptyTag('descriptor', 'v', $_);
                        }
                        for (exists $_->{weather}{phenomenon}
                                    ? @{$_->{weather}{phenomenon}} : ()) {
                            $writer->emptyTag('phenomenon', 'v', $_);
                        }
                        for (@{$_->{start_end}}) {
                            if (exists $_->{startTime}) {
                                _printXMLTag $_, 'startTime';
                            } else {
                                _printXMLTag $_, 'endTime';
                            }
                        }
                        $writer->endTag('precip');
                    }
                    _printXMLLocSpecs $e;
                    $writer->endTag($r_type);
                } elsif ($r_type =~ '^phenomenon(?:AtLoc|Only)$') {
                    $writer->startTag($r_type, 's', $e->{s});
                    for (@{$e->{phenomDescrPre}}) {
                        $writer->emptyTag('phenomDescrPre', 'v', $_);
                    }
                    for (@{$e->{phenomDescrPost}}) {
                        $writer->emptyTag('phenomDescrPost', 'v', $_);
                    }
                    _printXMLWeather [$e->{weather}]
                        if exists $e->{weather};
                    if (exists $e->{cloudType}) {
                        $writer->startTag('cloudTypeList');
                        for (@{$e->{cloudType}}) {
                            $writer->emptyTag('cloudType', 'v', $_);
                        }
                        $writer->endTag('cloudTypeList');
                    }
                    $writer->emptyTag('otherPhenom', 'v', $e->{otherPhenom})
                        if exists $e->{otherPhenom};
                    $writer->emptyTag('cloudCover', 'v', $e->{cloudCover})
                        if exists $e->{cloudCover};
                    if (exists $e->{lightningType}) {
                        $writer->startTag('lightningTypeList');
                        for (@{$e->{lightningType}}) {
                            $writer->emptyTag('lightningType', 'v', $_);
                        }
                        $writer->endTag('lightningTypeList');
                    }
                    _printXMLLocSpecs $e;
                    _printXMLEmptyTag $e, 'cloudTypeAsoctd';
                    _printXMLEmptyTag $e, 'cloudTypeEmbd';
                    $writer->endTag($r_type);
                } elsif ($r_type eq 'conditionMountain') {
                    $writer->startTag($r_type . 'List', 's', $e->{s});
                    for (@{$e->{condMoun}}) {
                        $writer->startTag($r_type);
                        $writer->emptyTag('condMounType', 'v',
                                          $_->{condMounType});
                        $writer->emptyTag('condMounChange', 'v',
                                          $_->{condMounChange})
                            if exists $_->{condMounChange};
                        _printXMLLocSpecs $_;
                        $writer->endTag($r_type);
                    }
                    $writer->endTag($r_type . 'List');
                } elsif ($r_type eq 'conditionValley') {
                    $writer->startTag($r_type . 'List', 's', $e->{s});
                    for (@{$e->{condVall}}) {
                        $writer->startTag($r_type);
                        $writer->emptyTag('condVallType', 'v',
                                          $_->{condVallType});
                        $writer->emptyTag('condVallChange', 'v',
                                          $_->{condVallChange})
                            if exists $_->{condVallChange};
                        _printXMLLocSpecs $_;
                        $writer->endTag($r_type);
                    }
                    $writer->endTag($r_type. 'List');
                } elsif ($r_type eq 'phenomOpacityList') {
                    $writer->startTag($r_type, 's', $e->{s});
                    for (@{$e->{phenomOpacity}}) {
                        $writer->startTag('phenomOpacity');
                        $writer->emptyTag('eights', 'v', $_->{eights});
                        if (exists $_->{weather}) {
                            _printXMLWeather [$_->{weather}];
                        } else {
                            $writer->emptyTag('cloudType', 'v',$_->{cloudType});
                        }
                        $writer->endTag('phenomOpacity');
                    }
                    _printXMLEmptyTag $e, 'cloudTypeAsoctd';
                    _printXMLEmptyTag $e, 'cloudTypeEmbd';
                    $writer->endTag($r_type);
                } elsif ($r_type eq 'cloudTrace') {
                    $writer->startTag($r_type, 's', $e->{s});
                    for (@{$e->{cloudType}}) {
                        $writer->emptyTag('cloudType', 'v', $_);
                    }
                    _printXMLEmptyTag $e, 'cloudTypeNotAvailable';
                    _printXMLEmptyTag $e, 'isLower';
                    $writer->endTag($r_type);
                } elsif ($r_type eq 'ceilVisVariable') {
                    $writer->startTag($r_type, 's', $e->{s});
                    _printXMLTag $e, 'visibilityFrom';
                    _printXMLTag $e, 'visibilityTo';
                    $writer->endTag($r_type);
                } elsif ($r_type eq 'rwySfcCondition') {
                    $writer->startTag($r_type, 's', $e->{s});
                    for (@{$e->{arr}}) {
                        _printXMLEmptyTag $_, keys %$_;
                    }
                    $writer->endTag($r_type);
                } elsif ($r_type eq 'climate') {
                    $writer->startTag($r_type, 's', $e->{s});
                    _printXMLEmptyTag $e, 'precip1Traces';
                    _printXMLEmptyTag $e, 'precipAmount1Inch';
                    _printXMLEmptyTag $e, 'precipAmount1MM';
                    _printXMLEmptyTag $e, 'precipAmount2Inch';
                    _printXMLEmptyTag $e, 'precipAmount2MM';
                    _printXMLTag $e, 'temp1';
                    _printXMLTag $e, 'temp2';
                    $writer->endTag($r_type);
                } elsif ($r_type eq 'tornadicActivity') {
                    $writer->startTag($r_type, 's', $e->{s});
                    _printXMLEmptyTag $e, 'tornadicActivityType';
                    for (@{$e->{start_end}}) {
                        if (exists $_->{startTime}) {
                            _printXMLTag $_, 'startTime';
                        } else {
                            _printXMLTag $_, 'endTime';
                        }
                    }
                    _printXMLLocSpecs $e;
                    $writer->endTag($r_type);
                } elsif ($r_type eq 'balloon') {
                    $writer->startTag($r_type, 's', $e->{s});
                    _printXMLTag $e, 'disappearedAt';
                    _printXMLTag $e, 'visibleTo';
                    $writer->endTag($r_type);
                } elsif ($r_type =~ '^(?:first|next|last)Obs$') {
                    $writer->startTag($r_type, 's', $e->{s});
                    _printXMLEmptyTag $e, 'isStaffed';
                    _printXMLEmptyTag $e, 'isManned';
                    _printXMLTag $e, 'obsAt';
                    $writer->endTag($r_type);
                } elsif ($r_type eq 'estimated') {
                    $writer->startTag($r_type, 's', $e->{s});
                    for (@{$e->{estimatedItem}}) {
                        $writer->emptyTag('estimatedItem', 'v', $_);
                    }
                    _printXMLEmptyTag $e, 'dueToIceAccretion';
                    $writer->endTag($r_type);
                } elsif ($r_type =~ '^(?:RSNK|LAG_PK)$') {
                    $writer->startTag($r_type, 's', $e->{s});
                    _printXMLTag $e, 'air';
                    _printXMLTag $e, 'dewpoint';
                    _printXMLWind $e->{wind}, 'wind';
                    $writer->endTag($r_type);
                } else {
                    $writer->emptyTag(_mkXMLTag $r, $r_type);
                }
            }
            $writer->endTag($rmk_label);
        }
        $writer->endTag($rmk_label . 'List');
    }

    _printXMLTag $metar, 'amendment';

    $writer->endTag($is_taf ? 'taf' : 'metar');
}

########################################################################
# variables shared by xml_start, xml_end
########################################################################
my ($opt_X, $output);

########################################################################
# xml_start
########################################################################

=item xml_start(B<opts>, B<versions>)

This function opens the XML file and writes the necessary meta data into it.

The following arguments are expected:

=over

=item B<opts>

Reference to hash of options. The following keys of the hash are recognised:

=over

=item B<x>, with value B<E<lt>filenameE<gt>>

print parsed content of message as XML to <filename>

=item B<X>

with option B<x>: produce complete XML file, not just data

=item B<D>

with option B<X>: add DOCTYPE, reference to DTD

=item B<S>, with value B<E<lt>filenameE<gt>>

with option B<X>: add reference to stylesheet <filename>

=item B<O>, with value B<E<lt>optionsE<gt>>

with option B<S>: add <options> (a space separated list of the options passed to
metaf2xml.pl) to XML file

=back

=item B<versions>

Reference to array of version strings. They are written to the XML file if the
option B<X> is given.

=back

=cut

sub xml_start {
    my ($opts, $versions) = @_;

    $output = new IO::File(">$opts->{x}");
    $writer = new XML::Writer(OUTPUT=>$output, DATA_MODE=>1, DATA_INDENT=>1);

    # XML header
    $writer->xmlDecl("UTF-8");

    # additional XML info, if required
    $opt_X = exists $opts->{X};
    if ($opt_X) {

        # XML meta data
        $writer->doctype('data', undef, 'metaf.dtd') if defined $opts->{D};
        $writer->pi('xml-stylesheet',
                    'href="' . $opts->{S} . '" type="text/xsl"')
            if defined $opts->{S};

        # opening data tag
        $writer->startTag('data');

        # version stuff
        $writer->startTag('versionList');
        for (@$versions) {
            $writer->dataElement('version', $_);
        }
        $writer->dataElement('version', 'XML: $Id: XML.pm,v 1.23 2006/11/09 19:42:20 metaf2xml Exp $');
        $writer->endTag('versionList');

        if (exists $opts->{O}) {
            my %options;
            @{$options{options}}{'type', 'lang', 'format', 'source', 'msg'}
                                                     = split / /, $opts->{O}, 5;
            _printXMLTag \%options, 'options';
        }
    }
    $writer->startTag('metafList');
}

########################################################################
# xml_end
########################################################################

=item xml_end()

This function completes the writing to the XML file.

No arguments are expected.

=back

=cut

sub xml_end {
    return unless $writer;

    $writer->endTag('metafList');

    # closing data tag, if required
    $writer->endTag('data') if $opt_X;

    # close XML writer and file
    $writer->end();
    $output->close();
    $writer = undef;
    $output = undef;
}

1;
__END__
