#!/usr/bin/perl

# copyright (c) metaf2xml 2006
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License as
# published by the Free Software Foundation; either version 2 of the
# License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA

use strict;
use warnings;
use Getopt::Std;
use IO::File;

# path may be changed by install
use lib '/opt/metaf2xml/lib'; # METAF2XML_LIB
use XML::Writer;
use metaf2xml::parser;
use metaf2xml::FGFS;
use metaf2xml::XML;

my (%metar, %opts, $strict_fgfs, $is_taf, $output);

sub go {
    my $msg = shift;

    if ($msg eq 'TAF') {
        $is_taf = 1;
        return;
    }
    if ($msg eq 'METAR') {
        $is_taf = 0;
        return;
    }

    %metar = parseReport($msg, $is_taf);

    print printMetar_FGFS(\%metar, $opts{m}, $strict_fgfs) . "\n"
        if exists $opts{f};
    printReport_XML \%metar, $is_taf if exists $opts{x};
}

sub usage {
    print STDERR
          (shift)
        . "Usage: metaf2xml.pl [ <options> ] [ <message> ]\n"
        . "  Options:\n"
        . "  -t           message is a TAF message\n"
        . "  -f           print parsed content of message similar to FGFS\n"
        . "  -F           with -f: print parsed content exactly like FGFS\n"
        . "  -m           with -f: print message before processing\n"
        . "  -x <file>    print parsed content of message as XML to <file>\n"
        . "  -X           with -x: produce complete XML file, not just data\n"
        . "  -D           with -X: add DOCTYPE, reference to DTD\n"
        . "  -S <file>    with -X: add reference to stylesheet <file>\n"
        . "  -O <options> with -S: add <options> to XML file\n";
    exit 1;
}

# get and check options
usage '' unless getopts('mtfFx:XDS:O:', \%opts);

# check dependencies of options
usage "option -F requires option -f\n" if exists $opts{F} && !exists $opts{f};
usage "option -m requires option -f\n" if exists $opts{m} && !exists $opts{f};
usage "option -X requires option -x\n" if exists $opts{X} && !exists $opts{x};
usage "option -D requires option -X\n" if exists $opts{D} && !exists $opts{X};
usage "option -S requires option -X\n" if exists $opts{S} && !exists $opts{X};
usage "option -O requires option -S\n" if exists $opts{O} && !exists $opts{S};

$strict_fgfs = exists $opts{F};
$is_taf = exists $opts{t};

xml_start \%opts,
        [ 'metaf2xml: $Id: metaf2xml.pl,v 1.12 2006/11/09 19:41:23 metaf2xml Exp $',
          version_parser() ]
    if exists $opts{x};

if ($#ARGV >= 0) {
    for (@ARGV) {
        for (split /\n/, $_) {
            go $_;
        }
    }
} else {
    while (<STDIN>) {
        chomp();
        go $_;
    }
}

xml_end;
