package metaf2xml::FGFS;

# copyright (c) metaf2xml 2006
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License as
# published by the Free Software Foundation; either version 2 of the
# License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA

use strict;
use warnings;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);
require Exporter;

use POSIX qw(mktime strftime);

# path may be changed by install
use lib '/opt/metaf2xml/lib'; # METAF2XML_LIB
use metaf2xml::parser;

@ISA = qw(Exporter);

@EXPORT = qw(
    printMetar_FGFS
);

my $strict_fgfs;

my $FT2M               = 0.3048;
my $STATUTE_MILE2METER = 1609.3412196;
my $NM2KM              = 1.852;
my $METER_PER_SEC2KT   = (3.6 / $NM2KM);

my %keywords = (
    '$'     => 'automated system needs maintenance',
    ABV     => 'above',
    AI      => 'AI',
    ALG_LK  => 'along lake',
    ALG_MT  => 'along mountains',
    ALG_RIVER => 'along river',
    ALL     => 'all',
    ALL_WNDS_GRID => 'all wind directions grid',
    ALOFT   => 'aloft',
    ALQDS   => 'all quadrants',
    ALSTG   => 'altimeter setting',
    ALTM    => 'altimeter',
    ALTM_MISG => 'altimeter value missing',
    AND     => 'and',
    APCH    => 'approach',
    ARND    => 'around',
    ASOCTD  => 'associated',
    AT      => 'at',
    AT_AP   => 'at airport',
    AURBO   => 'aurea borealis',
    BA_GOOD => 'braking action good',
    BA_POOR => 'braking action poor',
    BBLO    => 'cloud base below station level',
    BECMG   => 'becoming:',
    BINOVC  => 'breaks in overcast',
    BLW     => 'below',
    CAVOK   => 'cloud and visibility OK',
    CAVU    =>'clear or scattered clouds and visibility greater than ten miles',
    CHINO   => 'secondary ceiling height indicator',
    CIG_BLN => 'ceiling from balloon data',
    CIG_HYR => 'ceiling higher',
    CIG_LWR => 'ceiling lower',
    CIG     => 'ceiling',
    CIG_RAG => 'ceiling ragged',
    CIG_RGD => 'ceiling ragged',
    CLD     => 'cloud',
    CLDS    => 'clouds',
    CLD_EMBD => 'cloud embedded',
    CLD_HGTS => 'cloud heights',
    CLDS_EMBD => 'clouds embedded',
    CONTRAILS => 'condensation trails',
    CONS    => 'continuous',
    CVCTV   => 'convective',
    D       => 'decreasing',
    DA      => 'density altitude',
    DATA    => 'data',
    DEW     => 'dew',
    DSIPTD  => 'dissipated',
    DSNT    => 'in the distance',
    DRY     => 'dry',
    EMBD    => 'embedded',
    EPC     => 'estimated pass closed',
    EPM     => 'estimated pass marginal',
    EPO     => 'estimated pass open',
    ESTMD   => 'estimated',
    FG_BNK  => 'fog bank',
    FGBNK   => 'fog bank',
    FBL     => 'feeble',
    FIBI    => 'filed but impracticable to transmit',
    FIRE    => 'fire',
    FIRES   => 'fires',
    FOGGY   => 'foggy',
    FM      => 'from',
    FROIN   => 'frost on the indicator',
    FRQ     => 'frequent',
    FULYR   => 'smoke layer',
    FUOCTY  => 'smoke over city',
    FZRANO  => 'sensor to decect freezing rain not operating',
    GRASS_FIRE  => 'grass fire',
    GRASS_FIRES => 'grass fires',
    GRID    => 'grid',
    HALO    => 'halo',
    HIR_CLDS => 'higher clouds',
    HVY_FROIN => 'heavy frost on the indicator',
    HZY     => 'hazy',
    ICG     => 'icing',
    IN_VLY  => 'in valley',
    INTMT   => 'intermittent',
    IOVC    => 'in overcast',
    ISOL    => 'isolated',
    IR      => 'ice on runway',
    LGT     => 'light',
    LGT_FROIN => 'light frost on the indicator',
    LOW     => 'low',
    LTG     => 'lightning',
    LSR     => 'loose snow on runway',
    LWR     => 'lower',
    MDT     => 'moderate',
    METAR   => 'METeorological Aerodrome Report',
    MISG    => 'missing',
    MOV     => 'moving',
    MOVD    => 'moved',
    MTNS    => 'mountains',
    MTNS_OBSC => 'mountains obscured',
    N       => 'no change',
    NOSIG   => 'no significant change',
    NOSPECI => 'no SPECI reports taken at this station',
    OAT     => 'outside air temp.',
    OBSCG   => 'obscuring',
    OBSCURED => 'obscured',
    OCNL    => 'occasional',
    OHD     => 'overhead',
    OMTNS   => 'over mountains',
    OVR_AP  => 'over airport',
    OVR_LK  => 'over lake',
    OVR_RIVER  => 'over river',
    P       => 'patchy',
    PA      => 'pressure altitude',
    PAST_HR => 'in past hour',
    PCPN    => 'precipitation',
    PK      => 'peak',
    PNO     => 'sensor not operating: tipping bucket rain gauge',
    PR      => 'pretty',
    PRESFR  => 'pressure trend: falling rapidly',
    PRESRR  => 'pressure trend: rising rapidly',
    PSR     => 'packed snow on runway',
    PWINO   => 'sensor to identify present weather not operating',
   'PWR PLNT PLUME' => 'power plant plume',
    QUAD    => 'quadrant',
    RCRNR   => 'runway condition reading not reported',
    RTS     => 'return to service',
    RVRNO   => 'runway visual range should be reported but is missing',
    RWY     => 'runway',
    SANDED  => 'sanded',
    SFC     => 'surface',
    SH      => 'shower',
    SHS     => 'showers',
    SKY     => 'sky',
    SKY_OBSC => 'sky obscured',
    SLP     => 'sea level pressure',
    SLR     => 'slush on runway',
    SPTS    => 'spots',
    SST     => 'sea surface temp.',
    STNRY   => 'stationary',
    RH      => 'relative humidity',
    TEMPO   => 'temporarily:',
    THK     => 'thick',
    THN     => 'thin',
    THN_SPTS_IOVC => 'thin spots in overcast',
    THRU    => 'through',
    TL      => 'till',
    TR      => 'traces of',
    TSNO    => 'sensor to decect lightning not operating',
    TWLGT   => 'twilight',
    TWR     => 'tower',
    U       => 'increasing',
    UNKN    => 'unknown',
    VAL     => 'valleys',
    V_D     => 'to various directions',
    VC      => 'in the vicinity',
    VIA_PHONE => 'via phone',
    VIRGA   => 'virga',
    VISNO   => 'secondary visibility sensor',
    VIS     => 'visibility',
    VIS_HYR => 'visibility higher',
    VIS_LWR => 'visibility lower',
    VIS_RDCD => 'visibility reduced',
    VLY_FG  => 'valley fog',
    VRY     => 'very',
    VRBL_CONDS => 'variable conditions',
    WEA     => 'significant weather',
    WET     => 'wet',
    WR      => 'wet runway',
    WND     => 'wind',
    WNDS    => 'winds',
);
my %weather_descr = (
    MI => 'shallow',
    BC => 'patches of',
    PR => 'partial',
    DR => 'low drifting',
    BL => 'blowing',
    SH => 'showers of',
    TS => 'thunderstorm with',
    FZ => 'freezing',
);

my %weather_types = (
    DZ => 'drizzle',
    RA => 'rain',
    SN => 'snow',
    SG => 'snow grains',
    IC => 'ice crystals',
    PL => 'ice pellets',
    GR => 'hail',
    GS => 'small hail and/or snow pellets',
    UP => 'unknown precipitation',
    BR => 'mist',
    FG => 'fog',
    FU => 'smoke',
    VA => 'volcanic ash',
    DU => 'widespread dust',
    SA => 'sand',
    HZ => 'haze',
    PO => 'dust/sand whirls',
    SQ => 'squalls',
    FC => 'funnel cloud',
    SS => 'sandstorm',
    DS => 'duststorm',
    TS => 'thunderstorm',
    SH => 'showers',
    JP => 'adjacent precipitation',
);

my %ltg_types = (
    CA => 'cloud-air',
    CC => 'cloud-cloud',
    CG => 'cloud-ground',
    CW => 'cloud-water',
    IC => 'in-cloud',
);


my %cloud_types = (
    AC     => 'altocumulus',
    ACC    => 'altocumulus castellanus',
    ACSL   => 'altocumulus standing lenticular',
    AS     => 'altostratus',
    CB     => 'cumulonimbus',
    CBMAM  => 'cumulonimbus mammatus',
    CC     => 'cirrocumulus',
    CCSL   => 'cirrocumulus standing lenticular',
    CF     => 'cumulus fractus',
    CI     => 'cirrus',
    CS     => 'cirrostratus',
    CU     => 'cumulus',
    CUFRA  => 'cumulus fractus',
    NS     => 'nimbostratus',
    SAC    => 'stratoaltocumulus',
    SC     => 'stratocumulus',
    SCSL   => 'stratocumulus standing lenticular',
    SF     => 'stratus fractus',
    ST     => 'stratus',
    STFRA  => 'stratus fractus',
    TCU    => 'towering cumulus',
);

# see http://www.srh.noaa.gov/eyw/HTML/Jim_Clouds/CloudTypes/low.html
my %cloud_types_low = (
    '0' => 'no clouds',
    '1' => $cloud_types{CU} . ' (little vertical extent)',
    '2' => $cloud_types{CU} . ' (moderate/strong vertical extent or '
           . $cloud_types{TCU} . ')',
    '3' => $cloud_types{CB} . ' (tops not fibrous or in form of an anvil)',
    '4' => $cloud_types{SC} . ' (formed by spreading '
           . $cloud_types{CU} . ')',
    '5' => $cloud_types{SC} . ' (not formed by spreading '
           . $cloud_types{CU} . ')',
    '6' => $cloud_types{ST} . ' (fairly continuous sheet and/or layer)',
    '7' => $cloud_types{SF} . ' or ' . $cloud_types{CF} . ' of bad weather',
    '8' => $cloud_types{CU} . ' and ' . $cloud_types{SC}
           . ' (not formed by spreading ' . $cloud_types{CU} . ')',
    '9' => $cloud_types{CB} . ' (tops fibrous or in form of an anvil)',
);

my %cloud_types_middle = (
    '0' => 'no clouds',
    '1' => $cloud_types{AS} . ' (predominantly semi-transparent)',
    '2' => $cloud_types{AS} . ' (predominantly opaque)',
    '3' => $cloud_types{AC} . ' (not progressively invading the sky)',
    '4' => $cloud_types{AC} . ' (lenticularis)',
    '5' => $cloud_types{AC}
           . ' (progressively invading the sky or semi-transparent)',
    '6' => $cloud_types{AC} . ' (formed by spreading '
           . $cloud_types{CU} . ' or ' . $cloud_types{CB} . ')',
    '7' => $cloud_types{AC} . ' (mainly opaque, not expanding; or with '
           . $cloud_types{AS} . ')',
    '8' => $cloud_types{AC} . ' (with sproutings or tufts)',
    '9' => 'chaotic sky with ' . $cloud_types{AC} . ' at several levels',
);

my %cloud_types_high = (
    '0' => 'no clouds',
    '1' => $cloud_types{CI} . ' (filaments, strands, or hooks)',
    '2' => $cloud_types{CI} . ' (dense, with sproutings, or in tufts)',
    '3' => $cloud_types{CI} . ' (dense, remaining from cumulonimbus anvil)',
    '4' => $cloud_types{CI}
           . ' (hooks or filaments, progressively invading the sky)',
    '5' => $cloud_types{CI}
           . ' and/or ' . $cloud_types{SC}
           . ' (progressively invading the sky, veil <45° above horizon)',
    '6' => $cloud_types{CI}
           . ' and/or ' . $cloud_types{SC}
           . ' (progressively invading the sky, veil >45° above horizon)',
    '7' => $cloud_types{CS} . ' (covering the entire sky)',
    '8' => $cloud_types{CS} . ' (not invading or covering the entire sky)',
    '9' => $cloud_types{CC} . ' (as predominant high cloud)',
);

sub _dir2compass8 {
    my $dir = shift;
    my ($compass, $ii);

    $ii = sprintf "%.0f", $dir / 45; # with rounding
    while ($ii < 0) {
        $ii += 8;
    }
    while ($ii > 7) {
        $ii -= 8;
    }
    return ['N', 'NE', 'E', 'SE', 'S', 'SW', 'W', 'NW']->[$ii];
}

sub _dir2compass16 {
    my $dir = shift;
    my ($compass, $ii);

    $ii = sprintf "%.0f", $dir / 45 * 2; # with rounding
    while ($ii < 0) {
        $ii += 16;
    }
    while ($ii > 15) {
        $ii -= 16;
    }
    return ['N', 'NNE', 'NE', 'ENE', 'E', 'ESE', 'SE', 'SSE',
            'S', 'SSW', 'SW', 'WSW', 'W', 'WNW', 'NW', 'NNW']->[$ii];
}

sub _printVis_FGFS {
    my ($v_na, $vis, $unitThreshold) = @_;
    my ($msg, $gt_lt, $amountM);

    return "(invalid format: '$vis->{invalidFormat}')"
        if exists $vis->{invalidFormat};
    return $v_na if exists $vis->{notAvailable};
    $gt_lt = '';
    $gt_lt = '>=' if exists $vis->{isEqualGreater};
    $gt_lt = '<' if exists $vis->{isLess};
    if (exists $vis->{unitLength}) {
        if ($vis->{unitLength} eq 'M') {
            $amountM = $vis->{distance};
            if ($amountM == 0) {
                $amountM = 50;
                $gt_lt = '<';
            }
            $gt_lt  = '>=' if $amountM == 9999;
        } elsif ($vis->{unitLength} eq 'KM') {
            $amountM = $vis->{distance} * 1000;
        } elsif ($vis->{unitLength} eq 'SM') {
            $amountM = $vis->{distance} * $STATUTE_MILE2METER
        }
    } else {
        $amountM = $vis->{distance} * 100 * $FT2M;
    }
    $msg = $gt_lt;
    if ($unitThreshold && $amountM >= $unitThreshold) {
        $msg .= sprintf "%g km", rnd($amountM / 1000, 0.1);
    } else {
        $msg .= sprintf "%.0f m", rnd($amountM, 10);
    }
    $msg .= ' ' . $vis->{compassDir} if exists $vis->{compassDir};
    $msg .= sprintf "\t\t\t\t\t$gt_lt%g US-miles",
            rnd($amountM / $STATUTE_MILE2METER, 0.1);
    $msg .= ' ' . $vis->{compassDir} if exists $vis->{compassDir};
    $msg .= "\n\t\t\t(No Directional Variation reporting capabilty)"
        if exists $vis->{NDV};
    return $msg;
}

# out: (is_greater, kt, km/h, mph, m/s), all rounded to 1/10
sub _mkWindSpeedVal {
    my ($speed, $speedGreater) = @_;

    return ($speedGreater,
            rnd($speed, 0.1),
            rnd($speed * $NM2KM, 0.1),
            rnd($speed * $NM2KM / $STATUTE_MILE2METER * 1000, 0.1),
            rnd($speed * $NM2KM / 3.6, 0.1));
}

sub _printWind_FGFS {
    my $w = shift;
    my ($msg, $fmt1, $fmt2, $speedKT);

    $fmt1 = ' at %1$s%3$g km/h';
    $fmt2 = '%1$s%2$g kt = %1$s%4$g mph = %1$s%5$g m/s';

    if (exists $w->{notAvailable}) {
        $msg = '(not available)';
    } elsif (exists $w->{invalidFormat}) {
        $msg = "(invalid format: '$w->{invalidFormat}')";
    } else {
        if (  exists $w->{dir} && $w->{dir} == 0
            && exists $w->{speed} && $w->{speed} == 0
            && !exists $w->{gustSpeed})
        {
            $msg = 'calm';
        } else {
            if (exists $w->{dirNotAvailable}) {
                $msg = 'direction not available';
                $msg .= ',' unless exists $w->{speedNotAvailable};
            } elsif (exists $w->{dirVariable}) {
                $msg = 'from variable directions';
            } else {
                $msg = sprintf 'from the %s (%d°)',
                               _dir2compass16($w->{dir}), $w->{dir};
            }

            if (exists $w->{speedNotAvailable}) {
                $msg .= ', speed not available';
            } else {
                if ($w->{unitSpeed} eq 'KT') {
                    $speedKT = $w->{speed};
                } elsif ($w->{unitSpeed} eq 'KMH') {
                    $speedKT = $w->{speed} / $NM2KM;
                } else {
                    $speedKT = $w->{speed} * $METER_PER_SEC2KT;
                }
                $msg .= sprintf "$fmt1\t\t$fmt2",
                                _mkWindSpeedVal($speedKT,
                                                $w->{speedGreater} ? '>' : '');
            }
        }

        if (exists $w->{gustSpeed}) {
            if ($w->{unitSpeed} eq 'KT') {
                $speedKT = $w->{gustSpeed};
            } elsif ($w->{unitSpeed} eq 'KMH') {
                $speedKT = $w->{gustSpeed} / $NM2KM;
            } else {
                $speedKT = $w->{gustSpeed} * $METER_PER_SEC2KT;
            }
            $msg .= sprintf "\n\t\t\twith gusts$fmt1\t\t\t$fmt2",
                            _mkWindSpeedVal($speedKT,
                                            $w->{gustSpeedGreater} ? '>' : '');
        }
    }

    if (exists $w->{windVarLeft}) {
        $msg .= sprintf "\n\t\t\tvariable from %s to %s (%d°--%d°)",
                        _dir2compass16($w->{windVarLeft}),
                        _dir2compass16($w->{windVarRight}),
                        $w->{windVarLeft}, $w->{windVarRight};
    }

    return "$msg\n";
}

sub _printCloudCover {
    return {  SKC => $strict_fgfs ? 'clear skies' : 'sky clear',
              NSC => $strict_fgfs ? 'clear skies' : 'nil significant clouds',
              CLR => $strict_fgfs ? 'clear skies' : 'no clouds below 10000 ft',
              NCD => 'no cloud detected',
             '///'=> '(not available)',
              FEW => 'few clouds',
              SCT => 'scattered clouds',
              BKN => 'broken clouds',
              OVC => 'sky overcast' }->{(shift)};
}

sub _printWeather {
    my ($w, $recent) = @_;
    my ($msg, $int, $vc, $desc, $phen, $intensity, $delim);

    return '' unless defined $w;

    return "(not available)" if exists $w->{notAvailable};
    return "(invalid format: '$w->{invalidFormat}')"
        if exists $w->{invalidFormat};
    return 'nil significant weather' if exists $w->{NSW};

    return 'tornado/waterspout' if exists $w->{tornado};

    # EXTENSION:
    # allow FZ with (BC|PR)FG (Austria, WMO-No. 306, Vol. II)
    $delim = '';
    if (exists $w->{intensity}) {
        $int = $w->{intensity} eq 'LIGHT' ? '-' : '+';
    } else {
        $int = '';
    }
    $intensity = { ''  =>      defined $recent
                            || exists $w->{inVicinity}
                            || !exists $w->{phenomenon}
                            || join('', @{$w->{phenomenon}}) !~ 'DZ|RA|SN|SG|PL|GR|GS|DS|SS'
                          ? '' :'moderate',
                   '+' => 'heavy',
                   '-' => 'light' }->{$int};
    if (   $intensity
        && (   !exists $w->{descriptor}
            || join('', @{$w->{descriptor}}) !~ 'TS'))
    {
        $msg .= $intensity;
        $delim = ' ';
    }
    if (exists $w->{descriptor}) {
        for (@{$w->{descriptor}}) {
            $msg .= $delim . $weather_descr{$_};
            $delim = ' ';
        }
        if ($intensity && join('', @{$w->{descriptor}}) =~ 'TS') {
            $msg .= $delim . $intensity;
            $delim = ' ';
        }
    }

    for (exists $w->{phenomenon} ? @{$w->{phenomenon}} : ()) {
        $msg .= $delim . $weather_types{$_};
        $delim = ' ';
    }
    $msg .= ' ' . $keywords{VC} if exists $w->{inVicinity};

    $msg =~ s/(showers) of (snow|rain)$/$2 $1/;

    return $msg;
}

sub _printCloud {
    my ($metar) = @_;
    my ($hdr, $msg, $ii);

    return '' unless exists $metar->{cloud};

    for ($strict_fgfs ? () : @{$metar->{cloud}}) {
        if (exists $_->{isCeiling}) {
            $msg .= "ceiling:\t\tat " . $_->{cloudBase} * 100 . ' ft';
            $msg .=   "\t\t\t\t"
                    . rnd($_->{cloudBase} * 100 * $FT2M, 10) . " m\n";
            last;
        }
    }

    $hdr .= "Sky condition:\t\t";
    for (@{$metar->{cloud}}) {
        $msg .= $hdr;
        $hdr = "\t\t\t";
        if (exists $_->{notAvailable}) {
            $msg .= '(not available)';
        } elsif (exists $_->{invalidFormat}) {
            $msg .= "(invalid format: '$_->{invalidFormat}')";
        } elsif (exists $_->{noClouds}) {
            $msg .= _printCloudCover $_->{noClouds};
        } else {
            $msg .= _printCloudCover $_->{cloudCover};
        }
        if (exists $_->{cloudBase}) {
            $msg .= ' at ' . $_->{cloudBase} * 100 . ' ft'
        } elsif (exists $_->{baseBelowStation}) {
            $msg .= ", $keywords{BBLO}";
        }
        if (exists $_->{cloudTypeNotAvailable}) {
            $msg .= ' (cloud type not observable)';
        } elsif (exists $_->{cloudType}) {
            $msg .= ' (' . $cloud_types{$_->{cloudType}} . ')';
        }
        $msg .=   "\t\t\t"
                . rnd($_->{cloudBase} * 100 * $FT2M, 10) . " m"
            if exists $_->{cloudBase};
        $msg .= "\n";
    }

    return $msg;
}

sub _mkTempVal {
    my ($t, $t_na, $fmt_t, $d_na, $fmt_d, $fmt_rh) = @_;
    my (@ret, $sign, $temp, $dew);

    @ret = ('', '', '');
    if (exists $t->{air}{temp}) {
        if ($t->{air}{unitTemp} eq 'C') {
            $ret[0] = sprintf $fmt_t, $t->{air}{temp},
                                      rnd($t->{air}{temp} * 1.8 + 32, 0.1);
        } else {
            $ret[0] = sprintf $fmt_t, rnd(($t->{air}{temp} - 32) / 1.8, 0.1),
                                      $t->{air}{temp};
        }
    } else {
        $ret[0] = $t_na;
    }
    if (exists $t->{dewpoint}) {
        if (exists $t->{dewpoint}{temp}) {
            if ($t->{dewpoint}{unitTemp} eq 'C') {
                $ret[1] = sprintf $fmt_d, $t->{dewpoint}{temp},
                                  rnd($t->{dewpoint}{temp} * 1.8 + 32, 0.1);
            } else {
                $ret[1] = sprintf $fmt_d,
                                  rnd(($t->{dewpoint}{temp} - 32) / 1.8, 0.1),
                                  $t->{dewpoint}{temp} - 32;
            }
        } else {
            $ret[1] = $d_na;
        }
    }
    $ret[2] = sprintf $fmt_rh, $t->{relHumid1}
        if exists $t->{relHumid1};
    return @ret;
}

sub _printTemp_FGFS {
    my ($t, $hdr1, $hdr2) = @_;
    $hdr1 = "Temperature:\t\t" unless $hdr1;
    $hdr2 = "Dewpoint:\t\t" unless $hdr2;
    my @ret = _mkTempVal $t,
                        '', "$hdr1%s °C\t\t\t\t\t%s °F\n",
                        '', "$hdr2%s °C\t\t\t\t\t%s °F\n",
                        "relative humidity:\t%.0f%%\n";
    return $ret[0] . $ret[1] . $ret[2];
}

sub _mkRwyVisVal {
    my ($v, $v_gt, $v_lt) = @_;
    my $visVal;

    $visVal = $v->{distance};
    $visVal *= $FT2M if $v->{unitLength} eq 'FT';
    return (undef,
            exists $v->{isEqualGreater} ? $v_gt
                                        : (exists $v->{isLess} ? $v_lt : ''),
            rnd($visVal, 10),
            rnd($visVal / 1000, 0.1),
            rnd($visVal / $STATUTE_MILE2METER, 0.1));
}

sub _printVisRwy {
    my ($vis, $from_to) = @_;
    my (@ret, $msg);

    return ("(visibility not available)\n") if exists $vis->{notAvailable};

    @ret = _mkRwyVisVal $vis, '>=', '<';

    $msg = $from_to;
    $msg .= $ret[1];
    if ($ret[2] >= 1000) {
        $msg .= sprintf "%g km", $ret[3];
    } else {
        $msg .= sprintf "%.0f m", $ret[2];
    }
    $msg .= "\t\t\t\t\t" . $ret[1];
    $msg .= sprintf "%g US-miles\n", $ret[4];
    return $msg;
}

sub _printRwyState {
    my $r = shift;
    my ($msg);

    $msg = '';
    if (exists $r->{SNOCLO}) {
        $msg .= "aerodrome closed for snow\n";
    } else {
        my $delim;
        if (exists $r->{rwyDesigAll}) {
            $msg .= "state all runways";
        } elsif (exists $r->{rwyDesigRep}) {
            $msg .= "state runway REP";
        } else {
            $msg .= 'state runway ' . $r->{rwyDesig};
        }
        $msg .= ":\t";
        $msg .= '(not available)'
            if    exists $r->{depositType}
               && exists $r->{depositExtent}
               && exists $r->{depositDepth}
               && exists $r->{friction}
               && exists $r->{depositType}{notAvailable}
               && exists $r->{depositExtent}{notAvailable}
               && exists $r->{depositDepth}{notAvailable}
               && exists $r->{friction}{notAvailable};
        $delim = '';
        if (exists $r->{cleared}) {
            $msg .= 'cleared';
            $delim = ', ';
        }
        if (   exists $r->{depositType}
            && exists $r->{depositType}{depositTypeVal})
        {
            $msg .= $delim
                    . { 0 => 'clear and dry',
                        1 => 'damp',
                        2 => 'wet or puddles',
                        3 => 'frost',
                        4 => 'dry snow',
                        5 => 'wet snow',
                        6 => 'slush',
                        7 => 'ice',
                        8 => 'compacted snow',
                        9 => 'frozen ridges',
                      }->{$r->{depositType}{depositTypeVal}};
            $delim = ', ';
        }
        if (   exists $r->{depositExtent}
            && exists $r->{depositExtent}{depositExtentVal})
        {
            $msg .= $delim
                    . { 0 => '0%',
                        1 => '1-10%',
                        2 => '11-25%',
                        5 => '26-50%',
                        9 => '51-100%',
                      }->{$r->{depositExtent}{depositExtentVal}}
                    . ($strict_fgfs ? '' : ' of runway covered');
            $delim = ', ';
        }
        if (   exists $r->{depositDepth}
            && exists $r->{depositDepth}{depositDepthVal})
        {
            my $dh_str = $strict_fgfs ? '' : 'deposit depth ';
            $msg .= $delim;
            if ($r->{depositDepth}{depositDepthVal} == 0) {
                $msg .= $dh_str . '<1 mm';
            } elsif ($r->{depositDepth}{depositDepthVal} < 91) {
                $msg .= $dh_str
                        . ($r->{depositDepth}{depositDepthVal} + 0) . ' mm';
            } elsif ($r->{depositDepth}{depositDepthVal} < 99) {
                $msg .= $dh_str
                      . 50 * ($r->{depositDepth}{depositDepthVal} - 90) . ' mm';
            } else {
                $msg .= 'runway not in use';
            }
            $delim = ', ';
        }
        if (exists $r->{friction}) {
            if (exists $r->{friction}{invalidFormat}) {
                $msg .= $delim . "(invalid format: "
                               . "'$r->{friction}{invalidFormat}')";
            } elsif (exists $r->{friction}{coefficient}) {
                $msg .= $delim . 'friction: '
                        . sprintf "%.2f", $r->{friction}{coefficient} / 100;
            } elsif (exists $r->{friction}{brakingAction}) {
                $msg .=  $delim
                         . { 91 => 'poor braking action',
                             92 => 'poor/medium braking action',
                             93 => 'medium braking action',
                             94 => 'medium/good braking action',
                             95 => 'good braking action',
                           } ->{$r->{friction}{brakingAction}};
            } elsif (exists $r->{friction}{unreliable}) {
                $msg .=  $delim . 'friction: unreliable measurement';
            }
        }
        $msg .= "\n";
    }
    return $msg;
}

sub _printColourCode {
    my $c = shift;
    my ($msg, $hdr);
    my $cc = { BLUplus    => 'ceiling 20000 ft, visibility 8 km',
               BLU        => 'ceiling 2500 ft, visibility 8 km',
               WHT        => 'ceiling 1500 ft, visibility 5 km',
               GRN        => 'ceiling 700 ft, visibility 3700 m',
               YLO1       => 'ceiling 500 ft, visibility 2500 m',
               YLO2       => 'ceiling 300 ft, visibility 1600 m',
               YLO        => 'ceiling 300 ft, visibility 1600 m',
               AMB        => 'ceiling 200 ft, visibility 800 m',
               RED        => 'ceiling <200 ft, visibility <800 m',
               FCSTCANCEL => 'forecast cancelled',
    };

    $hdr .= "Colour code:\t\t";
    if (exists $c->{BLACK}) {
        $msg .= $hdr . "airport closed for technical reasons";
        $hdr = "\t\t\t";
    }
    if (exists $c->{currentColour}) {
        $msg .= "\n" if exists $c->{BLACK};
        $msg .= $hdr . $cc->{$c->{currentColour}}
    }
    if (exists $c->{predictedColour}) {
        $msg .= "\n\t\t\tpredicted:\n\t\t\t";
        if (   exists $c->{currentColour}
            && $c->{currentColour} eq $c->{predictedColour})
        {
            $msg .= 'no change of ceiling or visibility';
        } else {
            $msg .= $cc->{$c->{predictedColour}};
        }
    }

    return $msg;
}

sub _printQuadrants {
    my $q = shift;
    my ($delim, $txt);
    $delim = '';
    for (@$q) {
        $txt .= $delim . { 1 => 'first',
                           2 => 'second',
                           3 => 'third',
                           4 => 'forth',
                         }->{$_};
        $delim = ' and ';
    }
    return " $txt quadrant";
}

sub _printLocations {
    my ($r, $is_dir) = @_;
    my ($delim1, $is_first, $txt, $is_dir_txt);

    my %comp_dir = (
        N     => 'north',
        E     => 'east',
        S     => 'south',
        W     => 'west',
    );

    my %phen_loc = (
          N => $comp_dir{N},
        NNE => join('-', @comp_dir{'N', 'N', 'E'}),
         NE => join('-', @comp_dir{'N', 'E'}),
        ENE => join('-', @comp_dir{'E', 'N', 'E'}),
          E => $comp_dir{E},
        ESE => join('-', @comp_dir{'E', 'S', 'E'}),
         SE => join('-', @comp_dir{'S', 'E'}),
        SSE => join('-', @comp_dir{'S', 'S', 'E'}),
          S => $comp_dir{S},
        SSW => join('-', @comp_dir{'S', 'S', 'W'}),
         SW => join('-', @comp_dir{'S', 'W'}),
        WSW => join('-', @comp_dir{'W', 'S', 'W'}),
          W => $comp_dir{W},
        WNW => join('-', @comp_dir{'W', 'N', 'W'}),
         NW => join('-', @comp_dir{'N', 'W'}),
        NNW => join('-', @comp_dir{'N', 'N', 'W'}),
      ALQDS => $keywords{ALQDS},
      OHD   => $keywords{OHD},
    );

    $is_first = 1;
    $is_dir_txt = $is_dir ? ' to the' : '';
    for my $l (@$r) {
        my $had_grid;

        $txt .= (   scalar @$l == 1
                   && exists $l->[0]{locationSpec}
                   && $l->[0]{locationSpec} eq 'STNRY') ? ',' : ','
            unless $is_first;
        $is_first = 0;

        $delim1  = '';
        for (@$l) {
            $txt .= $delim1;
            $delim1 = ' through';
            $txt .= ' ' . $keywords{DSNT} if exists $_->{isDistant};
            $txt .= ' ' . $keywords{VC} if exists $_->{inVicinity};
            $txt .= ' ' . $keywords{$_->{locationSpec}}
                if exists $_->{locationSpec};
            $txt .= ' ' . "@keywords{'OBSCG', 'MTNS'}"
                if exists $_->{obscgMtns};
            $txt .= ' ' . $_->{distance}
                    . ($_->{unitLength} eq 'SM' ? ' US-miles' : ' km')
                if exists $_->{distance};
            if (exists $_->{compassDir}) {
                $txt .= $is_dir_txt;
                $txt .= ' ' . $keywords{GRID}
                    if exists $_->{isGrid} && !$had_grid;
                $had_grid = 1;
                $txt .= ' ' . $phen_loc{$_->{compassDir}};
                $is_dir_txt = '';
            }
            $txt .= _printQuadrants $_->{quadrant}
                if exists $_->{quadrant};
            $txt .= ' ' . $keywords{QUAD} if exists $_->{isQuadrant};
        }
    }
    return $txt;
}

sub _printCloudOpacityLvl {
    my $c = shift;
    my $msg;

    $msg = "phenomenon w. opacity:\t" . $c->{eights} . '/8 ';
    $msg .= $cloud_types{$c->{cloudType}}
        if exists $c->{cloudType};
    $msg .= _printWeather $c->{weather}, 1;
    $msg .= ' at ' . $c->{cloudBase} * 100 . ' ft';
    $msg .= _printLocations $c->{locationAndList}
        if exists $c->{locationAndList};
    $msg .= "\t\t" . rnd($c->{cloudBase} * 100 * $FT2M, 10) . " m\n";
    return $msg;
}

sub _printPhenomOpacityList {
    my $c = shift;
    my ($txt, $hdr);

    $hdr = "phenomenon w. opacity:\t";
    for (@{$c->{phenomOpacity}}) {
        $txt .= $hdr . $_->{eights} . "/8 ";
        if (exists $_->{weather}) {
            $txt .= _printWeather $_->{weather}, 1;
        } else {
            $txt .= $cloud_types{$_->{cloudType}};
        }
        $hdr = "\n\t\t\t";
    }
    $txt .= $hdr . $cloud_types{$c->{cloudTypeAsoctd}} . " $keywords{ASOCTD}"
        if exists $c->{cloudTypeAsoctd};
    $txt .= $hdr . $cloud_types{$c->{cloudTypeEmbd}} . " $keywords{EMBD}"
        if exists $c->{cloudTypeEmbd};
    $txt .= "\n";
    return $txt;
}

sub _printUSTemp {
    my ($t, $hdr) = @_;

    return sprintf "$hdr%s °C\t\t\t\t\t%s °F\n",
                    $t->{temp}, rnd($t->{temp} * 1.8 + 32, 0.1)
        if $t->{unitTemp} eq 'C';
    return sprintf "$hdr%.1f °C\t\t\t\t\t%s °F\n",
                    rnd(($t->{temp} - 32) / 1.8, 0.1), $t->{temp};
}

sub _mkQNHVal {
    my $qnh = shift;
    my ($qnhinHg);

    return () unless    exists $qnh->{hPa}
                     || exists $qnh->{inHg}
                     || exists $qnh->{mmHg};

    return (0, $qnh->{hPa}, rnd($qnh->{hPa} / $INHG2HPA, 0.01))
        if exists $qnh->{hPa};
    return (0, rnd($qnh->{inHg} * $INHG2HPA, 1), $qnh->{inHg})
        if exists $qnh->{inHg};
    $qnhinHg = $qnh->{mmHg} / ($FT2M / 12 * 1000);
    return (0, rnd($qnhinHg * $INHG2HPA, 1), rnd($qnhinHg, 0.01));
}


sub _printPressure {
    my ($qnh, $hdr, $fmt) = @_;
    my @ret;

    return $hdr . "(not available)\n" if exists $qnh->{notAvailable};
    return "$hdr(invalid format: '$qnh->{invalidFormat}')\n"
        if exists $qnh->{invalidFormat};
    @ret = _mkQNHVal $qnh;
    return $hdr . sprintf $fmt, $ret[1], $ret[2];
}

sub _printTime {
    my $time = shift;

    $time =~ '(..)?(..)';
    return ($2 + 0) . " minutes after the hour" unless defined $1;
    return "$1:$2 UTC";
}

sub _printPhenomDescr {
    my ($descr, $pre, $post) = @_;
    my ($msg, $hdr);

    return '' unless defined $descr;

    $msg = '';
    $hdr = $pre;
    for (@$descr) {
        $msg .= $hdr .
            { 'isFrequent'     =>  $keywords{FRQ},
              'isOccasional'   =>  $keywords{OCNL},
              'isIntermittent' =>  $keywords{INTMT},
              'isContinuous'   =>  $keywords{CONS},
              'isThick'        =>  $keywords{THK},
              'isPrettyThick'  =>  "@keywords{'PR', 'THK'}",
              'isVeryThick'    =>  "@keywords{'VRY', 'THK'}",
              'isThin'         =>  $keywords{THN},
              'isPrettyThin'   =>  "@keywords{'PR', 'THN'}",
              'isVeryThin'     =>  "@keywords{'VRY', 'THN'}",
              'isLight'        =>  $keywords{LGT},
              'isPrettyLight'  =>  "@keywords{'PR', 'LGT'}",
              'isVeryLight'    =>  "@keywords{'VRY', 'LGT'}",
              'isFeeble'       =>  $keywords{FBL},
              'isPrettyFeeble' =>  "@keywords{'PR', 'FBL'}",
              'isVeryFeeble'   =>  "@keywords{'VRY', 'FBL'}",
              'isModerate'     =>  $keywords{MDT},
              'isLow'          =>  $keywords{LOW},
              'isLower'        =>  $keywords{LWR},
              'isIsolated'     =>  $keywords{ISOL},
              'isConvective'   =>  $keywords{CVCTV},
              'isDissipated'   =>  $keywords{DSIPTD},
              'inPastHour'     =>  $keywords{PAST_HR},
              'baseBelowStation' => $keywords{BBLO},
              'isAloft'        =>  $keywords{ALOFT},
              'isAround'       =>  $keywords{ARND},
              'isFreezing'     =>  $weather_descr{FZ},
              'isPatchy'       =>  $keywords{P},
            }->{$_};
        $hdr = ' and ';
    }
    return $msg . $post;
}

sub printMetar_FGFS {
    my ($metar, $opt_m, $strict) = @_;
    my $msg = '';
    my (@ret, $ii, $td, $need_nl, $hdr, $delim);

    $strict_fgfs = $strict;

    $msg .= "msg: $metar->{msg}\n" x $opt_m if defined $opt_m;
    $msg .= $metar->{WARNING} if exists $metar->{WARNING} && !$strict_fgfs;
    if (exists $metar->{SPECI}) {
        $msg .= "SPECI Report";
    } else {
        $msg .= "METAR Report";
    }
    if (exists $metar->{reportModifier}) {
        my $rm = $metar->{reportModifier};
        $msg .= "\t\t(" . { NIL  => 'report missing',
                            AUTO => 'automatically generated',
                            COR  => 'manually corrected',
                            RTD  => 'routine delayed',
                            P    => 'segmented, ',
                            RR   => 'delayed',
                            CC   => 'corrected',
                            AA   => 'amended',
                          }->{$rm->{modifierType}};
        if ($rm->{modifierType} eq 'P') {
            $msg .= 'last ' if exists $rm->{isLastSegment};
            $msg .= "segment $rm->{segment}";
        }
        $msg .= ', bulletin over 24 hours after observation'
            if exists $rm->{over24hLate};
        $msg .= ', bulletin sequence lost'
            if exists $rm->{sequenceLost};
        $msg .= ", bulletin sequence $rm->{bulletinSeq}"
            if exists $rm->{bulletinSeq};
        $msg .= ')';
    }
    $msg .= "\n============\n";
    $msg .= "Airport-Id:\t\t" . $metar->{obsStation}{id} . "\n"
        if exists $metar->{obsStation};

    if (exists $metar->{obsTime} || exists $metar->{issueTime}) {
        my $rep_time;

        if (exists $metar->{obsTime}) {
            $msg .= "Report time:\t\t";
            $rep_time = $metar->{obsTime};
        } else {
            $msg .= "Issue time:\t\t";
            $rep_time = $metar->{issueTime};
        }
        if (exists $rep_time->{invalidFormat}) {
            $msg .= "(invalid format: '$rep_time->{invalidFormat}')\n";
        } else {
            $msg .= sprintf "on the %d., %02d:%02d UTC\n", @$rep_time{'day', 'hour', 'minute'};
        }
    }
    if (exists $metar->{fcstPeriod}) {
        $msg .= sprintf "Forecast period:\t%d., %02d:00 - %02d:00 UTC",
                        @{$metar->{fcstPeriod}}{'day', 'hourFrom', 'hourTill'};
        $msg .= ' on the following day'
            if $metar->{fcstPeriod}{hourFrom} >= $metar->{fcstPeriod}{hourTill};
        $msg .= "\n";
    }

    $msg .= "forecast not available:\t" .
                { NOOBS           => 'no observation',
                  INSUFFICIENTOBS => 'insufficient observation'
                }->{$metar->{fcstNotAvbl}{fcstNotAvblReason}}
            . "\n"
        if exists $metar->{fcstNotAvbl};

    $msg .= "Wind:\t\t\t" . _printWind_FGFS $metar->{sfcWind}{wind}
        if exists $metar->{sfcWind};

    $msg .= $keywords{CAVOK} . "\n"
        if !$strict_fgfs && exists $metar->{CAVOK};

    $msg .= "Visibility:\t\t"
            . _printVis_FGFS('(not available)', $metar->{visPrev}, 1000) . "\n"
        if exists $metar->{visPrev};
    $msg .= "\t\t\t" . _printVis_FGFS(0, $metar->{visMin}, 1000) . "\n"
        if exists $metar->{visMin};

    for (exists $metar->{visRwy} ? @{$metar->{visRwy}} : ()) {
        $need_nl = 0;
        $msg .= "visibility runway " . $_->{rwyDesig} . ":\t";
        if (exists $_->{notAvailable}) {
            $msg .= "(not available)\n";
        } else {
            $msg .= _printVisRwy($_->{RVR},
                    (exists $_->{RVRVariations} ? 'from ' : ''));
            if (exists $_->{RVRVariations}) {
                $need_nl = 1;
                $msg .= "\t\t\t" . _printVisRwy($_->{RVRVariations}, 'to ');
            }
        }
        if (!$strict_fgfs && exists $_->{visTrend}) {
            $need_nl = 1;
            $msg .= "\ttrend:\t\t" . $keywords{$_->{visTrend}} . "\n";
        }
        $msg .= "\n" if $need_nl;
    }
    $msg .= $keywords{RVRNO} . "\n"
        if !$strict_fgfs && exists $metar->{RVRNO};

    if (exists $metar->{weather}) {
        $delim = "Weather:\t\t";
        for (@{$metar->{weather}}) {
            $msg .= $delim . _printWeather $_;
            $delim = ', ';
        }
        $msg .= "\n";
    }

    $msg .= _printCloud $metar;

    $msg .= "Vert. visibility:\t"
            . _printVis_FGFS('impossible to determine', $metar->{visVert}, 1000)
            . "\n"
        if exists $metar->{visVert};

    $msg .= _printTemp_FGFS($metar->{temperature})
        if exists $metar->{temperature};

    if (exists $metar->{QNH}) {
        @ret = _mkQNHVal $metar->{QNH};
        $msg .= sprintf "Pressure:\t\t%.0f hPa\t\t\t\t%g in. Hg\n",
                        $ret[1], $ret[2]
            if $ret[1];
    }

    if (!$strict_fgfs && exists $metar->{somePressure}) {
        @ret = _mkQNHVal $metar->{somePressure};
        $msg .= sprintf "Some pressure:\t\t%.0f hPa\t\t\t\t%g in. Hg\n",
                        $ret[1], $ret[2]
            if $ret[1];
    }

    for (exists $metar->{windShear} ? @{$metar->{windShear}} : ()) {
        $msg .=   (exists $_->{rwyDesigAll}
                    ? 'all runways' : "runway $_->{rwyDesig}")
                . ":\t\tcritical wind shear\n";
    }

    for (exists $metar->{rwyState} ? @{$metar->{rwyState}} : ()) {
        $msg .= _printRwyState $_;
    }

    if (   $strict_fgfs
        && exists $metar->{trend}
        && $metar->{trend}[0]{trendType} eq 'NOSIG')
    {
        for (exists $metar->{trend}[0]{rwyState}
             ? @{$metar->{trend}[0]{rwyState}} : ())
        {
            $msg .= _printRwyState $_;
        }
    }

    if (!$strict_fgfs && exists $metar->{recWeather}) {
        $delim = "recent weather:\t\t";
        for (@{$metar->{recWeather}}) {
            $msg .= $delim . _printWeather($_, 1);
            $delim = ', ';
        }
        $msg .= "\n";
    }

    $msg .= _printColourCode($metar->{colourCode}) . "\n"
        if !$strict_fgfs && $metar->{colourCode};

    if (!$strict_fgfs && exists $metar->{cloudMaxCover}) {
        $msg .= "max. cloud cover:\t";
        if (exists $metar->{cloudMaxCover}{SKC}) {
            $msg .= _printCloudCover 'SKC';
        } else {
            $msg .= _printCloudCover $metar->{cloudMaxCover}{cloudCover};
        }
        $msg .= "\n";
    }

    if (exists $metar->{NEFO_PLAYA} && !$strict_fgfs) {
        $msg .= "NEFO PLAYA:\t\t";
        if (exists $metar->{NEFO_PLAYA}{SKC}) {
            $msg .= _printCloudCover 'SKC';
        } else {
            $msg .= 'at ' . $metar->{NEFO_PLAYA}{cloudBaseFT} . " ft\t\t\t"
                 . rnd($metar->{NEFO_PLAYA}{cloudBaseFT} * $FT2M, 10) . ' m';
        }
        $msg .= "\n";
    }

    if (exists $metar->{RH} && !$strict_fgfs) {
        $msg .= "$keywords{RH}:\t" . $metar->{RH}{relHumid} . "%\n";
    }

    for (exists $metar->{rwyWind} && !$strict_fgfs ? @{$metar->{rwyWind}} : ()){
        $msg .= "wind for runway $_->{rwyDesig}:\t"
                . _printWind_FGFS $_->{wind};
    }

    $hdr = "trends within the next 2 hours:\n===============================\n";
    for my $td (exists $metar->{trend} && !$strict_fgfs ? @{$metar->{trend}} : ())
    {
        $msg .= "\n$hdr";
        $hdr = '';
        $delim = '';
        if ($td->{trendType} eq 'NOSIG') {
            $msg .= $keywords{NOSIG};
        } else {
            $msg .= "with $td->{probability}% probability"
                if $td->{trendType} eq 'PROB';
            if (exists $td->{trendTime1}) {
                $delim = ' ';
                $msg .= ' ' if $td->{trendType} eq 'PROB';
                $msg .= $keywords{$td->{trendTime1}{timeSpec}} . ' '
                        . $td->{trendTime1}{hour} . ':'
                        . $td->{trendTime1}{minute} . ' UTC';
                $msg .= ' '. $keywords{$td->{trendTime2}{timeSpec}} . ' '
                        . $td->{trendTime2}{hour} . ':'
                        . $td->{trendTime2}{minute} . ' UTC'
                    if exists $td->{trendTime2};
            }
            $msg .= $td->{trendType} eq 'PROB'
                                   ? ':' : $delim . $keywords{$td->{trendType}};
        }
        $msg .= "\n";

        $msg .= "Wind:\t\t\t" . _printWind_FGFS $td->{sfcWind}{wind}
            if exists $td->{sfcWind};

        $msg .= $keywords{CAVOK} . "\n"
            if exists $td->{CAVOK};

        $msg .= "Visibility:\t\t" . _printVis_FGFS('(not available)',
                                                  $td->{visPrev}, 1000) . "\n"
            if exists $td->{visPrev};

        $msg .= "Vert. visibility:\t"
               . _printVis_FGFS('impossible to determine', $td->{visVert}, 1000)
               . "\n"
            if exists $td->{visVert};

        if (exists $td->{weather}) {
            $delim = "Weather:\t\t";
            for (@{$td->{weather}}) {
                $msg .= $delim . _printWeather $_;
                $delim = ', ';
            }
            $msg .= "\n";
        }

        $msg .= _printCloud $td;

        for (exists $td->{rwyState} ? @{$td->{rwyState}} : ()) {
            $msg .= _printRwyState $_;
        }

        $msg .= _printColourCode($td->{colourCode}) . "\n"
            if $td->{colourCode};
    }

    if (!$strict_fgfs && exists $metar->{remark}) {
        my $rem = $metar->{remark};
        $msg .= "\nRemarks:\n========\n";

        for (exists $metar->{remark} ? @{$metar->{remark}} : '') {
            my $r = $_;
            for (sort keys %$r) {
                my $e = $r->{$_};
                if ($_ eq 'notRecognised') {
                    $msg .= "NOT RECOGNISED:\t\t$e->{s}\n";
                } elsif ($_ eq 'obsStationType') {
                    $msg .= 'automated station with'
                            . { AO1  => 'out precipitation discriminator',
                                AO2  => ' precipitation discriminator',
                                AO2A =>
                          ' precipitation discriminator and manual augmentation'
                       }->{$e->{stationType}};
                    $msg .= "\n";
                } elsif ($_ eq 'visibilityAtLoc') {
                    $msg .= $keywords{VIS} .  ' at';
                    if (exists $e->{locationAt}) {
                        $msg .= ' sea'
                            if $e->{locationAt} eq 'MAR';
                        $msg .= " " . $keywords{$e->{locationAt}}
                            if $e->{locationAt} =~ 'SFC|TWR';
                    } elsif (exists $e->{rwyDesig}) {
                        $msg .= " $keywords{APCH}" if exists $e->{isApproach};
                        $msg .= ' runway ' . $e->{rwyDesig};
                    }
                    $msg .= ":\t";
                    $msg .= _printVis_FGFS(0, $e->{visibility}, 1000) ."\n";
                } elsif ($_ eq 'visVar1') {
                    $msg .= "visibility varies from:\t"
                            . _printVis_FGFS(0, $e, 1000) . "\n";
                } elsif ($_ eq 'visVar2') {
                    $msg .= "\t\tto:\t" . _printVis_FGFS(0, $e, 1000) . "\n";
                } elsif ($_ eq 'cloudMaxCover') {
                    $msg .= "max. cloud cover:\t"
                            . _printCloudCover($e->{cloudCover}) . "\n";
                } elsif ($_ eq 'cloudOpacityLvl') {
                    $msg .= _printCloudOpacityLvl $e;
                } elsif ($_ eq 'rwyState') {
                    $msg .= _printRwyState $e;
                } elsif ($_ eq 'recWeather') {
                    $msg .= "recent weather:\t\t"
                            . _printWeather($e->[0], 1) . "\n";
                } elsif ($_ eq 'visMin') {
                    $msg .= "minimum visibility:\t"
                            . _printVis_FGFS(0, $e, 1000) . "\n";
                } elsif ($_ eq 'visListLoc') {
                    $delim = $keywords{VIS};
                    for (@{$e->{arr}}) {
                        $msg .= $delim;
                        $delim = "\n" . $keywords{VIS};
                        $msg .= _printLocations($_->{locationAndList},1) . ":\t"
                                . _printVis_FGFS(0, $_->{visibility}, 1000);
                    }
                    $msg .= "\n";
                } elsif ($_ eq 'correctedAt') {
                    $msg .= "report corrected at:\t$e->{hour}:$e->{minute} UTC\n";
                } elsif ($_ eq 'colourCode') {
                    $msg .= _printColourCode($e) . "\n";
                } elsif ($_ eq 'QNH') {
                    $msg .= _printPressure $e,
                                     "Pressure:\t\t",
                                     "%.0f hPa\t\t\t\t%g in. Hg\n";
                } elsif ($_ =~ '^SLP') {
                    $msg .= _printPressure $e,
                                     $keywords{SLP} . ":\t",
                                     "%.1f hPa\t\t\t\t%g in. Hg\n";
                } elsif ($_ eq 'regQNH') {
                    $msg .= _printPressure $e,
                                          "reg. QNH:\t\t",
                                          "%.0f hPa\t\t\t\t%g in. Hg\n";
                } elsif ($_ eq 'QFE') {
                    $msg .= _printPressure $e,
                                          "QFE pressure:\t\t",
                                          "%.0f hPa\t\t\t\t%g in. Hg\n";
                } elsif ($_ eq 'QFF') {
                    $msg .= _printPressure $e,
                                          "QFF pressure:\t\t",
                                          "%.0f hPa\t\t\t\t%g in. Hg\n";
                } elsif ($_ eq 'reportConcerns') {
                    $msg .= "report concerns:\t\t"
                           . { M => 'deterioration',
                               B => 'improvement'
                             }->{$e->{change}}
                           . ' of '
                           . { 0 => 'gust',
                               1 => $keywords{WND},
                               2 => $keywords{VIS},
                               3 => $keywords{CLD},
                               4 => $keywords{PCPN},
                               5 => 'pressure',
                               6 => 'state of sea or swell',
                               7 => $weather_types{DS} . ', ' .
                                    $weather_types{SS} . ' or ' .
                                    $weather_descr{BL} . ' ' .
                                    $weather_types{SN},
                               8 => $weather_types{TS},
                               9 => $weather_types{SQ} . ' or tornado',
                             }->{$e->{subject}}
                           . " conditions\n";
                } elsif ($_ eq 'RH') {
                    $msg .= "$keywords{RH}:\t" . $e->{relHumid} . "%\n";
                } elsif ($_ eq 'AI') {
                    $msg .= "$keywords{AI}:\t\t\t$e->{AIVal}\n";
                } elsif ($_ eq 'SST' || $_ eq 'OAT') {
                    $msg .= _printUSTemp $e, "$keywords{$_}:\t";
                } elsif ($_ eq 'temp6hMax') {
                    $msg .= _printUSTemp $e, "6h max. temperature:\t";
                } elsif ($_ eq 'temp6hMin') {
                    $msg .= _printUSTemp $e, "6h min. temperature:\t", $e;
                } elsif ($_ eq 'temp24h') {
                    $msg .= _printUSTemp $e->{temp24hMax},
                                         "24h max. temperature:\t";
                    $msg .= _printUSTemp $e->{temp24hMin},
                                         "24h min. temperature:\t"
                        if exists $e->{temp24hMin};
                } elsif ($_ eq 'tempHourly') {
                    $msg .= _printTemp_FGFS $e, "1h avg. temperature:\t",
                                                "1h avg. dewpoint:\t";
                } elsif ($_ eq 'peakWind') {
                    $msg .= 'peak wind';
                    $msg .= ' at '
                            . _printTime((exists $e->{hour} ? $e->{hour} : '')
                                         . $e->{minute})
                        if exists $e->{minute};
                    $msg .= ":\t" . _printWind_FGFS $e->{wind};
                } elsif ($_ eq 'visRwy') {
                    $msg .= "visibility runway " . $e->{rwyDesig} . ":\t"
                            . _printVisRwy($e->{RVR}, '');
                } elsif ($_ eq 'rwyWind') {
                    $msg .= "wind for runway $e->{rwyDesig}:\t"
                            . _printWind_FGFS $e->{wind};
                } elsif ($_ eq 'thrWind') {
                    $msg .= "wind for runway $e->{rwyDesig}:\t"
                            . _printWind_FGFS $e->{wind};
                } elsif ($_ eq 'gridWind') {
                    $msg .= "Grid wind:\t\t" . _printWind_FGFS $e->{wind};
                } elsif ($_ eq 'cloudCoverVar') {
                    $msg .= "Sky condition:\t\t"
                            . _printCloudCover($e->{cloudCover})
                            . (exists $e->{cloudBase} ?
                               ' at ' . $e->{cloudBase} * 100 . " ft\t\t\t"
                               . rnd($e->{cloudBase} * 100 * $FT2M, 10) . ' m'
                               : '')
                            . "\nvarying to:\t\t\t"
                            . _printCloudCover($e->{cloudCover2}) . "\n"
                } elsif ($_ eq 'precipHourly') {
                    if (exists $e->{precipHoursNotAvailable}) {
                        $msg .= '3-/6-hour precip. amount';
                    } else {
                        $msg .= $e->{precipHours} . '-hour precip. amount';
                    }
                    $msg .= ":\t";
                    if (exists $e->{notAvailable}) {
                        $msg .= "(not available)\n";
                    } else {
                        $msg .= sprintf "%.0f mm\t\t\t\t\t%g in.\n",
                                $e->{precipAmountInch} * ($FT2M / 12 * 1000),
                                $e->{precipAmountInch};
                    }
                } elsif ($_ eq 'snowIncr') {
                    $msg .= "snow increasing rapidly:\t" . $e->{pastHour}
                            . " in. past hour, " . $e->{onGround}
                            . " in. on ground\n";
                } elsif ($_ eq 'keyword') {
                    if (exists $keywords{$e->{keyword}}) {
                        $msg .= $keywords{$e->{keyword}};
                        if ($e->{keyword} =~ 'EP[OCM]') {
                            $msg .= ' (Tahetna pass)'
                                if $metar->{obsStation}{id} eq 'PASP';
                        }
                        $msg .= "\n";
                    } else {
                        $msg .= $e->{s} . "\n";
                    }
                } elsif ($_ eq 'needMaint') {
                    $msg .= $keywords{'$'} . "\n";
                } elsif ($_ eq 'cloud') {
                    $msg .= _printCloud { cloud => [$e] };
                } elsif ($_ eq 'cloudTypeFamily') {
                    $msg .= "cloud types:\n\tlow:\t\t";
                    if (exists $e->{cloudTypeLowAboveOvercast}) {
                        $msg .= "layer above overcast\n";
                    } else {
                        $msg .= $cloud_types_low{$e->{cloudTypeLow}} . "\n";
                    }
                    $msg .= "\tmid-level:\t";
                    if (exists $e->{cloudTypeMiddleAboveOvercast}) {
                        $msg .= "layer above overcast\n";
                    } else {
                        $msg .= $cloud_types_middle{$e->{cloudTypeMiddle}}."\n";
                    }
                    $msg .= "\thigh:\t\t";
                    if (exists $e->{cloudTypeHighAboveOvercast}) {
                        $msg .= "layer above overcast\n";
                    } else {
                        $msg .= $cloud_types_high{$e->{cloudTypeHigh}} . "\n";
                    }
                } elsif ($_ eq 'windShift') {
                    $msg .= "wind shift at "
                            .   _printTime((exists $e->{hour} ? $e->{hour} : '')
                                          . $e->{minute});
                    $msg .= ' due to frontal passage' if exists $e->{FROPA};
                    $msg .= "\n";
                } elsif ($_ eq 'beginEndPrecip') {
                    my $txt_loc_mov;

                    if (   exists $e->{locationAndList}
                        || exists $e->{MOV}
                        || exists $e->{MOVD})
                    {
                        $txt_loc_mov = "\n\t\t\t";
                        if (exists $e->{locationAndList}) {
                            $txt_loc_mov .=
                                         _printLocations($e->{locationAndList});
                            $txt_loc_mov .= ','
                                if exists $e->{MOV} || exists $e->{MOVD};
                        }
                        $txt_loc_mov .= " $keywords{MOV}"
                                        . _printLocations($e->{MOV}, 1)
                            if exists $e->{MOV};
                        $txt_loc_mov .= " $keywords{MOVD}"
                                        . _printLocations($e->{MOVD}, 1)
                            if exists $e->{MOVD};
                    }

                    $hdr = "precipitation:\t";
                    for (@{$e->{precip}}) {
                        $msg .= $hdr . _printWeather($_->{weather}, 1) . ':';
                        for (@{$_->{start_end}}) {
                            $msg .= "\n\t\t\t\t";
                            if (exists $_->{startTime}) {
                                $msg .= 'began at '
                                  .   _printTime((exists $_->{startTime}{hour}
                                                 ? $_->{startTime}{hour} : '')
                                                . $_->{startTime}{minute});
                            } else {
                                $msg .= 'ended at '
                                  .   _printTime((exists $_->{endTime}{hour}
                                                 ? $_->{endTime}{hour} : '')
                                                . $_->{endTime}{minute});
                            }
                        }
                        $hdr = "\n\t\t\t";
                    }
                    $msg .= $txt_loc_mov if defined $txt_loc_mov;
                    $msg .= "\n";
                } elsif ($_ =~ '^phenomenon(?:AtLoc|Only)$') {
                    $msg .= _printPhenomDescr $e->{phenomDescrPre}, '', ' ';
                    $hdr = '';
                    for (exists $e->{cloudType} ? @{$e->{cloudType}} : ()) {
                        $msg .= $hdr . $cloud_types{$_};
                        $hdr = ' and ';
                    }
                    $msg .= _printWeather $e->{weather}, 1;
                    $msg .= _printCloudCover $e->{cloudCover}
                        if exists $e->{cloudCover};
                    if (exists $e->{lightningType}) {
                        $msg .= $keywords{LTG};
                        $delim = ' ';
                        for (@{$e->{lightningType}}) {
                            $msg .= $delim . $ltg_types{$_};
                            $delim = ' and ';
                        }
                    }
                    $msg .= $keywords{$e->{otherPhenom}}
                        if exists $e->{otherPhenom};
                    $msg .= _printPhenomDescr $e->{phenomDescrPost}, ' ', '';
                    if (exists $e->{locationAndList}) {
                        $msg .= ',' if exists $e->{phenomDescrPost};
                        $msg .= _printLocations($e->{locationAndList});
                        $msg .= ',' if exists $e->{MOV} || exists $e->{MOVD};
                    }
                    $msg .= " $keywords{MOV}" . _printLocations($e->{MOV}, 1)
                        if exists $e->{MOV};
                    $msg .= " $keywords{MOVD}" . _printLocations($e->{MOVD}, 1)
                        if exists $e->{MOVD};
                    $msg .= ', '. $cloud_types{$e->{cloudTypeAsoctd}}
                            . " $keywords{ASOCTD}"
                        if exists $e->{cloudTypeAsoctd};
                    $msg .= ', '. $cloud_types{$e->{cloudTypeEmbd}}
                            . " $keywords{EMBD}"
                        if exists $e->{cloudTypeEmbd};
                    $msg .= "\n";
                } elsif ($_ eq 'conditionMountain') {
                    my %specificatione_mon = (
                        LIB       => 'free of clouds',
                       'CLD_SCT'  => 'partially covered in isolated clouds',
                       'VERS_INC' => 'slopes in clouds, top is free',
                       'CNS_POST' => 'free from side of observer, other side in clouds',
                       'CLD_CIME' => 'clouds touch top',
                       'CIME_INC' => 'top in clouds, slopes free',
                       'GEN_INC'  => 'generally in clouds, some tops free',
                        INC       => 'in clouds',
                        INVIS     => 'not visible',
                    );
                    my %evoluzione_mon = (
                        NC          => 'no change',
                        CUF         => 'cumuli developing',
                       'ELEV_SLW'   => 'developing slowly',
                       'ELEV_RAPID' => 'developing rapidly',
                       'ELEV_STF'   => 'developing and stratification',
                       'ABB_SLW'    => 'sinking slowly',
                       'ABB_RAPID'  => 'sinking rapidly',
                        STF         => 'stratification',
                       'STF_ABB'    => 'stratification and sinking',
                       'VAR_RAPID'  => 'varying rapidly',
                    );

                    $msg .= "$keywords{MTNS}:\t\t";
                    $delim = '';
                    for (@{$e->{condMoun}}) {
                        $msg .= $delim;
                        $delim = "\n\t\t\t";
                        $msg .= _printLocations($_->{locationAndList}) . ': '
                            if defined $_->{locationAndList};
                        $msg .= $specificatione_mon{$_->{condMounType}};
                        $msg .= ', ' . $evoluzione_mon{$_->{condMounChange}}
                            if defined $_->{condMounChange};
                    }
                    $msg .= "\n";
                } elsif ($_ eq 'conditionValley') {
                    my %specificatione_val = (
                        NIL                 => 'no low clouds, fog or haze',
                        FOSCHIA             => 'haze',
                       'FOSCHIA_SKC_SUP'    => 'haze, sky clear above',
                        NEBBIA              => 'fog',
                       'NEBBIA_SCT'         => 'scattered fog banks',
                       'CLD_SCT'            => 'isolated clouds',
                       'CLD_SCT_NEBBIA_INF' => 'isolated clouds, fog below',
                       'MAR_CLD'            => 'sea in clouds',
                        INVIS               => 'not visible',
                    );
                    my %evoluzione_val = (
                        NC            => 'no change',
                        DIM           => 'decreasing',
                       'DIM_ELEV'     => 'decreasing and raising',
                       'DIM_ABB'      => 'decreasing and sinking',
                        AUM           => 'increasing',
                       'AUM_ELEV'     => 'increasing and raising',
                       'AUM_ABB'      => 'increasing and sinking',
                        ELEV          => 'raising',
                        ABB           => 'sinking',
                       'NEBBIA_INTER' => 'intermittent fog at station',
                    );

                    $msg .= "$keywords{VAL}:\t\t";
                    $delim = '';
                    for (@{$e->{condVall}}) {
                        $msg .= $delim;
                        $delim = "\n\t\t\t";
                        $msg .= _printLocations($_->{locationAndList}) . ': '
                            if defined $_->{locationAndList};
                        $msg .= $specificatione_val{$_->{condVallType}};
                        $msg .= ', ' . $evoluzione_val{$_->{condVallChange}}
                            if defined $_->{condVallChange};
                    }
                    $msg .= "\n";
                } elsif ($_ eq 'ceilingAtLoc') {
                    $msg .= $keywords{CIG};
                    $msg .= ' ' . $keywords{APCH} if exists $e->{isApproach};
                    $msg .= ' runway ' . $e->{rwyDesig}
                        if exists $e->{rwyDesig};
                    $msg .= _printLocations $e->{locationAndList}
                        if exists $e->{locationAndList};
                    $msg .= ":\t";
                    $msg .= 'at ' . $e->{cloudBase} * 100 . ' ft';
                    $msg .= "\t\t\t\t" . rnd($e->{cloudBase} * 100 * $FT2M, 10) . " m\n";
                } elsif ($_ eq 'phenomOpacityList') {
                    $msg .= _printPhenomOpacityList $e;
                } elsif ($_ eq 'cloudTypeLvl') {
                    $msg .= "Sky condition:\t\t"
                            . $cloud_types{$e->{cloudType}}
                            . ' at ' . $e->{cloudBase} * 100 . ' ft'
                            . "\t\t\t" . rnd($e->{cloudBase} * 100 * $FT2M, 10) . " m\n";
                } elsif ($_ eq 'cloudTrace') {
                    $msg .= "traces of:\t\t";
                    $msg .= "$keywords{LWR} " if exists $e->{isLower};
                    if (exists $e->{cloudTypeNotAvailable}) {
                        $msg .= '(cloud type not observable)';
                    } else {
                        $delim = '';
                        for (@{$e->{cloudType}}) {
                            $msg .= $delim . $cloud_types{$_};
                            $delim = ', ';
                        }
                    }
                    $msg .= "\n";
                } elsif ($_ eq 'seaCondition') {
                    $msg .= "condition of the sea:\t";
                    if (exists $e->{notAvailable}) {
                        $msg .= '(not available)';
                    } else {
                        $msg .= {
                        0 => 'calm (glassy)',
                        1 => 'calm (rippled)',
                        2 => 'smooth (wavelets)',
                        3 => 'slight waves',
                        4 => 'moderate waves',
                        5 => 'rough',
                        6 => 'very rough',
                        7 => 'high waves',
                        8 => 'very high waves',
                        9 => 'phenomenal waves',
                        }->{$e->{seaCondVal}};
                    }
                    $msg .= ', from the' . _printLocations $e->{locationAndList}
                        if exists $e->{locationAndList};
                    $msg .= "\n";
                } elsif ($_ eq 'swellCondition') {
                    $msg .= "condition of the swell:\t";
                    if (exists $e->{notAvailable}) {
                        $msg .= '(not available)';
                    } else {
                        $msg .= {
                        0 => 'none',
                        1 => 'length: short or average, height: low',
                        2 => 'length: long, height: low',
                        3 => 'length: short, height: moderate',
                        4 => 'length: average, height: moderate',
                        5 => 'length: ?, height: ?',
                        6 => 'length: ?, height: ?',
                        7 => 'length: ?, height: ?',
                        8 => 'length: ?, height: ?',
                        9 => 'length: ?, height: ?',
                        }->{$e->{swellCondVal}};
                    }
                    $msg .= ', from the' . _printLocations $e->{locationAndList}
                        if exists $e->{locationAndList};
                    $msg .= "\n";
                } elsif ($_ eq 'cloudAbove') {
                    if (exists $e->{isThin}) {
                        $msg .= "$keywords{THN} cloud layer:\t";
                    } else {
                        $msg .= "cloud layer:\t\t";
                    }
                    $msg .=   _printCloudCover($e->{cloudCover}) . ' '
                            . $keywords{ABV} . ' '
                            . $e->{cloudBase} * 100 . ' ft' . "\t\t\t"
                            . rnd($e->{cloudBase} * 100 * $FT2M, 10) . " m\n";
                } elsif ($_ eq 'obscuration') {
                    $msg .= 'due to ';
                    if (exists $e->{weather}) {
                        $msg .= _printWeather $e->{weather};
                    } else {
                        $msg .=
                            { PWR_PLNT       => 'power plant',
                              PWR_PLNT_PLUME => 'power plant plume'
                            }->{$e->{cloudPhenom}};
                    }
                    $msg .= ":\t" . _printCloudCover($e->{cloud}{cloudCover})
                            . ' at ' . $e->{cloud}{cloudBase} * 100 . ' ft'
                            . "\t\t\t"
                            . rnd($e->{cloud}{cloudBase} * 100 * $FT2M, 10)
                            . " m\n";
                } elsif ($_ eq 'variableCeiling') {
                    $msg .= "ceiling variable:\tfrom " . $e->{cloudBaseFrom} * 100 . " ft\t\t\t\t"
                            . rnd($e->{cloudBaseFrom} * 100 * $FT2M, 10) . " m\n"
                            . "\t\t\tto " . $e->{cloudBaseTo} * 100 . " ft\t\t\t\t"
                            . rnd($e->{cloudBaseTo} * 100 * $FT2M, 10) . " m\n";
                } elsif ($_ eq 'pressureTendency3h') {
                    $msg .= "3-h pressure tend.:\t";
                    if (exists $e->{notAvailable}) {
                        $msg .= "(not available)\n";
                    } else {
                        $msg .=
                         { 0 => 'increasing, then decreasing',
                           1 => 'increasing, then steady, or increasing then increasing more slowly',
                           2 => 'increasing steadily or unsteadily',
                           3 => 'decreasing or steady, then increasing; or increasing then increasing more rapidly',
                           4 => 'steady',
                           5 => 'decreasing then increasing',
                           6 => 'decreasing, then steady, or decreasing then decreasing more slowly',
                           7 => 'decreasing steadily or unsteadily',
                           8 => 'steady or increasing, then decreasing; or decreasing then decreasing more rapidly',
                         }->{$e->{pressureTendency}};
                        $msg .= ' by ' . $e->{pressureChange} . ' hPa'
                            if $e->{pressureChange} != 0;
                        $msg .= "\n";
                    }
                } elsif ($_ eq 'ceilVisVariable') {
                    $msg .= 'ceiling at or below 900 m and variable, '
                            . "visibility varies from:\t"
                            . _printVis_FGFS('(not available)',
                                             $e->{visibilityFrom}, 1000) . "\n"
                            . "\t\tto:\t"
                            . _printVis_FGFS('(not available)',
                                             $e->{visibilityTo}, 1000) . "\n";
                } elsif ($_ eq 'rwySfcCondition') {
                    $delim = "rwy surface condition:\t";
                    for (@{$e->{arr}}) {
                        my $key = (keys %$_)[0];
                        $msg .= $delim;
                        if ($key eq 'decelerometer') {
                            $msg .= "decelerometer reading $_->{$key}";
                        } else {
                            $msg .= $key eq 'notAvailable'
                                          ? 'no decelerometer reading available'
                                          : $keywords{$_->{$key}};
                        }
                        $delim = ', ';
                    }
                    $msg .= "\n";
                } elsif ($_ eq 'rainfall') {
                    $msg .= "rainfall:\t\t"
                            . $e->{rainfall10min}
                            . " mm in the last 10 minutes, "
                            . $e->{rainfall0900}
                            . " mm since 09:00 airport time\n";
                } elsif ($_ eq 'precipPastHour') {
                    $msg .= "1-hour precip. amount:\t"
                            . sprintf "%g mm\t\t\t\t\t%.2f in.\n",
                                $e->{precipAmountMM},
                                $e->{precipAmountMM} / ($FT2M / 12 * 1000);
                } elsif ($_ =~ '^(?:densityAlt|pressureAlt)$') {
                    $msg .= $keywords{ { densityAlt => 'DA',
                                         pressureAlt => 'PA'
                                        }->{$_}}
                            . sprintf ":\t%g ft\t\t\t\t\t%.0f m\n",
                                $e->{altitude}, $e->{altitude} * $FT2M;
                } elsif ($_ =~ '^(?:VISNO|CHINO)$') {
                    $msg .= "@keywords{$_, 'RWY'} $e->{rwyDesig} not operating\n";
                } elsif ($_ eq 'obsTimeOffset') {
                    $msg .= "offset for observation time:\t"
                            . $e->{minutes} . " minutes\n";
                } elsif ($_ eq 'nextFcstBy') {
                    $msg .= "next forecast by:\t$e->{hour}:00 UTC\n";
                } elsif ($_ eq 'nextFcstAt') {
                    $msg .= "next forecast:\t\ton the $e->{day}., "
                            . "$e->{hour}:$e->{minute} UTC\n";
                } elsif ($_ eq 'fcstAutoObs') {
                    $msg .= 'forecast based on automated observation';
                    $msg .= " $e->{hourFrom}:00 - $e->{hourTill}:00 UTC"
                        if exists $e->{hourFrom};
                    $msg .= "\n";
                } elsif ($_ eq 'fcstAutoMETAR') {
                    $msg .= "forecast based on automated METAR\n";
                } elsif ($_ eq 'waterEquivOfSnow') {
                    $msg .= "water equivalent of snow on ground:\t"
                            . sprintf "%.0f mm\t\t\t\t\t%g in.\n",
                                $e->{precipAmountInch} * ($FT2M / 12 * 1000),
                                $e->{precipAmountInch};
                } elsif ($_ eq 'snowOnGround') {
                    $msg .= "snow on ground:\t\t"
                            . sprintf "%.0f mm\t\t\t\t\t%g in.\n",
                                $e->{precipAmountInch} * ($FT2M / 12 * 1000),
                                $e->{precipAmountInch};
                } elsif ($_ eq 'snowCover') {
                    $msg .= "snow cover:\t\t"
                            . { ONE_LOOSE     => 'one loose',
                                MUCH_LOOSE    => 'much loose',
                                TRACE_LOOSE   => 'trace loose',
                                MEDIUM_PACKED => 'medium packed',
                                HARD_PACKED   => 'hard packed',
                                NIL           => 'nil'
                            }->{$e->{snowCoverType}}
                            . "\n";
                } elsif ($_ eq 'climate') {
                    $msg .= _printUSTemp $e->{temp1}, "temperature 1:\t\t";
                    $msg .= _printUSTemp $e->{temp2}, "temperature 2:\t\t";
                    $msg .= "precipitation 1:\ttraces\n"
                        if exists $e->{precip1Traces};
                    $msg .=   "precipitation 1:\t"
                            . sprintf("%.0f mm\t\t\t\t\t%g in.\n",
                                $e->{precipAmount1Inch} * ($FT2M / 12 * 1000),
                                $e->{precipAmount1Inch})
                        if exists $e->{precipAmount1Inch};
                    $msg .= "precipitation 1:\t"
                            . sprintf("%g mm\t\t\t\t\t%.2f in.\n",
                                $e->{precipAmount1MM},
                                $e->{precipAmount1MM} / ($FT2M / 12 * 1000))
                        if exists $e->{precipAmount1MM};
                    $msg .=   "precipitation 2:\t"
                            . sprintf("%.0f mm\t\t\t\t\t%g in.\n",
                                $e->{precipAmount2Inch} * ($FT2M / 12 * 1000),
                                $e->{precipAmount2Inch})
                        if exists $e->{precipAmount2Inch};
                    $msg .= "precipitation 2:\t"
                            . sprintf("%g mm\t\t\t\t\t%.2f in.\n",
                                $e->{precipAmount2MM},
                                $e->{precipAmount2MM} / ($FT2M / 12 * 1000))
                        if exists $e->{precipAmount2MM};
                } elsif ($_ eq 'tornadicActivity') {
                    $msg .= { tornado      => 'tornado',
                              funnel_cloud => 'funnel cloud',
                              waterspout   => 'waterspout'
                            }->{$e->{tornadicActivityType}};
                    if (exists $e->{locationAndList}) {
                        $msg .= _printLocations($e->{locationAndList});
                        $msg .= ',' if exists $e->{MOV} || exists $e->{MOVD};
                    }
                    $msg .= " $keywords{MOV}" . _printLocations($e->{MOV}, 1)
                        if exists $e->{MOV};
                    $msg .= " $keywords{MOVD}" . _printLocations($e->{MOVD}, 1)
                        if exists $e->{MOVD};
                    $msg .= ":";
                    for (@{$e->{start_end}}) {
                        $msg .= "\n\t\t\t\t";
                        if (exists $_->{startTime}) {
                            $msg .= 'began at '
                              .   _printTime((exists $_->{startTime}{hour}
                                             ? $_->{startTime}{hour} : '')
                                            . $_->{startTime}{minute});
                        } else {
                            $msg .= 'ended at '
                              .   _printTime((exists $_->{endTime}{hour}
                                             ? $_->{endTime}{hour} : '')
                                            . $_->{endTime}{minute});
                        }
                    }
                    $msg .= "\n";
                } elsif ($_ eq 'balloon') {
                    if (exists $e->{disappearedAt}) {
                        $msg .=   'balloon disappeared at '
                                . $e->{disappearedAt}{distance} . " ft\n";
                    } else {
                        $msg .=   'balloon visible to '
                                . $e->{visibleTo}{distance} . " ft\n";
                    }
                } elsif ($_ =~ '^(first|next|last)Obs$') {
                    $msg .= $1;
                    $msg .= ' manned' if exists $e->{isManned};
                    $msg .= ' staffed' if exists $e->{isStaffed};
                    $msg .= ' observation';
                    $msg .= sprintf ' on the %d., %02d:%02d UTC',
                                    @{$e->{obsAt}}{'day', 'hour', 'minute'}
                        if exists $e->{obsAt};
                    $msg .= "\n";
                } elsif ($_ eq 'estimated') {
                    $delim = '';
                    for (@{$e->{estimatedItem}}) {
                        $msg .= $delim . $keywords{$_};
                        $delim = ', ';
                    }
                    $msg .= ' ' . $keywords{ESTMD};
                    $msg .= ' due to ice accretion'
                        if exists $e->{dueToIceAccretion};
                    $msg .= "\n";
                } elsif ($_ eq 'RSNK') {
                    $msg .= "data for Rattlesnake Mountain:\n";
                    $msg .= _printTemp_FGFS $e;
                    $msg .= "Wind:\t\t\t" . _printWind_FGFS $e->{wind};
                } elsif ($_ eq 'LAG_PK') {
                    $msg .= "data for Laguna Peak:\n";
                    $msg .= _printTemp_FGFS $e;
                    $msg .= "Wind:\t\t\t" . _printWind_FGFS $e->{wind};
                } elsif ($_ eq 'RADAT') {
                    $msg .= "radiosonde observation data for freezing level:\n";
                    if (exists $e->{missing}) {
                        $msg .= "missing\n";
                    } else {
                        $msg .= sprintf "relative humidity:\t%.0f%%\n",
                                        $e->{relHumid};
                        $msg .= "freezing level:\t\t"
                                 . 'at ' . $e->{distance} * 100 . ' ft'
                                 . "\t\t\t\t"
                                 . rnd($e->{distance} * 100 * $FT2M, 10)
                                 . " m\n";
                    }
                } elsif ($_ eq 'tempMaxFQ') {
                    $msg .= _printUSTemp $e, "max. temperature:\t";
                } elsif ($_ eq 'durationOfSunshine') {
                    $msg .= "sunshine yesterday:\t";
                    if (exists $e->{durationMinutes}) {
                        if ($e->{durationMinutes} > 59) {
                            my $min = $e->{durationMinutes} % 60;
                            $msg .= (($e->{durationMinutes} - $min) / 60)
                                    . ' hour(s)';
                            $msg .= " $min min." if $min > 0;
                            $msg .= "\n";
                        } else {
                            $msg .= $e->{durationMinutes} . " min.\n";
                        }
                    } else {
                        $msg .= "(not available)\n";
                    }
                } elsif ($_ eq 'hailStones') {
                    $msg .=   "hail stones:\t\tbiggest diameter "
                            . (exists $e->{isLess} ? '<' : '')
                            . $e->{hailStoneSize} . " in.\n";
                }
            }
        }
    }

    $msg .= "ERROR: " . { obsStation => 'not a station code',
                          obsTime    => 'not an observation time',
                          other      => 'invalid format',
                        }->{$metar->{ERROR}{descr}}
                    . ': ' . $metar->{ERROR}{pos} . "\n"
        if exists $metar->{ERROR};

    return $msg;
}

1;
__END__
